package Swatchdog::Actions::Perl;

use strict;
use warnings qw(all);
use utf8;

use threads;
use threads::shared;

use Carp qw();

require 'Swatchdog/Actions/exec_command.pl';

use vars qw(
	$VERSION
	$Asynchronous
	$AsyncPropagation
	$UseThread
	$MaxForkRetryCount
	$ForkRetryInterval
);

BEGIN {
	$VERSION = '0.1';
}

$Asynchronous = 1;
$AsyncPropagation = 1;
$UseThread = 1;
$MaxForkRetryCount = 3;
$ForkRetryInterval = 5;

sub new($@) {
	my	$class = shift;
	my	$self = {};
	my	%attribute = @_;

	bless $self, $class;

	$self->{attribute} = {
		asynchronous		=> $Asynchronous,
		asyncPropagation	=> $AsyncPropagation,
		useThread			=> $UseThread,
		maxForkRetryCount	=> $MaxForkRetryCount,
		forkRetryInterval	=> $ForkRetryInterval,
		%attribute
	};
	$self->{subActionClasses} = {};
	$self->{property} = {};

	return $self;
}

sub attribute($$;$) {
	my	$self = shift;
	my	$name = shift;
	my	$oldValue = $self->{attribute}->{$name};

	if (0 < scalar @_) {
		$self->{attribute}->{$name} = $_[0];
	}

	return $oldValue;
}
sub asynchronous($;$) {
	return $_[0]->attribute('asynchronous', @_[1 .. $#_]);
}
sub asyncPropagation($;$) {
	return $_[0]->attribute('asyncPropagation', @_[1 .. $#_]);
}
sub useThread($;$) {
	return $_[0]->attribute('useThread', @_[1 .. $#_]);
}
sub maxForkRetryCount($;$) {
	return $_[0]->attribute('maxForkRetryCount', @_[1 .. $#_]);
}
sub forkRetryInterval($;$) {
	return $_[0]->attribute('forkRetryInterval', @_[1 .. $#_]);
}

sub log($$$@) {
	my	$self = shift;
	my	$level = shift;
	my	($format, @args) = @_;

	# You must implement.
}
sub logTrace($$@) { $_[0]->log('trace', @_[1 .. $#_]) }
sub logDebug($$@) { $_[0]->log('debug', @_[1 .. $#_]) }
sub logInfo($$@)  { $_[0]->log('info',  @_[1 .. $#_]) }
sub logWarn($$@)  { $_[0]->log('warn',  @_[1 .. $#_]) }
sub logError($$@) { $_[0]->log('error', @_[1 .. $#_]) }
sub logFatal($$@) { $_[0]->log('fatal', @_[1 .. $#_]) }

sub registerSubActionClass($$) {
	my	$self = shift;
	my	$subActionClass = $_[0];

	$self->{subActionClasses}->{$subActionClass} = undef;
}

sub unregisterSubActionClass($$) {
	my	$self = shift;
	my	$subActionClass = $_[0];

	delete $self->{subActionClasses}->{$subActionClass};
}

sub logEventOccurred($) {
	my	$self = shift;

	$self->logInfo('Something has occurred.');
}

sub logSkipOverDealing($) {
	my	$self = shift;

	$self->logInfo('It is not necessary to deal.');
}

sub onEventOccurred($@) {
	my	$self = shift;
	my	%variable = @_;
	$self->logTrace('BEGIN %s', (caller 0)[3]);

	$self->setProperties(%variable);
	$self->logEventOccurred();

	if ($self->asynchronous()) {
		if ($self->useThread) {
			my	$thr =  threads->create(\&doit, $self);

			if (defined $thr) {
				$self->logDebug('create a thread #%d', $thr->tid());
				$thr->detach();
			}
			else {
				$self->logWarn("Can't create a thread: %s", $!);
			}
		}
		else {
			my	$retryCount = 0;

			CREATE_CHILD: {
				my	$pid = fork;

				if (defined $pid) {
					if ($pid) {
						$self->logDebug('create sub process #%d', $pid);
					}
					else {
						$self->doit();
						exit 0;
					}
				}
				elsif ($retryCount < $self->maxForkRetryCount) {
					$retryCount++;
					$self->logWarn("Can't fork: %s", $!);
					$self->logWarn("Retry to fork - %dth", $retryCount);
					sleep $self->forkRetryInterval;
					redo CREATE_CHILD;
				}
				else {
					$self->logError("Can't fork: %s", $!);
					$self->logError('Give up retring to fork.');
				}
			}
		}
	}
	else {
		$self->doit();
	}

	foreach my $subAction ($self->_subActions()) {
		if ($self->asyncPropagation()) {
			$subAction->asynchronous($self->asynchronous());
		}
		$subAction->onEventOccurred(%variable);
	}

	$self->logTrace('END   %s', (caller 0)[3]);
}

sub setProperties($@) {
	my	$self = shift;
	my	%prop = @_;

	$self->{property} = {};
	while (my ($key, $value) = each %prop) {
		$self->{property}->{$key} = $value;
	}
}

sub _subActions($) {
	my	$self = shift;
	my	@subActions;
	$self->logTrace('BEGIN %s', (caller 0)[3]);

	@subActions = map {
		my	$subActionClass = $_;

		$self->logTrace('create sub event: %s', $subActionClass);
		eval "$subActionClass->new"
	} keys %{$self->{subActionClasses}};

	$self->logTrace('END   %s', (caller 0)[3]);
	return @subActions;
}

sub doit($) {
	my	$self = shift;
	$self->logTrace('BEGIN %s', (caller 0)[3]);

	if ($self->mustEventBeDealtWith()) {
		$self->dealWithEvent();
	}
	else {
		$self->logSkipOverDealing();
	}

	$self->logTrace('END   %s', (caller 0)[3]);
}

sub mustEventBeDealtWith($) {
	my	$self = shift;
	$self->logTrace('BEGIN %s', (caller 0)[3]);

	$self->logWarn('You must implement in a sub class.');

	$self->logTrace('END   %s', (caller 0)[3]);
	return 0;
}

sub dealWithEvent($) {
	my	$self = shift;
	$self->logDebug('BEGIN %s', (caller 0)[3]);

	$self->logWarn('You must implement in a sub class.');

	$self->logDebug('END   %s', (caller 0)[3]);
}

1;

=head1 NAME

Swatchdog::Actions::Perl - perl object for Swatchdog action

=head1 SYNOPSIS

In your perl module file:

    package Swatchdog::Actions::Perl::Login;
    use threads;
    use threads::shared;
    use Swatchdog::Actions::Perl;
    
    use vars qw(
        @ISA
        %TrustedUser
    );
    
    @ISA = qw(Swatchdog::Actions::Perl);
    %TrustedUser = ();
    share(%TrustedUser);
    
    sub new($@) {
        my $class = shift;
        my $self = Swatchdog::Actions::Perl->new(@_);
    
        bless $self, $class;
    
        return $self;
    }
    
    sub logEventOccurred($) {
        my $self    = shift;
    
        $self->logInfo('The use %s has logged in a %s manner from %s.',
            $self->{property}->{user},
            $self->{property}->{secure} ? 'secure' : 'non-secure',
            $self->{property}->{from}
        );
    }
    
    sub mustEventBeDealtWith($) {
        my $self = shift;
        my $value;
        $self->logTrace('BEGIN %s', (caller 0)[3]);
    
        $value =
            exists $TrustedUser{$self->{property}->{user}}
                &&
            $self->{property}->{secure}
            ;
        $self->logInfo('The user %s from %s %s a trusted user.',
            $self->{property}->{user},
            $self->{property}->{from},
            $self->{property}->{secure} ? 'is' : "isn't"
        );
    
        $self->logTrace('END   %s', (caller 0)[3]);
        return $value;
    }
    
    sub dealWithEvent($) {
        my $self = shift;
        $self->logDebug('BEGIN %s', (caller 0)[3]);
    
        AddTrustedClientHost($self->{property}->{from});
    
        $self->logDebug('END   %s', (caller 0)[3]);
    }
    
    sub AddTrustedClientHost($) {
    	# Do something.
    }

In the configuration file of Swatchdog:

    perlcode 0 use Swatchdog::Actions::Perl::Login;
    
    perlcode 0 $Swatchdog::Actions::Perl::Login::TrustedUser{john} = 1;
    perlcode 0 $Swatchdog::Actions::Perl::Login::TrustedUser{mark} = 1;
    perlcode 0 $Swatchdog::Actions::Perl::Login::TrustedUser{becky} = 1;
    perlcode 0 my $Login = Swatchdog::Actions::Perl::Login->new();
    watchfor /^\w{3}\s+\d{1-2} \d{2}:\d{2}:\d{2} \S+ sshd\[\d+\]: Accepted publickey for (\w+) from: (\S+) port /
        perlcode 2 $Login->onEventOccurred(
        perlcode 2    user => $1, from => $2,
        perlcode 2    secure => 1
        perlcode 2 );

=head1 DESCRIPTION

This module is a base class, which provides Swatchdog actions.
In the configuration file of Swatchdog, to run a complex process in a exec action, you need to create a command.
By using this module, you will be able to define the action in the perl code instead of the command.

=head1 METHODS

=head2 Methods to be used in the configuration file of Swatchdog

=over 4

=item Swatchdog::Actions::Perl->new(%attributes)

This class method returns a new Swatchdog::Actions::Perl object.
You can describe attributes of this object by the hash C<%attributes>.

See L<Set or Get object attributes> for attributes.

=item $action->onEventOccurred(%properties)

Run the process when a event occurres.
The event are described by C<%properties>.

See $action->setProperties(%properties)

=back

=head2 Methods that must be overridden

=over 4

=item $action->mustEventBeDealtWith()

This method returns whether the event must be deal with.

You must override this method in your sub class, because Swatchdog::Actions::Perl::mustEventBeDealtWith() allways return 0.

=item $action->dealWithEvent()

This method deals with the event.
It is called, only if the method mustEventBeDealtWith() returns true value.

You must override this method in you sub class, because Swatchdog::Actions::Perl::dealWithEvent do nothing.

=back

=head2 Methods that is often overridden

=over 4

=item $action->setProperties(%properties)

Set the event properties.
This method is called by the method onEventOccurred().

You can refer the event properties by the following manner:

    $self->{property}->{PROPERTY_NAME}

If you wants complex properties, override this method in your sub class.

Example:

    use Time::Local;
    
    sub setProperties($@) {
        my $self = shift;
    
        $self->SUPER::setProperties(@_);
        $self->{property}->{detectedAt} = time
            unless (defined $self->{property}->{detectedAt});
        $self->{property}->{occurredAt} = timelocal(
            $self->{property}->{sec},
            $self->{property}->{min},
            $self->{property}->{hour},
            $self->{property}->{mday},
            $self->{property}->{mon} - 1,
            $self->{property}->{year},
        )   unless (defined $self->{property}->{occurredAt});
    }

=item $action->log($level, $format, @args)

This method does nothing. But it used to output log messages. When you need logs, override this method in your sub class.

C<$level> is a log level. It is one of C<trace>, C<debug>, C<info>, C<warn>, C<error>, C<fatal>.
The log message is a string which is generated by the code C<sprintf($format, @args)>.

=item $action->logEventOccurred()

This method is called by the method onEventOccurred().
Its purpose is to log that an event has occurred.

When you override the method log(), you should also override this method.

=item $action->logSkipOverDealing()

This method is called when the method mustEventBeDealtWith() returns false value.
Its purpose is to log that dealing with the event is skipped.

When you override the method log(), you should also override this method.

=back

=head2 Set or Get object attributes

Object attributes are an inherent part of the object.

=over 4

=item $action->attribute($name)

=item $action->attribute($name, $value)

This method returns a value at the time of the call of the attributes of the named C<$name>.
If C<$value> is given, this method sets the value of the attribute named C<$name> to C<$value>.

Other methods of this section are shortcuts for this method.
When you wants new attributes for the object of your sub class, you can use this method to define new attribute methods. 

Example:

    sub logger($$) { $_[0]->attribute('logger', @_[1 .. $#_]) }

=item $action->asynchronous()

=item $action->asynchronous($BOOL)

When the attribute S<asynchronous> is set true value, the method onEventOccurred() creates a new parallel task, in which such as method mustEventBeDealtWith() and dealWithEvent() is executed.

Even if it takes time for the execution of these methods, the log monitoring of Swatchdog isn't interfered, when it is set true value.

default: C<$Swatchdog::Actions::Perl::Asynchronou>

=item $action->asyncPropagation()

=item $action->asyncPropagation($BOOL)

When the attribute S<asyncPropagation> is set true value, the attribute S<asynchronous> of sub-action is also set to same value as the attribute S<asynchronous> of this object.

default: C<$Swatchdog::Actions::Perl::AsyncPropagation>

=item $action->useThread()

=item $action->useThread($BOOL)

When the attribute S<useThread> is set true value, a thread is used for the task which is run in parallel with the main processing.
Otherwise, a sub-process is used instead of a thread.

default: C<$Swatchdog::Actions::Perl::UseThread>

=item $action->maxForkRetryCount()

=item $action->maxForkRetryCount($INTEGER)

The attribute S<maxForkRetryCount> is the number of times to retry when the fork for the concurrent task fails.

default: C<$Swatchdog::Actions::Perl::MaxForkRetryCount>

=item $action->forkRetryInterval()

=item $action->forkRetryInterval($seconds)

The attribute S<forkRetryInterval> is delay time to retry when the fork for the concurrent task fails.
Its unit is second.

default: C<$Swatchdog::Actions::Perl::ForkRetryInterval>

=back

=head2 Logging Methods

The following methods use the method log().
Implement the method log(), when you want to log actualy.

=over 4

=item $action->logTrace($format, @args)

Log the string which is generated with C<sprintf($format, @args)> as trace level message.

=item $action->logDebug($format, @args)

Log the string which is generated with C<sprintf($format, @args)> as debug level message.

=item $action->logInfo($format, @args)

Log the string which is generated with C<sprintf($format, @args)> as info level message.

=item $action->logWarn($format, @args)

Log the string which is generated with C<sprintf($format, @args)> as warn level message.

=item $action->logError($format, @args)

Log the string which is generated with C<sprintf($format, @args)> as error level message.

=item $action->logFatal($format, @args)

Log the string which is generated with C<sprintf($format, @args)> as fatal level message.

=back

=head2 Methods for sub-action

You may want to do more than one action against one event.
One way is to discribe more than one action in the configuration file of Swatchdog.
Another way is to add multipe actions to one action as sub-action.

You can define a bundle action of multiple actions.

=over 4

=item $action->registerSubActionClass($className)

Register a sub-action class which is specified with class name.
The class must be a derived class of Swatchdog::Actions::Perl.

=item $action->unregisterSubActionClass($className)

Unregist a sub-action class which is specified with class name.

=back

=head1 VARIABLES

=over 4

=item $Swatchdog::Actions::Perl::Asynchronous

This is default value of the object attribute S<asynchronous>.

=item $Swatchdog::Actions::Perl::AsyncPropagation

This is default value of the object attribute S<asyncPropagation>.

=item $Swatchdog::Actions::Perl::UseThread

This is default value of the object attribute S<useThread>.

=item $Swatchdog::Actions::Perl::MaxForkRetryCount

This is default value of the object attribute S<maxForkRetryCount>.

=item $Swatchdog::Actions::Perl::ForkRetryInterval

This is default value of the object attribute S<forkRetryInterval>.

=back

=head1 VERSION

    0.01

=head1 AUTHOR

Yuji Okamura <okamura[at]users.osdn.me>

=head1 COPYRIGHT

Copyright (C) 2015 Yuji OKamura. All Rights Reserved.

This module is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

