/* cdedit2 -- class diagram creation/manipulation program
 * Copyright (C) 2001 Touge Kamisimo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "math.h"

#include "cd_define.h"
#include "cd_node.h"
#include "cd_canvas.h"

#include "cd_util.h"

extern GdkFont *cd_font;
extern GdkGC *cd_black_gc;
extern GdkGC *cd_white_gc;
extern GdkGC *cd_select_gc;

Node cd_node;
NodeItem cd_node_item_last_deleted;

/*
 * cd_node_item_init
 */
void cd_node_item_init(int i)
{
  cd_node.item[i].id = 0;

  sprintf(cd_node.item[i].name, "");

  cd_node.item[i].name_point.x = 0;
  cd_node.item[i].name_point.y = 0;

  cd_node.item[i].rect.x = 0;
  cd_node.item[i].rect.y = 0;
  cd_node.item[i].rect.width = 0;
  cd_node.item[i].rect.height = 0;

  if(cd_node.item[i].attributes){
    g_list_free(cd_node.item[i].attributes);
    cd_node.item[i].attributes = NULL;
  }
  cd_node.item[i].attr_y = 0;
      
  if(cd_node.item[i].operations){
    g_list_free(cd_node.item[i].operations);
    cd_node.item[i].operations = NULL;
  }
  cd_node.item[i].op_y = 0;

  cd_node.item[i].select = FALSE;
}

/*
 * cd_node_item_new
 */
void cd_node_item_new(int i, int x, int y)
{
  cd_node_item_init(i);

  cd_node.item[i].id = cd_node.seq;

  sprintf(cd_node.item[i].name, CD_NODE_ITEM_NAME_DEFAULT);

  cd_node.item[i].rect.x = x;
  cd_node.item[i].rect.y = y;
  cd_node.item[i].rect.width = CD_NODE_ITEM_WIDTH_DEFAULT;
  cd_node.item[i].rect.height = CD_NODE_ITEM_HEIGHT_DEFAULT;

  if(cd_node_check_intersect(&cd_node.item[i].rect)){
    g_print(CD_NODE_MESSAGE1);
    cd_node_item_init(i);    
    return;
  }

  cd_node_item_calc_name_point(i);
  cd_node_item_calc_attr_y(i);
  cd_node_item_calc_op_y(i);

  cd_node.seq++;
  cd_node.count++;
}

/*
 * cd_node_item_copy
 */
void cd_node_item_copy(int dst, int src)
{
  cd_node.item[dst].id          = cd_node.item[src].id;

  cd_node.item[dst].rect.x       = cd_node.item[src].rect.x;
  cd_node.item[dst].rect.y       = cd_node.item[src].rect.y;
  cd_node.item[dst].rect.width   = cd_node.item[src].rect.width;
  cd_node.item[dst].rect.height  = cd_node.item[src].rect.height;

  sprintf(cd_node.item[dst].name, "%s", cd_node.item[src].name);

  cd_node.item[dst].name_point.x = cd_node.item[src].name_point.x;
  cd_node.item[dst].name_point.y = cd_node.item[src].name_point.y;

  if(cd_node.item[dst].attributes){
    g_list_free(cd_node.item[dst].attributes);
    cd_node.item[dst].attributes = NULL;
  }
  cd_node.item[dst].attributes   = g_list_copy(cd_node.item[src].attributes);
  cd_node.item[dst].attr_y       = cd_node.item[src].attr_y;
      
  if(cd_node.item[dst].operations){
    g_list_free(cd_node.item[dst].operations);
    cd_node.item[dst].operations = NULL;
  }
  cd_node.item[dst].operations   = g_list_copy(cd_node.item[src].operations);
  cd_node.item[dst].op_y       = cd_node.item[src].op_y;

  cd_node.item[dst].select       = cd_node.item[src].select;
}  

/*
 * cd_node_item_delete
 */
gboolean cd_node_item_delete()
{
  int i,j;

  if(cd_node.count <= 0)
    return FALSE;

  j = cd_node.count-1;

  if(cd_node.item[j].select){
    cd_node_item_escape(j);
    cd_node_item_init(j);
    cd_node.count--;
    cd_node.selecting = CD_SELECTING_CLEAR;
    cd_node.selected = CD_SELECTING_CLEAR;
    return TRUE;
  }

  for(i=cd_node.count-2; i>=0; i--){
    if(cd_node.item[i].select){
      cd_node_item_escape(i);
      cd_node_item_copy(i,j);
      cd_node_item_init(j);
      cd_node.count--;
      cd_node.selecting = CD_SELECTING_CLEAR;
      cd_node.selected = CD_SELECTING_CLEAR;
      return TRUE;
    }
  }

  return FALSE;
}

/*
 * cd_node_item_escape
 */
void cd_node_item_escape(int i)
{
  cd_node_item_last_deleted.id = cd_node.item[i].id;

  cd_node_item_last_deleted.rect.x       = cd_node.item[i].rect.x;
  cd_node_item_last_deleted.rect.y       = cd_node.item[i].rect.y;
  cd_node_item_last_deleted.rect.width   = cd_node.item[i].rect.width;
  cd_node_item_last_deleted.rect.height  = cd_node.item[i].rect.height;

  sprintf(cd_node_item_last_deleted.name, "%s", cd_node.item[i].name);
  cd_node_item_last_deleted.name_point.x = cd_node.item[i].name_point.x;
  cd_node_item_last_deleted.name_point.y = cd_node.item[i].name_point.y;

  if(cd_node_item_last_deleted.attributes){
    g_list_free(cd_node_item_last_deleted.attributes);
    cd_node_item_last_deleted.attributes = NULL;
  }

  cd_node_item_last_deleted.attributes =
    g_list_copy(cd_node.item[i].attributes);
    
  cd_node_item_last_deleted.attr_y       = cd_node.item[i].attr_y;

  if(cd_node_item_last_deleted.operations){
    g_list_free(cd_node_item_last_deleted.operations);
    cd_node_item_last_deleted.operations = NULL;
  }
  cd_node_item_last_deleted.operations
    = g_list_copy(cd_node.item[i].operations);

  cd_node_item_last_deleted.op_y         = cd_node.item[i].op_y;

  cd_node_item_last_deleted.select       = cd_node.item[i].select;
}

/*
 * cd_node_item_calc_rect
 */
void cd_node_item_calc_rect(int i)
{
  Attribute *attr;
  Operation *op;
  int j, width;

  cd_node.item[i].rect.width = CD_NODE_ITEM_WIDTH_DEFAULT;
  cd_node.item[i].rect.height = CD_NODE_ITEM_HEIGHT_DEFAULT;

  cd_node.item[i].rect.height +=
    g_list_length(cd_node.item[i].attributes) * 15 +
    g_list_length(cd_node.item[i].operations) * 15;

  width = gdk_string_width(cd_font, cd_node.item[i].name) + 20;
  if(cd_node.item[i].rect.width < width)
    cd_node.item[i].rect.width = width;
  
  for(j=0; j<g_list_length(cd_node.item[i].attributes); j++){
    attr = g_list_nth_data(cd_node.item[i].attributes, j);
    width = gdk_string_width(cd_font, attr->name) + 20;
    if(cd_node.item[i].rect.width < width)
      cd_node.item[i].rect.width = width;
  }

  for(j=0; j<g_list_length(cd_node.item[i].operations); j++){
    op = g_list_nth_data(cd_node.item[i].operations, j);
    width = gdk_string_width(cd_font, op->name) + 20;
    if(cd_node.item[i].rect.width < width)
      cd_node.item[i].rect.width = width;
  }

  if(cd_node.item[i].rect.width < CD_NODE_ITEM_WIDTH_DEFAULT)
    cd_node.item[i].rect.width = CD_NODE_ITEM_WIDTH_DEFAULT;
}

/*
 * cd_node_item_calc_name_point
 */
void cd_node_item_calc_name_point(int i)
{
  int margin; 

  margin = (cd_node.item[i].rect.width -
	    gdk_string_width(cd_font, cd_node.item[i].name)) /2;
  
  cd_node.item[i].name_point.x = cd_node.item[i].rect.x + margin;
  cd_node.item[i].name_point.y =
    cd_node.item[i].rect.y +
    gdk_string_height(cd_font, cd_node.item[i].name) + 3;
}

/*
 * cd_node_item_calc_attr_y
 */
void cd_node_item_calc_attr_y(int i)
{
  cd_node.item[i].attr_y = cd_node.item[i].name_point.y + 5;
}

/*
 * cd_node_item_calc_op_y
 */
void cd_node_item_calc_op_y(int i)
{
  cd_node.item[i].op_y =
    cd_node.item[i].attr_y + 
    g_list_length(cd_node.item[i].attributes) * 15 +
    5;
}

/*
 * cd_node_item_calc_center
 */
void cd_node_item_calc_center(int i, GdkPoint *point)
{
  point->x = cd_node.item[i].rect.x + cd_node.item[i].rect.width/2;
  point->y = cd_node.item[i].rect.y + cd_node.item[i].rect.height/2;
}

/*
 * cd_node_item_calc_term_mult_rect
 */
void cd_node_item_calc_term_mult_rect(ArcTerm *term)
{
  int length;

  switch(term->mult){
  case m0:
    length = CD_MULT_ZERO_LEN;
    break;
  case m0_1:
    length = CD_MULT_ZERO_ONE_LEN;
    break;
  case m1:
    length = CD_MULT_ONE_LEN;
    break;
  case m0_n:
    length = CD_MULT_ZERO_N_NEN;
    break;
  case m1_n:
    length = CD_MULT_ONE_N_EN;
    break;
  }

  switch(term->face){
  case face_first:
    term->mult_rect.x = term->point.x - length - 20;
    term->mult_rect.y = term->point.y;
    break;
  case face_second:
    term->mult_rect.x = term->point.x;
    term->mult_rect.y = term->point.y - 20;
    break;
  case face_third:
    term->mult_rect.x = term->point.x + 25;
    term->mult_rect.y = term->point.y - 2;
    break;
  case face_fourth:
    term->mult_rect.x = term->point.x + 3;
    term->mult_rect.y = term->point.y + CD_MULT_HEIGHT + 20;
    break;
  }

  term->mult_rect.width = length;
  term->mult_rect.height = CD_MULT_HEIGHT;
}

/*
 * cd_node_item_calc_arc_point
 */
void cd_node_item_calc_arc_point(double theta,
				 int x1, int y1, int x2, int y2,
				 RectCorner cn, ArcTerm *term)
{
  int m, n, m_n;
  double theta1, theta2, theta3, theta4;

  theta1 = cd_util_get_theta(x1, y1, cn.p1.x, cn.p1.y);
  theta2 = cd_util_get_theta(x1, y1, cn.p2.x, cn.p2.y);
  theta3 = cd_util_get_theta(x1, y1, cn.p3.x, cn.p3.y);
  theta4 = cd_util_get_theta(x1, y1, cn.p4.x, cn.p4.y);

  if(theta1 < theta && theta < theta2){
    m   = cn.p1.x - x2;
    n   = x1      - cn.p1.x;
    m_n = x1      - x2;

    if(m_n == 0)
      return;
    else
      term->point.y = (n*y2 + m*y1)/m_n;

    term->point.x = cn.p1.x - 1;
    term->face = face_first;
  }else if(theta4 < theta && theta < theta1){
    m   = cn.p1.y - y2;
    n   = y1      - cn.p1.y;
    m_n = y1      - y2;

    if(m_n == 0)
      return;
    else
      term->point.x = (n*x2 + m*x1)/m_n;

    term->point.y = cn.p1.y - 1;
    term->face = face_second;
  }else if(theta3 < theta && theta < theta4){
    m   = cn.p3.x - x1;
    n   = x2      - cn.p3.x;
    m_n = x2      - x1;

    if(m_n == 0)
      return;
    else
      term->point.y = (n*y1 + m*y2)/m_n;

    term->point.x = cn.p3.x + 1;
    term->face = face_third;
  }else{
    m   = cn.p3.y - y1;
    n   = y2      - cn.p3.y;
    m_n = y2      - y1;

    if(m_n == 0)
      return;
    else
      term->point.x = (n*x1 + m*x2)/m_n;

    term->point.y = cn.p3.y + 1;
    term->face = face_fourth;
  }

  cd_node_item_calc_term_mult_rect(term);
}

/*
 * cd_node_item_calc_points
 */
void cd_node_item_calc_arc_points(int start_node_id, int end_node_id,
				  ArcItem *arc)
{
  int i;
  int start_i = 0;
  int end_i   = 0;
  int start_x, start_y;
  int end_x, end_y;

  double theta;

  RectCorner start_cn, end_cn;

  for(i=0; i<cd_node.count; i++){
    if(start_node_id == cd_node.item[i].id)
      start_i = i;
    if(end_node_id == cd_node.item[i].id)
      end_i = i;
    if(start_i != 0 && end_i != 0)
      break;
  }

  start_x = cd_node.item[start_i].rect.x + cd_node.item[start_i].rect.width/2;
  start_y = cd_node.item[start_i].rect.y + cd_node.item[start_i].rect.height/2;

  end_x = cd_node.item[end_i].rect.x + cd_node.item[end_i].rect.width/2;
  end_y = cd_node.item[end_i].rect.y + cd_node.item[end_i].rect.height/2;

  cd_util_get_rect_corner(&cd_node.item[start_i].rect, &start_cn);
  cd_util_get_rect_corner(&cd_node.item[end_i].rect, &end_cn);

  arc->start.theta = theta = cd_util_get_theta(start_x, start_y, end_x, end_y);

  cd_node_item_calc_arc_point(theta,
			      start_x, start_y, end_x, end_y,
			      start_cn, &arc->start);

  arc->end.theta = theta = cd_util_get_theta(end_x, end_y, start_x, start_y);

  cd_node_item_calc_arc_point(theta,
			      end_x, end_y, start_x, start_y, 
			      end_cn, &arc->end);
}

/*
 * cd_node_item_erase
 */
void cd_node_item_erase(GdkPixmap *pixmap, GdkGC *gc, int i )
{
  gdk_draw_rectangle(pixmap, gc, TRUE,
		     cd_node.item[i].rect.x-1,
		     cd_node.item[i].rect.y-1, 
		     cd_node.item[i].rect.width+3,
		     cd_node.item[i].rect.height+3);
}  

/*
 * cd_node_item_erase_deleted
 */
void cd_node_item_erase_deleted(GdkPixmap *pixmap, GdkGC *gc)
{
  gdk_draw_rectangle(pixmap, gc, TRUE,
		     cd_node_item_last_deleted.rect.x-1,
		     cd_node_item_last_deleted.rect.y-1,
		     cd_node_item_last_deleted.rect.width+3,
		     cd_node_item_last_deleted.rect.height+3);
}  

void cd_node_item_erase_corner_deleted(GdkPixmap *pixmap, GdkGC *gc)
{
  int x1,x2,y1,y2;

  x1 = cd_node_item_last_deleted.rect.x;
  x2 = cd_node_item_last_deleted.rect.x + cd_node_item_last_deleted.rect.width;
  y1 = cd_node_item_last_deleted.rect.y;
  y2 = cd_node_item_last_deleted.rect.y + cd_node_item_last_deleted.rect.height;
  gdk_draw_rectangle(pixmap, gc, TRUE, x1-4, y1-4, 4, 4);
  gdk_draw_rectangle(pixmap, gc, TRUE, x1-4, y2,   4, 4);
  gdk_draw_rectangle(pixmap, gc, TRUE, x2,   y1-4, 4, 4);
  gdk_draw_rectangle(pixmap, gc, TRUE, x2,   y2,   4, 4);
}

/*
 * cd_node_item_draw
 */
void cd_node_item_draw(GdkPixmap *pixmap, GdkGC *gc, int i )
{
  int j;
  Attribute *attr;
  Operation *op;
  char vis[2];

  gdk_draw_rectangle(pixmap, gc, FALSE,
		     cd_node.item[i].rect.x,
		     cd_node.item[i].rect.y, 
		     cd_node.item[i].rect.width,
		     cd_node.item[i].rect.height);

  if(strlen(cd_node.item[i].name) > 0) {
    gdk_draw_string(pixmap, cd_font, gc,
		    cd_node.item[i].name_point.x,
		    cd_node.item[i].name_point.y,
		    cd_node.item[i].name);
  }

  gdk_draw_line(pixmap, gc,
		cd_node.item[i].rect.x,
		cd_node.item[i].attr_y,
		cd_node.item[i].rect.x + cd_node.item[i].rect.width,
		cd_node.item[i].attr_y);

  for(j=0; j<g_list_length(cd_node.item[i].attributes); j++){
    attr = g_list_nth_data(cd_node.item[i].attributes, j);

    switch(attr->vis){
    case vis_private:
      sprintf(vis, CD_VIS_PRIVATE_DQ);
      break;
    case vis_protected:
      sprintf(vis, CD_VIS_PROTECTED_DQ);
      break;
    case vis_public:
      sprintf(vis, CD_VIS_PUBLIC_DQ);
      break;
    }

    gdk_draw_string(pixmap, cd_font, gc,
		    cd_node.item[i].rect.x+3,
		    cd_node.item[i].attr_y+15+j*15,
		    vis);

    gdk_draw_string(pixmap, cd_font, gc,
		    cd_node.item[i].rect.x+12,
		    cd_node.item[i].attr_y+15+j*15,
		    attr->name);
  }
    
  for(j=0; j<g_list_length(cd_node.item[i].operations); j++){
    op = g_list_nth_data(cd_node.item[i].operations, j);

    switch(op->vis){
    case vis_private:
      sprintf(vis, CD_VIS_PRIVATE_DQ);
      break;
    case vis_protected:
      sprintf(vis, CD_VIS_PROTECTED_DQ);
      break;
    case vis_public:
      sprintf(vis, CD_VIS_PUBLIC_DQ);
      break;
    }

    gdk_draw_string(pixmap, cd_font, gc,
		    cd_node.item[i].rect.x+3,
		    cd_node.item[i].op_y+15+j*15,
		    vis);

    gdk_draw_string(pixmap, cd_font, gc,
		    cd_node.item[i].rect.x+12,
		    cd_node.item[i].op_y+15+j*15,
		    op->name);
  }
    

  gdk_draw_line(pixmap, gc,
		cd_node.item[i].rect.x,
		cd_node.item[i].op_y,
		cd_node.item[i].rect.x + cd_node.item[i].rect.width,
		cd_node.item[i].op_y);
}

/*
 * cd_node_item_draw_corner
 */
void cd_node_item_draw_corner(GdkPixmap *pixmap, GdkGC *gc, int i)
{
  int x1,x2,y1,y2;

  x1 = cd_node.item[i].rect.x;
  x2 = cd_node.item[i].rect.x + cd_node.item[i].rect.width+1;
  y1 = cd_node.item[i].rect.y;
  y2 = cd_node.item[i].rect.y + cd_node.item[i].rect.height+1;

  gdk_draw_rectangle(pixmap, gc, TRUE, x1-4, y1-4, 4, 4);
  gdk_draw_rectangle(pixmap, gc, TRUE, x1-4, y2,   4, 4);
  gdk_draw_rectangle(pixmap, gc, TRUE, x2,   y1-4, 4, 4);
  gdk_draw_rectangle(pixmap, gc, TRUE, x2,   y2,   4, 4);
}

/*
 * cd_node_item_draw_selecting
 */
void cd_node_item_draw_selecting(GdkPixmap *pixmap, GdkGC *gc)
{
  int i;
  if(cd_node.selecting >= cd_node.count )
    return;

  i = cd_node.selecting;

  cd_node_item_draw_corner(pixmap, gc, i);
}

/*
 * cd_node_item_draw_selected
 */
void cd_node_item_draw_selected(GdkPixmap *pixmap, GdkGC *gc)
{
  int i;
  if(cd_node.selected >= cd_node.count)
    return;

  i = cd_node.selected;

  cd_node_item_draw_corner(pixmap, gc, i);
}

/*
 * cd_node_item_selection
 */
int cd_node_item_selection(int x, int y)
{
  int i;

  for(i=0; i<cd_node.count; i++){
    if(cd_util_select_rect(&cd_node.item[i].rect, x, y)){
      cd_node.selecting = i;
      cd_node.item[i].select = TRUE;
      return i;
    }
  }

  return CD_SELECTING_CLEAR;
}

/*
 * cd_node_item_selection_clear_all
 */
void cd_node_item_selection_clear_all()
{
  int i;

  for(i=0; i<cd_node.count; i++)
    cd_node.item[i].select = FALSE;
}

/*
 * cd_node_item_move
 */
void cd_node_item_move(int dx, int dy)
{
  int i = cd_node.selecting;

  cd_node.item[i].rect.x += dx;
  cd_node.item[i].rect.y += dy;
  
  cd_node_item_calc_name_point(i);
  cd_node_item_calc_attr_y(i);
  cd_node_item_calc_op_y(i);
}  

/*
 * cd_node_item_get_id
 */
int cd_node_item_get_id(int i)
{
  return cd_node.item[i].id;
}

/*
 * cd_node_init
 */
void cd_node_init()
{
  int i;

  cd_node.seq = 0;
  cd_node.count = 0;
  cd_node.selecting = CD_SELECTING_CLEAR;
  cd_node.selected = CD_SELECTING_CLEAR;

  for(i=0; i<CD_NODE_ITEM_MAX; i++)
    cd_node_item_init(i);
}

/*
 * cd_op_load
 */
void cd_op_load(FILE *fp, int i)
{
  Operation *op;
  char line[BUFSIZ];
  char str[4];

  op = g_new(Operation, 1);
  
  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %s %d", str, op->name, &op->vis);

  cd_node.item[i].operations =
    g_list_append(cd_node.item[i].operations, op);
}

/*
 * cd_attr_load
 */
void cd_attr_load(FILE *fp, int i)
{
  Attribute *attr;
  char line[BUFSIZ];
  char str[6];

  attr = g_new(Attribute, 1);
  
  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %s %d", str, attr->name, &attr->vis);

  cd_node.item[i].attributes =
    g_list_append(cd_node.item[i].attributes, attr);
}

/*
 * cd_node_item_load
 */
void cd_node_item_load(FILE *fp, int i)
{
  char line[BUFSIZ];
  int j, count;
  char str[11];

  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %d %d %d %d %d %s %d %d",
	 str,
	 &cd_node.item[i].id,
	 &cd_node.item[i].rect.x,
	 &cd_node.item[i].rect.y,
	 &cd_node.item[i].rect.width,
	 &cd_node.item[i].rect.height,
	 cd_node.item[i].name,
	 &cd_node.item[i].name_point.x,
	 &cd_node.item[i].name_point.y);

  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %d %d", str, &count, &cd_node.item[i].attr_y);
  for(j=0; j<count; j++)
    cd_attr_load(fp, i);
  
  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %d %d", str, &count, &cd_node.item[i].op_y);
  for(j=0; j<count; j++)
    cd_op_load(fp, i);
}

/*
 * cd_node_load
 */
void cd_node_load(FILE *fp)
{
  char line[BUFSIZ];
  int count, seq;
  char str[6];

  int i;

  fgets(line, BUFSIZ, fp);
  sscanf(line, "%s %d %d", str, &count, &seq);

  cd_node.count = count;
  cd_node.seq = seq;

  for(i=0; i<cd_node.count; i++)
    cd_node_item_load(fp, i);
}

/*
 * cd_op_save
 */
void cd_op_save(FILE *fp, int i)
{
  int j, op_count;
  Operation *op;

  op_count = g_list_length(cd_node.item[i].operations);

  fprintf(fp, " Ops!: %d %d\n", op_count, cd_node.item[i].op_y);

  for(j=0; j<op_count; j++){
    fprintf(fp, " op:");
    op = g_list_nth_data(cd_node.item[i].operations, j);
    if(strlen(op->name) > 0)
      fprintf(fp, " %s", op->name);
    else
      fprintf(fp, " None");

    fprintf(fp, " %d", op->vis);
    fprintf(fp, "\n");
  }
}

/*
 * cd_attr_save
 */
void cd_attr_save(FILE *fp, int i)
{
  int j, attr_count;
  Attribute *attr;

  attr_count = g_list_length(cd_node.item[i].attributes);

  fprintf(fp, " Attrs!: %d %d\n", attr_count, cd_node.item[i].attr_y);

  for(j=0; j<attr_count; j++){
    fprintf(fp, " attr:");
    attr = g_list_nth_data(cd_node.item[i].attributes, j);
    if(strlen(attr->name) > 0)
      fprintf(fp, " %s", attr->name);
    else
      fprintf(fp, " None");

    fprintf(fp, " %d", attr->vis);
    fprintf(fp, "\n");
  }
}

/*
 * cd_node_item_save
 */
void cd_node_item_save(FILE *fp)
{
  int i;

  for(i=0; i<cd_node.count; i++){
    fprintf(fp, "node_item: %d", cd_node.item[i].id);
    
    fprintf(fp, " %d %d %d %d",
	    cd_node.item[i].rect.x,
	    cd_node.item[i].rect.y,
	    cd_node.item[i].rect.width,
	    cd_node.item[i].rect.height);

    if(strlen(cd_node.item[i].name) > 0)
      fprintf(fp, " %s", cd_node.item[i].name);
    else
      fprintf(fp, " None");

    fprintf(fp, " %d %d",
	    cd_node.item[i].name_point.x,
	    cd_node.item[i].name_point.y);

    fprintf(fp, "\n");

    cd_attr_save(fp, i);
    cd_op_save(fp, i);
  }
}

/*
 * cd_node_save
 */
void cd_node_save(FILE *fp)
{
  int i;

  fprintf(fp, "NODE! %d %d\n", cd_node.count, cd_node.seq);

  cd_node_item_save(fp);
}

/*
 * cd_node_selecting_end
 */
void cd_node_selecting_end()
{
  cd_node.selected = cd_node.selecting;
  cd_node.selecting = CD_SELECTING_CLEAR;
}

/*
 * cd_node_selection_clear
 */
void cd_node_selection_clear()
{
  cd_node.selected = CD_SELECTING_CLEAR;
  cd_node.selecting = CD_SELECTING_CLEAR;
}

/*
 * cd_node_check_intersect
 */
gboolean cd_node_check_intersect(GdkRectangle *rect)
{
  int i;
  GdkRectangle rect3;

  for(i=0; i<cd_node.count; i++)
    if(gdk_rectangle_intersect(rect, &cd_node.item[i].rect, &rect3))
      return TRUE;

  return FALSE;
}
