/*
 * @file  protomod_sessionless.c
 * @brief protocol module of any protocol.
 * @brief this module never keep session persistence.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#define __STDC_LIMIT_MACROS
#include <stdlib.h>
#include <time.h>
#include <getopt.h>
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_dest.h"
#include "l7vs_module.h"
#include "module_http.h"

#define SERVICE_ARG_MAXSIZE    (512)
#define SESSIONLESS_SERVICE_NUMBER (128)
#define X_FORWARDED_FOR_LENGTH (48)

struct l7vs_sessionless_service {
	handle_t service_handle;
	int forwarded_for;
	int reschedule;
};

struct  l7vs_sessionless_service_arg {
	int forwarded_for;
	int reschedule;
};

static void  fini(void);
static int   create(void*, handle_t);
static void* create_sa(struct l7vs_service_arg*);
static int   compare(handle_t, handle_t);
static int   match_cldata(struct l7vs_service*, struct l7vs_conn*,
	    char*, size_t*, struct l7vs_dest**, int*);
static int   analyze_rsdata(struct l7vs_service*, struct l7vs_conn*,
	    char*, size_t*);
static int   destroy(handle_t);
static void  destroy_sa(void**);
static int   service_arg(struct l7vs_service_arg_multi*, handle_t);
static int   parse(void*, int, char**);

static struct l7vs_sessionless_service *l7vs_protomod_sessionless_search_service(handle_t);
static struct l7vs_sessionless_service *l7vs_protomod_sessionless_create_service();
static struct l7vs_sessionless_service *l7vs_protomod_sessionless_create_temp_service();

static void l7vs_sessionless_service_c_str(char*, struct l7vs_sessionless_service*);
static void l7vs_sessionless_service_arg_c_str(char*, struct l7vs_sessionless_service_arg*);

struct l7vs_sessionless_service *sessionless_service_list[SESSIONLESS_SERVICE_NUMBER];

static struct l7vs_protomod sessionless_protomod = {
	NULL,           /* handle */
	"sessionless",      /* modname */
	0,              /* refcnt */
	create,         /* create function */
	compare,        /* compare function */
	match_cldata,   /* match_cldata function */
	analyze_rsdata, /* analyze_rsdata function */
	destroy,        /* destroy function */
	fini,           /* fini function */
	create_sa,      /* create_sa function */
	service_arg,    /* service_arg function */
	parse,          /* parse function */
	destroy_sa,     /* destroy_sa function */
	NULL,           /* initialize function */
	NULL,           /* finalize function */
	NULL,           /* get_log_level function */
	NULL,           /* put_log_debug function */
	NULL,           /* put_log_info function */
	NULL,           /* put_log_warn function */
	NULL,           /* put_log_error function */
	NULL            /* put_log_fatal function */
};

/*!
 * Protocol module initialize function. This function run when dlopen and dlsym at first time.
 * @param[in] handle dlopen's handle
 * @return l7vs_protomod struct
 */
extern "C" struct l7vs_protomod *
init(void *handle)
{
	struct l7vs_protomod* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,59,
	            "in_function: struct l7vs_protomod* init(void* handle): handle=%p", handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (handle == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,51, "Arg(handle) is NULL pointer.");
		goto init_out;
	}

	/* initialize sessionless service list */
	memset(sessionless_service_list, 0, sizeof(struct l7vs_sessionless_service *) * SESSIONLESS_SERVICE_NUMBER);
	/* set dlopen's handle */
	sessionless_protomod.handle = handle;

	return_value = &sessionless_protomod;

init_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char protomod_str[DEBUG_STR_LEN] = {0};
		l7vs_protomod_c_str(protomod_str, &sessionless_protomod);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,60,
		    "out_function: struct l7vs_protomod* init(void* handle): return=&(%s)", protomod_str);
	}
	/*------ DEBUG LOG END ------*/
	return return_value;
}

/*!
 * Protocol module finalize function. free all sessionless service list just in case.
 * @param   void
 * @return  void
 */
static void
fini(void)
{
	/* sessionless service list counter */
	int service_number = 0;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,61, "in_function: void fini(void)");
	}
	/*------ DEBUG LOG END ------*/

	/* check all sessionless service list */
	for (service_number = 0; service_number < SESSIONLESS_SERVICE_NUMBER; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (sessionless_service_list[service_number] != NULL) {
			/*-------- DEBUG LOG --------*/
			if (sessionless_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,20, "free: %p",
				    sessionless_service_list[service_number]);
			}
			/*------ DEBUG LOG END ------*/

			/* free and points NULL */
			free(sessionless_service_list[service_number]);
			sessionless_service_list[service_number] = NULL;
		}
	}

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,62, "out_function: void fini(void)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create sessionless service struct.
 * @param[in] sessionless_arg    sessionless service argument struct
 * @param[in] service_handle a unique service ID
 * @retval 0  successfully create sessionless service.
 * @retval -1 some errors occur.
 */
static int
create(void *sessionless_arg, handle_t service_handle)
{
	struct l7vs_sessionless_service *sessionless_service;
	struct l7vs_sessionless_service_arg *sessionless_service_arg;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sessionless_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_arg_c_str(sessionless_arg_str, (struct l7vs_sessionless_service_arg*) sessionless_arg);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,63,
		    "in_function: int create(void* sessionless_arg, handle_t service_handle):sessionless_arg=&(%s), "
		    "service_handle=%d", sessionless_arg_str, service_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (sessionless_arg == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,52, "Arg(sessionless_arg) is NULL pointer.");
		return_value = -1;
		goto create_out;
	}


	if (service_handle != TEMP_SERVICEHANDLE) {
		/* search empty sessionless service list and create sessionless service */
		sessionless_service = l7vs_protomod_sessionless_create_service();
	} else {
		/* create temporary sessionless service */
		sessionless_service = l7vs_protomod_sessionless_create_temp_service();
	}

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sessionless_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_c_str(sessionless_str, sessionless_service);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,64, "pointer assign: sessionless_service=&(%s)",
		    sessionless_str);
	}
	/*------ DEBUG LOG END ------*/

	if (sessionless_service == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,53, "Could not make sessionless service.");
		return_value = -1;
		goto create_out;
	}

	sessionless_service_arg = (struct l7vs_sessionless_service_arg *) sessionless_arg;

	/* set service handle */
	sessionless_service->service_handle = service_handle;
	/* set x-forwarded-for flag */
	sessionless_service->forwarded_for = sessionless_service_arg->forwarded_for;
	/* set reschedule  */
	sessionless_service->reschedule = sessionless_service_arg->reschedule;

create_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,65,
		    "out_function: int create(void* sessionless_arg, handle_t service_handle):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create sessionless service argument struct.
 * @param[out] srv_arg service argument struct
 * @return sessionless service argument struct
 */
static void *
create_sa(struct l7vs_service_arg *srv_arg)
{
	struct l7vs_sessionless_service_arg *sessionless_service_arg;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char service_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_service_arg_c_str(service_arg_str, srv_arg);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,66,
		    "in_function: void* create_sa(struct l7vs_service_arg* srv_arg):srv_arg=&(%s)",
		    service_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv_arg == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,54, "Arg(srv_arg) is NULL pointer.");
		sessionless_service_arg = NULL;
		goto create_sa_out;
	}

	/* create sessionless service argument struct */
	sessionless_service_arg = (struct l7vs_sessionless_service_arg *) calloc(1, sizeof(struct l7vs_sessionless_service_arg));

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,21, "calloc: addr=%p, size=%ld",
		    sessionless_service_arg, (unsigned long int) sizeof(struct l7vs_sessionless_service_arg));
	}
	/*------ DEBUG LOG END ------*/

	if (sessionless_service_arg == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,19, "Could not allocate memory.");
		goto create_sa_out;
	}

	/* set sessionless service argument size and protomod name "sessionless" */
	srv_arg->len = sizeof(struct l7vs_sessionless_service_arg);
	strcpy(srv_arg->protomod, sessionless_protomod.modname);

create_sa_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sessionless_service_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_arg_c_str(sessionless_service_arg_str, sessionless_service_arg);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,67,
		    "out_function: void* create_sa(struct l7vs_service_arg* srv_arg):return_value=&(%s)",
		    sessionless_service_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	return (void *) sessionless_service_arg;
}

/*!
 * Compare two service.
 * @param[in] srv_handle1 one of a unique service ID
 * @param[in] srv_handle2 one of a unique service ID
 * @retval 0  they matched perfectly.
 * @retval -1 they are different.
 */
static int
compare(handle_t srv_handle1, handle_t srv_handle2)
{
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,68,
		    "in_function: int compare(handle_t srv_handle1, handle_t srv_handle2):"
		    "srv_handle1=%u, srv_handle2=%u", srv_handle1, srv_handle2);
	}
	/*------ DEBUG LOG END ------*/

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,69,
		    "out_function: int compare(handle_t srv_handle1, handle_t srv_handle2):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Do not check the request packet.
 * @param[in]  srv service struct include service handle, protocol module and schedule module.
 * @param[in]  conn connection data.
 * @param[in]  request packet data from client
 * @param[in]  len length of packet data
 * @param[out] dest destination (real server) list
 * @param[out] tcps TCP Splicer flag
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *request, size_t *len, struct l7vs_dest **dest, int *tcps)
{
	struct l7vs_sessionless_service *sessionless_service;
	int ret;
	int offset_length;
	char *x_forwarded_value;
	char *next_line = NULL;
	char x_forwarded_for_header[X_FORWARDED_FOR_LENGTH];
	size_t uri_len = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN] = {0};
		char conn_str[DEBUG_STR_LEN] = {0};
		char dest_str[DEBUG_STR_LEN] = {0};
		char len_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (dest != NULL) {
			l7vs_dest_c_str(dest_str, *dest);
		}
		else {
			strncpy(dest_str, "NULL", DEBUG_STR_LEN);
		}
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,70,
		    "in_function: int match_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):srv=&(%s), conn=&(%s), "
		    "request=\"%s\", len=&(%s), dest=&(&(%s)), tcps=&(%d)",
		    srv_str, conn_str, request, len_str, dest_str, *tcps);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,55, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (srv->pm == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,56, "Arg(srv->pm) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (request == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,57, "Arg(request) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,58, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (dest == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,59, "Arg(dest) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}
	if (tcps == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,60, "Arg(tcps) is NULL pointer.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* search service that has such a service ID */
	sessionless_service = l7vs_protomod_sessionless_search_service(srv->handle);

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sessionless_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_c_str(sessionless_str, sessionless_service);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,71, "pointer assign: sessionless_service=&(%s)",
		    sessionless_str);
	}
	/*------ DEBUG LOG END ------*/

	if (sessionless_service == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,61, "Could not find such service handle's sessionless service.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* initialize protocol module ... clear destination list */
	ret = srv->pm->initialize(srv, conn, request, *len, dest);
	if (ret != 0){
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,62, "Could not initialize protomod.");
		return_value = -1;
		goto match_cldata_out;
	}

	/* set X-Forwarded-For field */
	if (sessionless_service->forwarded_for) {
		uri_len = *len;
		/* check request */
		if (http_check_request_method(request, &uri_len) == NULL)
			goto match_cldata_finalize;

		x_forwarded_value = http_search_header_field(request, "X-Forwarded-For");

		/* already exists X-Forwarded-For field */
		if (x_forwarded_value) {
			next_line = http_skip_header_line(x_forwarded_value);
			/* backtrack to look up insert point */
			if (next_line) {
				next_line--; // *next_line == '\n'
				if (*(next_line - 1) == '\r') {
					next_line--;
				}
				/* append client IP address */
				snprintf(x_forwarded_for_header, X_FORWARDED_FOR_LENGTH, ", %s", inet_ntoa(conn->caddr.sin_addr));
			}
		}

		/* not exists X-Forwarded-For field */
		if (!next_line) {
			/* construct new X-Forwarded-For header item */
			snprintf(x_forwarded_for_header, X_FORWARDED_FOR_LENGTH, "X-Forwarded-For: %s\r\n", inet_ntoa(conn->caddr.sin_addr));
	
			next_line = http_skip_header_line(request);
		}

		/* when insert point exist */
		if (next_line != NULL) {
			offset_length = (int) (next_line - request);
	
			/* insert X-Forwarded-For header field */
			http_insert_field(request, offset_length, x_forwarded_for_header, *len);
		
			/* add header length */
			*len += strlen(x_forwarded_for_header);
		}
	}

match_cldata_finalize:
	*tcps = 0;

	/* finalize, always set reschedule flag */
	ret = srv->pm->finalize(srv, conn, request, *len, dest, 1);

	if (ret != 0){
		PUT_LOG_INFO(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,10, "Could not finalize protomod. (Realserver decision failure)");
		return_value = -1;
		goto match_cldata_out;
	}

match_cldata_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,72,
		    "out_function: int match_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Do nothing.
 * @param[in] srv service struct include service handle, protocol module and schedule module.
 * @param[in] conn connection data.
 * @param[in] response packet data from real server
 * @param[in] len length of packet data. it will be lengthened.
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
	char *response, size_t *len)
{
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN] = {0};
		char conn_str[DEBUG_STR_LEN] = {0};
		char len_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,73,
		    "in_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* response, size_t* len):srv=&(%s), conn=&(%s), response=\"%s\", len=&(%s)",
		    srv_str, conn_str, response, len_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,64, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (conn == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,65, "Arg(conn) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (conn->dest == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,66, "Arg(conn->dest) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (response == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,67, "Arg(response) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,68, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}

	/* sorry flag check */
	if (conn->sorry_conn_flag == 1) {
		/*-------- DEBUG LOG --------*/
		if (sessionless_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
			PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,74, "Response from sorry server.");
		}
		/*------ DEBUG LOG END ------*/
		goto analyze_rsdata_out;
	}

analyze_rsdata_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,75,
		    "out_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* response, size_t* len):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Destroy sessionless service
 * @param[in] srv_handle a unique service ID
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
destroy(handle_t srv_handle)
{
	/* sessionless service list counter */
	int service_number = 0;
	int free_flag = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,76,
		    "in_function: int destroy(handle_t srv_handle):srv_handle=%u",
		    srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check all sessionless service list */
	for (service_number = 0; service_number < SESSIONLESS_SERVICE_NUMBER; ++service_number) {
		/* found sessionless service that has srv_handle */
		if (sessionless_service_list[service_number] != NULL && 
		    sessionless_service_list[service_number]->service_handle == srv_handle) {

			/* free and NULL */
			free(sessionless_service_list[service_number]);
			sessionless_service_list[service_number] = NULL;

			free_flag = 1;
			break;
		}
	}
	
	/* sessionless service was not found */
	if (free_flag == 0) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,69, "Could not find such service handle's sessionless service.");
		return_value = -1;
		goto destroy_out;
	}

destroy_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,77,
		    "out_function: int destroy(handle_t srv_handle):return_value=%d",
		    srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Destroy sessionless service argument
 * @param[in] sessionless_arg sessionless service argument
 * @return void
 */
static void
destroy_sa(void **sessionless_arg)
{
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sessionless_arg_str[DEBUG_STR_LEN] = {0};
		if (sessionless_arg != NULL) {
			l7vs_sessionless_service_arg_c_str(sessionless_arg_str, (struct l7vs_sessionless_service_arg*) *sessionless_arg);
		}
		else {
			strncpy(sessionless_arg_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,78,
		    "in_function: void destroy_sa(void** sessionless_arg):sessionless_arg=&(&(%s))",
		    sessionless_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (sessionless_arg == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,70, "Arg(sessionless_arg) is NULL pointer.");
	}
	else if (*sessionless_arg == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,71, "Arg(*sessionless_arg) is NULL pointer.");
	}
	else {
		/*-------- DEBUG LOG --------*/
		if (sessionless_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
			PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,22, "free: %p",
			    *sessionless_arg);
		}
		/*------ DEBUG LOG END ------*/

		/* free and NULL */
		free((struct l7vs_sessionless_service_arg*) *sessionless_arg);
		*sessionless_arg = NULL;
	}

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,79,
		    "out_function: void destroy_sa(void** sessionless_arg)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create strings for service list of l7vsadm
 * @param[out] srv_arg service argument struct
 * @param[in]  srv_handle a unique service ID
 * @retval 0  successfully create strings
 * @retval -1 some errors occur.
 */
static int
service_arg(struct l7vs_service_arg_multi *srv_arg_mt, handle_t srv_handle)
{
	struct l7vs_sessionless_service *sessionless_service;
	struct l7vs_sessionless_service_arg c_sarg;
	char sessionless_argument[SERVICE_ARG_MAXSIZE];
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_arg_mt_str[DEBUG_STR_LEN] = {0};
		l7vs_service_arg_multi_c_str(srv_arg_mt_str, srv_arg_mt);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,80,
		    "in_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
		    "handle_t srv_handle):srv_arg_mt=&(%s), srv_handle=%u",
		    srv_arg_mt_str, srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv_arg_mt == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,72, "Arg(srv_arg_mt) is NULL pointer.");
		return_value = -1;
		goto service_arg_out;
	}

	/* search service that has such a service ID */
	sessionless_service = l7vs_protomod_sessionless_search_service(srv_handle);

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sessionless_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_c_str(sessionless_str, sessionless_service);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,81, "pointer assign: sessionless_service=&(%s)",
		    sessionless_str);
	}
	/*------ DEBUG LOG END ------*/

	if (sessionless_service == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,73, "Could not find such service handle's sessionless service.");
		return_value = -1;
		goto service_arg_out;
	}

	/* initialize argument strings */
	memset(sessionless_argument, 0, SERVICE_ARG_MAXSIZE);

	/* set sessionless args to service argument struct */
	srv_arg_mt->srv_arg.reschedule = sessionless_service->reschedule;

	/* create long argument (l7vsadm option -L/-l) */
	strncpy(srv_arg_mt->srv_arg.protomod_key_string, "", 256);

	/* create verbose argument (l7vsadm option -V/-v) */
	strncpy(srv_arg_mt->srv_arg.protomod_opt_string, "", 512);
	if (sessionless_service->forwarded_for) {
		strncpy(srv_arg_mt->srv_arg.protomod_opt_string, "--forwarded-for", 512);
	}

	c_sarg.reschedule = sessionless_service->reschedule;

	memcpy(srv_arg_mt->protomod_arg, &c_sarg, sizeof(struct l7vs_sessionless_service_arg));

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sessionless_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_arg_c_str(sessionless_arg_str, &c_sarg);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,82,
		    "pointer assign: srv_arg_mt->protomod_arg=&(%s)", sessionless_arg_str);
	}
	/*------ DEBUG LOG END ------*/

service_arg_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,83,
		    "out_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
		    "handle_t srv_handle):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Parse l7vsadm options to sessionless argument
 * @param[out] sessionless_arg sessionless service argument struct
 * @param[in]  argc number of l7vsadm argument
 * @param[in]  argv l7vsadm argument list
 * @retval 0  successfully parse argument
 * @retval -1 some errors occur.
 */
static int
parse(void *sessionless_arg, int argc, char *argv[])
{
	struct l7vs_sessionless_service_arg *sessionless_service_arg;
	static struct option opt[] = {
		{"forwarded-for", no_argument,       NULL, 'F'},
		{NULL,            0,                 NULL, 0  }
	};
	int c;
	int return_value = 0;
	int forwarded_for_flag = 0;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		int i;
		char argv_str[DEBUG_STR_LEN] = {0};
		char sessionless_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_arg_c_str(sessionless_arg_str, (struct l7vs_sessionless_service_arg*) sessionless_arg);
		argv_str[0] = '\0';
		if (argv == NULL)
			snprintf(argv_str, DEBUG_STR_LEN, "NULL");
		else {
			for (i = 0; i < argc; i++) {
				snprintf(argv_str, DEBUG_STR_LEN, "%sargv[%d]=\"%s\", ", argv_str, i, argv[i]);
			}
			i = strnlen(argv_str, DEBUG_STR_LEN);
			if (i > 1)
				argv_str[i - 2] = '\0';
		}
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,84,
		    "in_function: int parse(void* sessionless_arg, int argc, char* argv[]):sessionless_arg=&(%s), "
		    "argc=%d, %s", sessionless_arg_str, argc, argv_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (sessionless_arg == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,74, "Arg(sessionless_arg) is NULL pointer.");
		return_value = -1;
		goto parse_out;
	}
	if (argv == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,75, "Arg(argv) is NULL pointer.");
		return_value = -1;
		goto parse_out;
	}

	sessionless_service_arg = (struct l7vs_sessionless_service_arg *) sessionless_arg;
	optind = 0;

	/* check all argument */
	while ((c = getopt_long(argc, argv, "F", opt, NULL)) != -1) {
		switch (c) {
		/* --forwarded-for / -F */
		case 'F':
			/* x-forwarded-for on */
			sessionless_service_arg->forwarded_for = 1;
			forwarded_for_flag++;
			break;
		/* else error */
		default:
			PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,76, "Option error.");
			return_value = -1;
			goto parse_out;
		}
	}

	/* set default reschedule */
	sessionless_service_arg->reschedule = 1;

	/* set default no forwarded_for */
	if (forwarded_for_flag == 0) {
		sessionless_service_arg->forwarded_for = 0;
	}

parse_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,85,
		    "out_function: int parse(void* sessionless_arg, int argc, char* argv[]):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Search sessionless service from sessionless service list using service handle
 * @param[in] service_handle a unique service ID
 * @return sessionless service struct when service was found. NULL when service was not found.
 */
static struct l7vs_sessionless_service *
l7vs_protomod_sessionless_search_service(handle_t service_handle)
{
	/* sessionless service list counter */
	int service_number = 0;
	struct l7vs_sessionless_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,86,
		    "in_function: struct l7vs_sessionless_service* l7vs_protomod_sessionless_search_service(handle_t service_handle):"
		    "service_handle=%d", service_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check all sessionless service list */
	for (service_number = 0; service_number < SESSIONLESS_SERVICE_NUMBER; ++service_number) {
		/* found the service has same service handle */
		if (sessionless_service_list[service_number] != NULL && 
		    sessionless_service_list[service_number]->service_handle == service_handle) {
			return sessionless_service_list[service_number];
		}
	}
	
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,87,
		    "out_function: struct l7vs_sessionless_service* l7vs_protomod_sessionless_search_service(handle_t service_handle):"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create sessionless service.
 * @param void
 * @return sessionless service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_sessionless_service *
l7vs_protomod_sessionless_create_service()
{
	/* sessionless service list counter */
	int service_number = 0;
	struct l7vs_sessionless_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,88,
		    "in_function: struct l7vs_sessionless_service* l7vs_protomod_sessionless_create_service()");
	}
	/*------ DEBUG LOG END ------*/

	/* check all sessionless service list */
	for (service_number = 0; service_number < SESSIONLESS_SERVICE_NUMBER - 1; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (sessionless_service_list[service_number] == NULL) {
			/* create a service at empty pointer */
			sessionless_service_list[service_number] = (struct l7vs_sessionless_service *) calloc(1, sizeof(struct l7vs_sessionless_service));

			/*-------- DEBUG LOG --------*/
			if (sessionless_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,23, "calloc: addr=%p, size=%ld",
				    sessionless_service_list[service_number], (unsigned long int) sizeof(struct l7vs_sessionless_service));
			}
			/*------ DEBUG LOG END ------*/

			if (sessionless_service_list[service_number] == NULL) {
				PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,20, "Could not allocate memory.");
				goto create_service_out;
			}
			return_value = sessionless_service_list[service_number];
			goto create_service_out;
		}
	}
	
	/* all service list is full */
	PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,77, "sessionless service list is full.");

create_service_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,89,
		    "out_function: struct l7vs_sessionless_service* l7vs_protomod_sessionless_create_service():"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create temporary sessionless service.
 * @param  void
 * @return sessionless service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_sessionless_service *
l7vs_protomod_sessionless_create_temp_service()
{
	struct l7vs_sessionless_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,90,
		    "in_function: struct l7vs_sessionless_service* l7vs_protomod_sessionless_create_temp_service()");
	}
	/*------ DEBUG LOG END ------*/

	/* if pointer that does not point NULL exists ... */
	if (sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1] != NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,78, "Temporary sessionless service is being used by other process.");
		goto create_temp_service_out;
	}

	/* create temp service */
	sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1] = (struct l7vs_sessionless_service *) calloc(1, sizeof(struct l7vs_sessionless_service));

	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,24, "calloc: addr=%p, size=%ld",
		    sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1], (unsigned long int) sizeof(struct l7vs_sessionless_service));
	}
	/*------ DEBUG LOG END ------*/

	if (sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1] == NULL) {
		PUT_LOG_ERROR(sessionless_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,21, "Could not allocate memory");
		goto create_temp_service_out;
	}

	return_value = sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1];

create_temp_service_out:
	/*-------- DEBUG LOG --------*/
	if (sessionless_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sessionless_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN] = {0};
		l7vs_sessionless_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(sessionless_protomod, LOG_CAT_L7VSD_PROTOCOL,91,
		    "out_function: struct l7vs_sessionless_service* l7vs_protomod_sessionless_create_service():"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Serialize struct l7vs_sessionless_service for debug log.
 * @param[out] buf   serialized string
 * @param[in]  sessionless l7vs_sessionless_service struct
 */
static void l7vs_sessionless_service_c_str(char* buf, struct l7vs_sessionless_service* sessionless) {
	if (sessionless == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		snprintf(buf, DEBUG_STR_LEN, "service_handle=%d, forwarded_for=%d, reschedule=%d",
		    sessionless->service_handle, sessionless->forwarded_for, sessionless->reschedule);
	}
}

/*!
 * Serialize struct l7vs_sessionless_service_arg for debug log.
 * @param[out] buf       serialized string
 * @param[in]  sessionless_arg l7vs_sessionless_service_arg struct
 */
void l7vs_sessionless_service_arg_c_str(char* buf, struct l7vs_sessionless_service_arg* sessionless_arg) {
	if (sessionless_arg == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		snprintf(buf, DEBUG_STR_LEN, "forwarded_for=%d, reschedule=%d",
		    sessionless_arg->forwarded_for, sessionless_arg->reschedule);
	}
}
