#define _GNU_SOURCE
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fnmatch.h>
#include <getopt.h>
#include <glib.h>
#include <time.h>
#include <ctype.h>
#include "vanessa_logger.h"
#include "l7vs.h"
#include "sync_session.h" 

struct l7vs_urla_service {
	struct l7vs_service srv;
	char key[129];
	int search_len;
	int timeout;
	int max_list;
	int reschedule;
	GList *list;
};

struct  l7vs_urla_service_arg {
	struct l7vs_service_arg arg;
	char key[129];
	int search_len;
	int timeout;
	int max_list;
	int reschedule;
	GList *list;
};

int process_state;

static void fini(void);
static struct l7vs_service *create(struct l7vs_service_arg *arg);
static struct l7vs_service_arg *create_sa(void);
static int compare(struct l7vs_service *s1, struct l7vs_service *s2);
static int match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
		char *buf, size_t *len, struct l7vs_dest **dest, int *tcps);
static int analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
		char *buf, size_t *len);
static void destroy(struct l7vs_service *srv);
static struct l7vs_service_arg *service_arg(struct l7vs_service *srv);
static int parse(struct l7vs_service_arg *arg, int argc, char *argv[]);

char *l7vs_protomod_urla_search(const char *s1, const char *s2, const char *s3);
char *l7vs_protomod_urla_toupper_search(const char *s1, const char *s2, const char *s3);
int l7vs_protomod_urla_del_list(struct l7vs_urla_service *as, time_t t);
int l7vs_protomod_urla_add_list(struct l7vs_urla_service *as, struct l7vs_conn *conn, char *match, int key_len, time_t t);
int l7vs_protomod_urla_str_len(const char *s1, const char s2);
int l7vs_protomod_urla_dest_check(struct l7vs_service *srv, struct l7vs_conn *conn, struct l7vs_dest **dest);


static char *get_servicename(struct sockaddr_in *addr, int isnumeric);

struct url_list {
	char pattern[129];
	struct sockaddr_in raddr;
	time_t date;
};

static struct l7vs_protomod urla_protomod = {
		NULL,			/* handle */
		"urla",			/* modname */
        	0,                      /* refcnt */
		create,			/* create function */
		compare,		/* compare function */
		match_cldata,		/* match_cldata function */
		analyze_rsdata,		/* analyze_rsdata function */
	        destroy,                /* destroy function */
	        fini,                   /* fini function */
	        create_sa,              /* create_sa function */
	        service_arg,            /* service_arg function */
	        parse,                  /* parse function */
};

struct l7vs_protomod *
init(void *handle)
{
        urla_protomod.handle = handle;
        return &urla_protomod;
}

static void 
fini(void)
{
        /* XXX  maybe some work needed */
}

static struct l7vs_service *
create(struct l7vs_service_arg *arg)
{
        struct l7vs_urla_service *as;
        struct l7vs_urla_service_arg *aa;

        as = (struct l7vs_urla_service *)calloc(1, sizeof(*as));
        if (as == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service *)as;
        }

        aa = (struct l7vs_urla_service_arg *)arg;
        strcpy(as->key, aa->key);
        as->search_len = aa->search_len;
	as->timeout = aa->timeout;
	as->reschedule = aa->reschedule;
	as->list = aa->list;
	as->max_list = aa->max_list;
        return (struct l7vs_service *)as;
}

static struct l7vs_service_arg *
create_sa(void)
{
        struct l7vs_urla_service_arg *aa;

        aa = (struct l7vs_urla_service_arg *)calloc(1, sizeof(*aa));
        if (aa == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service_arg *)aa;
        }

        aa->arg.len = sizeof(*aa);
        strcpy(aa->arg.protomod, urla_protomod.modname);

        return (struct l7vs_service_arg *)aa;
}

static int
compare(struct l7vs_service *s1, struct l7vs_service *s2)
{
        struct l7vs_urla_service *a1, *a2;

        a1 = (struct l7vs_urla_service *)s1;
        a2 = (struct l7vs_urla_service *)s2;

        return strcmp(a1->key, a2->key);

}

static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *buf, size_t *len, struct l7vs_dest **dest, int *tcps)
{
        struct l7vs_urla_service *as;
	struct l7vs_dest d;
	struct url_list *ul, *ul_temp;
	time_t t;
	char *temp, temp2;
	int ret, f = -1, key_len;
	GList *l;

	int tindx=0;
	char * srvname;
	void *shared_memory=(void *)0;
	int sema_id,shmid;
	struct sync_info_st *sync_info;
	struct cookie_list *ckl, *cl_temp;
	unsigned vip_existence=0;
	unsigned vip_existence2=0;
	unsigned add_data=0;

	as = (struct l7vs_urla_service *)srv;
	ret = srv->pm->initialize(srv, conn, buf, *len, dest);
	if (ret != 0) {
		VANESSA_LOGGER_ERR("Could not initialize protomod");
		return -1;
	}

	/* check weather l7syncd_master is running or not if running it returns int 2 */
	process_state=check_process_state("/var/run/l7syncd_master.pid",1);
	
	if(process_state!=2)
	{
		VANESSA_LOGGER_ERR("l7syncd_master is not running");
	}

	if(process_state==2)
	{
		sema_id=initialize_sem(key_sem_ck);
		shmid=initialize_shm(key_shm_ck,sizeof(struct sync_info_st));
        	if (shmid == -1)
	        {
			VANESSA_LOGGER_ERR("urla shmget failed");
                	exit(EXIT_FAILURE);
	        }
	
		shared_memory = shmat(shmid, (void *)0, 0);
	        if (shared_memory == (void *)-1)
	        {
			VANESSA_LOGGER_ERR("urla shmat failed");
	        }
		sync_info=(struct sync_info_st *)shared_memory;
	}

	srvname=get_servicename(&srv->lsock->addr,1);

 	/* reading from the shared memory and filling the glist  */
        if(process_state==2)  
	{
		int i=0;
		int lvipindx=0;
		int lvipindx2=0;
	        semaphore_lock(sema_id);
                
		for (i=0;i<VIP_SZ;i++)
                {
			if(sync_info->standby_flag==0)
			{
				add_data=1;
				if ((strcmp(srvname,sync_info->ckal[i].vip)==0) && (strcmp("urla",sync_info->ckal[i].protomod)==0))
				{
					break;
				}
				else if(strcmp(sync_info->ckal[i].vip,"")==0)
				{
					strcpy(sync_info->ckal[i].vip,srvname);
					strcpy(sync_info->ckal[i].protomod,"urla");
					break;
				}
			}
		}
		for (i=0;i<VIP_SZ;i++)
		{
			if (strcmp(srvname,sync_info->ckal[i].vip)==0)
			{
				lvipindx=i;
				vip_existence=1;
				break;
			}
		}

		if(sync_info->standby_flag==1)
		{
			for (i=0;i<VIP_SZ;i++)
			{
				if (strcmp(srvname,sync_info->vp[i].vip)==0)
				{
					lvipindx2=i;
					vip_existence2=1;
					break;
				}
			}

			if((vip_existence==1) && (vip_existence2==1))
			{
				if(strcmp(sync_info->ckal[lvipindx].protomod,sync_info->vp[lvipindx2].protomod)==0)
					add_data=1;
			}
		}
		if(add_data==1)
		{

 			/* If status changed i.e l7syc daemon changed master to backup vice versa */
        		if(sync_info->status_flag[lvipindx]==1)  
        		{
                		int shi;
				if(strlen(sync_info->ckal[lvipindx].protomod)!=0)
				{
					for(shi=0;shi<SES_SZ;shi++)
        				{
						if(strlen(sync_info->ckal[lvipindx].cklist_ary[shi].pattern)!=0)
						{
                                			cl_temp = (struct cookie_list *)
                                        		calloc(1, sizeof(struct cookie_list));
                                			if (cl_temp == NULL) {
                                        			VANESSA_LOGGER_ERR("Could not allocate memory");
                                        			return -1;
                                			}

                                			memcpy(cl_temp,&sync_info->ckal[lvipindx].cklist_ary[shi],sizeof(struct cookie_list));
                                			as->list = g_list_append(as->list, cl_temp);
						}
					}
				}
        			sync_info->status_flag[lvipindx]=0; 
			}
		}
       		semaphore_unlock(sema_id);
        }

	t = time((time_t *)0);

	ret = l7vs_protomod_urla_del_list(as, t);
	if (ret != 0) {
		VANESSA_LOGGER_ERR("Could not del list");
		return -1;
	}

	if (as->search_len <= *len) {
		temp2 = buf[as->search_len];
		buf[as->search_len] = '\0';
		temp = l7vs_protomod_urla_search(buf, "\r\n\r\n", as->key);
		buf[as->search_len] = temp2;
	} else {
		temp = l7vs_protomod_urla_search(buf, "\r\n\r\n", as->key);
	}

	if (temp != 0){
		if ((ul_temp = (struct url_list *)calloc(1, sizeof(*ul_temp))) == NULL){
			VANESSA_LOGGER_ERR("Could not allocate memory");
                	return -1;
		}
		key_len = l7vs_protomod_urla_str_len(temp, '\r');
		strncpy(ul_temp->pattern, temp, key_len);
		ul_temp->pattern[key_len] = '\0';
		ul_temp->date = (t + as->timeout);
	
		l = g_list_first(as->list);

		while (l != NULL) {
			ul = (struct url_list *)l->data;
			if (ul->date < ul_temp->date) {
				if (strcmp(ul->pattern, ul_temp->pattern) == 0) {
					ul_temp->raddr = ul->raddr;
					d.addr = ul_temp->raddr;
					*dest = &d;
					ret = l7vs_protomod_urla_dest_check(srv, conn, dest);
					if (ret == 1) {
						ul_temp->raddr = (*dest)->addr;
					}
					as->list = g_list_remove(as->list ,ul);
					free(ul);
					if (g_list_length(as->list) == 0) {
						as->list = g_list_append(as->list, ul_temp);
						goto OUT;
					}
					for (l = g_list_last(as->list); l != NULL; l = g_list_previous(l)) {
						ul = (struct url_list *)l->data;
						if (ul->date < ul_temp->date) {
							f = g_list_index(as->list, ul);
							as->list = g_list_insert(as->list, ul_temp, (f+1));
							goto OUT;
						}
					}
					as->list = g_list_prepend(as->list, ul_temp);
					goto OUT;				
				}
				l = g_list_next(l);
			} else {
				if (f == -1) {
					f = g_list_index(as->list, ul);
				}
				if (strcmp(ul->pattern, ul_temp->pattern) == 0) {
					ul_temp->raddr = ul->raddr;
					d.addr = ul_temp->raddr;
					*dest = &d;
					ret = l7vs_protomod_urla_dest_check(srv, conn, dest);
					if (ret == 1) {
						ul_temp->raddr = (*dest)->addr;
					}
					as->list = g_list_remove(as->list, ul);
					free(ul);
					as->list = g_list_insert(as->list, ul_temp, f);
					goto OUT;
				}
				l = g_list_next(l);
			}
		}

		if (g_list_length(as->list) > as->max_list){	
			VANESSA_LOGGER_ERR("Could not save url_list");
			l = g_list_first(as->list);
			ul = (struct url_list *)l->data;
			as->list = g_list_remove(as->list, ul);
			free(ul);
		}

		*dest = srv->scheduler->schedule(srv, conn);
		ul_temp->raddr = (*dest)->addr;

		if (f == -1) {
			as->list = g_list_append(as->list, ul_temp);
		} else {
			as->list = g_list_insert(as->list, ul_temp, f);
		}
	}
OUT:
	if (shmdt(shared_memory) == -1)
	{
		VANESSA_LOGGER_ERR("urla shmdt failed");
	}

	*tcps = 0;
	ret = srv->pm->finalize(srv, conn, buf, *len, dest, as->reschedule);
	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not finalize protomod");
		return -1;
	}
	return 0;
}

static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
	char *buf, size_t *len)
{
        struct l7vs_urla_service *as;
	struct url_list *ul, *ul_temp;
	time_t t;
	char *temp, *temp2, *temp3, temp5;
	int ret, f = -1, key_len;
	GList *l;

	as = (struct l7vs_urla_service *)srv;

	int retval=0;
	int tindx=0;
	char * srvname;
	void *shared_memory=(void *)0;
	int sema_id,shmid;
	struct sync_info_st *sync_info;
	struct cookie_list *ckl;	
	unsigned vip_existence=0;
	unsigned vip_existence2=0;
	unsigned add_data=0;

	if(process_state==2)
	{
		sema_id=initialize_sem(key_sem_ck);
		shmid=initialize_shm(key_shm_ck,sizeof(struct sync_info_st));
        	if (shmid == -1)
	        {
			VANESSA_LOGGER_ERR("urla shmget failed");
                	exit(EXIT_FAILURE);
	        }

		shared_memory = shmat(shmid, (void *)0, 0);
	        if (shared_memory == (void *)-1)
	        {
			VANESSA_LOGGER_ERR("urla shmat failed");
	        }
		sync_info=(struct sync_info_st *)shared_memory;
	}

	srvname=get_servicename(&srv->lsock->addr,1);

	t = time((time_t *)0);

	ret = l7vs_protomod_urla_del_list(as, t);
	if (ret != 0) {
		VANESSA_LOGGER_ERR("Could not del list");
		retval=-1;
		goto OUT;
	}

	if (as->search_len <= *len) {
		temp5 = buf[as->search_len];
		buf[as->search_len] = '\0';
	}
	temp = strchr(buf, '<');
	while (temp != NULL) {
		temp2 = l7vs_protomod_urla_toupper_search(temp, ">", "HREF=");
		if (temp2 != 0) {
			if (temp2[0] == '"') temp2++;
			temp2 = l7vs_protomod_urla_search(temp2, " ", as->key);
			if (temp2 != 0) {
				key_len = l7vs_protomod_urla_str_len(temp2, '"');
				ret = l7vs_protomod_urla_add_list(as, conn, temp2, key_len, t);
				if (ret != 0) {
					VANESSA_LOGGER_ERR("Could not add list");
					retval=-1;
					goto OUT;
				}
			}
		}

		temp3 = l7vs_protomod_urla_toupper_search(temp, ">", "TYPE=");
		if (temp3 != 0) {
			if (temp3[0] == '"') temp3++;
			temp3 = l7vs_protomod_urla_toupper_search(temp3, ">", "NAME=");
			if (temp3 != 0) {
				if (temp3[0] == '"') temp3++;
				if (strncmp(temp3, as->key, strlen(as->key)) == 0) {
					if ((temp3[strlen(as->key)] == '"') || (temp3[strlen(as->key)] == ' ')) {
						temp3 += size(as->key);
						temp3 = l7vs_protomod_urla_toupper_search(temp3, ">", "VALUE=");
						if (temp3 != 0) {
							if (temp3[0] == '"') temp3++;
							key_len = l7vs_protomod_urla_str_len(temp3, '"');
							ret = l7vs_protomod_urla_add_list(as, conn, temp3, key_len, t);
							if (ret != 0) {
								VANESSA_LOGGER_ERR("Could not add list");
								retval=-1;
								goto OUT;
							}
						}
					}
				}
			}
		}
		temp = strchr(temp, '>');
		temp = strchr(temp, '<');
	}

	if (as->search_len <= *len) {
		buf[as->search_len] = temp5;
	}

OUT:
	VANESSA_LOGGER_INFO_UNSAFE("URLA glist length %d\n",g_list_length(as->list));
        if(process_state==2)  
	{
		int i=0;
		int lvipindx=0;
		semaphore_lock(sema_id);

                for (i=0;i<VIP_SZ;i++)
                {
			if (strcmp(srvname,sync_info->ckal[i].vip)==0)
			{
				lvipindx=i;
			        break;
		        }
		}
		strcpy(sync_info->ckal[lvipindx].cookie_name,as->key);

	        // Reading the session information and writing into the shared memory
	        for(l=g_list_first(as->list);l!=NULL;l=g_list_next(l))
	        {
		        ckl=(struct cookie_list *)l->data;
		        sync_info->ckal[lvipindx].cklist_ary[tindx]=*ckl;
			if(tindx<SES_SZ)
				tindx++;
	        }
		semaphore_unlock(sema_id);

		if (shmdt(shared_memory) == -1)
	        {
			VANESSA_LOGGER_ERR("urla shmdt failed");
	        }
	}
 
	return retval;
}

static void
destroy(struct l7vs_service *srv)
{
        free(srv);
}

static struct l7vs_service_arg *
service_arg(struct l7vs_service *srv)
{
        struct l7vs_urla_service *as;
        struct l7vs_urla_service_arg *aa;
	char temp[512];
	char *ptr;

        as = (struct l7vs_urla_service *)srv;

        aa = (struct l7vs_urla_service_arg *)malloc(sizeof(*aa));
        memset(aa, 0, sizeof(*aa));
        if (aa == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service_arg *)aa;
        }

        aa->arg.len = sizeof(*aa);
        strcpy(aa->key, as->key);
        aa->search_len = as->search_len;
        aa->timeout = as->timeout;
        aa->reschedule = as->reschedule;
	aa->max_list = as->max_list;
	aa->list = as->list;

	ptr = temp;
	aa->arg.reschedule = as->reschedule;
	sprintf(ptr, "--key-name %s\0", as->key);
	strcpy(aa->arg.protomod_key_string, ptr);
	sprintf(ptr + strlen(ptr), " --search-len %d --timeout %d --max-list %d\0", as->search_len, as->timeout, as->max_list);
	strcpy(aa->arg.protomod_opt_string, ptr);

        return (struct l7vs_service_arg *)aa;
}

static int
parse(struct l7vs_service_arg *arg, int argc, char *argv[])
{
        static struct option opt[] = {
                {"key-name",  	required_argument,      NULL, 'K'},
                {"search-len",	required_argument,      NULL, 'L'},
		{"timeout",	required_argument,	NULL, 'T'},
		{"max-list",	required_argument,	NULL, 'M'},
		{"reschedule",	no_argument,		NULL, 'F'},
		{"no-reschedule",no_argument,		NULL, 'N'},
                {NULL,         	0,                 	NULL, 0}
        };
        struct l7vs_urla_service_arg *aa = (struct l7vs_urla_service_arg *)arg;
        int c;
	int ret;
	int c1 = 0, c2 = 0, c3 = 0, c4 = 0, c5 = 0;

        optind = 0;

	char *srvname;
	void *shared_memory=(void *)0;
	int sema_id,shmid;
	struct sync_info_st *sync_info;
	int i=0;

	srvname=get_servicename(&arg->addr,1);

	/* check weather l7syncd_master is running or not if running it returns 2 */
	process_state=check_process_state("/var/run/l7syncd_master.pid",1);
	
	if(process_state!=2)
	{
		VANESSA_LOGGER_ERR("l7syncd_master is not running");
	}

	if(process_state==2)
	{
		sema_id=initialize_sem(key_sem_ck);
		shmid=initialize_shm(key_shm_ck,sizeof(struct sync_info_st));
        	if (shmid == -1)
	        {
			VANESSA_LOGGER_ERR("urla shmget failed");
                	exit(EXIT_FAILURE);
	        }

		shared_memory = shmat(shmid, (void *)0, 0);
	        if (shared_memory == (void *)-1)
	        {
			VANESSA_LOGGER_ERR("urla shmat failed");
	        }
		sync_info=(struct sync_info_st *)shared_memory;
	}

        while ((c = getopt_long(argc, argv, "K:L:T:M:", opt, NULL)) != -1) {
                switch (c) {
                case 'K':
                        if (strlen(optarg) >= sizeof(aa->key)) {
                                VANESSA_LOGGER_ERR_UNSAFE(
                                                "%s: path too long",
                                                optarg);
                                return -1;
                        }
                        strcpy(aa->key, optarg);
			c1++;
                        break;
                case 'L':
			if (atoi(optarg) < 0) {
				VANESSA_LOGGER_ERR("You can't specify less than 0");
			}
			aa->search_len = atoi(optarg);
			c2++;
                        break;
                case 'T':
			if (atoi(optarg) < 0) {
				VANESSA_LOGGER_ERR("You can't specify less than 0");
			}
			aa->timeout = atoi(optarg);
			c3++;
                        break;
		case 'M':
			if (atoi(optarg) < 1) {
				VANESSA_LOGGER_ERR("You can't specify less than 1");
			}
			aa->max_list = atoi(optarg);
			c4++;
			break;
		case 'F':
			aa->reschedule = 1;
			c5++;
			break;
		case 'N':
			aa->reschedule = 0;
			c5++;
			break;
                default:
                        return -1;
                }
        }

	if (c5 == 2) {
		VANESSA_LOGGER_ERR("You should choose either of reschdule or no-reschedule");
		return -1;
	}
	if (c5 == 0) {
		aa->reschedule = 1;
	}

	if (c2 == 0) {
		aa->search_len = 65536;
	}

	if (c3 == 0) {
		aa->timeout = 86400;
	}

	if (c4 == 0) {
		aa->max_list = 1000;
	}

	if (c1 == 1) {

	if(process_state==2)
	{
	        semaphore_lock(sema_id);
                
		if(sync_info->standby_flag==1)
                {
			for (i=0;i<VIP_SZ;i++)
			{
				if ((strcmp(srvname,sync_info->vp[i].vip)==0) && (strcmp("urla",sync_info->vp[i].protomod)==0))
				{
				        break;
			        }
				else if(strcmp(sync_info->vp[i].vip,"")==0)
				{
					strcpy(sync_info->vp[i].vip,srvname);
					strcpy(sync_info->vp[i].protomod,"urla");
					break;
				}
			}
		}
		semaphore_unlock(sema_id);
		if (shmdt(shared_memory) == -1)
	        {
			VANESSA_LOGGER_ERR("urla shmdt failed");
	        }
	}	

		return 0;
	} else {
		VANESSA_LOGGER_ERR("you must write only key");
		return -1;
	}
}

char *
l7vs_protomod_urla_search(const char *s1, const char *s2, const char *s3)
{
        int i, j;

        if (*s2 == 0 || *s3 == 0) {
                return 0;
        }

        while (*s1) {
                i = 0;
                while (1) {
                        if (s2[i] == 0) return 0;
                        if (s2[i] != s1[i]) break;
                        else i++;
                }

                j = 0;
                if ((s1[0] == '?') || s1[0] == '&') {
                        while (1) {
                                if (s3[j] == 0){
					if (s1[j +1 ] == '=') return (char *) (s1 + j + 2);
					else break;
				}
                                if ((s1[j + 1] == '?') || (s1[j + 1] == '&')) break;
                                if (s3[j] != s1[j + 1]) break;
                                else j++;
                        }
                }
                s1++;
        }
        return 0;
}

char *
l7vs_protomod_urla_toupper_search(const char *s1, const char *s2, const char *s3)
{
	int i, j;
	char c0, c1, c2, c3;

	if (*s2 == 0 || *s3 == 0) {
		return 0;
	}

	while (*s1) {
		i = 0; j = 0;
		c0 = toupper(s1[i]);
		c1 = toupper(s1[j]);
		c2 = toupper(s2[i]);
		c3 = toupper(s3[j]);

		while (1) {
			if (s3[j] == 0) return (char *) (s1 + j);
			if (s2[i] == 0) return 0;

			if (c2 != c0 && c3 != c1) break;
			if (c2 == c0) {
				i++;
				c0 = toupper(s1[i]);
				c2 = toupper(s2[i]);
			}
			if (c3 == c1) {
				j++;
				c1 = toupper(s1[j]);
				c3 = toupper(s3[j]);
			}
		}
		s1++;
	}
	return 0;
}

int
l7vs_protomod_urla_del_list(struct l7vs_urla_service *as, time_t t)
{
	struct url_list *ul;
	GList *l;

	l = g_list_first(as->list);

	while (l != NULL) {
		ul = (struct url_list *)l->data;
		if (ul->date < t) {
			l = g_list_next(l);
			as->list = g_list_remove(as->list, ul);
			free(ul);
		} else {
			return 0;
		}
	}	
	return 0;
}

int
l7vs_protomod_urla_add_list(struct l7vs_urla_service *as, struct l7vs_conn *conn, char *match, int key_len, time_t t)
{
	struct url_list *ul, *ul_temp;
	GList *l;
	int f = -1;

	if ((ul_temp = (struct url_list *)calloc(1, sizeof(*ul_temp))) == NULL){
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return -1;
	}
	strncpy(ul_temp->pattern ,match ,key_len);
	ul_temp->pattern[key_len] = '\0';
	ul_temp->raddr = conn->dest->addr;
	ul_temp->date = (t + as->timeout);

	l = g_list_first(as->list);
	
	while (l != NULL) {
		ul = (struct url_list *)l->data;
		if (ul->date < ul_temp->date) {
			if (strcmp(ul->pattern, ul_temp->pattern) == 0) {
				as->list = g_list_remove(as->list ,ul);
				free(ul);
				if (g_list_length(as->list) == 0) {	
					as->list = g_list_append(as->list, ul_temp);
					return 0;;
				}
				for (l = g_list_last(as->list); l != NULL; l = g_list_previous(l)) {
					ul = (struct url_list *)l->data;
					if (ul->date < ul_temp->date) {
						f = g_list_index(as->list, ul);
						as->list = g_list_insert(as->list, ul_temp, (f+1));
						return 0;
					}
				}				
				as->list = g_list_prepend(as->list, ul_temp);
				return 0;
			}
			l = g_list_next(l);
		} else {
			if (f == -1) {
				f = g_list_index(as->list, ul);
			}
			if (strcmp(ul->pattern, ul_temp->pattern) == 0) {
				as->list = g_list_remove(as->list, ul);
				free(ul);
				as->list = g_list_insert(as->list, ul_temp, f);
				return 0;
			}
			l = g_list_next(l);
		}
	}

	if (g_list_length(as->list) > as->max_list){	
		VANESSA_LOGGER_ERR("Could not save url_list");
		l = g_list_first(as->list);
		ul = (struct url_list *)l->data;
		as->list = g_list_remove(as->list, ul);
		free(ul);
	}

	if (f == -1) as->list = g_list_append(as->list, ul_temp);
	else as->list = g_list_insert(as->list, ul_temp, f);
	
	return 0;
}	

int
l7vs_protomod_urla_str_len(const char *s1, const char s2)
{
	int i=0;

	while(*s1){
		if (s1[0] == ' ')
			return i;
		if (s1[0] == '?')
			return i;
		if (s1[0] == '&')
			return i;
		if (s1[0] == s2)
			return i;
		s1++;
		i++;
	}
	return i;
}

int
l7vs_protomod_urla_dest_check(struct l7vs_service *srv, struct l7vs_conn *conn, struct l7vs_dest **dest)
{
	GList *l;
	struct l7vs_dest *d;
	if ((*dest) != NULL) {
		for (l = g_list_first(srv->dest_list); l != NULL; l = g_list_next(l)) {
			d = (struct l7vs_dest *)l->data;
			if ((d->addr.sin_addr.s_addr == (*dest)->addr.sin_addr.s_addr) &&
			    (d->addr.sin_port == (*dest)->addr.sin_port)) {
				*dest = d;
				return 0;
			}
		}
	}
	*dest = srv->scheduler->schedule(srv, conn);
	if ((*dest) == NULL) {
		VANESSA_LOGGER_ERR("realserver nonexistence");
		return -1;
	}
	return 1;
}

/* reference from l7vsadm.c */
static char *get_servicename(struct sockaddr_in *addr, int isnumeric) 
{
        char *sname;
        char hostname[40], portname[20];
        int flags;
        int ret;

        flags = 0;
        if (isnumeric) {
                flags = NI_NUMERICHOST | NI_NUMERICSERV;
        }

        ret = getnameinfo((struct sockaddr *)addr, sizeof(*addr),
                          hostname, sizeof(hostname),
                          portname, sizeof(portname), flags);
        if (ret != 0) {
                VANESSA_LOGGER_ERR_UNSAFE("getnameinfo: %s",
                                          gai_strerror(ret));
                return NULL;
        }

        ret = asprintf(&sname, "%s:%s", hostname, portname);
        if (ret < 0) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return NULL;
        }

        return sname;
}

