#define _GNU_SOURCE
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fnmatch.h>
#include <getopt.h>
#include <glib.h>
#include <time.h>
#include <ctype.h>
#include <math.h>
#include "vanessa_logger.h"
#include "l7vs.h"

struct l7vs_crewrite_service {
        struct l7vs_service srv;
	char cookie_name[129];
	int reschedule;
};

struct  l7vs_crewrite_service_arg {
	struct l7vs_service_arg arg;
	char cookie_name[129];
	int reschedule;
};

static void fini(void);
static struct l7vs_service *create(struct l7vs_service_arg *arg);
static struct l7vs_service_arg *create_sa(void);
static int compare(struct l7vs_service *s1, struct l7vs_service *s2);
static int match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
        char *buf, size_t *len, struct l7vs_dest **dest, int *tcps);
static int analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
        char *buf, size_t *len);
static void destroy(struct l7vs_service *srv);
static struct l7vs_service_arg *service_arg(struct l7vs_service *srv);
static int parse(struct l7vs_service_arg *arg, int argc, char *argv[]);

char *l7vs_protomod_crewrite_search(const char *s1, const char *s2, const char *s3);
int l7vs_protomod_crewrite_str_len(const char *s1, const char s2);

static struct l7vs_protomod crewrite_protomod = {
        NULL,                   /* handle */
        "crewrite",             /* modname */
	0,                      /* refcnt */
        create,                 /* create function */
        compare,                /* compare function */
        match_cldata,           /* match_cldata function */
	analyze_rsdata,		/* analyze_rsdata function */
        destroy,                /* destroy function */
        fini,                   /* fini function */
        create_sa,              /* create_sa function */
        service_arg,            /* service_arg function */
        parse,                  /* parse function */
};

struct l7vs_protomod *
init(void *handle)
{
        crewrite_protomod.handle = handle;
        return &crewrite_protomod;
}

static void 
fini(void)
{
        /* XXX  maybe some work needed */
}

static struct l7vs_service *
create(struct l7vs_service_arg *arg)
{
        struct l7vs_crewrite_service *crs;
        struct l7vs_crewrite_service_arg *cra;

        crs = (struct l7vs_crewrite_service *)calloc(1, sizeof(*crs));
        if (crs == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service *)crs;
        }

        cra = (struct l7vs_crewrite_service_arg *)arg;
        strcpy(crs->cookie_name, cra->cookie_name);
	crs->reschedule =cra->reschedule;
        return (struct l7vs_service *)crs;
}

static struct l7vs_service_arg *
create_sa(void)
{
        struct l7vs_crewrite_service_arg *cra;

        cra = (struct l7vs_crewrite_service_arg *)calloc(1, sizeof(*cra));
        if (cra == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service_arg *)cra;
        }

        cra->arg.len = sizeof(*cra);
        strcpy(cra->arg.protomod, crewrite_protomod.modname);

        return (struct l7vs_service_arg *)cra;
}

static int
compare(struct l7vs_service *s1, struct l7vs_service *s2)
{
        struct l7vs_crewrite_service *c1, *c2;

        c1 = (struct l7vs_crewrite_service *)s1;
        c2 = (struct l7vs_crewrite_service *)s2;

        return strcmp(c1->cookie_name, c2->cookie_name);

}

static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *buf, size_t *len, struct l7vs_dest **dest, int *tcps)
{
        struct l7vs_crewrite_service *crs;
	struct l7vs_dest *d;
	char *temp, test[11];
	int ret;

	crs = (struct l7vs_crewrite_service *)srv;

	ret = srv->pm->initialize(srv, conn, buf, *len, dest);
	if (ret != 0) {
		VANESSA_LOGGER_ERR("Could not initialize protomod");
		return -1;
	}

	temp = l7vs_protomod_crewrite_search(buf, "\r\n\r\n", "Cookie:");
	if (temp == 0) goto OUT;

	do {
		temp = l7vs_protomod_crewrite_search(temp, "\r\n", crs->cookie_name);
		if (temp == 0) goto OUT;
	} while (temp[0] != '=');

	temp++;

	if (temp !=0) {
		ret = l7vs_protomod_crewrite_str_len(temp, '\r');
		if (ret == 15) {
			d = (struct l7vs_dest *)calloc(1, sizeof(*d));
			if (d == NULL) {
				VANESSA_LOGGER_ERR("Could not allocate memory");
				return -1;
			}

			strncpy(test, temp, 10);
			test[10] = '\0';
			d->addr.sin_addr.s_addr = (u_long)(strtoul(test, NULL, 10));

			temp += 10;
			strncpy(test, temp, 5);
			test[5] = '\0';
			d->addr.sin_port = (u_short)(atoi(test));

			*dest =d;

		} else {
			VANESSA_LOGGER_INFO("Cookie value is not 15 byte");
		}	
	}
	
OUT:
	*tcps = 0;
	ret = srv->pm->finalize(srv, conn, buf, *len, dest, crs->reschedule);
	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not finalize protomod");
		return -1;
	}

	return 0;
}

static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
	char *buf, size_t *len)
{
        struct l7vs_crewrite_service *crs;
	char *temp ,*temp2;
	int ret, l = 0;

	crs = (struct l7vs_crewrite_service *)srv;

	temp = l7vs_protomod_crewrite_search(buf, "\r\n\r\n", "Set-Cookie:");
	while (1) {
		if (temp == 0) return 0;
		temp2 = l7vs_protomod_crewrite_search(temp, "\r\n", crs->cookie_name);
		if ((temp2 != 0) && (temp2[0] == '=')) {
			temp = temp2 + 1;
			break;
		}
		else temp = l7vs_protomod_crewrite_search(temp, "\r\n\r\n", "Set-Cookie:");
	}

	if (temp !=0) {
		while (temp[l] == '0') {
			l++;
		}
		if (l == 15) {
 			sprintf(temp, "%010lu%05u%s", conn->dest->addr.sin_addr.s_addr, conn->dest->addr.sin_port, temp + 15);
		} else {
			VANESSA_LOGGER_INFO("Server must have 15 byte size of Cookie value");
		}
	}
	return 0;
}

static void
destroy(struct l7vs_service *srv)
{
        free(srv);
}

static struct l7vs_service_arg *
service_arg(struct l7vs_service *srv)
{
        struct l7vs_crewrite_service *crs;
        struct l7vs_crewrite_service_arg *cra;
	char temp[512];
	char *ptr;

        crs = (struct l7vs_crewrite_service *)srv;

        cra = (struct l7vs_crewrite_service_arg *)malloc(sizeof(*cra));
        if (cra == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service_arg *)cra;
        }

        cra->arg.len = sizeof(*cra);
        strcpy(cra->cookie_name, crs->cookie_name);
        cra->reschedule = crs->reschedule;

	ptr = temp;
	cra->arg.reschedule = crs->reschedule;
	sprintf(ptr, "--cookie-name %s\0", crs->cookie_name);
	strcpy(cra->arg.protomod_key_string, ptr);
	sprintf(ptr + strlen(ptr), "\0");
	strcpy(cra->arg.protomod_opt_string, ptr);

        return (struct l7vs_service_arg *)cra;
}

static int
parse(struct l7vs_service_arg *arg, int argc, char *argv[])
{
        static struct option opt[] = {
                {"cookie-name", required_argument,      NULL, 'C'},
		{"reschedule",	no_argument,		NULL, 'F'},
		{"no-reschedule",no_argument,		NULL, 'N'},
                {NULL,         	0,                 	NULL, 0}
        };
        struct l7vs_crewrite_service_arg *cra = (struct l7vs_crewrite_service_arg *)arg;
        int c;
	int ret;
	int c1 = 0, c2 = 0;

        optind = 0;
        while ((c = getopt_long(argc, argv, "C:", opt, NULL)) != -1) {
                switch (c) {
                case 'C':
                        if (strlen(optarg) >= sizeof(cra->cookie_name)) {
                                VANESSA_LOGGER_ERR_UNSAFE(
                                                "%s: path too long",
                                                optarg);
                                return -1;
                        }
                        strcpy(cra->cookie_name, optarg);
			c1++;
                        break;
		case 'F':
			cra->reschedule = 1;
			c2++;
			break;
		case 'N':
			cra->reschedule = 0;
			c2++;
			break;
                default:
                        return -1;
                }
        }

	if (c2 == 2) {
		VANESSA_LOGGER_ERR("You should choose either of reschdule or no-reschedule");
		return -1;
	}
	if (c2 == 0) {
		cra->reschedule = 1;
	}

	if (c1 == 1) {
		return 0;
	} else {
		VANESSA_LOGGER_ERR("you must write only cookie-name");
		return -1;
	}
}

char *
l7vs_protomod_crewrite_search(const char *s1, const char *s2, const char *s3)
{
        int i, j;
	char c0, c1, c2, c3;

        if (*s2 == 0 || *s3 == 0) {
                return 0;
        }

        while (*s1) {
                i = 0; j = 0;
		c0 = toupper(s1[i]);
		c1 = toupper(s1[j]);
		c2 = toupper(s2[i]);
		c3 = toupper(s3[j]);

                while (1) {
                        if (s3[j] == 0) return (char *) (s1 + j);
                        if (s2[i] == 0) return 0;

			if (c2 != c0 && c3 != c1) break;
			if (c2 == c0) {
				i++;
				c0 = toupper(s1[i]);
				c2 = toupper(s2[i]);
			}
			if (c3 == c1) {
				j++;
				c1 = toupper(s1[j]);
				c3 = toupper(s3[j]);
			}
                }
		s1++;
	}
        return 0;
}

int
l7vs_protomod_crewrite_str_len(const char *s1, const char s2)
{
	int i=0;

	while(*s1){
		if (s1[0] == ' ')
			return i;
		if (s1[0] == ';')
			return i;
		if (s1[0] == '"')
			return i;
		if (s1[0] == '\t')
			return i;
		if (s1[0] == s2)
			return i;
		s1++;
		i++;
	}
	return i;
}
