#define _GNU_SOURCE
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fnmatch.h>
#include <getopt.h>
#include <glib.h>
#include <time.h>
#include <ctype.h>
#include <math.h>
#include <stdlib.h>
#include "vanessa_logger.h"
#include "l7vs.h"

struct l7vs_cinsert_service {
        struct l7vs_service srv;
	char cookie_name[129];
	int cookie_expire;
	int reschedule;
};

struct  l7vs_cinsert_service_arg {
	struct l7vs_service_arg arg;
	char cookie_name[129];
	int cookie_expire;
	int reschedule;
};

static void fini(void);
static struct l7vs_service *create(struct l7vs_service_arg *arg);
static struct l7vs_service_arg *create_sa(void);
static int compare(struct l7vs_service *s1, struct l7vs_service *s2);
static int match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
        char *buf, size_t *len, struct l7vs_dest **dest, int *tcps);
static int analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
        char *buf, size_t *len);
static void destroy(struct l7vs_service *srv);
static struct l7vs_service_arg *service_arg(struct l7vs_service *srv);
static int parse(struct l7vs_service_arg *arg, int argc, char *argv[]);

char *l7vs_protomod_cinsert_search(const char *s1, const char *s2, const char *s3);
time_t l7vs_protomod_cinsert_timeout(int timeout);
char * l7vs_protomod_cinsert_bufadd(char *buf, int offset, char *str, int length);
char * l7vs_protomod_cinsert_bufshift(char *buf, int offset, int shift_len, int buf_len);
char * l7vs_protomod_cinsert_expire(time_t expire_t);
int l7vs_protomod_cinsert_httpstatusck(char *buf);


#define EXPIRE_MAXSIZE 64
char expire_c[EXPIRE_MAXSIZE];

static struct l7vs_protomod cinsert_protomod = {
        NULL,                   /* handle */
        "cinsert",              /* modname */
        0,                      /* refcnt */
        create,                 /* create function */
        compare,                /* compare function */
        match_cldata,           /* match_cldata function */
	analyze_rsdata,		/* analyze_rsdata function */
        destroy,                /* destroy function */
        fini,                   /* fini function */
        create_sa,              /* create_sa function */
        service_arg,            /* service_arg function */
        parse,                  /* parse function */
};

struct l7vs_protomod *
init(void *handle)
{
        cinsert_protomod.handle = handle;
        return &cinsert_protomod;
}

static void 
fini(void)
{
        /* XXX  maybe some work needed */
}

static struct l7vs_service *
create(struct l7vs_service_arg *arg)
{
        struct l7vs_cinsert_service *cis;
        struct l7vs_cinsert_service_arg *cia;

        cis = (struct l7vs_cinsert_service *)calloc(1, sizeof(*cis));
        if (cis == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service *)cis;
        }

        cia = (struct l7vs_cinsert_service_arg *)arg;
        strcpy(cis->cookie_name, cia->cookie_name);
	cis->cookie_expire = cia->cookie_expire;
	cis->reschedule =cia->reschedule;
        return (struct l7vs_service *)cis;
}

static struct l7vs_service_arg *
create_sa(void)
{
        struct l7vs_cinsert_service_arg *cia;

        cia = (struct l7vs_cinsert_service_arg *)calloc(1, sizeof(*cia));
        if (cia == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service_arg *)cia;
        }

        cia->arg.len = sizeof(*cia);
        strcpy(cia->arg.protomod, cinsert_protomod.modname);

        return (struct l7vs_service_arg *)cia;
}

static int
compare(struct l7vs_service *s1, struct l7vs_service *s2)
{
        struct l7vs_cinsert_service *c1, *c2;

        c1 = (struct l7vs_cinsert_service *)s1;
        c2 = (struct l7vs_cinsert_service *)s2;

        return strcmp(c1->cookie_name, c2->cookie_name);
}

static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *buf, size_t *len, struct l7vs_dest **dest, int *tcps)
{
        struct l7vs_cinsert_service *cis;
	struct l7vs_dest *d;
	int ret, key_len, i;
	char *temp, test[11];
	u_long ad;
	u_short pt;

	cis = (struct l7vs_cinsert_service *)srv;

	ret = srv->pm->initialize(srv, conn, buf, *len, dest);
	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not initialize protomod");
		return -1;
	}
		
	temp = l7vs_protomod_cinsert_search(buf, "\r\n\r\n", "Cookie:");
	if (temp == 0) goto OUT;
	temp = l7vs_protomod_cinsert_search(temp, "\r\n", cis->cookie_name);
	if (temp == 0) goto OUT;
	if (temp[0] == '=') temp++;
	else goto OUT;

	d = (struct l7vs_dest *)calloc(1, sizeof(*d));
	if (d == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
        }

	strncpy(test, temp, 10);
	test[10] = '\0';
	d->addr.sin_addr.s_addr = (u_long)(strtoul(test, NULL, 10));

	temp += 10;
	strncpy(test, temp, 5);
	test[5] = '\0';
	d->addr.sin_port = (u_short)(atoi(test));

	*dest = d;

OUT:
	*tcps = 0;
	ret = srv->pm->finalize(srv, conn, buf, *len, dest, cis->reschedule);
	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not finalize protomod");
		return -1;
	}

	return 0;
}

static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
	char *buf, size_t *len)
{
        struct l7vs_cinsert_service *cis;
	char *temp, *temp2;
	char buf_temp[L7VS_PROTOMOD_MAX_ADD_BUFSIZE];
	int l = 0, cmp_temp;
	time_t time_time;
	char *str_time;
	char *ptr;

	cis = (struct l7vs_cinsert_service *)srv;

	cmp_temp = l7vs_protomod_cinsert_httpstatusck(buf);
	if (cmp_temp != 0) return 0;

	ptr = buf_temp;

	strncpy(ptr + l, "Set-Cookie: ", 12);
	l += 12;
	strncpy(ptr + l, cis->cookie_name, strlen(cis->cookie_name));
	l += strlen(cis->cookie_name);
	sprintf(ptr + l, "=%010lu%05u; expires=", conn->dest->addr.sin_addr.s_addr, conn->dest->addr.sin_port);
	l += 26;
	time_time = l7vs_protomod_cinsert_timeout(cis->cookie_expire);
	str_time = l7vs_protomod_cinsert_expire(time_time);
	strncpy(ptr + l, str_time, strlen(str_time));
	l += strlen(str_time);
	strncpy(ptr + l, "\r\n\0", 3);
	l += 2;

 	temp = l7vs_protomod_cinsert_search(buf, "\r\n\r\n", "\r\n");
	if (temp == 0) return 0;
	temp2 = l7vs_protomod_cinsert_bufshift(buf, (int)(temp - buf), l, *len);
	if (temp2 == NULL) return 0;
	temp = l7vs_protomod_cinsert_bufadd(temp, 0, buf_temp, l);

	*len += l;

	return 0;
}

static void
destroy(struct l7vs_service *srv)
{
        free(srv);
}

static struct l7vs_service_arg *
service_arg(struct l7vs_service *srv)
{
        struct l7vs_cinsert_service *cis;
        struct l7vs_cinsert_service_arg *cia;
	char temp[512];
	char *ptr;

        cis = (struct l7vs_cinsert_service *)srv;

        cia = (struct l7vs_cinsert_service_arg *)malloc(sizeof(*cia));
        if (cia == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return (struct l7vs_service_arg *)cia;
        }

        cia->arg.len = sizeof(*cia);
        strcpy(cia->cookie_name, cis->cookie_name);
        cia->cookie_expire = cis->cookie_expire;
        cia->reschedule = cis->reschedule;

	ptr = temp;
	cia->arg.reschedule = cis->reschedule;
	sprintf(ptr, "--cookie-name %s\0", cis->cookie_name);
	strcpy(cia->arg.protomod_key_string, ptr);
	sprintf(ptr + strlen(ptr), " --cookie-expire %d\0", cis->cookie_expire);
	strcpy(cia->arg.protomod_opt_string, ptr);

        return (struct l7vs_service_arg *)cia;
}

static int
parse(struct l7vs_service_arg *arg, int argc, char *argv[])
{
        static struct option opt[] = {
                {"cookie-name", required_argument,      NULL, 'C'},
		{"cookie-expire",required_argument,	NULL, 'E'},
		{"reschedule",	no_argument,		NULL, 'F'},
		{"no-reschedule",no_argument,		NULL, 'N'},
                {NULL,         	0,                 	NULL, 0}
        };
        struct l7vs_cinsert_service_arg *cia = (struct l7vs_cinsert_service_arg *)arg;
        int c;
	int ret;
	int c1 = 0, c2 = 0, c3 = 0;

        optind = 0;
        while ((c = getopt_long(argc, argv, "C:E:", opt, NULL)) != -1) {
                switch (c) {
                case 'C':
                        if (strlen(optarg) >= sizeof(cia->cookie_name)) {
                                VANESSA_LOGGER_ERR_UNSAFE(
                                                "%s: path too long",
                                                optarg);
                                return -1;
                        }
                        strcpy(cia->cookie_name, optarg);
			c1++;
                        break;
		case 'E':
			if (atoi(optarg) < 0) {
				VANESSA_LOGGER_ERR("You can't specify less than 0");
				return -1;
			}
			cia->cookie_expire = atoi(optarg);
			c2++;
			break;
		case 'F':
			cia->reschedule = 1;
			c3++;
			break;
		case 'N':
			cia->reschedule = 0;
			c3++;
			break;
                default:
                        return -1;
                }
        }

	if (c3 == 2) {
		VANESSA_LOGGER_ERR("You should choose either of reschdule or no-reschedule");
		return -1;
	}
	if (c3 == 0) {
		cia->reschedule = 0; 
	}

	if (c1 == 0) {
		strcpy(cia->cookie_name, "CookieName");
	}

	if (c2 == 0) {
		cia->cookie_expire = 86400;
	}
	return 0;
}

char *
l7vs_protomod_cinsert_search(const char *s1, const char *s2, const char *s3)
{
        int i, j;
	char c0, c1, c2, c3;

        if (*s2 == 0 || *s3 == 0) {
                return 0;
        }

        while (*s1) {
                i = 0; j = 0;
		c0 = toupper(s1[i]);
		c1 = toupper(s1[j]);
		c2 = toupper(s2[i]);
		c3 = toupper(s3[j]);

                while (1) {
                        if (s3[j] == 0) return (char *) (s1 + j);
                        if (s2[i] == 0) return 0;

			if (c2 != c0 && c3 != c1) break;
			if (c2 == c0) {
				i++;
				c0 = toupper(s1[i]);
				c2 = toupper(s2[i]);
			}
			if (c3 == c1) {
				j++;
				c1 = toupper(s1[j]);
				c3 = toupper(s3[j]);
			}
                }
		s1++;
	}
        return 0;
}

time_t
l7vs_protomod_cinsert_timeout(int timeout)
{
	time_t timer;

	timer = time((time_t *)0);

	timer += timeout;

	return timer;
}

char *
l7vs_protomod_cinsert_bufadd(char *buf, int offset, char *str, int length)
{
	memcpy(buf + offset, str, length);
	return (buf + length);
}

char *
l7vs_protomod_cinsert_bufshift(char *buf, int offset, int shift_len, int buf_len)
{
	char *temp;

	temp = (char *)calloc(1,(buf_len - offset));
	if (temp == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return NULL;
	}
	memcpy(temp, buf + offset, (buf_len - offset));
	memcpy(buf + offset + shift_len, temp, (buf_len - offset));

	free(temp);

	return (buf + shift_len);
}

char *
l7vs_protomod_cinsert_expire(time_t expire_t)
{
	struct tm *date;

	date = localtime(&expire_t);

	strftime(expire_c, EXPIRE_MAXSIZE, "%a, %d-%b-%Y %H:%M:%S GMT", date);

	return expire_c;
}

int
l7vs_protomod_cinsert_httpstatusck(char *buf)
{
	int i, temp;

	temp = strncmp("HTTP/", buf, 5);
        if (temp != 0) return 1;

        temp = isdigit(buf[5]);
        if (temp == 0) return 1;

        temp = strncmp(".", buf + 6, 1);
        if (temp != 0) return 1;

        temp = isdigit(buf[7]);
        if (temp == 0) return 1;


        temp = isgraph(buf[8]);
        if (temp != 0) return 1;

        for (i = 9; i <= 11; i++) {
                temp = isdigit(buf[i]);
                if (temp == 0) return 1;
        }

        temp = isgraph(buf[12]);
        if (temp != 0) return 1;

        temp = strncmp("4", buf + 9, 1);
        if (temp == 0) return 1;

        temp = strncmp("5", buf + 9, 1);
        if (temp == 0) return 1;

        return 0;
}
