/**********************************************************************
 * l7vs.h                                                   August 2005
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#ifndef L7VS_H
#define L7VS_H

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "tcps.h"

#define L7VS_MODNAME_LEN                16
#define L7VS_VERSION_STRING             VERSION         /* by autoconf */

#ifndef L7VS_MODULE_PATH
#define L7VS_MODULE_PATH        "."
#endif

/* I/O multiplexing support. */
#define L7VS_IOMUX_READ         0x0001
#define L7VS_IOMUX_WRITE        0x0002
#define L7VS_IOMUX_EXCEPT       0x0004

#define L7VS_IOMUX_LIST_UNCHANGED       0
#define L7VS_IOMUX_LIST_REMOVED_MYSELF  1
#define L7VS_IOMUX_LIST_REMOVED_OTHER   2


struct l7vs_iomux {
        int fd;
        int (*callback)(struct l7vs_iomux *iom, int flags);
        void *data;     /* user data */
};

/* listen socket datatype. accepts client connections. */
struct l7vs_lsock {
        struct sockaddr_in addr;
        u_int8_t proto;
        struct l7vs_iomux iom;
        int refcnt;
        GList *srv_list;
};

/* real-server datatype */
struct l7vs_dest {
        struct sockaddr_in addr;
        int weight;
        int nactive;
        int ninact;
};

/* connection management datatype */
#define L7VS_CLDATA_CHUNKSIZE           512
#define L7VS_CLDATA_BUFSIZE(L)          \
        (((L)->cldata_len + L7VS_CLDATA_CHUNKSIZE - 1) & \
         ~(L7VS_CLDATA_CHUNKSIZE - 1))
#define L7VS_CLDATA_REST(L)             \
        (L7VS_CLDATA_BUFSIZE(L) - (L)->cldata_len)

struct l7vs_conn {
        struct l7vs_lsock *lsock;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        int splice;
        struct sockaddr_in caddr;
        struct sockaddr_in raddr;
        struct sockaddr_in vaddr;
        struct sockaddr_in laddr;
        struct l7vs_iomux ciom;         /* client-side socket */
        struct l7vs_iomux riom;         /* realserver-side socket */
        u_int8_t proto;                 /* XXX  needed? */
        struct timeval timo;
#define L7VS_CONN_S_CL_CONNECTED        0
#define L7VS_CONN_S_RS_CONNECTING       1
#define L7VS_CONN_S_RS_CONNECTED        2
        u_int state;
        char *cldata;
        size_t cldata_len;
        size_t cldata_bufsize;
        struct tcps_tcpopt tcpopt;
        int cmss;
};

/* virtual service definition datatype */
struct l7vs_service {
        struct l7vs_lsock *lsock;
        GList *dest_list;
        GList *conn_list;
        struct l7vs_scheduler *scheduler;
        void *sched_data;
        struct l7vs_protomod *pm;

        /* ... and per-module match-data follows */
        char protomod_opt_string[512]; 
        char protomod_key_string[256];
        int reschedule;
};

/* vitual service definition, external representation. */
struct l7vs_service_arg {
        size_t len;                     /* structure size */
        struct sockaddr_in addr;        /* server address */
        u_int8_t proto;                 /* server transport protocol */
        int persist;                    /* persistency */
        int backlog;                    /* listen(2) backlog length */
        char protomod[L7VS_MODNAME_LEN];/* protocol module name */
        char schedmod[L7VS_MODNAME_LEN];/* scheduler */

        /* ... and protocol-module defined data follows */
        char protomod_opt_string[512];
        char protomod_key_string[256];
        int reschedule;
};

#define L7VS_PROTOMOD_MAX_ADD_BUFSIZE 256

/* protocol module datatype */
struct l7vs_protomod {
        void *handle;
        char modname[L7VS_MODNAME_LEN+1];
        int refcnt;

        /* member functions */
        struct l7vs_service *(*create)(struct l7vs_service_arg *);
        int (*compare)(struct l7vs_service *, struct l7vs_service *);
        int (*match_cldata)(struct l7vs_service *, struct l7vs_conn *, 
                            char *, size_t *, struct l7vs_dest **, int *);
        int (*analyze_rsdata)(struct l7vs_service *, struct l7vs_conn *,
                               char *, size_t *);
        void (*destroy)(struct l7vs_service *);
        void (*fini)(void);
        struct l7vs_service_arg *(*create_sa)(void);
        struct l7vs_service_arg *(*service_arg)(struct l7vs_service *);
        int (*parse)(struct l7vs_service_arg *, int, char **);
	int (*initialize)(struct l7vs_service *, struct l7vs_conn *, char *, size_t, struct l7vs_dest **);
	int (*finalize)(struct l7vs_service *, struct l7vs_conn *, char *, size_t, struct l7vs_dest **,int);
};

/* scheduler datatype */
struct l7vs_scheduler {
        void *handle;
        char modname[L7VS_MODNAME_LEN+1];
        int refcnt;
        
        struct l7vs_dest *(*schedule)(struct l7vs_service *,
                                      struct l7vs_conn *);
        void (*bind)(struct l7vs_service *);
        void (*unbind)(struct l7vs_service *);
        void (*fini)(void);
};

/* real-server datatype, external representation. */
struct l7vs_dest_arg {
        struct sockaddr_in addr;        /* real-server address */
        int weight;
        int nactive;
        int ninact;
};

extern char l7vs_module_path[];

/* configuration datatypes */
#define L7VS_CONFIG_ERR_INVALID_COMMAND 1
#define L7VS_CONFIG_ERR_NOMEM           2
#define L7VS_CONFIG_ERR_VS_EXISTS       3
#define L7VS_CONFIG_ERR_RS_EXISTS       4
#define L7VS_CONFIG_ERR_NOVS            5
#define L7VS_CONFIG_ERR_NORS            6
#define L7VS_CONFIG_ERR_NOSCHED         7
#define L7VS_CONFIG_ERR_NOSOCK          8

#ifndef L7VS_CONFIG_SOCK_PATH
#define L7VS_CONFIG_SOCK_PATH           "/var/run/l7vs"
#endif

#define L7VS_CONFIG_SOCKNAME            L7VS_CONFIG_SOCK_PATH "/l7vs"

enum l7vs_config_cmd {
        L7VS_CONFIG_LIST_VS,
        L7VS_CONFIG_LIST_RS,
        L7VS_CONFIG_ADD_VS,
        L7VS_CONFIG_ADD_RS,
        L7VS_CONFIG_DEL_VS,
        L7VS_CONFIG_DEL_RS,
        L7VS_CONFIG_EDIT_VS,
        L7VS_CONFIG_EDIT_RS,
        L7VS_CONFIG_FLUSH_VS,
        L7VS_CONFIG_NONE
};

struct l7vs_config_req_list_vs {
        enum l7vs_config_cmd cmd;
};

struct l7vs_config_req_list_rs {
        enum l7vs_config_cmd cmd;
        /* service_arg follows */
};

struct l7vs_config_req_operate_vs {
        enum l7vs_config_cmd cmd;
        /* service_arg follows */
};

struct l7vs_config_req_operate_rs {
        enum l7vs_config_cmd cmd;
        struct l7vs_dest_arg darg;
        /* service_arg follows */
};

struct l7vs_config_req_flush_vs {
        enum l7vs_config_cmd cmd;
};

union l7vs_config_req {
        enum l7vs_config_cmd cmd;
        struct l7vs_config_req_list_vs list_vs;
        struct l7vs_config_req_list_rs list_rs;
        struct l7vs_config_req_operate_vs operate_vs;
        struct l7vs_config_req_operate_rs operate_rs;
        struct l7vs_config_req_flush_vs flush_vs;
        char _buf[1024];
};

struct l7vs_config_rsp_list_vs {
        enum l7vs_config_cmd cmd;
        int code;
        int num;
        /* number of "num" l7vs_service_args follow... */
};

struct l7vs_config_rsp_list_rs {
        enum l7vs_config_cmd cmd;
        int code;
        int num;
        /* number of "num" l7vs_dest_args follow... */
};

struct l7vs_config_rsp_operate_vs {
        enum l7vs_config_cmd cmd;
        int code;
};

struct l7vs_config_rsp_operate_rs {
        enum l7vs_config_cmd cmd;
        int code;
};

struct l7vs_config_rsp_flush_vs {
        enum l7vs_config_cmd cmd;
        int code;
};

struct l7vs_config_rsp_unknown_cmd {
        enum l7vs_config_cmd cmd;
        int code;
};

/* 
 * Functions
 */
/* protocol module functions */
extern struct l7vs_protomod *l7vs_protomod_get(char *name);
extern void l7vs_protomod_put(struct l7vs_protomod *pmod);
extern struct l7vs_protomod *l7vs_protomod_lookup(char *modname);

/* scheduler functions */
extern struct l7vs_scheduler *l7vs_sched_get(char *name);
extern void l7vs_sched_put(struct l7vs_scheduler *sched);
extern void l7vs_sched_bind(struct l7vs_scheduler *sched,
                            struct l7vs_service *svc);
extern void l7vs_sched_unbind(struct l7vs_scheduler *sched,
                              struct l7vs_service *svc);

/* listen socket functions */
extern int l7vs_lsock_init(void);
extern void l7vs_lsock_fini(void);
extern struct l7vs_lsock *l7vs_lsock_get(struct sockaddr_in *sin,
                                        u_int8_t proto, int backlog);
extern void l7vs_lsock_put(struct l7vs_lsock *lsock);
extern struct l7vs_lsock *l7vs_lsock_table_lookup(struct sockaddr_in *sin,
                                                  u_int8_t proto);
extern void l7vs_lsock_add_service(struct l7vs_lsock *lsock,
                                   struct l7vs_service *srv);
extern void l7vs_lsock_remove_service(struct l7vs_lsock *lsock,
                                      struct l7vs_service *srv);
extern int l7vs_lsock_select_service(
                struct l7vs_lsock *lsock, 
                struct l7vs_conn *conn,
                char *buf,
                size_t len,
                struct l7vs_service **srv_ret,
                struct l7vs_dest **dest_ret,
                int *tcps);

/* service functions */
extern struct l7vs_service *l7vs_service_create(struct l7vs_service_arg *arg,
                                                int *err);
extern void l7vs_service_destroy(struct l7vs_service *srv);
extern struct l7vs_service *l7vs_service_lookup(struct l7vs_service_arg *arg);
extern int l7vs_service_add_dest(struct l7vs_service *srv,
                                 struct l7vs_dest_arg *darg);
extern int l7vs_service_remove_dest(struct l7vs_service *srv,
                                    struct l7vs_dest_arg *darg);
extern int l7vs_service_schedule(struct l7vs_service *srv, 
                                 struct l7vs_conn *conn);
extern int l7vs_service_establish(struct l7vs_service *srv, 
                                  struct l7vs_conn *conn);
extern struct l7vs_dest *l7vs_service_lookup_dest(struct l7vs_service *srv,
                                                  struct sockaddr_in *sin);
extern int l7vs_service_list_service_arg(struct l7vs_service_arg **sas,
                                         int *num);
extern int l7vs_service_list_dest_arg(struct l7vs_service *srv,
                                      struct l7vs_dest_arg **das);
extern void l7vs_service_register_conn(struct l7vs_service *srv,
                                       struct l7vs_conn *conn);
extern void l7vs_service_remove_conn(struct l7vs_service *srv,
                                     struct l7vs_conn *conn);
extern void l7vs_service_flush_all(void);

/* connection entry functions */
extern struct l7vs_conn *l7vs_conn_create(int lfd, struct l7vs_lsock *lsock);
extern void l7vs_conn_destroy(struct l7vs_conn *conn);
extern int l7vs_conn_recv_client(struct l7vs_conn *conn);
extern int l7vs_conn_connect_rs(struct l7vs_conn *conn, 
                                struct l7vs_dest *dest);
extern int l7vs_conn_splice(struct l7vs_conn *conn);
extern void l7vs_conn_close_csock(struct l7vs_conn *conn);
extern void l7vs_conn_close_rsock(struct l7vs_conn *conn);
extern int l7vs_conn_closed(struct l7vs_conn *conn);

/* dest functions */
extern struct l7vs_dest *l7vs_dest_create(struct sockaddr_in *sin, int weight);
extern void l7vs_dest_destroy(struct l7vs_dest *dest);
extern void l7vs_dest_to_arg(struct l7vs_dest *dest,
                             struct l7vs_dest_arg *darg);

/* module functions */
extern int l7vs_module_init(char *modpath);
extern void l7vs_module_fini(void);
extern void *l7vs_module_load(char *modname, char *type);
extern void l7vs_module_unload(void *handle);
extern void l7vs_module_register(GList **modlist, void *mod);
extern void l7vs_module_remove(GList **modlist, void *mod);
extern void *l7vs_module_lookup(GList *modlist, void *key, GCompareFunc cmp);

/* config functions */
extern int l7vs_config_init(void);
extern void l7vs_config_fini(void);

/* iomux functions */
extern void l7vs_iomux_init(void);
extern void l7vs_iomux_fini(void);
extern void l7vs_iomux_add(struct l7vs_iomux *iom, int flags);
extern void l7vs_iomux_remove(struct l7vs_iomux *iom);
extern void l7vs_iomux_change_flags(struct l7vs_iomux *iom, int flags);
extern int l7vs_iomux_select(struct timeval *timo);

#endif /* L7VS_H */
