/**********************************************************************
 * config.c                                                 August 2005
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdlib.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"

/* Added headers for sending messages to l7vsagent */
#include <stdio.h>
#include <errno.h>
#include <sys/msg.h>
#include <string.h>
#include <signal.h>

/* Added header for getnameinfo() */
#include <netdb.h>
#include "sync_session.h" 

static void shm_sem_remove(); 
static void stop_l7sync_daemons();

static int l7vs_config_callback(struct l7vs_iomux *iom, int flags);
static void l7vs_config_dispatch(union l7vs_config_req *req,
                                 struct sockaddr_un *addr);
static void l7vs_config_list_vs(union l7vs_config_req *arg,
                                struct sockaddr_un *addr);
static void l7vs_config_list_rs(union l7vs_config_req *arg,
                                struct sockaddr_un *addr);
static void l7vs_config_add_vs(union l7vs_config_req *arg,
                               struct sockaddr_un *addr);
static void l7vs_config_del_vs(union l7vs_config_req *arg,
                               struct sockaddr_un *addr);
static void l7vs_config_edit_vs(union l7vs_config_req *arg,
                                struct sockaddr_un *addr);
static void l7vs_config_add_rs(union l7vs_config_req *arg,
                               struct sockaddr_un *addr);
static void l7vs_config_del_rs(union l7vs_config_req *arg,
                               struct sockaddr_un *addr);
static void l7vs_config_edit_rs(union l7vs_config_req *arg,
                                struct sockaddr_un *addr);
static void l7vs_config_flush_vs(union l7vs_config_req *arg,
                                 struct sockaddr_un *addr);
static int l7vs_config_send_response(struct sockaddr_un *addr,
                                     struct iovec *iov,
                                     int iovlen);


/* For message queue / Trap */
int check_process_l7vsagent(char *);
static char * get_servicename(struct sockaddr_in *, int);
char trap_message[BUFSIZ]; /* Assuming IP address, Port and module name */
char *srvname, *srvname_temp;

const char l7vs_config_sockname[] = L7VS_CONFIG_SOCKNAME;

static struct l7vs_iomux l7vs_config_iomux;

int
l7vs_config_init(void)
{
        int s;
        int ret;
        int opt;
        mode_t mode;
        struct sockaddr_un addr;
        socklen_t len;

        if (sizeof(addr.sun_path) < sizeof(l7vs_config_sockname)) {
                VANESSA_LOGGER_ERR("Internal error."
                                   " The config socket name is too large.");
                return -1;
        }

        s = socket(PF_LOCAL, SOCK_DGRAM, 0);
        if (s < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("socket: %s", strerror(errno));
                return s;
        }

        opt = 1;
        ret = setsockopt(s, SOL_SOCKET, SO_PASSCRED, &opt, sizeof(opt));
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("setsockopt SO_PASSCRED: %s",
                                          strerror(errno));
                close(s);
                return ret;
        }

        memset(&addr, 0, sizeof(addr));
        addr.sun_family = AF_LOCAL;
        strcpy(addr.sun_path, l7vs_config_sockname);
        len = SUN_LEN(&addr);

        mode = umask(0077);
        if (mode < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("umask: %s", strerror(errno));
                close(s);
                return -1;
        }

        ret = bind(s, (struct sockaddr *)&addr, len);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("bind: %s", strerror(errno));
                if (errno == EADDRINUSE) {
                        VANESSA_LOGGER_ERR("Config socket already exists. "
                                           "You could remove it if"
                                           " the l7vs daemon is not running");
                }
                close(s);
                umask(mode);
                return ret;
        }
        umask(mode);

        l7vs_config_iomux.fd = s;
        l7vs_config_iomux.data = NULL;
        l7vs_config_iomux.callback = l7vs_config_callback;
        l7vs_iomux_add(&l7vs_config_iomux, L7VS_IOMUX_READ);

        return 0;
}

void
l7vs_config_fini(void)
{
        l7vs_iomux_remove(&l7vs_config_iomux);
        close(l7vs_config_iomux.fd);
        unlink(l7vs_config_sockname);
	stop_l7sync_daemons();
	shm_sem_remove();
}

static int
l7vs_config_callback(struct l7vs_iomux *iom, int flags)
{
        struct msghdr msg;
        struct sockaddr_un addr;
        struct iovec iov;
        struct ucred *cred;
        struct cmsghdr *cmsg;
        unsigned char cbuf[CMSG_LEN(sizeof(struct ucred))];
        union l7vs_config_req req;
        int ret;

        if ((flags & L7VS_IOMUX_READ) == 0) {
                return L7VS_IOMUX_LIST_UNCHANGED;
        }

        cmsg = (struct cmsghdr *)cbuf;
        cmsg->cmsg_len = sizeof(cbuf);

        memset(&msg, 0, sizeof(msg));
        msg.msg_name = &addr;
        msg.msg_namelen = sizeof(addr);
        msg.msg_iov = &iov;
        msg.msg_iovlen = 1;
        msg.msg_control = cmsg;
        msg.msg_controllen = sizeof(cbuf);

        iov.iov_base = &req;
        iov.iov_len = sizeof(req);

        ret = recvmsg(iom->fd, &msg, 0);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("recvmsg on config socket: %s",
                                          strerror(errno));
                return L7VS_IOMUX_LIST_UNCHANGED;
        }

        if (msg.msg_flags & (MSG_TRUNC|MSG_CTRUNC)) {
                VANESSA_LOGGER_ERR("Unknown message on config socket"
                                   " (message too large)");
                VANESSA_LOGGER_DEBUG_UNSAFE("msg_flags=%x", msg.msg_flags);
                return L7VS_IOMUX_LIST_UNCHANGED;
        }

        if (! (cmsg->cmsg_level == SOL_SOCKET &&
               cmsg->cmsg_type == SCM_CREDENTIALS)) {
                VANESSA_LOGGER_ERR("Could not receive a remote credential");
                return L7VS_IOMUX_LIST_UNCHANGED;
        }

        /*
         * actually credential check is unnecessary,
         * at least for Linux 2.4.20.
         */
        cred = (struct ucred *)CMSG_DATA(cmsg);
        if (cred->uid != 0) {
                VANESSA_LOGGER_ERR("Request from unprivileged user");
                return L7VS_IOMUX_LIST_UNCHANGED;
        }

        l7vs_config_dispatch(&req, &addr);

        switch (req.cmd) {
        case L7VS_CONFIG_DEL_VS:
        case L7VS_CONFIG_FLUSH_VS:
                return L7VS_IOMUX_LIST_REMOVED_OTHER;
        default:
                break;
        }

        return L7VS_IOMUX_LIST_UNCHANGED;
}

struct l7vs_config_table {
        enum l7vs_config_cmd cmd;
        void (*func)(union l7vs_config_req *req, struct sockaddr_un *addr);
};

struct l7vs_config_table l7vs_config_table[] = {
        {L7VS_CONFIG_LIST_VS,   l7vs_config_list_vs},
        {L7VS_CONFIG_LIST_RS,   l7vs_config_list_rs},
        {L7VS_CONFIG_ADD_VS,    l7vs_config_add_vs},
        {L7VS_CONFIG_DEL_VS,    l7vs_config_del_vs},
        {L7VS_CONFIG_EDIT_VS,   l7vs_config_edit_vs},
        {L7VS_CONFIG_ADD_RS,    l7vs_config_add_rs},
        {L7VS_CONFIG_DEL_RS,    l7vs_config_del_rs},
        {L7VS_CONFIG_EDIT_RS,   l7vs_config_edit_rs},
        {L7VS_CONFIG_FLUSH_VS,  l7vs_config_flush_vs},
        {L7VS_CONFIG_NONE, NULL}
};

static void
l7vs_config_dispatch(union l7vs_config_req *req, struct sockaddr_un *addr)
{
        struct l7vs_config_table *t;
        struct iovec iov;
        struct l7vs_config_rsp_unknown_cmd rsp;

        for (t = l7vs_config_table; t->cmd != L7VS_CONFIG_NONE; t++) {
                if (t->cmd == req->cmd) {
                        break;
                }
        }

        if (t->cmd == L7VS_CONFIG_NONE) {
                VANESSA_LOGGER_ERR("Unknown command from config socket");
                rsp.cmd = req->cmd;
                rsp.code = L7VS_CONFIG_ERR_INVALID_COMMAND;
                iov.iov_base = &rsp;
                iov.iov_len = sizeof(rsp);
                l7vs_config_send_response(addr, &iov, 1);
                return;
        }

        t->func(req, addr);
}

static void
l7vs_config_list_vs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_list_vs *req = &arg->list_vs;
        struct l7vs_config_rsp_list_vs rsp;
        struct l7vs_service_arg *sas;
        struct iovec iov[2];
        int iovlen;
        int len;

        sas = NULL;
        rsp.cmd = req->cmd;
        rsp.code = 0;
        rsp.num = 0;
        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);

        len = l7vs_service_list_service_arg(&sas, &rsp.num);
        if (len < 0) {
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

        if (rsp.num == 0) {
                iovlen = 1;
        } else {
                iov[1].iov_base = sas;
                iov[1].iov_len = len;
                iovlen = 2;
        }

        l7vs_config_send_response(addr, iov, iovlen);
        if (sas != NULL) {
                free(sas);
        }
        return;
}

static void
l7vs_config_list_rs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_list_rs *req = &arg->list_rs;
        struct l7vs_config_rsp_list_rs rsp;
        struct l7vs_service_arg *sarg;
        struct l7vs_service  *srv;
        struct l7vs_dest_arg *das;
        struct iovec iov[2];
        int iovlen;
        int ret;

        rsp.cmd = req->cmd;
        rsp.code = 0;
        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg *)(req + 1);

        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

        ret = l7vs_service_list_dest_arg(srv, &das);
        if (ret < 0) {
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

        rsp.num = ret;
        if (ret == 0) {
                iovlen = 1;
        } else {
                iov[1].iov_base = das;
                iov[1].iov_len = ret * sizeof(*das);
                iovlen = 2;
        }

        l7vs_config_send_response(addr, iov, iovlen);
        if (das != NULL) {
                free(das);
        }
}

static void
l7vs_config_add_vs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg *sarg;
        struct l7vs_service *srv;
        struct iovec iov;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv != NULL) {
                VANESSA_LOGGER_ERR("ADD_VS: service already exists");
                rsp.code = L7VS_CONFIG_ERR_VS_EXISTS;
                goto out;
        }
        /* Keep the message ready to be sent */
        srvname = get_servicename(&sarg->addr, 1);
        sprintf(trap_message, "Add Virtual Service:%s module %s scheduler %s", srvname, sarg->protomod, sarg->schedmod);

        srv = l7vs_service_create(sarg, &rsp.code);
        if (srv == NULL) {
                VANESSA_LOGGER_ERR("ADD_VS: failed to create a service");
        } else {
		VANESSA_LOGGER_INFO("ADD_VS: IFRM001: created a service ");

                /* Send Trap message to l7vsagent process */
                check_process_l7vsagent(trap_message);
	}

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_del_vs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg *sarg;
        struct l7vs_service *srv;
	struct l7vs_dest *dest;
        struct iovec iov;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
                VANESSA_LOGGER_DEBUG("DEL_VS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

        /* Keep the message ready to be sent */
        srvname = get_servicename(&sarg->addr, 1);
        sprintf(trap_message, "Delete Virtual Service:%s module %s scheduler %s", srvname, sarg->protomod, sarg->schedmod);

	while (g_list_length(srv->dest_list) > 0) {
		dest = (g_list_first(srv->dest_list))->data;
		srv->dest_list = g_list_remove(srv->dest_list, dest);
		l7vs_dest_destroy(dest);
	}

        l7vs_service_destroy(srv);

	VANESSA_LOGGER_INFO("DEL_VS: IFRM003: released virtual service");

        /* Send Trap message to l7vsagent process */
        check_process_l7vsagent(trap_message);

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_edit_vs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg *sarg;
        struct l7vs_service *srv;
        struct l7vs_scheduler *sched, *sched_old;
        struct iovec iov;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
                VANESSA_LOGGER_DEBUG("EDIT_VS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

        sched = l7vs_sched_get(sarg->schedmod);
        if (sched == NULL) {
                VANESSA_LOGGER_DEBUG("EDIT_VS: unknown scheduler");
                rsp.code = L7VS_CONFIG_ERR_NOSCHED;
                goto out;
        } else {
		VANESSA_LOGGER_INFO("EDIT_VS: IFRM002: changed virtual service");
	}
        sched_old = srv->scheduler;
        if (sched != sched_old) {
                l7vs_sched_unbind(sched_old, srv);
                l7vs_sched_put(sched_old);
                l7vs_sched_bind(sched, srv);

        /* Send Trap message to l7vsagent process */
        srvname = get_servicename(&sarg->addr, 1);
        sprintf(trap_message, "Edit Virtual Service:%s module %s scheduler %s", srvname, sarg->protomod, sarg->schedmod);
        check_process_l7vsagent(trap_message);
        }

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_add_rs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg *sarg;
        struct l7vs_service *srv;
        struct iovec iov;
        int ret;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
                VANESSA_LOGGER_DEBUG("ADD_RS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

        if (l7vs_service_lookup_dest(srv, &req->darg.addr) != NULL) {
                VANESSA_LOGGER_DEBUG("ADD_RS: real server already exists");
                rsp.code = L7VS_CONFIG_ERR_RS_EXISTS;
                goto out;
        }

        /* Keep the message ready to be sent */
        srvname = get_servicename(&req->darg.addr, 1);
        srvname_temp = get_servicename(&sarg->addr, 1);
        sprintf(trap_message, "Add Real Server:%s for Virtual Service %s module %s", srvname, srvname_temp, sarg->protomod);

        ret = l7vs_service_add_dest(srv, &req->darg);
        if (ret < 0) {
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
        }

        /* Send Trap message to l7vsagent process */
        check_process_l7vsagent(trap_message);

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_del_rs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg *sarg;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        struct iovec iov;
        int ret;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
                VANESSA_LOGGER_DEBUG("DEL_RS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

        dest = l7vs_service_lookup_dest(srv, &req->darg.addr);
        if (dest == NULL) {
                VANESSA_LOGGER_DEBUG("DEL_VS: destination nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NORS;
                goto out;
        }

        /* Keep the message ready to be sent */
        srvname = get_servicename(&req->darg.addr, 1);
        srvname_temp = get_servicename(&sarg->addr, 1);
        sprintf(trap_message, "Delete Real Server:%s for Virtual Service %s module %s", srvname, srvname_temp, sarg->protomod);

        ret = l7vs_service_remove_dest(srv, &req->darg);
        if (ret < 0) {
                /* XXX actually impossible */
                rsp.code = L7VS_CONFIG_ERR_NORS;
        }

        /* Send Trap message to l7vsagent process */
        check_process_l7vsagent(trap_message);

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_edit_rs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg *sarg;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        struct iovec iov;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
                VANESSA_LOGGER_DEBUG("EDIT_RS: service nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

        dest = l7vs_service_lookup_dest(srv, &req->darg.addr);
        if (dest == NULL) {
                VANESSA_LOGGER_DEBUG("EDIT_VS: destination nonexistent");
                rsp.code = L7VS_CONFIG_ERR_NORS;
                goto out;
        }

        dest->weight = req->darg.weight;

	/* Send signal SIGUSR1 to l7vsagent process if required */
	/*
	srvname = get_servicename(&req->darg.addr, 1);
	srvname_temp = get_servicename(&sarg.addr, 1);
	sprintf(trap_message, "Edit Real Server:%s for Virtual Service %s module %s", srvname, srvname_temp, sarg->protomod);
	check_process_l7vsagent(trap_message);
	*/

out:
        l7vs_config_send_response(addr, &iov, 1);
}

static void
l7vs_config_flush_vs(union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_req_flush_vs *req = &arg->flush_vs;
        struct l7vs_config_rsp_flush_vs rsp;
        struct iovec iov;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        l7vs_service_flush_all();


        /* Send signal SIGUSR1 to l7vsagent process */
        check_process_l7vsagent("Clear L7VS Table");

        l7vs_config_send_response(addr, &iov, 1);
}

static int
l7vs_config_send_response(struct sockaddr_un *addr,
                          struct iovec *iov,
                          int iovlen)
{
        struct  msghdr msg;
        int ret;

        memset(&msg, 0, sizeof(msg));
        msg.msg_name = addr;
        msg.msg_namelen = sizeof(*addr);
        msg.msg_iov = iov;
        msg.msg_iovlen = iovlen;

        ret = sendmsg(l7vs_config_iomux.fd, &msg, 0);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("sendmsg on config socket: %s",
                                          strerror(errno));
        }

        return ret;
}

/*
This method checks whether the process l7vsagent - L7VS SNMP subagent
*  is running or not.
*  If l7vsagent is running, then the function puts the appropriate TRAP message in the
*  message queue, sends the signal SIGUSR1 to the l7vsagent process and returns 1.
*  If l7vsagent is not running, then the function simply returns 0 and exits
*/
int check_process_l7vsagent(char *trap_message)
{
        FILE *fp;
        char l7vsagent_pid_folder[30];
        char l7vsagent_string[100];
        char l7vsagent_pid[10];

        /* message related */
        char trap_string[BUFSIZ];

        key_t key;
        int msgid;

        /* Open the PID file to see whether the snmp agent process is running */
        fp = fopen("/var/run/l7vsagent.pid", "r");
        if (fp != NULL)
                fscanf(fp, "%s", l7vsagent_pid);
        else
                return 0;

        fclose(fp);

        sprintf(l7vsagent_pid_folder, "/proc/%s/cmdline", l7vsagent_pid);

        /* Confirm whether the existing PID file is not stale */
        fp = fopen(l7vsagent_pid_folder, "r");
        if (fp != NULL)
                        fscanf(fp, "%s", l7vsagent_string);
        else
        return 0;

        fclose(fp);

        /* Using strstr, confirm that the PID file is not stale or does not
        belong to some other process */
        if (strstr(l7vsagent_string, "l7vsagent") != NULL)
        {
                /* Create key for the message queue */
                key = ftok("/var/run/l7vsagent.pid", 'b');
                if ((int) key == -1)
                                return 0;

                /* Create or Get the message ID for the queue */
                msgid = msgget(key, 0666 | IPC_CREAT);
                if (msgid == -1)
                        return 0;

                /* Send the message to the queue */
                strcpy(trap_string, trap_message); 
                if (msgsnd(msgid, (void *) &trap_string, sizeof(trap_string), 0) == -1) 
                        return 0;

                /* Sending SIGUSR1 to l7vsagent */
				kill((pid_t) atoi(l7vsagent_pid), SIGUSR1);
                return 1;
        }

        /* In case of any errors or something, just return 0 */
        return 0;
}

/*
*  Method added for getting the address and port as character strings.
*  This method was copied from l7vsadm.c
*/
static char * get_servicename(struct sockaddr_in *addr, int isnumeric)
{
        char *sname;
        char hostname[40], portname[20];
        int flags;
        int ret;

        flags = 0;
        if (isnumeric) {
                flags = NI_NUMERICHOST | NI_NUMERICSERV;
        }

        ret = getnameinfo((struct sockaddr *)addr, sizeof(*addr),
                          hostname, sizeof(hostname),
                          portname, sizeof(portname), flags);
        if (ret != 0) {
                VANESSA_LOGGER_ERR_UNSAFE("getnameinfo: %s",
                                          gai_strerror(ret));
                return NULL;
        }

        ret = asprintf(&sname, "%s:%s", hostname, portname);
        if (ret < 0) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return NULL;
        }

        return sname;
}

static void stop_l7sync_daemons()
{
	FILE *fp;
	fp=fopen("/etc/init.d/l7syncd","r");
        if (fp != NULL)
		system("/etc/init.d/l7syncd stopall");
        fclose(fp);
}

void shared_mem_sem_remove(key_t shmid, key_t sema_id)
{
        if (semctl(sema_id, IPC_RMID, 0) == -1)
        {
                fprintf(stderr, "semctl(IPC_RMID) failed reason %s\n",strerror(errno));
                fprintf(stderr, "semctl(IPC_RMID) failed\n");
        }

        if (shmctl(shmid, IPC_RMID, 0) == -1)
        {
                fprintf(stderr, "shmctl(IPC_RMID) failed reason %s\n",strerror(errno));
                fprintf(stderr, "shmctl(IPC_RMID) failed\n");
        }
}

static void shm_sem_remove() 
{
        int ck_semid,ck_shmid;
	int evs_semid,evs_shmid;

        ck_semid=initialize_sem(key_sem_ck);
        ck_shmid=initialize_shm(key_shm_ck,sizeof(struct sync_info_st));
	shared_mem_sem_remove(ck_shmid, ck_semid);

        evs_semid=initialize_sem(key_sem_evs);
	evs_shmid=initialize_shm(key_shm_evs,sizeof(struct evasion));
	shared_mem_sem_remove(evs_shmid, evs_semid);
}

