﻿'Copyright (C) 2010 pepetaro, All rights reserved.
'This library is free software; you can redistribute it and/or
'modify it under the terms of the GNU Lesser General Public
'License as published by the Free Software Foundation; either
'version 3 of the License, or (at your option) any later version.

'This library is distributed in the hope that it will be useful,
'but WITHOUT ANY WARRANTY; without even the implied warranty of
'MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
'Lesser General Public License for more details.

'You should have received a copy of the GNU Lesser General Public
'License along with this library; if not, write to the Free Software
'Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

''' <summary>Twitterのユーザーを表します。</summary>
<DebuggerDisplay("User {ScreenName} / {Name}")> _
Public Class TwitterUser
    Implements ITwitterIdData, IEquatable(Of TwitterUser)

    Friend Sub New()
    End Sub

    ''' <summary>取得したXMLから情報を読み取ります。</summary>
    Protected Overridable Sub LoadAll(ByVal doc As XmlDocument)
        On Error Resume Next '読み込める物だけ読み込む
        _userid = Long.Parse(doc.SelectSingleNode("/user/id").InnerText)
        _screenname = doc.SelectSingleNode("/user/screen_name").InnerText
        _name = doc.SelectSingleNode("/user/name").InnerText
        _location = doc.SelectSingleNode("/user/location").InnerText
        _description = doc.SelectSingleNode("/user/description").InnerText
        profile_image_url = doc.SelectSingleNode("/user/profile_image_url").InnerText
        url = doc.SelectSingleNode("/user/url").InnerText
        _protected = Boolean.Parse(doc.SelectSingleNode("/user/protected").InnerText)
        followers_count = Integer.Parse(doc.SelectSingleNode("/user/followers_count").InnerText)
        friends_count = Integer.Parse(doc.SelectSingleNode("/user/friends_count").InnerText)
        created_at = Date.SpecifyKind(Date.ParseExact(doc.SelectSingleNode("/user/created_at").InnerText, "ddd MMM dd HH:mm:ss +0000 yyyy", New Globalization.CultureInfo("en-US")), DateTimeKind.Utc)
        favourites_count = Integer.Parse(doc.SelectSingleNode("/user/favourites_count").InnerText)
        statuses_count = Integer.Parse(doc.SelectSingleNode("/user/statuses_count").InnerText)
        _status = GetStatus(doc.SelectSingleNode("/user/status").OuterXml)
    End Sub

    ''' <summary>XMLから新しいインスタンスを作成します。</summary>
    Public Shared Function FromXml(ByVal doc As XmlDocument) As TwitterUser
        Dim re As New TwitterUser
        re.LoadAll(doc)
        Return re
    End Function

    Private Function GetStatus(ByVal xml As String) As TwitterStatus
        Try
            Dim xd As New XmlDocument
            xd.LoadXml(xml)
            Return TwitterStatus.FromXml(xd)
        Catch
            Return Nothing
        End Try
    End Function

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend _userid As Long
    ''' <summary>ユーザーのID（番号）</summary>
    Public ReadOnly Property Id As Long Implements ITwitterIdData.Id
        Get
            Return _userid
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend _screenname As String
    ''' <summary>表示名（@○○）</summary>
    Public ReadOnly Property ScreenName As String
        Get
            Return _screenname
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend _name As String
    ''' <summary>名称</summary>
    Public ReadOnly Property Name As String
        Get
            Return _name
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend _location As String
    ''' <summary>住んでいる場所</summary>
    Public ReadOnly Property Location As String
        Get
            Return _location
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend _description As String
    ''' <summary>自己紹介</summary>
    Public ReadOnly Property Description As String
        Get
            Return _description
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend profile_image_url As String
    ''' <summary>プロフィール画像</summary>
    Public ReadOnly Property ProfileImageUri As Uri
        Get
            Return New Uri(profile_image_url)
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend url As String
    ''' <summary>ユーザーのWebサイトのURI</summary>
    Public ReadOnly Property WebSite As Uri
        Get
            Return New Uri(url)
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend _protected As Boolean
    ''' <summary>つぶやきを非公開にしているか</summary>
    Public ReadOnly Property [Protected] As Boolean
        Get
            Return _protected
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend followers_count As Integer
    ''' <summary>フォロワー数</summary>
    Public ReadOnly Property FollowersCount As Integer
        Get
            Return followers_count
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend friends_count As Integer
    ''' <summary>フレンド（フォロー）数</summary>
    Public ReadOnly Property FriendsCount As Integer
        Get
            Return friends_count
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend created_at As Date
    ''' <summary>Twitter登録日時（UTC）</summary>
    Public ReadOnly Property CreatedAtUtc As Date Implements ITwitterIdData.CreatedAtUtc
        Get
            Return created_at
        End Get
    End Property
    ''' <summary>Twitter登録日時</summary>
    Public ReadOnly Property CreatedAt As Date Implements ITwitterIdData.CreatedAt
        Get
            Return created_at.ToLocalTime
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend favourites_count As Integer
    ''' <summary>お気に入りの数</summary>
    Public ReadOnly Property FavouritesCount As Integer
        Get
            Return favourites_count
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend statuses_count As Integer
    ''' <summary>ツイート数</summary>
    Public ReadOnly Property StatusesCount As Integer
        Get
            Return statuses_count
        End Get
    End Property

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Protected Friend _status As TwitterStatus
    ''' <summary>このユーザーの最新のつぶやき</summary>
    Public ReadOnly Property Status As TwitterStatus
        Get
            Return _status
        End Get
    End Property

    ''' <summary>常にIdObjectType.Userを返します。</summary>
    Private ReadOnly Property ObjectType As IdObjectType Implements ITwitterIdData.ObjectType
        Get
            Return IdObjectType.User
        End Get
    End Property

    Public Overrides Function ToString() As String
        Return "@" & _screenname
    End Function

    Public Shared Operator =(ByVal ts1 As TwitterUser, ByVal ts2 As TwitterUser) As Boolean
        Return ts1._userid = ts2._userid
    End Operator
    Public Shared Operator <>(ByVal ts1 As TwitterUser, ByVal ts2 As TwitterUser) As Boolean
        Return ts1._userid <> ts2._userid
    End Operator

    Public Overloads Function Equals(ByVal other As TwitterUser) As Boolean Implements System.IEquatable(Of TwitterUser).Equals
        Return Me = other
    End Function
End Class