﻿'Copyright (C) 2010 pepetaro, All rights reserved.
'This library is free software; you can redistribute it and/or
'modify it under the terms of the GNU Lesser General Public
'License as published by the Free Software Foundation; either
'version 3 of the License, or (at your option) any later version.

'This library is distributed in the hope that it will be useful,
'but WITHOUT ANY WARRANTY; without even the implied warranty of
'MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
'Lesser General Public License for more details.

'You should have received a copy of the GNU Lesser General Public
'License along with this library; if not, write to the Free Software
'Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

''' <summary>Twitterを扱います。</summary>
''' <remarks>テキトーに作ってあるのでAPIWikiなどを見て調査してください。</remarks>
Public NotInheritable Class Twitter
    Public Sub New()
    End Sub

    Public Sub New(ByVal consumerKey As String, ByVal consumerKeySecret As String, ByVal accessToken As String, ByVal accessTokenSecret As String)
        Me.ConsumerKey = New PublicSecretPair(consumerKey, consumerKeySecret)
        Me.AccessToken = New PublicSecretPair(accessToken, accessTokenSecret)
    End Sub

    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Friend _twiCon As TwitterConnection
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Friend ReadOnly Property twiCon As TwitterConnection
        Get
            If _twiCon Is Nothing Then _twiCon = New TwitterConnection(Me)
            Return _twiCon
        End Get
    End Property
    ''' <summary>つぶやくことができる最大文字数</summary>
    Public Const MAX_LENGTH As Integer = 140

    ''' <summary>アクセストークン</summary>
    Public Property AccessToken As New PublicSecretPair
    ''' <summary>コンシューマー鍵</summary>
    Public Property ConsumerKey As New PublicSecretPair

    ''' <summary>ユーザーの画像を取得します。</summary>
    ''' <param name="id">画像を取得するユーザーのscreen_nameかid</param>
    Public Overloads Function GetProfileImage(ByVal id As String) As Image
        Dim tu = Users.GetUser(id)
        Return GetProfileImage(tu.ProfileImageUri)
    End Function
    ''' <summary>指定したURIから画像を取得します。</summary>
    Public Overloads Shared Function GetProfileImage(ByVal uri As Uri) As Image
        Using wc As New WebClient
            GetProfileImage = Image.FromStream(wc.OpenRead(uri))
        End Using
    End Function

    ''' <summary>OAuth認証を使用して通信します。使用後にはCloseしてください。</summary>
    ''' <param name="uri">呼び出すAPIのURI</param>
    ''' <param name="method">HTTP通信メソッド（GET/HEAD/POST/PUT/DELETE）</param>
    ''' <param name="param">GET時のクエリ、またはPOST時のエンティティボディ</param>
    ''' <param name="header">[IN/OUT]HTTP応答のヘッダ情報</param>
    ''' <returns>レスポンスのStream</returns>
    Public Overloads Function GetStream(ByVal uri As Uri, ByVal method As String, ByVal param As Dictionary(Of String, String), ByVal header As Dictionary(Of String, String)) As Stream
        Return twiCon.GetStream(uri, method, param, header)
    End Function
    ''' <summary>OAuth認証を使用してバイナリをアップロードします。使用後にはCloseしてください。</summary>
    ''' <param name="uri">呼び出すAPIのURI</param>
    ''' <param name="binarys">アップロードするファイルの情報</param>
    ''' <param name="header">[IN/OUT]HTTP応答のヘッダ情報</param>
    ''' <returns>レスポンスのStream</returns>
    Public Overloads Function GetStream(ByVal uri As Uri, ByVal param As Dictionary(Of String, String), ByVal binarys As Dictionary(Of String, UploadFile), ByVal header As Dictionary(Of String, String)) As Stream
        Return twiCon.GetStream(uri, param, binarys, header)
    End Function

    ''' <summary>TwitterAPI用のURIを作成します。</summary>
    ''' <param name="version">TwitterAPIのバージョン。通常は 1 を指定します。</param>
    ''' <param name="reqUri">https://api.twitter.com/version/ 以降を指定してください。</param>
    Public Shared Function CreateUri(ByVal version As Integer, ByVal reqUri As String) As Uri
        Return New Uri(String.Format("https://api.twitter.com/{0}/{1}", version, reqUri))
    End Function

    ''' <summary>プロフィール画像用に加工します。</summary>
    ''' <param name="image">加工する画像</param>
    Public Shared Function CreateProfileImage(ByVal image As Image) As Image
        Dim img_size As Integer = Math.Min(500, Math.Max(image.Width, image.Height))
        Dim width As Integer
        Dim height As Integer
        Dim x As Integer
        Dim y As Integer
        If image.Width >= image.Height Then
            width = img_size
            height = CInt(width * (image.Height / image.Width))
            x = 0
            If image.Width = image.Height Then
                y = 0
            Else
                y = CInt((img_size - height) / 2)
            End If
        Else
            height = img_size
            width = CInt(height * (image.Width / image.Height))
            y = 0
            x = CInt((img_size - width) / 2)
        End If
        Dim bmp As New Bitmap(img_size, img_size, Imaging.PixelFormat.Format32bppArgb)
        Dim g = Graphics.FromImage(bmp)
        g.InterpolationMode = Drawing2D.InterpolationMode.HighQualityBicubic
        g.DrawImage(image, x, y, width, height)
        g.Dispose()
        Return bmp
    End Function

    ''' <summary>API情報を取得します。</summary>
    ''' <param name="tw">Twitterクラスのインスタンス。nullだと認証をせずに取得します。</param>
    Public Shared Function GetRateLimitStatus(ByVal tw As Twitter) As RateLimitStatusResult
        Return New RateLimitStatusResult(tw, tw IsNot Nothing)
    End Function

    ''' <summary>スパム報告をし、ブロックします。</summary>
    ''' <param name="id">スパムとして報告するユーザーのscreen_nameまたはid</param>
    Public Sub ReportSpam(ByVal id As String)
        Dim param As New Dictionary(Of String, String)
        param.Add("id", id)
        GetStream(CreateUri(1, "report_spam.xml"), "POST", param, Nothing).Close()
    End Sub

    ''' <summary>アクセストークンが正しいかを調べます。</summary>
    ''' <returns>成功すればtrueを、失敗ならfalseを返します。また、SelfInfoで自分自身を取得できるようになります。</returns>
    ''' <remarks>初めての通信時にも実行されます。</remarks>
    Public Function VerifyCredentials() As Boolean
        Try
            Dim stream = twiCon.GetStream(CreateUri(1, "account/verify_credentials.xml"), "GET", Nothing, Nothing)
            Dim xd As New XmlDocument
            xd.Load(stream)
            stream.Close()
            _self = TwitterUser.FromXml(xd)
            Return True
        Catch
            Return False
        End Try
    End Function
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> Dim _self As TwitterUser
    ''' <summary>自分自身の情報。取得するにはVerifyCredentials()を呼び出してください。</summary>
    Public ReadOnly Property SelfInfo As TwitterUser
        Get
            Return _self
        End Get
    End Property

#Region "クラスのインスタンス"
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Dim _DirectMessages As New DirectMessagesService(Me)
    ''' <summary>DMを操作します。</summary>
    Public ReadOnly Property DirectMessages As DirectMessagesService
        Get
            Return _DirectMessages
        End Get
    End Property
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Dim _Authenticate As New AuthenticateService(Me)
    ''' <summary>xAuthやPINを使った認証を行います。</summary>
    Public ReadOnly Property Authenticate As AuthenticateService
        Get
            Return _Authenticate
        End Get
    End Property
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Dim _Timelines As New TimelinesServices(Me)
    ''' <summary>タイムラインを取得します。</summary>
    Public ReadOnly Property Timelines As TimelinesServices
        Get
            Return _Timelines
        End Get
    End Property
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Dim _Statuses As New StatusesService(Me)
    ''' <summary>ステータスを操作します。</summary>
    Public ReadOnly Property Statuses As StatusesService
        Get
            Return _Statuses
        End Get
    End Property
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Dim _Users As New UsersService(Me)
    ''' <summary>ユーザー情報を操作します。</summary>
    Public ReadOnly Property Users As UsersService
        Get
            Return _Users
        End Get
    End Property
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Dim _Friends As New FriendsService(Me)
    ''' <summary>フレンドを操作します。</summary>
    Public ReadOnly Property Friends As FriendsService
        Get
            Return _Friends
        End Get
    End Property
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
    Dim _Blocks As New BlocksService(Me)
    ''' <summary>ブロック関係を操作します。お取り扱いにはご注意を</summary>
    Public ReadOnly Property Blocks As BlocksService
        Get
            Return _Blocks
        End Get
    End Property
    <DebuggerBrowsable(DebuggerBrowsableState.Never)> Dim _Lists As New ListsService(Me)
    ''' <summary>リストを操作します。</summary>
    Public ReadOnly Property Lists As ListsService
        Get
            Return _Lists
        End Get
    End Property
#End Region
End Class
