/*
 * Copyright 2006-2007 Sxip Identity Corporation
 */

package jp.sourceforge.tsukuyomi.openid.message.ax;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;

import jp.sourceforge.tsukuyomi.openid.message.MessageException;
import jp.sourceforge.tsukuyomi.openid.message.Parameter;
import jp.sourceforge.tsukuyomi.openid.message.ParameterList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Implements the extension for Attribute Exchange fetch requests.
 * 
 * @see AxMessage Message
 * @author Marius Scurtescu, Johnny Bufu
 */
public class FetchRequest extends AxMessage {
	private static final Log LOG = LogFactory.getLog(FetchRequest.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	/**
	 * Constructs a Fetch Request with an empty parameter list.
	 */
	protected FetchRequest() {
		parameters.set(new Parameter("mode", "fetch_request"));

		if (DEBUG) {
			LOG.debug("Created empty fetch request.");
		}
	}

	/**
	 * Constructs a Fetch Request with an empty parameter list.
	 */
	public static FetchRequest createFetchRequest() {
		return new FetchRequest();
	}

	/**
	 * Constructs a FetchRequest from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	protected FetchRequest(ParameterList params) {
		parameters = params;
	}

	/**
	 * Constructs a FetchRequest from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	public static FetchRequest createFetchRequest(ParameterList params)
			throws MessageException {
		FetchRequest req = new FetchRequest(params);

		if (!req.isValid()) {
			throw new MessageException("Invalid parameters for a fetch request");
		}

		if (DEBUG) {
			LOG.debug("Created fetch request from parameter list:\n" + params);
		}

		return req;
	}

	/**
	 * Adds an attribute to the fetch request.
	 * 
	 * @param alias
	 *            The attribute alias that will be associated with the attribute
	 *            type URI
	 * @param typeUri
	 *            The attribute type URI
	 * @param required
	 *            If true, marks the attribute as 'required'; 'if_available'
	 *            otherwise.
	 * @param count
	 *            The number of attribute values requested.
	 */
	public void addAttribute(String alias, String typeUri, boolean required,
			int count) {
		parameters.set(new Parameter("type." + alias, typeUri));

		String level = required ? "required" : "if_available";

		Parameter levelParam = parameters.getParameter(level);
		Parameter newParam;

		if (levelParam == null) {
			newParam = new Parameter(level, multivalEncode(alias));
		} else {
			newParam =
				new Parameter(level, levelParam.getValue()
					+ ","
					+ multivalEncode(alias));
			parameters.removeParameters(level);
		}

		parameters.set(newParam);

		if (count > 1) {
			parameters.set(new Parameter("count." + alias, Integer
				.toString(count)));
		}

		if (DEBUG) {
			LOG.debug("Added new attribute to fetch request; type: "
				+ typeUri
				+ " alias: "
				+ alias
				+ " count: "
				+ count
				+ " required: "
				+ required);
		}
	}

	/**
	 * Adds an attribute to the fetch request, with a default value-count of 1.
	 * 
	 * @see #addAttribute(String, String, boolean, int)
	 */
	public void addAttribute(String alias, String typeUri, boolean required) {
		addAttribute(alias, typeUri, required, 1);
	}

	/**
	 * Sets the desired number of attribute vaules requested for the specified
	 * attribute alias.
	 * 
	 * @param alias
	 *            The attribute alias.
	 */
	public void setCount(String alias, int count) {
		if (count > 1) {
			parameters.set(new Parameter("count." + alias, Integer
				.toString(count)));
		}
	}

	/**
	 * Returns the number of values requested for the specified attribute alias,
	 * or 1 (the default number) if the count parameter is absent.
	 * 
	 * @param alias
	 *            The attribute alias.
	 */
	public int getCount(String alias) {
		if (parameters.hasParameter("count." + alias)) {
			return Integer.parseInt(parameters.getParameterValue("count."
				+ alias));
		} else {
			return 1;
		}
	}

	/**
	 * Sets the optional 'update_url' parameter where the OP can later re-post
	 * fetch-response updates to the values of the requested attributes.
	 * 
	 * @param updateUrl
	 *            The URL where the RP accepts later updates to the requested
	 *            attributes.
	 */
	public void setUpdateUrl(String updateUrl) throws MessageException {
		try {
			new URL(updateUrl);
		} catch (MalformedURLException e) {
			throw new MessageException("Invalid update_url: " + updateUrl);
		}

		if (DEBUG) {
			LOG.debug("Setting fetch request update_url: " + updateUrl);
		}

		parameters.set(new Parameter("update_url", updateUrl));
	}

	/**
	 * Gets the optional 'update_url' parameter if available, or null otherwise.
	 */
	public String getUpdateUrl() {
		return parameters.hasParameter("update_url") ? parameters
			.getParameterValue("update_url") : null;
	}

	/**
	 * Returns a map with the requested attributes.
	 * 
	 * @param required
	 *            If set to true the list of 'required' attributes is returned,
	 *            otherwise the list of 'if_available' attributes.
	 * @return Map of attribute aliases -> attribute type URIs.
	 */
	public Map<String, Object> getAttributes(boolean required) {
		Map<String, Object> attributes = new LinkedHashMap<String, Object>();

		String level = required ? "required" : "if_available";

		Parameter param = parameters.getParameter(level);
		if (param != null) {
			String[] values = param.getValue().split(",");
			for (String element : values) {
				String alias = multivalDecode(element);
				attributes.put(alias, parameters.getParameterValue("type."
					+ alias));
			}
		}

		return attributes;
	}

	/**
	 * Gets all requested attributes (required and optional).
	 * 
	 * @return Map of attribute aliases -> attribute type URIs.
	 */
	public Map<String, Object> getAttributes() {
		Map<String, Object> attributes = getAttributes(true);
		attributes.putAll(getAttributes(false));

		return attributes;
	}

	/**
	 * Checks the validity of the extension.
	 * <p>
	 * Used when constructing a extension from a parameter list.
	 * 
	 * @return True if the extension is valid, false otherwise.
	 */
	public boolean isValid() {
		if (!parameters.hasParameter("required")
			&& !parameters.hasParameter("if_available")) {
			LOG
				.warn("One of 'required' or 'if_available' parameters must be present.");
			return false;
		}

		if (!parameters.hasParameter("mode")
			|| !"fetch_request".equals(parameters.getParameterValue("mode"))) {
			LOG.warn("Invalid mode value in fetch_request: "
				+ parameters.getParameterValue("mode"));
			return false;
		}

		if (parameters.hasParameter("required")) {
			String[] values =
				parameters.getParameterValue("required").split(",");
			for (String element : values) {
				String value = multivalDecode(element);
				if (!parameters.hasParameter("type." + value)) {
					LOG.warn("Type missing for attribute alias: " + value);
					return false;
				}
			}
		}

		if (parameters.hasParameter("if_available")) {
			String[] values =
				parameters.getParameterValue("if_available").split(",");
			for (String element : values) {
				String value = multivalDecode(element);
				if (!parameters.hasParameter("type." + value)) {
					LOG.warn("Type missing for attribute alias: " + value);
					return false;
				}
			}
		}

		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();
			if (!paramName.equals("mode")
				&& !paramName.startsWith("type.")
				&& !paramName.equals("required")
				&& !paramName.equals("if_available")
				&& !paramName.equals("update_url")) {
				LOG.warn("Invalid parameter name in fetch request: "
					+ paramName);
			}
		}

		return true;
	}
}
