package jp.sourceforge.tsukuyomi.openid.impl;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import jp.sourceforge.tsukuyomi.openid.AuthenticationRequest;
import jp.sourceforge.tsukuyomi.openid.Consts;
import jp.sourceforge.tsukuyomi.openid.OpenIDConsumer;
import jp.sourceforge.tsukuyomi.openid.OpenIDException;
import jp.sourceforge.tsukuyomi.openid.OpenIDRuntimeException;
import jp.sourceforge.tsukuyomi.openid.discovery.DiscoveryException;
import jp.sourceforge.tsukuyomi.openid.discovery.DiscoveryInformation;
import jp.sourceforge.tsukuyomi.openid.discovery.Identifier;
import jp.sourceforge.tsukuyomi.openid.discovery.IdentifierException;
import jp.sourceforge.tsukuyomi.openid.message.AuthRequest;
import jp.sourceforge.tsukuyomi.openid.message.ParameterList;
import jp.sourceforge.tsukuyomi.openid.rp.RelayParty;
import jp.sourceforge.tsukuyomi.openid.rp.RelayPartyConfig;
import jp.sourceforge.tsukuyomi.openid.rp.VerificationResult;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * OpenID コンシューマを提供するクラスです。インスタンスは、DI コンテナによって管理され、HTTP
 * リクエスト毎にインスタンスが生成されることを期待しています。
 * 
 * @author asuka
 */
public class OpenIDConsumerImpl implements OpenIDConsumer {
	private static final Log LOG = LogFactory.getLog(OpenIDConsumerImpl.class);

	private DiscoveryInformation discovered;

	private List<DiscoveryInformation> discoveryList;

	private HttpServletRequest request;

	private HttpServletResponse response;

	private String version;

	private RelayParty relayParty;

	private RelayPartyConfig relayPartyConfig;

	/**
	 * 新たにインスタンスを生成します。
	 */
	public OpenIDConsumerImpl() {
	}

	/**
	 * OpenID Identifier で OpenID プロバイダを探しに行った際、OpenID
	 * プロバイダを取得できるかを調べます。調べる際に実際にプロバイダを取得します。
	 * 
	 * @param identifier
	 *            OpenID Identifier
	 * @return 取得できた場合 true
	 */
	public boolean findProvider(String identifier) {
		if (identifier == null) {
			return false;
		}
		try {

			// ユーザーから入力のあった OpenID プロパイダを探しに行って、プロパイダの情報を取得してきます。
			discoveryList = relayParty.discover(identifier);

			return true;
		} catch (DiscoveryException e) {
			LOG.debug("identifier=" + identifier, e);
			return false;
		} catch (IdentifierException e) {
			LOG.debug("identifier=" + identifier, e);
			return false;
		}
	}

	/**
	 * アソシエーション確立を行います。
	 * 
	 * @return 認証サーバが要求する OpenID のバージョン
	 */
	public String associate() {
		if (discoveryList == null) {
			String s =
				Consts.BUNDLE.getString(Consts.KEY_PREFIX
					+ "DiscoveryListIsNull");
			throw new IllegalStateException(s);
		}

		discovered = relayParty.associate(discoveryList);

		// セッションに保存
		HttpSession session = request.getSession(true);
		session.setAttribute(Consts.SN_DISCOVERED, discovered);

		version = discovered.getVersion();

		return version;
	}

	/**
	 * 認証リクエストを取得します。
	 * 
	 * @return 認証リクエスト
	 */

	public AuthenticationRequest getAuthenticationRequest() {
		if (discovered == null) {
			String s =
				Consts.BUNDLE.getString(Consts.KEY_PREFIX + "DiscoveredIsNull");
			throw new IllegalStateException(s);
		}

		AuthRequest authReq;

		try {
			authReq = relayParty.authenticate(discovered, getURL());

			AuthenticationRequest ar =
				new AuthenticationRequestImpl(response, authReq, version);

			return ar;
		} catch (OpenIDException e) {
			String s =
				Consts.BUNDLE.getString(Consts.KEY_PREFIX
					+ "AuthenticateException");
			LOG.warn(s, e);
			throw new OpenIDRuntimeException(e);
		}
	}

	/**
	 * HTTP サーブレットレスポンスオブジェクトを設定します。
	 * 
	 * @param response
	 *            HTTP サーブレットレスポンスオブジェクト
	 */
	public void setResponse(HttpServletResponse response) {
		this.response = response;
	}

	/**
	 * HTTP サーブレットリクエストオブジェクトを設定します。
	 * 
	 * @param request
	 *            HTTP サーブレットリクエストオブジェクト
	 */
	public void setRequest(HttpServletRequest request) {
		this.request = request;
	}

	private String getURL() {
		String scheme = request.getScheme();
		String serverName = request.getServerName();
		String context = request.getContextPath();
		int port = request.getServerPort();
		String path = context + relayPartyConfig.getReturnURL();
		String res;

		try {
			URL url = new URL(scheme, serverName, port, path);
			if (port == url.getDefaultPort()) {
				url = new URL(scheme, serverName, -1, path);
			}
			res = url.toString();
		} catch (MalformedURLException e) {
			throw new OpenIDRuntimeException(e);
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("Scheme = " + scheme);
			LOG.debug("Server name = " + serverName);
			LOG.debug("Port = " + port);
			LOG.debug("Path = " + path);
			LOG.debug("URL = " + res);
		}

		return res;
	}

	/**
	 * 認証応答結果に対するベリファイを行います。
	 * 
	 * @return ベリファイに成功した場合、ベリファイされた OpenID Identifier。失敗した場合は null
	 */
	public String verify() {
		// プロパイダから遷移してきた際にURLに付加されているパラメータを取得します。
		ParameterList openidResp = new ParameterList(request.getParameterMap());

		HttpSession session = request.getSession(true);

		// セッションからアソシエーション情報を取得します。
		DiscoveryInformation discovered =
			(DiscoveryInformation) session.getAttribute(Consts.SN_DISCOVERED);

		String queryString = request.getQueryString();
		String url = getURL() + "?" + queryString;

		VerificationResult verification = null;

		try {
			// ベリファイを行います。
			verification = relayParty.verify(url, openidResp, discovered);
		} catch (OpenIDException e) {
			LOG.error(Consts.BUNDLE + "VerifyException");
			throw new OpenIDRuntimeException(e);
		}

		Identifier verifiedId = null;
		if (verification != null) {
			// ユーザーIDを取得します。ユーザーIDが取得できない場合は失敗です。
			verifiedId = verification.getVerifiedId();
		}

		if (verifiedId == null) {
			return null;
		}

		return verifiedId.getIdentifier();
	}

	/**
	 * プロバイダに対してリクエストを送るまでの一連の処理を行います。
	 */
	public void request() {
		associate();

		AuthenticationRequest ar = getAuthenticationRequest();

		ar.goNext();
	}

	public void setRelayParty(RelayParty relayParty) {
		this.relayParty = relayParty;
	}

	public void setRelayPartyConfig(RelayPartyConfig relayPartyConfig) {
		this.relayPartyConfig = relayPartyConfig;
	}
}
