/*
 * DefaultQueryTransaction class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.util.Map;
import java.util.HashMap;
import java.util.LinkedHashMap;

/**
 * ひとまとまりの処理を表すトランザクション・クラス。
 *
 * @author 佐藤隆之
 * @version $Id: DefaultQueryTransaction.java,v 1.5 2011-09-18 15:59:37 tayu Exp $
 */
public class DefaultQueryTransaction implements QueryTransaction
{
  /** トランザクションの状態。 */
  private State state;

  /** トランザクションの開始時刻[msec]。 */
  private long beginTimeMillis = -1L;

  /** トランザクションのタイムアウトのリミット時刻[msec]。 */
  private long limitTimeMillis = -1L;

  /** トランザクションのデフォルトのタイムアウト時間[msec]。 */
  private long timeoutMillis = -1L;

  /** コネクション・オブジェクトを格納するマップ。 */
  private Map<String,QueryConnection> connectionMap
    = new LinkedHashMap<String,QueryConnection>();

  /** コネクション・ファクトリ・オブジェクトを格納するマップ。 */
  private final Map<String,QueryConnectionFactory> connectionFactoryMap
    = new HashMap<String,QueryConnectionFactory>();

  /** このトランザクションの中で実行されたクエリ実行結果を格納するリスト。 */
  private final QueryResultList resultLst = newQueryResultList();

  /**
   * デフォルト・コンストラクタ。
   * <br>
   * このクラスのインスタンスは{@link QueryTransactionManager}クラスによって
   * 生成されることを意図しているため、アクセス指定子を<tt>protected</tt>に
   * している。
   */
  protected DefaultQueryTransaction()
  {
    this.state = State.Created;
  }

  /**
   * このトランザクションの中で実行されたクエリ実行結果を格納するリストを作成
   * する。
   *
   * @return このトランザクションの中で実行されたクエリ実行結果のリスト。
   */
  protected QueryResultList newQueryResultList()
  {
    return new QueryResultList();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void begin()
  {
    begin(this.timeoutMillis);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void begin(long timeoutMillis)
  {
    changeState(new State[]{ State.Created }, State.Begining);

    this.beginTimeMillis = System.currentTimeMillis();

    if (timeoutMillis >= 0L) {
      this.limitTimeMillis = this.beginTimeMillis + timeoutMillis;
    }
    else {
      this.limitTimeMillis = -1L;
    }

    this.state = State.Begined;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void commit()
  {
    changeState(new State[]{State.Begined, State.Committed, State.Rollbacked},
      State.Committing);

    for (QueryConnection conn : this.connectionMap.values()) {
      conn.commit();
    }

    this.state = State.Committed;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void rollback()
  {
    changeState(new State[]{State.Begined, State.Committing, State.Committed,
      State.Rollbacked}, State.Rollbacking);

    for (QueryConnection conn : this.connectionMap.values()) {
      conn.rollback();
    }

    this.state = State.Rollbacked;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void end()
  {
    State s = this.state;
    this.state = State.Ending;

    Map<String,QueryConnection> prevMap = this.connectionMap;
    this.connectionMap = new LinkedHashMap<String,QueryConnection>();

    if (s != State.Created && s != State.Committed && s != State.Rollbacked) {
      for (QueryConnection conn : prevMap.values()) {
        conn.rollback();
      }
    }

    for (QueryConnection conn : prevMap.values()) {
      conn.close();
    }

    this.state = State.Ended;
  }

  /**
   * トランザクションの状態を取得する。
   *
   * @return トランザクションの状態。
   */
  @Override
  public State getState()
  {
    return this.state;
  }

  /**
   * デフォルトのトランザクション・タイムアウト値を取得する。
   *
   * @return デフォルトのトランザクション・タイムアウト値[msec]。
   */
  protected long getTimeoutMillis()
  {
    return this.timeoutMillis;
  }

  /**
   * デフォルトのトランザクション・タイムアウト値を設定する。
   *
   * @param millis デフォルトのトランザクション・タイムアウト値[msec]。
   */
  public void setTimeoutMillis(long millis)
  {
    checkState(new State[]{ State.Created });
    this.timeoutMillis = millis;
  }

  /**
   * トランザクションの開始時刻を取得する。
   *
   * @return トランザクションの開始時刻[msec]。
   */
  @Override
  public long getBeginTimeMillis()
  {
    return this.beginTimeMillis;
  }

  /**
   * トランザクションのタイムアウトのリミット時刻を取得する。
   *
   * @return トランザクションのタイムアウトのリミット時刻[msec]。
   */
  @Override
  public long getLimitTimeMillis()
  {
    return this.limitTimeMillis;
  }

  /**
   * トランザクションの状態を変更する。
   * <br>
   * 現在の状態が、第一引数に指定された状態に含まれていない場合は、例外をスロー
   * する。
   *
   * @param froms 許される現在の状態。
   * @param to 変更後の状態。
   */
  protected final void changeState(State[] froms, State to)
  {
    checkState(froms);
    this.state = to;
  }

  /**
   * 現在の状態が、指定された状態のいずれかに含まれるかどうかを判定する。
   *
   * @param allows 許される現在の状態。
   */
  protected final void checkState(State[] allows)
  {
    for (State st : allows) {
      if (st == this.state)
        return;
    }

    throw new ReasonedRuntimeException(Error.IllegalState, this.state.name());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public QueryConnection getConnection(String connId) throws ReasonedException
  {
    checkState(new State[]{ State.Begined });

    QueryConnection conn = this.connectionMap.get(connId);
    if (conn == null) {
      QueryConnectionFactory factory = this.connectionFactoryMap.get(connId);
      if (factory == null)
        throw new ReasonedException(Error.ConnectionNotFound, connId);

      conn = factory.create(connId, this);
      conn.open();
      this.connectionMap.put(connId, conn);
    }

    return conn;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addConnectionFactory(
    String connId, QueryConnectionFactory factory)
  {
    assert (connId != null && factory != null) : (connId == null) ?
      "@param:connId is null." : "@param:factory is null.";

    checkState(new State[]{ State.Created });

    this.connectionFactoryMap.put(connId, factory);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public QueryResultList getQueryResultList()
  {
    return this.resultLst;
  }
}
