/*
 * StringSequence class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.text;

import java.util.Collection;

/**
 * R[hE|CgPʂɏANZX邽߂̃NXB
 * <br>
 * Unicode̕⏕̑Ή(JSR-204)ɂA<code>char</code>ϐ
 * KΉȂȂA񑀍ɂCfbNẌG
 * ȂB̃NX́ÃCfbNẌȒPɂ邽߂̎dg݂
 * {@link ts.util.text.StringOperation StringOperation}ƂƂɗpӂB
 * <br>
 * 񒆂̃CfbNXAR[hE|CgƂɐi߂߂肵āA
 * ̃CfbNXɈʒu镶擾ÃCfbNXJn
 * 擾肷邱ƂłB
 * <br>
 * CfbNX̎蓾l͈̔͂́A<tt>-1`n</tt> in ̓R[hE|CgP
 * ̕񒷁jłACfbNX<tt>-1</tt><tt>n</tt>̏ꍇ́A
 * ʒu镶Ƃċ󕶎ԂB
 * <br>
 * <br>
 * 擪當ԂɎ擾@́Aȉ̂悤ɍsƂłF<code><pre>
 *     for (StringSequence seq = new StringSequence(aStr);
 *         seq.validIndex(); seq.next()) {
 *       String chStr = seq.character();     // Ɏ擾B
 *       String subStr = seq.substring();    // ݂̃CfbNXJn镔擾B
 *       ...
 *     }
 *
 *     邢
 *
 *     StringSequence seq = new StringSequence(aStr, -1);
 *     while (seq.hasNext()) {
 *       seq.next();
 *       String ch = seq.character();      // Ɏ擾B
 *       String subStr = seq.substring();  // ݂̃CfbNXJn镔擾B
 *       ...
 *     } 
 *
 *     邢
 *
 *     if (! StringOperation.isEmpty(aStr)) {
 *       StringSequence seq = new StringSequence(aStr, -1);
 *       do {
 *         String chStr = StringOperation.characterAt(seq.index() + 1); // Ɏ擾B
 *         String subStr = seq.followingString();  // ݂̃CfbNX̎Jn镔擾B
 *         ...
 *       } while (seq.next().validIndex());
 *     }</pre></code>
 * Ĺ̗A╶̒lB
 * 
 * @author  V.
 * @version $Revision: 1.3 $, $Date: 2007/06/25 16:20:46 $
 */
public final class StringSequence implements Cloneable
{
  /** Ώۂ̕B */
  private final String str_ ;

  /** Ώۂ̃̕R[hE|CgB */
  private final int codePointCount_ ;

  /** <code>char</code>CfbNXB */
  private int charIndex_ ;

  /** R[hE|CgECfbNXB */
  private int codePointIndex_ ;

  /**
   * Ώۂ̕ɂƂRXgN^B
   *
   * @param  str Ώۂ̕B
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public StringSequence(String str)
  {
    assert (str != null) : "@param:str is null.";
    
    str_ = str;
    codePointCount_ = str.codePointCount(0, str.length());
    charIndex_ = 0;
    codePointIndex_ = 0;
  }

  /**
   * Ώۂ̕ƊJnCfbNXɂƂRXgN^B
   * <br>
   * ȂAłCfbNXƂ́AR[hE|CgPʂ̐擪̏
   * łBCfbNX̎蓾͈͂́A<tt>-1`n</tt>in̓R[hE|Cg
   * Pʂ̕񒷁jłB
   *
   * @param  str Ώۂ̕B
   * @param  beginCodePointIndex JnCfbNXB
   * @throws IndexOutOfBoundsException JnCfbNX͈͊ȌꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public StringSequence(String str, int beginCodePointIndex)
    throws IndexOutOfBoundsException
  {
    assert (str != null) : "@param:str is null.";

    int ind;
    if (beginCodePointIndex == -1) {
      ind = -1;
    }
    else {
      ind = StringOperation.offsetByCodePoints(str, 0, beginCodePointIndex);
    }

    str_ = str;
    codePointCount_ = str.codePointCount(0, str.length());
    charIndex_ = ind;
    codePointIndex_ = beginCodePointIndex;
  }

  /**
   * Rs[ERXgN^B
   *
   * @param seq {@link ts.util.text.StringSequence StringSequence}IuWFNgB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  protected StringSequence(StringSequence seq)
  {
    assert (seq != null) : "@param:seq is null.";

    this.str_ = seq.str_;
    this.codePointCount_ = seq.codePointCount_ ;
    this.charIndex_ = seq.charIndex_;
    this.codePointIndex_ = seq.codePointIndex_;
  }

  /**
   * ̃IuWFNg̃Rs[쐬B
   *
   * @return ̃IuWFNg̃Rs[B
   */
  public StringSequence copy()
  {
    return new StringSequence(this);
  }

  /**
   * ̓eǂmFB
   * <br>
   * ݂̃CfbNXJn镔̒lrāAꂪꍇ
   * <tt>true</tt>AȂꍇ<tt>false</tt>ԂB
   * <br>
   *  {@link ts.util.text.StringSequence StringSequence}IuWFNg
   * Ȃꍇ͏<tt>false</tt>ԂB
   *
   * @param  obj rs{@link ts.util.text.StringSequence StringSequence}
   *           IuWFNgB
   * @return ݂̃CfbNXɂlꍇ<tt>true</tt>ԂB
   */
  public boolean equals(Object obj)
  {
    if (obj != null && obj instanceof StringSequence) {
      return substring().equals(StringSequence.class.cast(obj).substring());
    }

    return false;
  }

  /**
   * ̃IuWFNg̃nbVER[hl擾B
   *
   * @return nbVER[hlB
   */
  public int hashCode()
  {
    return substring().hashCode();
  }

  /**
   * ݂̃CfbNX͈͓i0`n-1An̓R[hE|CgPʂ̕񒷁j
   * ɂ邩ǂmFB
   * <br>
   * ȂAłCfbNXƂ́AR[hE|CgPʂ̒lB
   *
   * @return ݂̃CfbNX͈͓ɂꍇ<tt>true</tt>ԂB
   */
  public boolean validIndex()
  {
    return (0 <= charIndex_ && charIndex_ < str_.length()) ? true : false;
  }

  /**
   * 㑱݂̕邩ǂmFB
   *
   * @return 㑱݂̕<tt>true</tt>ԂB
   */
  public boolean hasNext()
  {
    return (codePointIndex_ < codePointCount_ - 1);
  }

  /**
   * Oɕ݂邩ǂmFB
   *
   * @return Oɕ݂<tt>true</tt>ԂB
   */
  public boolean hasPrevious()
  {
    return (codePointIndex_ > 0);
  }

  /**
   * CfbNXAR[hE|CgPɐi߂B
   * <br>
   * CfbNX̎蓾͈͂-1`nin̓R[hE|CgPʂ̕񒷁j
   * Aɐi߂Ƃ͈̔͂Oꍇ́ACfbNXړȂB
   * <br>
   * ȂAłCfbNXƂ́AR[hE|CgPʂ̒lB
   *
   * @return CfbNXړ̂̃IuWFNgB
   */
  public StringSequence next()
  {
    return next(1);
  }

  /**
   * CfbNXAw肳ꂽɐi߂B
   * <br>
   * ɕ̒lw肵ꍇ́Aw肳ꂽOɖ߂B
   * <br>
   * CfbNX̎蓾͈͂-1`nin̓R[hE|CgPʂ̕񒷁j
   * Aɐi߂Ƃ͈̔͂Oꍇ́A-1nɃ~bgB
   * <br>
   * ȂAłCfbNX╶́AR[hE|CgPʂ̒lB
   *
   * @param  codePointCount ɐi߂镶B
   * @return CfbNXړ̂̃IuWFNgB
   */
  public StringSequence next(int codePointCount)
  {
    if (codePointCount > 0) {
      _next(codePointCount);
    }
    else if (codePointCount < 0){
      _prev(-codePointCount);
    }

    return this;
  }

  /**
   * CfbNXR[hE|CgPOɖ߂B
   * <br>
   * CfbNX̎蓾͈͂-1`nin̓R[hE|CgPʂ̕񒷁j
   * AOɖ߂Ƃ͈̔͂Oꍇ́ACfbNXړȂB
   * <br>
   * ȂAłCfbNXƂ́AR[hE|CgPʂ̒lB
   *
   * @return CfbNXړ̂̃IuWFNgB
   */
  public StringSequence previous()
  {
    return previous(1);
  }

  /**
   * CfbNXAw肳ꂽOɖ߂B
   * <br>
   * CfbNXɕ̒lw肵ꍇ́Aw肳ꂽɐi߂B
   * <br>
   * CfbNX̎蓾͈͂-1`nin̓R[hE|CgPʂ̕񒷁j
   * AOɖ߂Ƃ͈̔͂Oꍇ́A-1nɃ~bgB
   * <br>
   * ȂAłCfbNX╶́AR[hE|CgPʂ̒lB
   *
   * @param  codePointCount Oɖ߂B
   * @return CfbNXړ̂̃IuWFNgB
   */
  public StringSequence previous(int codePointCount)
  {
    if (codePointCount > 0) {
      _prev(codePointCount);
    }
    else if (codePointCount < 0){
      _next(-codePointCount);
    }

    return this;
  }

  /**
   * CfbNXAw肳ꂽɐi߂B
   * <br>
   * ɂ͕K̒lw肳邱ƂOƂĂB
   * <br>
   * ɐi߂ƃCfbNX̎蓾l͈̔́i<tt>-1`n</tt>A<tt>n</tt>
   * ̓R[hE|CgPʂ̕񒷁jOꍇ́A<tt>n</tt>ݒ肷B
   * <br>
   * ȂAłCfbNX╶́AR[hE|CgPʂ̒lB
   *
   * @param  codePointCount ɐi߂镶B
   */
  private void _next(int codePointCount)
  {
    assert (codePointCount > 0);

    int cpind = codePointIndex_ + codePointCount;
    try {
      charIndex_ = StringOperation.offsetByCodePoints(str_, 0, cpind);
      codePointIndex_ = cpind;
    }
    catch (Exception e) {
      charIndex_ = str_.length();
      codePointIndex_ = codePointCount_ ;
    }
  }

  /**
   * CfbNXAw肳ꂽOɖ߂B
   * <br>
   * ɂ͕K̒lw肳邱ƂOƂĂB
   * <br>
   * ɐi߂ƃCfbNX̎蓾l͈̔́i<tt>-1`n</tt>A<tt>n</tt>
   * ̓R[hE|CgPʂ̕񒷁jOꍇ́A<tt>-1</tt>ݒ肷B
   * <br>
   * ȂAłCfbNX╶́AR[hE|CgPʂ̒lB
   *
   * @param  codePointCount ɐi߂镶B
   */
  private void _prev(int codePointCount)
  {
    assert (codePointCount > 0);

    int cpind = codePointIndex_ - codePointCount;
    try {
      charIndex_ = StringOperation.offsetByCodePoints(str_, 0, cpind);
      codePointIndex_ = cpind;
    }
    catch (Exception e) {
      charIndex_ = -1;
      codePointIndex_ = -1;
    }
  }

  /**
   * ݂̃CfbNX擾B
   * <br>
   * ȂAłCfbNXƂ́AR[hE|CgPʂ̒lB
   *
   * @return ݂̃CfbNXB
   */
  public int index()
  {
    return codePointIndex_ ;
  }

  /**
   * ݂̃CfbNXɈʒu镶擾B
   * <br>
   * ȂAłƂ́AR[hE|CgɑΉlB
   *
   * @return ݂̃CfbNXɈʒu镶B
   */
  public String character()
  {
    try {
      return str_.substring(charIndex_, 
        StringOperation.offsetByCodePoints(str_, charIndex_, 1));
    }
    catch (IndexOutOfBoundsException e) {
      return "";
    }
  }

  /**
   * ݂̃CfbNXJn镔擾B
   *
   * @return ݂̃CfbNXJn镔B
   */
  public String substring()
  {
    return (charIndex_ <= 0) ? str_ : str_.substring(charIndex_);
  }

  /**
   * ݂̃CfbNXAw肳ꂽ{@link ts.util.text.StringSequence 
   * StringSequence}̎CfbNX̑O܂ł̕擾B
   * <br>
   * ݂̃CfbNXɈʒu镶́AɊ܂܂B
   * ܂ÃIuWFNgCfbNXɈʒu镶́A
   * ܂܂ȂB
   * <br>
   * ̃\bh́Aȉ̑Ɠlł:<code><pre>
   *     StringOperatin.substring(this.substring(), 0, endSeq.index())
   * </pre></code>
   *
   * @param  endSeq 擾镔̏ICfbNXێ{@link
   *           ts.util.text.StringSequence StringSequence}IuWFNgB
   * @return ̃IuWFNgQƂ镶̕B
   * @throws IndexOutOfBoundsException {@link ts.util.text.StringSequence
   *           StringSequence}̃IuWFNgÕCfbNXw
   *           ꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public String substring(StringSequence endSeq)
    throws IndexOutOfBoundsException
  {
    assert (endSeq != null) : "@param:endSeq is null.";

    return this.str_.substring(
      Math.max(charIndex_,0), Math.max(endSeq.charIndex_, 0));
  }

  /**
   * ݂̃CfbNX̎Jn镔擾B
   *
   * @return ݂̃CfbNX̎Jn镔B
   */
  public String followingString()
  {
    if (charIndex_ < 0) {
      return str_ ;
    }
    else if (charIndex_ < str_.length()) {
      return str_.substring(
        StringOperation.offsetByCodePoints(str_, charIndex_, 1));
    }
    else {
      return "";
    }
  }

  /**
   * ݂̃CfbNX̎Aw肳ꂽ{@link ts.util.text.StringSequence 
   * StringSequence}̎CfbNX̑O܂ł̕擾B
   * <br>
   * ݂̃CfbNXɈʒu镶́AɊ܂܂ȂB
   * ܂ÃIuWFNgCfbNXɈʒu镶́A
   * ܂܂ȂB
   * <br>
   * ̃\bh́Aȉ̑Ɠlł:<code><pre>
   *     StringOperatin.substring(this.substring(), 1, endSeq.index())
   * </pre></code>
   * AÃ݂CfbNXƈ̃IuWFNg̎CfbNXꍇ
   * ͋󕶎ԂB
   *
   * @param  endSeq 擾镔̏ICfbNXێ{@link
   *           ts.util.text.StringSequence StringSequence}IuWFNgB
   * @return ̃IuWFNgQƂ镶̕B
   * @throws IndexOutOfBoundsException {@link ts.util.text.StringSequence
   *           StringSequence}̃IuWFNgÕCfbNXw
   *           ꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public String followingString(StringSequence endSeq)
    throws IndexOutOfBoundsException
  {
    assert (endSeq != null) : "@param:endSeq is null.";

    int b = this.charIndex_ ;
    int e = endSeq.charIndex_ ;
    if (b == e) {
      return "";
    }

    int n = str_.length();

    b = (b < 0) ? 0 : ((b >= n) ?
      b : StringOperation.offsetByCodePoints(str_, b, 1));

    e = (e < 0) ? 0 : e;
    return str_.substring(b, e);
  }

  /**
   * ݂̃CfbNX當܂ł̒̕擾B
   * <br>
   * ȂAłCfbNXƂ́AR[hE|CgPʂ̒lB
   *
   * @return ݂̃CfbNX當܂ł̒̕B
   */
  public int restLength()
  {
    return codePointCount_ - codePointIndex_ ;
  }

  /**
   * ݂̃CfbNX當܂ł̕񒆂ŁATɍŏɈv
   * CfbNX擾B
   * <br>
   * T񂪌Ȃꍇ́A̒lԂB
   * <br>
   * ȂAłCfbNXƂ́AR[hE|CgPʂ̐擪̏
   * iOJnjłB
   * <br>
   * ̃\bh́Aȉ̑Ɠlł
   * iT񂪌ꍇj:<code><pre>
   *     this.index() + StringOperation.indexOf(this.substring(), searched)
   * </pre></code>
   *
   * @param  searched TB
   * @return TɍŏɈvCfbNXB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public int indexOf(String searched)
  {
    return StringOperation.indexOf(str_, searched, codePointIndex_);
  }

  /**
   * ݂̃CfbNX當܂ł̕񒆂ŁATɍŌɈv
   * CfbNX擾B
   * <br>
   * T񂪌Ȃꍇ́A̒lԂB
   * <br>
   * ȂAłCfbNXƂ́AR[hE|CgPʂ̐擪̏
   * iOJnjłB
   * <br>
   * ̃\bh́Aȉ̑Ɠlł
   * iT񂪌ꍇj:<code><pre>
   *     this.index() + StringOperation.lastIndexOf(this.substring(), searched)
   * </pre></code>
   *
   * @param  searched TB
   * @return TɍŌɈvCfbNXB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public int lastIndexOf(String searched)
  {
    int cpind = StringOperation.lastIndexOf(this.substring(), searched);
    return (cpind < 0) ? cpind : (Math.max(index(), 0) + cpind); 
  }

  /**
   * ݂̃CfbNX當܂ł̐̕擪AvtBbNXƈv
   * 邩ǂ𔻒肷B
   *
   * @param  prefix vtBbNXB
   * @return vtBbNXƈvꍇ<tt>true</tt>ԂB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public boolean startsWith(String prefix)
  {
    return StringOperation.startsWith(this.substring(), prefix);
  }

  /**
   * ݂̃CfbNX當܂ł̖̕ATtBbNXƈv
   * ǂ𔻒肷B
   *
   * @param  suffix TtBbNXB
   * @return TtBbNXvꍇ<tt>true</tt>ԂB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public boolean endsWith(String suffix)
  {
    return StringOperation.endsWith(this.substring(), suffix);
  }

  /**
   * ݂̃CfbNX當܂ł̐̕擪Aw肳ꂽW
   * ̗vf̂ꂩƈv邩ǂ𔻒肷B
   * <br>
   * vꍇ͂̕ԂAv镶񂪑݂Ȃꍇ̓k
   * ԂB
   *
   * @param  strs WB
   * @return vB
   * @throws AssertionError k̏ꍇA͕W̒Ƀk
   *           󕶎񂪊܂܂ĂꍇifobOE[ĥ݁jB
   */
  public String startsWithOneOf(Collection<String> strs)
  {
    assert (strs != null) : "@param:strs is null.";
    assert (! strs.contains(null)) : "@param:strs contains null.";
    assert (! strs.contains("")) : "@param:strs contains an empty string.";

    for (String s : strs) {
      if (s != null && startsWith(s) && ! StringOperation.isEmpty(s)) {
        return s;
      }
    }

    return null;
  }

  /**
   * ݂̃CfbNX當܂ł̐̕擪Aw肳ꂽW
   * ̗vf̂ꂩƈv邩ǂ𔻒肷B
   * <br>
   * vꍇ͂̕ԂAv镶񂪑݂Ȃꍇ̓k
   * ԂB
   *
   * @param  seqs WB
   * @return vB
   * @throws AssertionError k̏ꍇA͕W̒Ƀk
   *           󕶎񂪊܂܂ĂꍇifobOE[ĥ݁jB
   */
  public StringSequence startsWithOneOf(Collection<StringSequence> seqs)
  {
    assert (seqs != null) : "@param:seqs is null.";
    assert (! seqs.contains(null)) : "@param:seqs contains null.";
    assert (! seqs.contains(new StringSequence(""))) :
      "@param:seqs contains an empty string.";

    for (StringSequence s : seqs) {
      if (s != null && startsWith(s.substring()) && s.restLength() > 0) {
        return s;
      }
    }

    return null;
  }

  /**
   * ݂̃CfbNX當܂ł̕ɁAT񂪊܂܂Ă邩
   * ǂ𔻒肷B
   *
   * @param  searched TB
   * @return T񂪊܂܂Ăꍇ<tt>true</tt>ԂB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public boolean contains(String searched)
  {
    return StringOperation.contains(this.substring(), searched);
  }

  /**
   * w肳ꂽT񂪌܂ŁACfbNXɐi߂B
   * <br>
   * T񂪌Ȃꍇ́ACfbNX̒lniR[hE
   * |CgPʂ̕񒷁jɂȂB
   *
   * @param  searched TB
   * @return CfbNXړ̂̃IuWFNgB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public StringSequence nextUntil(String searched)
  {
    assert (searched != null) : "@param:searched is null.";

    int findIndex = str_.indexOf(searched, charIndex_);
    if (findIndex < 0) {
      charIndex_ = str_.length();
      codePointIndex_ = codePointCount_;
    }
    else if (findIndex > 0) {
      charIndex_ = findIndex;
      codePointIndex_ = str_.codePointCount(0, findIndex);
    }
    else {
      charIndex_ = 0;
      codePointIndex_ = 0;
    }
    return this;
  }

  /**
   * 󔒂łȂ܂ŁACfbNXɐi߂B
   * <br>
   * ړ̃CfbNX́AXLbvJnĂŏɌꂽ󔒂łȂ
   * wĂB
   * <br>
   * ̍Ō܂ŋ󔒕ꍇ́ACfbNX̒lniR[hE
   * |CgPʂ̕񒷁jɂȂB
   *
   * @return CfbNXړ̂̃IuWFNgB
   */
  public StringSequence skipWhitespaces()
  {
    for (int i=Math.max(charIndex_, 0);
        i<str_.length(); i+=str_.codePointCount(i, i+1)) {

      if (! Character.isWhitespace(str_.codePointAt(i))) {
        charIndex_ = i;
        codePointIndex_ = str_.codePointCount(0, i);
        return this;
      }
    }

    charIndex_ = str_.length();
    codePointIndex_ = codePointCount_ ;
    return this;
  }
}

