/*
 * PropertyResourceTest class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.resource;

import ts.tester.function.coverage.FunctionTester;
import ts.tester.function.print.*;
import java.io.*;
import java.util.*;

/**
 * <code>PropertyResource</code>NX̋@\NXB
 *
 * @author  V. 
 * @version $Revision: 1.3 $, $Date: 2007/05/27 16:13:15 $
 */
public class PropertyResourceTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      PrinterGroup group = new PrinterGroup();
      group.addPrinter(new ConsolePrinter());
      group.addPrinter(new HtmlPrinter("SATOH Takayuki"));
      setPrinter(group);

      run(PropertyResourceTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /* -- test case -- */

  public void constructor()
  {
    MSG("ftHgRXgN^̊mFB");

    PropertyResource res = new PropertyResource();
    NOTNULL(res);
    NOTNULL(res.getRootElement());
    EQUAL(res.getBaseElement(), res.getRootElement());
    NULL(res.getPath());
  }

  public void constructor_path()
  {
    MSG("\[Xt@CpXɂƂRXgN^̊mFB");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    try {
      PropertyResource res = new PropertyResource(PATH);
      NOTNULL(res.getRootElement());
      EQUAL(res.getBaseElement(), res.getRootElement());
      EQUAL(res.getPath(), PATH);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void constructor_path_1()
  {
    MSG("w肳ꂽt@C݂Ȃꍇ");

    final String PATH = "test/sample.properties";

    try {
      PropertyResource res = new PropertyResource(PATH);
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void constructor_path_2()
  {
    MSG("k̏ꍇ");

    try {
      PropertyResource res = new PropertyResource((String) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copy_constructor()
  {
    MSG("Rs[RXgN^̊mF");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    try {
      PropertyResource res2 = new PropertyResource(res);
      EQUAL(res2.getRootElement(), res.getRootElement());
      EQUAL(res2.getBaseElement(), res.getBaseElement());
      EQUAL(res2.getPath(), res.getPath());
    } catch (Exception e) {
      NG();
    }
  }

  public void copy_constructor_1()
  {
    MSG("k̏ꍇ");

    try {
       PropertyResource res = new PropertyResource((PropertyResource) null);
       NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getFirstValue()
  {
    MSG("w肳ꂽL[ɍŏɊYl擾郁\bh̊mFB");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";
    
    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getFirstValue(""), "");
    EQUAL(res.getFirstValue("sample"), "");
    EQUAL(res.getFirstValue("sample.title"), "^Cg");
    EQUAL(res.getFirstValue("sample.body"), "{B");
    EQUAL(res.getFirstValue("sample.body.a"), "");
    EQUAL(res.getFirstValue("sample.body.b"), "");
    EQUAL(res.getFirstValue("sample.body.c"), "");
    EQUAL(res.getFirstValue("aaaa.bbbb.cccc"), "");
  }

  public void getFirstValue_0()
  {
    MSG("L[ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getFirstValue("sample.body(type=typeA)"), "{B");
    EQUAL(res.getFirstValue("sample.body.a(id=a1)"), "");
    EQUAL(res.getFirstValue("sample.body.a(name=A1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b1)"), ""); 
    EQUAL(res.getFirstValue("sample.body.b(id=b2)"), "т"); 
    EQUAL(res.getFirstValue("sample.body.b(id=b3)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c1)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c2)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c3)"), "           ");
  }

  public void getFirstValue_1()
  {
    MSG("k̏ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.getFirstValue(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getFirstValue_2()
  {
    MSG("݂ȂL[w肵ꍇ");

    PropertyResource res = new PropertyResource();
    EQUAL(res.getFirstValue("AAAA.BBBB.CCCC"), "");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getFirstValue("AAAA.BBBB.CCCC"), "");
  }

  public void getFirstValue_3()
  {
    MSG("q\[Xɑ΂Ďsꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    Resource res2 = res.getFirstChild("sample.body");
    EQUAL(res2.getFirstValue(""), "{B");
    EQUAL(res2.getFirstValue("a"), "");
    EQUAL(res2.getFirstValue("b"), "");
    EQUAL(res2.getFirstValue("c"), "");
    EQUAL(res2.getFirstValue("a(id=a1)"), "");
    EQUAL(res2.getFirstValue("a(name=A1)"), "");
    EQUAL(res2.getFirstValue("b(id=b1)"), "");
    EQUAL(res2.getFirstValue("b(id=b2)"), "т");
    EQUAL(res2.getFirstValue("b(id=b3)"), "");
    EQUAL(res2.getFirstValue("c(id=c1)"), "");
    EQUAL(res2.getFirstValue("c(id=c2)"), "");
    EQUAL(res2.getFirstValue("c(id=c3)"), "           ");
  }

  public void getValues()
  {
    MSG("w肳ꂽL[ɊYSĂ̒l擾郁\bh̊mFB");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    List<String> vL = res.getValues("");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample.title");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "^Cg");

    vL = res.getValues("sample.body");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "{B");

    vL = res.getValues("sample.body.a");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample.body.b");
    EQUAL(vL.size(), 3);
    EQUAL(vL.get(0), "");
    EQUAL(vL.get(1), "т");
    EQUAL(vL.get(2), "");

    vL = res.getValues("sample.body.c");
    EQUAL(vL.size(), 3);
    EQUAL(vL.get(0), "");
    EQUAL(vL.get(1), "");
    EQUAL(vL.get(2), "           ");

    vL = res.getValues("aaaa.bbbb.cccc");
    EQUAL(vL.size(), 3);
    EQUAL(vL.get(0), "");
    EQUAL(vL.get(1), "");
    EQUAL(vL.get(2), "");
  }

  public void getValues_1()
  {
    MSG("k̏ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.getValues(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getValues_2()
  {
    MSG("݂ȂL[w肵ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    EQUAL(res.getValues("AAAA.BBBB.CCCC").size(), 0);

    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getValues("AAAA.BBBB.CCCC").size(), 0);
  }

  public void getValues_3()
  {
    MSG("L[w肵ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    List<String> vL = res.getValues("sample.body(type=typeA)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "{B");

    vL = res.getValues("sample.body.a(id=a1)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample.body.a(name=A1)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample.body.b(id=b1)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample.body.b(id=b2)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "т");

    vL = res.getValues("sample.body.b(id=b3)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample.body.c(id=c1)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample.body.c(id=c2)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res.getValues("sample.body.c(id=c3)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "           ");
  }

  public void getValues_4()
  {
    MSG("q\[Xɑ΂Ďsꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    Resource res2 = res.getFirstChild("sample.body");

    List<String> vL = res2.getValues("");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "{B");

    vL = res2.getValues("a(id=a1)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res2.getValues("a(name=A1)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res2.getValues("b(id=b1)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res2.getValues("b(id=b2)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "т");

    vL = res2.getValues("b(id=b3)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res2.getValues("c(id=c1)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res2.getValues("c(id=c2)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "");

    vL = res2.getValues("c(id=c3)");
    EQUAL(vL.size(), 1);
    EQUAL(vL.get(0), "           ");
  }

  public void setFirstValue()
  {
    MSG("w肳ꂽL[ɊYvfɒlݒ肷郁\bh̊mFB");

    PropertyResource res = new PropertyResource();
    EQUAL(res.getFirstValue("sample.aaa"), "");
    EQUAL(res.getFirstValue("sample.bbb"), "");

    res.setFirstValue("sample.aaa", "");
    EQUAL(res.getFirstValue("sample.aaa"), "");
    EQUAL(res.getFirstValue("sample.bbb"), "");

    res.setFirstValue("sample.bbb", "");
    EQUAL(res.getFirstValue("sample.aaa"), "");
    EQUAL(res.getFirstValue("sample.bbb"), "");

    res.setFirstValue("sample.aaa", "    ");
    EQUAL(res.getFirstValue("sample.aaa"), "    ");
    EQUAL(res.getFirstValue("sample.bbb"), "");

    res.setFirstValue("sample.aaa", "");
    EQUAL(res.getFirstValue("sample.aaa"), "");
    EQUAL(res.getFirstValue("sample.bbb"), "");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    try {
       res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getFirstValue("sample.title"), "^Cg");
    EQUAL(res.getFirstValue("sample.aaa"), "");
    EQUAL(res.getFirstValue("sample.bbb"), "");

    res.setFirstValue("sample.title", "");
    EQUAL(res.getFirstValue("sample.title"), "");
    EQUAL(res.getFirstValue("sample.aaa"), "");
    EQUAL(res.getFirstValue("sample.bbb"), "");

    res.setFirstValue("sample.title", "");
    EQUAL(res.getFirstValue("sample.title"), "");
    EQUAL(res.getFirstValue("sample.aaa"), "");
    EQUAL(res.getFirstValue("sample.bbb"), "");
  }

  public void setFirstValue_1()
  {
    MSG("k̏ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.setFirstValue(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.setFirstValue(null, "");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.setFirstValue("sample.aaa", null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void setFirstValue_2()
  {
    MSG("̏ꍇ");

    PropertyResource res = new PropertyResource();
    EQUAL(res.getFirstValue(""), "");

    try {
      res.setFirstValue("", "");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    EQUAL(res.getFirstValue(""), "");
  }

  public void setFirstValue_3()
  {
    MSG("sȏꍇ");

    PropertyResource res = new PropertyResource();

    try {
      res.setFirstValue("aaa.", "xyz");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.setFirstValue("aaa.bbb(X=", "");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void setFirstValue_4()
  {
    MSG("̗vfYL[w肵ꍇ");

    PropertyResource res = new PropertyResource();

    res.setFirstValue("aaa.bbb(x=0)", "XXX");
    EQUAL(res.getFirstValue("aaa.bbb"), "XXX");
    EQUAL(res.getFirstValue("aaa.bbb(x=0)"), "XXX");
    EQUAL(res.getFirstValue("aaa.bbb(x=1)"), "");
    EQUAL(res.getFirstValue("aaa.bbb(x=2)"), "");

    res.setFirstValue("aaa.bbb(x=1)", "YYY");
    EQUAL(res.getFirstValue("aaa.bbb"), "XXX");
    EQUAL(res.getFirstValue("aaa.bbb(x=0)"), "XXX");
    EQUAL(res.getFirstValue("aaa.bbb(x=1)"), "YYY");
    EQUAL(res.getFirstValue("aaa.bbb(x=2)"), "");

    res.setFirstValue("aaa.bbb(x=2)", "ZZZ");
    EQUAL(res.getFirstValue("aaa.bbb"), "XXX");
    EQUAL(res.getFirstValue("aaa.bbb(x=0)"), "XXX");
    EQUAL(res.getFirstValue("aaa.bbb(x=1)"), "YYY");
    EQUAL(res.getFirstValue("aaa.bbb(x=2)"), "ZZZ");

    res.setFirstValue("aaa.bbb", "ABC");
    EQUAL(res.getFirstValue("aaa.bbb"), "ABC");
    EQUAL(res.getFirstValue("aaa.bbb(x=0)"), "ABC");
    EQUAL(res.getFirstValue("aaa.bbb(x=1)"), "YYY");
    EQUAL(res.getFirstValue("aaa.bbb(x=2)"), "ZZZ");
  }

  public void setFirstValue_5()
  {
    MSG("qvf̏ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      final String PATH =
        "test/data/ts/util/resource/PropertyResource/sample.properties";
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getFirstValue("sample.body"), "{B");
    EQUAL(res.getFirstValue("sample.body.a"), "");
    EQUAL(res.getFirstValue("sample.body.b"), "");
    EQUAL(res.getFirstValue("sample.body.c"), "");
    EQUAL(res.getFirstValue("sample.body.a(id=a1)"), "");
    EQUAL(res.getFirstValue("sample.body.a(name=A1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b2)"), "т");
    EQUAL(res.getFirstValue("sample.body.b(id=b3)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c1)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c2)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c3)"), "           ");

    Resource res2 = res.getFirstChild("sample.body");
    res2.setFirstValue("d", "͂ɂقւ");
    EQUAL(res.getFirstValue("sample.body"), "{B");
    EQUAL(res.getFirstValue("sample.body.a"), "");
    EQUAL(res.getFirstValue("sample.body.b"), "");
    EQUAL(res.getFirstValue("sample.body.c"), "");
    EQUAL(res.getFirstValue("sample.body.a(id=a1)"), "");
    EQUAL(res.getFirstValue("sample.body.a(name=A1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b2)"), "т");
    EQUAL(res.getFirstValue("sample.body.b(id=b3)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c1)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c2)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c3)"), "           ");
    EQUAL(res.getFirstValue("sample.body.d"), "͂ɂقւ");

    res2.setFirstValue("a", "");
    EQUAL(res.getFirstValue("sample.body"), "{B");
    EQUAL(res.getFirstValue("sample.body.a"), "");
    EQUAL(res.getFirstValue("sample.body.b"), "");
    EQUAL(res.getFirstValue("sample.body.c"), "");
    EQUAL(res.getFirstValue("sample.body.a(id=a1)"), "");
    EQUAL(res.getFirstValue("sample.body.a(name=A1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b2)"), "т");
    EQUAL(res.getFirstValue("sample.body.b(id=b3)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c1)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c2)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c3)"), "           ");
    EQUAL(res.getFirstValue("sample.body.d"), "͂ɂقւ");
  }

  public void removeChildren()
  {
    MSG("w肳ꂽL[ɊYSĂ̗vf폜郁\bh̊mFB");

    PropertyResource res = new PropertyResource();
    EQUAL(res.getFirstValue("aaa"), "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    EQUAL(res.getFirstChild("aaa"), Resource.EMPTY);
    EQUAL(res.getFirstChild("aaa.bbb"), Resource.EMPTY);

    res.setFirstValue("aaa.bbb", "");
    EQUAL(res.getFirstValue("aaa"), "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    FALSE(res.getFirstChild("aaa") == Resource.EMPTY);
    FALSE(res.getFirstChild("aaa.bbb") == Resource.EMPTY);

    res.removeChildren("aaa.bbb");
    EQUAL(res.getFirstValue("aaa"), "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    FALSE(res.getFirstChild("aaa") == Resource.EMPTY);
    EQUAL(res.getFirstChild("aaa.bbb"), Resource.EMPTY);

    res.setFirstValue("aaa.bbb", "");
    EQUAL(res.getFirstValue("aaa"), "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    FALSE(res.getFirstChild("aaa") == Resource.EMPTY);
    FALSE(res.getFirstChild("aaa.bbb") == Resource.EMPTY);

    res.removeChildren("aaa");
    EQUAL(res.getFirstValue("aaa"), "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    EQUAL(res.getFirstChild("aaa"), Resource.EMPTY);
    EQUAL(res.getFirstChild("aaa.bbb"), Resource.EMPTY);
  }

  public void removeChildren_1()
  {
    MSG("󕶎̏ꍇ");

    PropertyResource res = new PropertyResource();
    EQUAL(res.getFirstValue(""), "");

    PropertyResource res2 = new PropertyResource(res);
    EQUAL(res2.getFirstValue(""), "");
    EQUAL(res2.getRootElement(), res.getRootElement());
    EQUAL(res2.getBaseElement(), res.getBaseElement());

    res.removeChildren("");
    EQUAL(res.getFirstValue(""), "");
    FALSE(res.getBaseElement() == res2.getBaseElement());
    FALSE(res.getRootElement() == res2.getRootElement());

    res2 = new PropertyResource(res);
    EQUAL(res.getFirstValue(""), "");
    EQUAL(res2.getFirstValue(""), "");
    EQUAL(res2.getRootElement(), res.getRootElement());
    EQUAL(res2.getBaseElement(), res.getBaseElement());

    res.setFirstValue("aaa.bbb", "");
    EQUAL(res.getFirstValue(""), "");
    EQUAL(res2.getFirstValue(""), "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    EQUAL(res2.getFirstValue("aaa.bbb"), "");
    EQUAL(res2.getRootElement(), res.getRootElement());
    EQUAL(res2.getBaseElement(), res.getBaseElement());

    res.removeChildren("");
    EQUAL(res.getFirstValue(""), "");
    EQUAL(res2.getFirstValue(""), "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    EQUAL(res2.getFirstValue("aaa.bbb"), "");
    FALSE(res.getBaseElement() == res2.getBaseElement());
    FALSE(res.getRootElement() == res2.getRootElement());
  }

  public void removeChildren_2()
  {
    MSG("k̏ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.removeChildren(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void removeChildren_3()
  {
    MSG("sȏꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.removeChildren("aaa.");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.removeChildren("aaa.bbb(X=");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }
  
  public void removeChildren_4()
  {
    MSG("̗vf폜ꍇ");

    PropertyResource res = new PropertyResource();
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    EQUAL(res.getFirstChild("aaa.bbb"), Resource.EMPTY);

    res.removeChildren("aaa.bbb");

    res.setFirstValue("aaa.bbb(X=1)", "");
    res.setFirstValue("aaa.bbb(X=2)", "");
    res.setFirstValue("aaa.bbb(X=3)", "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    EQUAL(res.getFirstValue("aaa.bbb(X=1)"), "");
    EQUAL(res.getFirstValue("aaa.bbb(X=2)"), "");
    EQUAL(res.getFirstValue("aaa.bbb(X=3)"), "");

    res.removeChildren("aaa.bbb");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    EQUAL(res.getFirstChild("aaa.bbb"), Resource.EMPTY);

    EQUAL(res.getFirstValue("aaa.bbb(X=1)"), "");
    EQUAL(res.getFirstValue("aaa.bbb(X=2)"), "");
    EQUAL(res.getFirstValue("aaa.bbb(X=3)"), "");
  }

  public void removeChildren_6()
  {
    MSG("qvf̏ꍇ");

    PropertyResource res = new PropertyResource();

    res.setFirstValue("aaa.bbb(X=1)", "");
    res.setFirstValue("aaa.bbb(X=2)", "");
    res.setFirstValue("aaa.bbb(X=3)", "");
    EQUAL(res.getFirstValue("aaa.bbb"), "");
    EQUAL(res.getFirstValue("aaa.bbb(X=1)"), "");
    EQUAL(res.getFirstValue("aaa.bbb(X=2)"), "");
    EQUAL(res.getFirstValue("aaa.bbb(X=3)"), "");

    Resource res2 = res.getFirstChild("aaa");

    EQUAL(res2.getFirstValue("bbb"), "");
    EQUAL(res2.getFirstValue("bbb(X=1)"), "");
    EQUAL(res2.getFirstValue("bbb(X=2)"), "");
    EQUAL(res2.getFirstValue("bbb(X=3)"), "");

    res2.removeChildren("bbb");
    EQUAL(res2.getFirstValue("bbb"), "");
    EQUAL(res2.getFirstChild("bbb"), Resource.EMPTY);

    EQUAL(res2.getFirstValue("bbb(X=1)"), "");
    EQUAL(res2.getFirstValue("bbb(X=2)"), "");
    EQUAL(res2.getFirstValue("bbb(X=3)"), "");
  }

  public void getFirstChild()
  {
    MSG("w肳ꂽL[ɊYŏ̗vf擾郁\bh̊mFB");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getFirstValue("sample.title"), "^Cg");
    EQUAL(res.getFirstValue("sample.body"), "{B");
    EQUAL(res.getFirstValue("sample.body.a(id=a1,name=A1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b1,name=B1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b2,name=B2)"), "т");
    EQUAL(res.getFirstValue("sample.body.b(id=b3)"), "");

    PropertyResource child = (PropertyResource) res.getFirstChild("");
    EQUAL(child.getBaseElement(), res.getBaseElement());
    EQUAL(child.getFirstValue("sample.title"), "^Cg");
    EQUAL(child.getFirstValue("sample.body"), "{B");
    EQUAL(child.getFirstValue("sample.body.a(id=a1,name=A1)"), "");
    EQUAL(child.getFirstValue("sample.body.b(id=b1,name=B1)"), "");
    EQUAL(child.getFirstValue("sample.body.b(id=b2,name=B2)"), "т");
    EQUAL(child.getFirstValue("sample.body.b(id=b3)"), "");

    child = (PropertyResource) res.getFirstChild("sample");
    EQUAL(child.getBaseElement().getName(), "sample");
    EQUAL(child.getFirstValue("title"), "^Cg");
    EQUAL(child.getFirstValue("body"), "{B");
    EQUAL(child.getFirstValue("body.a(id=a1,name=A1)"), "");
    EQUAL(child.getFirstValue("body.b(id=b1,name=B1)"), "");
    EQUAL(child.getFirstValue("body.b(id=b2,name=B2)"), "т");
    EQUAL(child.getFirstValue("body.b(id=b3)"), "");

    child = (PropertyResource) child.getFirstChild("body.b");
    EQUAL(child.getFirstValue(""), "");
  }

  public void getFirstChild_1()
  {
    MSG("k̏ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.getFirstChild(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getFirstChild_2()
  {
    MSG("󕶎̏ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    PropertyResource res2 = (PropertyResource) res.getFirstChild("");
    EQUAL(res.getRootElement(), res2.getRootElement());
    EQUAL(res.getBaseElement(), res2.getBaseElement());
  }

  public void getFirstChild_3()
  {
    MSG("sȏꍇ");
    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.getFirstChild("aaa.");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.getFirstChild("aaa.bbb(X=");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getFirstChild_4()
  {
    MSG("܂ރL[w肵ꍇ");
    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    PropertyResource child;
    child = (PropertyResource) res.getFirstChild("sample.body(type=typeA)");
    EQUAL(child.getBaseElement().getName(), "body");
    EQUAL(child.getBaseElement().getAttributes().get("type"), "typeA");
    EQUAL(child.getBaseElement().getValue(), "{B");

    child = (PropertyResource) res.getFirstChild("sample.body.a(id=a1)");
    EQUAL(child.getBaseElement().getName(), "a");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "a1");
    EQUAL(child.getBaseElement().getAttributes().get("name"), "A1");
    EQUAL(child.getBaseElement().getValue(), "");

    child = (PropertyResource) res.getFirstChild("sample.body.b(id=b1)");
    EQUAL(child.getBaseElement().getName(), "b");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "b1");
    EQUAL(child.getBaseElement().getAttributes().get("name"), "B1");
    EQUAL(child.getBaseElement().getValue(), "");

    child = (PropertyResource)res.getFirstChild("sample.body.b(id=b2,name=B2)");
    EQUAL(child.getBaseElement().getName(), "b");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "b2");
    EQUAL(child.getBaseElement().getAttributes().get("name"), "B2");
    EQUAL(child.getBaseElement().getValue(), "т");

    child = (PropertyResource) res.getFirstChild("sample.body.b(id=b3)");
    EQUAL(child.getBaseElement().getName(), "b");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "b3");
    EQUAL(child.getBaseElement().getAttributes().get("name"), null);
    EQUAL(child.getBaseElement().getValue(), "");
  }

  public void getFirstChild_5()
  {
    MSG("̗vfYL[w肵ꍇ");
    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    } 

    Resource res2 = res.getFirstChild("sample.body.a");
    EQUAL(res2.getFirstValue(""), "");

    res2 = res.getFirstChild("sample.body.b");
    EQUAL(res2.getFirstValue(""), "");

    res2 = res.getFirstChild("sample.body.c");
    EQUAL(res2.getFirstValue(""), "");
  }

  public void getFirstChild_6()
  {
    MSG("݂ȂL[w肵ꍇ");
    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getFirstChild("sample.label"), Resource.EMPTY);
    EQUAL(res.getFirstChild("sample.body.b(id=A1)"), Resource.EMPTY);
  }

  public void getFirstChild_7()
  {
    MSG("qvf̏ꍇ");
    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    Resource res2 = res.getFirstChild("sample.body");

    PropertyResource child;
    child = (PropertyResource) res2.getFirstChild("");
    EQUAL(child.getBaseElement().getName(), "body");
    EQUAL(child.getBaseElement().getAttributes().get("type"), "typeA");
    EQUAL(child.getBaseElement().getValue(), "{B");

    child = (PropertyResource) res2.getFirstChild("a");
    EQUAL(child.getBaseElement().getName(), "a");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "a1");
    EQUAL(child.getBaseElement().getAttributes().get("name"), "A1");
    EQUAL(child.getBaseElement().getValue(), "");

    child = (PropertyResource) res2.getFirstChild("a(id=a1)");
    EQUAL(child.getBaseElement().getName(), "a");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "a1");
    EQUAL(child.getBaseElement().getAttributes().get("name"), "A1");
    EQUAL(child.getBaseElement().getValue(), "");

    child = (PropertyResource) res2.getFirstChild("b");
    EQUAL(child.getBaseElement().getName(), "b");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "b1");
    EQUAL(child.getBaseElement().getAttributes().get("name"), "B1");
    EQUAL(child.getBaseElement().getValue(), "");

    child = (PropertyResource) res2.getFirstChild("b(id=b1)");
    EQUAL(child.getBaseElement().getName(), "b");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "b1");
    EQUAL(child.getBaseElement().getAttributes().get("name"), "B1");
    EQUAL(child.getBaseElement().getValue(), "");

    child = (PropertyResource) res2.getFirstChild("b(id=b2,name=B2)");
    EQUAL(child.getBaseElement().getName(), "b");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "b2");
    EQUAL(child.getBaseElement().getAttributes().get("name"), "B2");
    EQUAL(child.getBaseElement().getValue(), "т");

    child = (PropertyResource) res2.getFirstChild("b(id=b3)");
    EQUAL(child.getBaseElement().getName(), "b");
    EQUAL(child.getBaseElement().getAttributes().get("id"), "b3");
    EQUAL(child.getBaseElement().getAttributes().get("name"), null);
    EQUAL(child.getBaseElement().getValue(), "");
  }

  public void getChildren()
  {
    MSG("w肳ꂽL[ɊYSĂ̗vf擾郁\bh̊mFB");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    List<Resource> childL = res.getChildren("sample.body.a");
    EQUAL(childL.size(), 1);
    EQUAL(childL.get(0).getFirstValue(""), "");

    childL = res.getChildren("sample.body.b");
    EQUAL(childL.size(), 3);
    EQUAL(childL.get(0).getFirstValue(""), "");
    EQUAL(childL.get(1).getFirstValue(""), "т");
    EQUAL(childL.get(2).getFirstValue(""), "");

    childL = res.getChildren("sample.body.c");
    EQUAL(childL.size(), 3);
    EQUAL(childL.get(0).getFirstValue(""), "");
    EQUAL(childL.get(1).getFirstValue(""), "");
    EQUAL(childL.get(2).getFirstValue(""), "           ");
  }

  public void getChildren_1()
  {
    MSG("k̏ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.getChildren(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getChildren_2()
  {
    MSG("󕶎̏ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    List<Resource> childL = res.getChildren("");
    EQUAL(childL.size(), 1);
    PropertyResource child = (PropertyResource) childL.get(0);
    EQUAL(child.getBaseElement(), res.getBaseElement());
  }

  public void getChildren_3()
  {
    MSG("sȏꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }
    
    try {
      res.getChildren("aaa.bbb.");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    }

    try {
      res.getChildren(".bbb.ccc");
    } catch (IllegalKeyException e) {
      OK(e);
    }
  }

  public void getChildren_4()
  {
    MSG("܂ރL[w肵ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    List<Resource> childL = res.getChildren("sample.body.a(id=a1)");
    EQUAL(childL.size(), 1);
    EQUAL(childL.get(0).getFirstValue(""), "");

    childL = res.getChildren("sample.body.a(id=a1,name=A1)");
    EQUAL(childL.size(), 1);
    EQUAL(childL.get(0).getFirstValue(""), "");
  }

  public void getChildren_5()
  {
    MSG("݂ȂL[w肵ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    List<Resource> childL = res.getChildren("sample.body.k");
    EQUAL(childL.size(), 0);

    childL = res.getChildren("sample.body.a(id=a2)");
    EQUAL(childL.size(), 0);
  }

  public void getChildren_6()
  {
    MSG("qvf̏ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    Resource child = res.getFirstChild("sample.body");

    List<Resource> cL = child.getChildren("b");
    EQUAL(cL.size(), 3);
    EQUAL(cL.get(0).getFirstValue(""), "");
    EQUAL(cL.get(1).getFirstValue(""), "т");
    EQUAL(cL.get(2).getFirstValue(""), "");

    cL = child.getChildren("b(name=B2)");
    EQUAL(cL.size(), 1);
    EQUAL(cL.get(0).getFirstValue(""), "т");
  }

  public void getOrCreateFirstChild()
  {
    MSG("w肳ꂽL[ɊYvf΂ԂAȂΗvf쐬郁\bh̊mFB");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    Resource child = res.getOrCreateFirstChild("sample.title");
    EQUAL(child.getFirstValue(""), "^Cg");

    child = res.getOrCreateFirstChild("sample.name");
    EQUAL(child.getFirstValue(""), "");
    FALSE(child.equals(Resource.EMPTY));
  }

  public void getOrCreateFirstChild_1()
  {
    MSG("k̏ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.getOrCreateFirstChild(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getOrCreateFirstChild_2()
  {
    MSG("󕶎̏ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    PropertyResource child = (PropertyResource) res.getOrCreateFirstChild("");
    EQUAL(child.getBaseElement(), res.getBaseElement());
    EQUAL(child.getRootElement(), res.getRootElement());
  }

  public void getOrCreateFirstChild_3()
  {
    MSG("sȏꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.getOrCreateFirstChild(".aaa");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res.getOrCreateFirstChild("aaa.bbb.");
      NG();
    } catch (IllegalKeyException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getOrCreateFirstChild_4()
  {
    MSG("܂ރL[w肵ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    PropertyResource c1, c2;
    c1 = (PropertyResource) res.getFirstChild("sample.body.b(id=b2)");
    c2 = (PropertyResource) res.getOrCreateFirstChild("sample.body.b(id=b2)");
    EQUAL(c1.getBaseElement(), c2.getBaseElement());
    EQUAL(c1.getRootElement(), c2.getRootElement());
    EQUAL(c2.getFirstValue(""), "т");
  }

  public void getOrCreateFirstChild_5()
  {
    MSG("̗vfYL[w肵ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    PropertyResource c1, c2;
    c1 = (PropertyResource) res.getFirstChild("sample.body.c(id=c1)");
    c2 = (PropertyResource) res.getOrCreateFirstChild("sample.body.c");
    EQUAL(c1.getBaseElement(), c2.getBaseElement());
    EQUAL(c1.getRootElement(), c2.getRootElement());
    EQUAL(c2.getFirstValue(""), "");
  }

  public void getOrCreateFirstChild_7()
  {
    MSG("݂ȂL[w肵ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    PropertyResource c2, c3;
    Resource c1 = res.getFirstChild("sample.body.d");
    EQUAL(c1, Resource.EMPTY);
    c2 = (PropertyResource) res.getOrCreateFirstChild("sample.body.d");
    FALSE(c2.equals(Resource.EMPTY));

    c3 = (PropertyResource) res.getFirstChild("sample.body.d");
    FALSE(c3.equals(Resource.EMPTY));
    EQUAL(c3.getBaseElement(), c2.getBaseElement());
    EQUAL(c3.getRootElement(), c2.getRootElement());
  }

  public void getOrCreateFirstChild_8()
  {
    MSG("qvf̏ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    PropertyResource child, c1, c2;
    child = (PropertyResource) res.getFirstChild("sample.body");

    c1 = (PropertyResource) res.getFirstChild("sample.body.b(id=b2)");
    c2 = (PropertyResource) child.getOrCreateFirstChild("b(id=b2)");
    EQUAL(c1.getRootElement(), c2.getRootElement());
    EQUAL(c1.getBaseElement(), c2.getBaseElement());
    EQUAL(c1.getFirstValue(""), "т");
    EQUAL(c2.getFirstValue(""), "т");

    EQUAL(res.getFirstChild("sample.body.d(id=d1)"), Resource.EMPTY);

    c2 = (PropertyResource) child.getOrCreateFirstChild("d(id=d1)");
    c1 = (PropertyResource) res.getFirstChild("sample.body.d(id=d1)");
    EQUAL(c1.getRootElement(), c2.getRootElement());
    EQUAL(c1.getBaseElement(), c2.getBaseElement());
  }

  public void load()
  {
    MSG("t@C̓ǂݍ݂̊mFB");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/sample.properties";

    PropertyResource res = new PropertyResource();
    try {
      res.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res.getFirstValue("sample.title"), "^Cg");
    EQUAL(res.getFirstValue("sample.body(type=typeA)"), "{B");
    EQUAL(res.getFirstValue("sample.body.a(id=a1,name=A1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b1,name=B1)"), "");
    EQUAL(res.getFirstValue("sample.body.b(id=b2,name=B2)"), "т");
    EQUAL(res.getFirstValue("sample.body.b(id=b3)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c1,name=C1)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c2,name=C2)"), "");
    EQUAL(res.getFirstValue("sample.body.c(id=c3)"), "           ");
  }

  public void load_1()
  {
    MSG("k̏ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.load((InputStream) null);
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void load_2()
  {
    MSG("݂Ȃt@Cw肵ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.load("aaa.properties");
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void save()
  {
    MSG("vpeBt@Cۑ郁\bh̊mFB");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/aaa.properties";

    File file = new File(PATH);
    FALSE(file.exists());

    PropertyResource res = new PropertyResource();
    res.setFirstValue("abc.def.ghi", "");
    EQUAL(res.getFirstValue("abc.def.ghi"), "");
    try {
      res.save(PATH);
    } catch (Exception e) {
      NG(e);
      e.printStackTrace();
    }

    TRUE(file.exists());

    PropertyResource res2 = new PropertyResource();
    try {
      res2.load(PATH);
    } catch (Exception e) {
      NG(e);
    }

    EQUAL(res2.getFirstValue("abc.def.ghi"), "");

    file.delete();
    FALSE(file.exists());
  }

  public void save_1()
  {
    MSG("k̏ꍇ");

    PropertyResource res = new PropertyResource();
    try {
      res.save((OutputStream) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void save_2()
  {
    MSG("ɑ݂t@Cw肵ꍇ");

    final String PATH =
      "test/data/ts/util/resource/PropertyResource/bbb.properties";

    PropertyResource res = new PropertyResource();
    res.setFirstValue("aaa.bbb(id=b,nm=B).ccc(id=C)", "");
    try {
      res.save(PATH);
    } catch (Exception e) {
      NG(e);
    }

    PropertyResource res2 = new PropertyResource();
    try {
      res2.load(PATH);
    } catch (Exception e) {
      NG(e);
    }
    EQUAL(res2.getFirstValue("aaa.bbb(id=b,nm=B).ccc(id=C)"), "");
    
    res.setFirstValue("aaa.bbb(id=b,nm=B).ccc(id=C)", "");
    try {
      res.save(PATH);
    } catch (Exception e) {
      NG(e);
    }

    try {
      res2.load(PATH);
    } catch (Exception e) {
      NG(e);
    }
    EQUAL(res2.getFirstValue("aaa.bbb(id=b,nm=B).ccc(id=C)"), "");
  }

  public void save_3()
  {
    MSG("ۑɗOꍇB");

    final String PATH = "/cygdrive/XXX/aaa.properties";

    PropertyResource res = new PropertyResource();
    res.setFirstValue("aaa.bbb.ccc", "");
    try {
      res.save(PATH);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }
}
