/*
 * editpolicy.c
 *
 * An editor for editing TOMOYO Linux's policy.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.1.2   2006/06/02
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/types.h>
#include <curses.h>

#define ROOT_NAME "<kernel>"

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(127);
}

/*
 * Check whether the given filename follows the naming rules.
 * Returns nonzero if follows, zero otherwise.
 */
static int IsCorrectPath(const char *filename, const int may_contain_pattern) {
	if (filename && *filename == '/') {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c == '\\') {
				switch ((c = *filename++)) {
				case '\\':  /* "\\" */
					continue;
				case '$':   /* "\$" */
				case '+':   /* "\+" */
				case '?':   /* "\?" */
					if (may_contain_pattern) continue;
					break;
				case '*':   /* "\*" */
					if (may_contain_pattern) {
						while (*filename && *filename != '/') filename++;
						if (!*filename || *filename == '/') continue;
					}
					break;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				return 0;
			} else if (c <= ' ' || c >= 127) {
				return 0;
			}
		}
		return 1;
	}
	return 0;
}

static int IsCorrectDomain(const unsigned char *domainname) {
	unsigned char c, d, e;
	if (!domainname || strncmp(domainname, ROOT_NAME, strlen(ROOT_NAME))) goto out;
	domainname += strlen(ROOT_NAME);
	if (!*domainname) return 1;
	do {
		if (*domainname++ != ' ') goto out;
		if (*domainname++ != '/') goto out;
		while ((c = *domainname) != '\0' && c != ' ') {
			domainname++;
			if (c == '\\') {
				switch ((c = *domainname++)) {
				case '\\':  /* "\\" */
					continue;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *domainname++) >= '0' && d <= '7' && (e = *domainname++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				goto out;
			} else if (c < ' ' || c >= 127) {
				goto out;
			}
		}
	} while (*domainname);
	return 1;
 out:
	return 0;
}

static inline int strendswith(const char *name, const char *tail) {
	int len;
	if (!name || !tail) return 0;
	len = strlen(name) - strlen(tail);
	return len >= 0 && strcmp(name + len, tail) == 0;
}

static void ReadGenericPolicy(void);
static void ReadDomainPolicy(void);
static int FindOrAssignNewDomain(const char *domainname);
static int FindDomain(const char *domainname);
static void NormalizeLine(unsigned char *buffer);
static char *DomainName(const int index);
static int IsTrustedDomain(const int index);
static int IsInitializer(const char *filename);
static int IsInitializerDomain(const int index);
static int IsVirtualDomain(const int index);

#define SYSTEM_POLICY_FILE    "system_policy"
#define EXCEPTION_POLICY_FILE "exception_policy"
#define DOMAIN_POLICY_FILE    "domain_policy"

static const char *policy_file = DOMAIN_POLICY_FILE;
static const char *list_caption = NULL;
static char *current_domain = NULL;

#define SCREEN_SYSTEM_LIST    0
#define SCREEN_EXCEPTION_LIST 1
#define SCREEN_DOMAIN_LIST    2
#define SCREEN_ACL_LIST       3
#define MAXSCREEN             4

static int current_screen = SCREEN_DOMAIN_LIST;

// List for generic policy.
static char **generic_acl_list = NULL;
static int generic_acl_list_count = 0;

// List for domain policy.
typedef struct domain_info {
	char *domainname;
	int is_virtual;
} DOMAIN_INFO;
static DOMAIN_INFO *domain_list = NULL;
static int domain_list_count = 0;
static char **trusted_domainnames = NULL;
static int trusted_domainnames_count = 0;
static char **initializers = NULL;
static int initializers_count = 0;

///////////////////////////  ACL HANDLER  //////////////////////////////

static int IsTrustedDomain(const int index) {
	int i;
	char *domainname = domain_list[index].domainname;
	for (i = 0; i < trusted_domainnames_count; i++) {
		const int len = strlen(trusted_domainnames[i]);
		if (strncmp(domainname, trusted_domainnames[i], len)) continue;
		if (domainname[len] == ' ' || domainname[len] == '\0') return 1;
	}
	return 0;
}

static int IsInitializer(const char *filename) {
	if (filename) {
		int i;
		for (i = 0; i < initializers_count; i++) {
			if (strcmp(filename, initializers[i]) == 0) return 1;
		}
	}
	return 0;
}

static int IsInitializerDomain(const int index) {
	char *cp = strchr(domain_list[index].domainname, ' ');
	if (cp) return IsInitializer(cp + 1);
	return 0;
}

static int IsVirtualDomain(const int index) {
	return domain_list[index].is_virtual;
}

////////////////////////////  DOMAIN HANDLER  ////////////////////////////

static int FindDomain(const char *domainname) {
	int i;
	for (i = 0; i < domain_list_count; i++) {
		if (strcmp(domainname, domain_list[i].domainname) == 0) {
			return i;
		}
	}
	return EOF;
}

static int FindOrAssignNewDomain(const char *domainname) {
	int index;
	if ((index = FindDomain(domainname)) == EOF) {
		if (IsCorrectDomain(domainname)) {
			char *saved_domainname = strdup(domainname);
			if (!saved_domainname || (domain_list = (DOMAIN_INFO *) realloc(domain_list, (domain_list_count + 1) * sizeof(DOMAIN_INFO))) == NULL) OutOfMemory();
			memset(&domain_list[domain_list_count], 0, sizeof(DOMAIN_INFO));
			domain_list[domain_list_count].domainname = saved_domainname;
			index = domain_list_count++;
		} else {
			fprintf(stderr, "%s: Invalid domainname '%s'\n",  __FUNCTION__, domainname);
		}
	}
	return index;
}

static char *DomainName(const int index) {
	return domain_list[index].domainname;
}

///////////////////////////  UTILITY FUNCTIONS  //////////////////////////////

static int IsDomainDef(const unsigned char *buffer) {
	return strncmp(buffer, ROOT_NAME, strlen(ROOT_NAME)) == 0;
}

static void NormalizeLine(unsigned char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}

static int buffer_lock = 0;
static void get(void) {
	if (buffer_lock) OutOfMemory();
	buffer_lock++;
}
static void put(void) {
	if (buffer_lock != 1) OutOfMemory();
	buffer_lock--;
}

#define MAXBUFSIZE  8192
static char buffer[MAXBUFSIZE];

static void ReadGenericPolicy(void) {
	FILE *fp;
	int i, j;
	while (generic_acl_list_count) free(generic_acl_list[--generic_acl_list_count]);
	if ((fp = fopen(policy_file, "r")) != NULL) {
		if (current_screen == SCREEN_ACL_LIST) {
			int flag = 0;
			get();
			while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
				char *cp = strchr(buffer, '\n');
				if (cp) *cp = '\0';
				else if (!feof(fp)) break;
				NormalizeLine(buffer);
				if (IsDomainDef(buffer)) {
					flag = strcmp(buffer, current_domain) == 0 ? 1 : 0;
				} else if (flag && buffer[0]) {
					if ((generic_acl_list = (char **) realloc(generic_acl_list, (generic_acl_list_count + 1) * sizeof(char *))) == NULL
						|| (generic_acl_list[generic_acl_list_count++] = strdup(buffer)) == NULL) OutOfMemory();
				}
			}
			put();
			for (i = 0; i < generic_acl_list_count; i++) {
				for (j = i + 1; j < generic_acl_list_count; j++) {
					char *a = generic_acl_list[i], *b = generic_acl_list[j];
					if (*a && *b && strcmp(a + 1, b + 1) > 0) {
						generic_acl_list[i] = b; generic_acl_list[j] = a;
					}
				}
			}
		} else {
			get();
			while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
				char *cp = strchr(buffer, '\n');
				if (cp) *cp = '\0';
				else if (!feof(fp)) break;
				NormalizeLine(buffer);
				if ((generic_acl_list = (char **) realloc(generic_acl_list, (generic_acl_list_count + 1) * sizeof(char *))) == NULL
					|| (generic_acl_list[generic_acl_list_count++] = strdup(buffer)) == NULL) OutOfMemory();
			}
			put();
			for (i = 0; i < generic_acl_list_count; i++) {
				for (j = i + 1; j < generic_acl_list_count; j++) {
					char *a = generic_acl_list[i], *b = generic_acl_list[j];
					if (strcmp(a, b) > 0) {
						generic_acl_list[i] = b; generic_acl_list[j] = a;
					}
				}
			}
		}
		fclose(fp);
	}
}

static void ReadDomainPolicy(void) {
	FILE *fp;
	int i, j;
	while (domain_list_count) free(domain_list[--domain_list_count].domainname);
	while (initializers_count) free(initializers[--initializers_count]);
	while (trusted_domainnames_count) free(trusted_domainnames[--trusted_domainnames_count]);
	FindOrAssignNewDomain(ROOT_NAME);

	// Load initializer list, trusted domain list.
	if ((fp = fopen(EXCEPTION_POLICY_FILE, "r")) != NULL) {
		get();
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			if (strncmp(buffer, "initializer ", 12) == 0) {
				memmove(buffer, buffer + 12, strlen(buffer + 12) + 1);
				if (!IsCorrectPath(buffer, 0)) continue;
				if ((initializers = (char **) realloc(initializers, (initializers_count + 1) * sizeof(char *))) == NULL
					|| (initializers[initializers_count++] = strdup(buffer)) == NULL) OutOfMemory();
			} else if (strncmp(buffer, "trust_domain ", 13) == 0) {
				memmove(buffer, buffer + 13, strlen(buffer + 13) + 1);
				if (!IsCorrectDomain(buffer)) continue;
				if ((trusted_domainnames = (char **) realloc(trusted_domainnames, (trusted_domainnames_count + 1) * sizeof(char *))) == NULL
					|| (trusted_domainnames[trusted_domainnames_count++] = strdup(buffer)) == NULL) OutOfMemory();
			}
		}
		put();
		fclose(fp);
	}

	// Load all domain list.
	if ((fp = fopen(DOMAIN_POLICY_FILE, "r")) != NULL) {
		int index = EOF;
		get();
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			if (IsDomainDef(buffer)) {
				index = FindOrAssignNewDomain(buffer);
			} else if (index >= 0 && (atoi(buffer) & 1) == 1 && (cp = strchr(buffer, ' ')) != NULL &&
					   IsCorrectPath(cp + 1, 0) && !strendswith(cp + 1, "/") && IsInitializer(cp + 1)) {
				int virtual_index;
				static char buffer2[MAXBUFSIZE];
				memset(buffer2, 0, sizeof(buffer2));
				snprintf(buffer2, sizeof(buffer2) - 1, "%s %s", domain_list[index].domainname, cp + 1);
				NormalizeLine(buffer2);
				if ((virtual_index = FindOrAssignNewDomain(buffer2)) == EOF) {
					fprintf(stderr, "ERROR: Can't create domain for '%s'.\n", buffer2);
					exit(127);
				}
				domain_list[virtual_index].is_virtual = 1;
			}
		}
		put();
		fclose(fp);
	}
	// Clear is_virtual flag for initialized domains.
	for (i = 0; i < domain_list_count; i++) {
		char *cp;
		if (!domain_list[i].is_virtual) continue;
		if ((cp = strchr(domain_list[i].domainname, ' ')) != NULL && strchr(cp + 1, ' ') == NULL) domain_list[i].is_virtual = 0;
	}
	// Sort by domain name.
	for (i = 0; i < domain_list_count; i++) {
		for (j = i + 1; j < domain_list_count; j++) {
			if (strcmp(domain_list[i].domainname, domain_list[j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[i]; domain_list[i] = domain_list[j]; domain_list[j] = tmp;
			}
		}
	}
}

////////////////////////////  UI HANDLER  ////////////////////////////

static int DomainListLoop(void);
static int GenericListLoop(void);
static void ShowList(void);

static void ResizeWindow(void);
static void UpArrowKey(void);
static void DownArrowKey(void);
static void PageUpKey(void);
static void PageDownKey(void);
static void ShowCurrent(void);
static int GetCurrent(void);

static int window_width = 0, window_height = 0;
static int current_y[MAXSCREEN], current_item_index[MAXSCREEN], list_item_count[MAXSCREEN];

static struct {
	int current_y[MAXSCREEN];
	int current_item_index[MAXSCREEN];
	char search_buffer[4096];
} history;

static const int header_lines = 3;
static int body_lines = 0;

static void ShowList(void) {
	const int offset = current_item_index[current_screen];
	int i;
	if (current_screen == SCREEN_DOMAIN_LIST) list_item_count[SCREEN_DOMAIN_LIST] = domain_list_count;
	else list_item_count[current_screen] = generic_acl_list_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	if (current_screen == SCREEN_DOMAIN_LIST) mvprintw(0, 0, "<<< Domain Transition Editor >>>      %d domain%c", list_item_count[SCREEN_DOMAIN_LIST], list_item_count[SCREEN_DOMAIN_LIST] > 1 ? 's' : ' ');
	else mvprintw(0, 0, "<<< %s Editor >>>      %d entr%s", list_caption, list_item_count[current_screen], list_item_count[current_screen] > 1 ? "ies" : "y");
	if (current_screen == SCREEN_DOMAIN_LIST) mvprintw(1, 0, "Commands =  Q:quit  A:append  D:delete  R:refresh  Enter:edit  @:search  Tab:switch");
	else if (current_screen == SCREEN_ACL_LIST) mvprintw(1, 0, "Commands =  Q:quit  A:append  D:delete  R:refresh  Enter:list");
	else mvprintw(1, 0, "Commands =  Q:quit  A:append  D:delete  R:refresh  Tab:switch");
	if (current_screen == SCREEN_ACL_LIST) mvprintw(2, 0, "%s", current_domain);
	for (i = 0; i < body_lines; i++) {
		const int index = offset + i;
		if (index >= list_item_count[current_screen]) break;
		if (current_screen == SCREEN_DOMAIN_LIST) {
			char *sp, *cp;
			mvprintw(header_lines + i, 0, "%5d: %c%c ", index, IsTrustedDomain(index) ? '#' : ' ', IsInitializerDomain(index) ? '*' : ' ');
			sp = DomainName(index);
			while ((cp = strchr(sp, ' ')) != NULL) { printw("    "); sp = cp + 1; }
			printw("%s", sp);
			if (IsVirtualDomain(index)) {
				get();
				memset(buffer, 0, sizeof(buffer));
				snprintf(buffer, sizeof(buffer) - 1, ROOT_NAME "%s", strrchr(DomainName(index), ' '));
				printw(" ( -> %d )", FindDomain(buffer));
				put();
			}
		} else {
			mvprintw(header_lines + i, 0, "%5d: %s", index, generic_acl_list[index]);
		}
	}
	ShowCurrent();
}

static void ResizeWindow(void) {
	getmaxyx(stdscr, window_height, window_width);
	body_lines = window_height - header_lines;
	if (body_lines <= current_y[current_screen]) current_y[current_screen] = body_lines - 1;
	if (current_y[current_screen] < 0) current_y[current_screen] = 0;
}

static void UpArrowKey(void) {
	if (current_y[current_screen] > 0) {
		current_y[current_screen]--;
		ShowCurrent();
	} else if (current_item_index[current_screen] > 0) {
		current_item_index[current_screen]--;
		ShowList();
	}
}

static void DownArrowKey(void) {
	if (current_y[current_screen] < body_lines - 1) {
		if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
			current_y[current_screen]++;
			ShowCurrent();
		}
	} else if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
		current_item_index[current_screen]++;
		ShowList();
	}
}

static void PageUpKey(void) {
	if (current_item_index[current_screen] + current_y[current_screen] > body_lines) {
		current_item_index[current_screen] -= body_lines;
		if (current_item_index[current_screen] < 0) current_item_index[current_screen] = 0;
		ShowList();
	} else if (current_item_index[current_screen] + current_y[current_screen] > 0) {
		current_item_index[current_screen] = 0;
		current_y[current_screen] = 0;
		ShowList();
	}
}

static void PageDownKey(void) {
	if (list_item_count[current_screen] - current_item_index[current_screen] > body_lines) {
		current_item_index[current_screen] += body_lines;
		if (current_item_index[current_screen] + current_y[current_screen] > list_item_count[current_screen] - 1) current_y[current_screen] = list_item_count[current_screen] - 1 - current_item_index[current_screen];
		ShowList();
	} else if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
		current_y[current_screen] = list_item_count[current_screen] - current_item_index[current_screen] - 1;
		ShowCurrent();
	}
}

static int GetCurrent(void) {
	if (list_item_count[current_screen] == 0) return EOF;
	if (current_item_index[current_screen] + current_y[current_screen] < 0 || current_item_index[current_screen] + current_y[current_screen] >= list_item_count[current_screen]) {
		fprintf(stderr, "ERROR: current_item_index=%d current_y=%d\n", current_item_index[current_screen], current_y[current_screen]);
		exit(127);
	}
	return current_item_index[current_screen] + current_y[current_screen];
}

static void ShowCurrent(void) {
	if (current_screen == SCREEN_DOMAIN_LIST) {
		const int current = GetCurrent();
		get();
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "%s", DomainName(current));
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		move(2, 0);
		clrtoeol();
		printw("%s", buffer);
		put();
	}
	move(header_lines + current_y[current_screen], 0);
	refresh();
}

static void LoadCursorPos(const int current_screen) {
	current_item_index[current_screen] = history.current_item_index[current_screen];
	current_y[current_screen] = history.current_y[current_screen];
}

static void AdjustCursorPos(const int item_count) {
	if (item_count == 0) {
		current_item_index[current_screen] = current_y[current_screen] = 0;
	} else {
		while (current_item_index[current_screen] + current_y[current_screen] >= item_count) {
			if (current_y[current_screen] > 0) current_y[current_screen]--;
			else if (current_item_index[current_screen] > 0) current_item_index[current_screen]--;
		}
	}
}

static void SaveCursorPos(const int current_screen) {
	history.current_item_index[current_screen] = current_item_index[current_screen];
	history.current_y[current_screen] = current_y[current_screen];
}

static int GenericListLoop(void) {
	if (current_screen == SCREEN_SYSTEM_LIST) {
		policy_file = SYSTEM_POLICY_FILE;
		list_caption = "System Policy";
	} else if (current_screen == SCREEN_EXCEPTION_LIST) {
		policy_file = EXCEPTION_POLICY_FILE;
		list_caption = "Exception Policy";
	} else {
		policy_file = DOMAIN_POLICY_FILE;
		list_caption = "Domain Policy";
	}
	LoadCursorPos(current_screen);
 start:
	ReadGenericPolicy();
	AdjustCursorPos(generic_acl_list_count);
	ShowList();
	while (1) {
		const int c = getch();
		SaveCursorPos(current_screen);
		if (c == 'q' || c == 'Q') return MAXSCREEN;
		if (current_screen == SCREEN_ACL_LIST) {
			if (c == '\r') return SCREEN_DOMAIN_LIST;
		} else {
			if (c == '\t') {
				if (current_screen == SCREEN_SYSTEM_LIST) return SCREEN_EXCEPTION_LIST;
				else if (current_screen == SCREEN_EXCEPTION_LIST) return SCREEN_DOMAIN_LIST;
			}
		}
		if (c == ERR) continue; // Ignore invalid key.
		switch(c) {
		case KEY_RESIZE:
			ResizeWindow();
			ShowList();
			break;
		case KEY_UP:
			UpArrowKey();
			break;
		case KEY_DOWN:
			DownArrowKey();
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey();
			break;
		case KEY_NPAGE:
			PageDownKey();
			break;
		case 'd':
		case 'D':
			{
				const int index = GetCurrent();
				if (index >= 0 && index < generic_acl_list_count) {
					const int fd = open(policy_file, O_WRONLY);
					if (fd != EOF) {
						if (current_screen == SCREEN_ACL_LIST) {
							write(fd, "select ", 7); write(fd, current_domain, strlen(current_domain)); write(fd, "\n", 1);
						}
						write(fd, "delete ", 7); write(fd, generic_acl_list[index], strlen(generic_acl_list[index])); write(fd, "\n", 1); close(fd);
						close(fd);
					}
					goto start;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				get();
				mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
				memset(buffer, 0, sizeof(buffer));
				getnstr(buffer, sizeof(buffer) - 1); noecho();
				NormalizeLine(buffer);
				if (buffer[0]) {
					const int fd = open(policy_file, O_WRONLY);
					if (fd != EOF) {
						if (current_screen == SCREEN_ACL_LIST) {
							write(fd, "select ", 7); write(fd, current_domain, strlen(current_domain)); write(fd, "\n", 1);
						}
						write(fd, buffer, strlen(buffer)); write(fd, "\n", 1); close(fd);
					}
				}
				put();
				goto start;
			}
			break;
		case 'r':
		case 'R':
			goto start;
		}
	}
}

static int DomainListLoop(void) {
	LoadCursorPos(current_screen);
 start: ;
	ReadDomainPolicy();
	AdjustCursorPos(domain_list_count);
	ShowList();
	while (1) {
		const int c = getch();
		SaveCursorPos(current_screen);
		if (c == 'q' || c == 'Q') return MAXSCREEN;
		if (c == '\t') return SCREEN_SYSTEM_LIST;
		if (c == ERR) continue; // Ignore invalid key.
		switch (c) {
		case KEY_RESIZE:
			ResizeWindow();
			ShowList();
			break;
		case KEY_UP:
			UpArrowKey();
			break;
		case KEY_DOWN:
			DownArrowKey();
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey();
			break;
		case KEY_NPAGE:
			PageDownKey();
			break;
		case '@':
			{
				int i;
			input_path:
				mvprintw(window_height - 1, 0, "Search> "); clrtoeol(); refresh(); echo();
				memset(history.search_buffer, 0, sizeof(history.search_buffer));
				getnstr(history.search_buffer, sizeof(history.search_buffer) - 1); noecho();
				NormalizeLine(history.search_buffer);
				if (history.search_buffer[0]) {
					for (i = 0; i < list_item_count[current_screen]; i++) {
						char *cp = DomainName(i);
						if (strchr(history.search_buffer, '/')) {
							if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
						} else {
							if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
						}
						if (strcmp(cp, history.search_buffer)) continue;
						while (i < current_y[current_screen] + current_item_index[current_screen]) {
							if (current_y[current_screen] > 0) current_y[current_screen]--;
							else current_item_index[current_screen]--;
						}
						while (i > current_y[current_screen] + current_item_index[current_screen]) {
							if (current_y[current_screen] < body_lines - 1) current_y[current_screen]++;
							else current_item_index[current_screen]++;
						}
						break;
					}
				}
				ShowList();
			}
			break;
		case KEY_LEFT:
			{
				int i;
				if (!history.search_buffer[0]) goto input_path;
				for (i = GetCurrent() - 1; i >= 0; i--) {
					char *cp = DomainName(i);
					if (strchr(history.search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, history.search_buffer)) continue;
					while (i < current_y[current_screen] + current_item_index[current_screen]) {
						if (current_y[current_screen] > 0) current_y[current_screen]--;
						else current_item_index[current_screen]--;
					}
					ShowList();
					break;
				}
			}
			break;
		case KEY_RIGHT:
			{
				int i;
				if (!history.search_buffer[0]) goto input_path;
				for (i = GetCurrent() + 1; i < list_item_count[current_screen]; i++) {
					char *cp = DomainName(i);
					if (strchr(history.search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, history.search_buffer)) continue;
					while (i > current_y[current_screen] + current_item_index[current_screen]) {
						if (current_y[current_screen] < body_lines - 1) current_y[current_screen]++;
						else current_item_index[current_screen]++;
					}
					ShowList();
					break;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				const int current = GetCurrent();
				if (!IsVirtualDomain(current)) {
					get();
					mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
					memset(buffer, 0, sizeof(buffer));
					getnstr(buffer, sizeof(buffer) - 1); noecho();
					NormalizeLine(buffer);
					if (IsCorrectPath(buffer, 0) && !IsInitializer(buffer)) {
						const int fd = open(DOMAIN_POLICY_FILE, O_WRONLY);
						if (fd != EOF) {
							char *domainname = domain_list[current].domainname;
							write(fd, domainname, strlen(domainname)); write(fd, " ", 1); write(fd, buffer, strlen(buffer)); write(fd, "\n", 1);
							close(fd);
						}
					}
					put();
					goto start;
				}
			}
			break;
		case 'd':
		case 'D':
			{
				const int current = GetCurrent();
				if (current > 0 && !IsVirtualDomain(current)) { // Never delete ROOT_NAME
					char *key = DomainName(current);
					int c;
					mvprintw(header_lines, 0, "Delete '%s' and its descendant domains? (Y/N):", key);
					clrtoeol();
					refresh();
					do {
						c = getch();
					} while (!(c == 'Y' || c == 'y' || c == 'N' || c == 'n' || c == EOF));
					ResizeWindow();
					if (c == 'Y' || c == 'y') {
						const int fd = open(DOMAIN_POLICY_FILE, O_WRONLY);
						if (fd != EOF) {
							const int key_len = strlen(key);
							int index;
							for (index = 0; index < domain_list_count; index++) {
								char *cp = DomainName(index);
								if (strncmp(cp, key, key_len) || (cp[key_len] != '\0' && cp[key_len] != ' ')) continue;
								write(fd, "delete ", 7); write(fd, cp, strlen(cp)); write(fd, "\n", 1);
							}
							close(fd);
						}						
						goto start;
					}
					ShowList();
				}
			}
			break;
		case 'r':
		case 'R':
			goto start;
		case '\r':
			{
				const int current = GetCurrent();
				if (IsVirtualDomain(current)) {
					int redirect_index;
					get();
					memset(buffer, 0, sizeof(buffer));
					snprintf(buffer, sizeof(buffer) - 1, ROOT_NAME "%s", strrchr(DomainName(current), ' '));
					redirect_index = FindDomain(buffer);
					put();
					if (redirect_index != EOF) {
						current_item_index[current_screen] = redirect_index - current_y[current_screen];
						while (current_item_index[current_screen] < 0) {
							current_item_index[current_screen]++; current_y[current_screen]--;
						}
						ShowList();
					}
				} else {
					free(current_domain);
					if ((current_domain = strdup(domain_list[current].domainname)) == NULL) OutOfMemory();
					return SCREEN_ACL_LIST;
				}
			}
			break;
		}
	}
}

int main(int argc, char *argv[]) {
	memset(&history, 0, sizeof(history));
	memset(current_y, 0, sizeof(current_y));
	memset(current_item_index, 0, sizeof(current_item_index));
	memset(list_item_count, 0, sizeof(list_item_count));
	if (argc > 1) {
		if (strcmp(argv[1], "s") == 0) current_screen = SCREEN_SYSTEM_LIST;
		else if (strcmp(argv[1], "e") == 0) current_screen = SCREEN_EXCEPTION_LIST;
		else if (strcmp(argv[1], "d") == 0) current_screen = SCREEN_DOMAIN_LIST;
		else {
			printf("Usage: %s [s|e|d]\n", argv[0]);
			return 1;
		}
	}
	if (chdir("/proc/ccs/policy/")) {
		fprintf(stderr, "You can't use this editor for this kernel.\n");
		return 1;
	}
	{
		const int fd1 = open(SYSTEM_POLICY_FILE, O_RDWR), fd2 = open(EXCEPTION_POLICY_FILE, O_RDWR), fd3 = open(DOMAIN_POLICY_FILE, O_RDWR);
		if ((fd1 != EOF && write(fd1, "", 0) != 0) || (fd2 != EOF && write(fd2, "", 0) != 0) || (fd3 != EOF && write(fd3, "", 0) != 0)) {
			fprintf(stderr, "You need to register this program to /root/security/manager.txt and reboot to run this program.\n");
			return 1;
		}
		close(fd1); close(fd2); close(fd3);
	}
	setenv("TERM", "linux", 1);
	initscr();
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);
	getmaxyx(stdscr, window_height, window_width);
	while (current_screen < MAXSCREEN) {
		if (current_screen == SCREEN_DOMAIN_LIST && access(DOMAIN_POLICY_FILE, F_OK)) current_screen = SCREEN_SYSTEM_LIST;
		else if (current_screen == SCREEN_SYSTEM_LIST && access(SYSTEM_POLICY_FILE, F_OK)) current_screen = SCREEN_EXCEPTION_LIST;
		else if (current_screen == SCREEN_EXCEPTION_LIST && access(EXCEPTION_POLICY_FILE, F_OK)) current_screen = SCREEN_DOMAIN_LIST;
		ResizeWindow();
		if (current_screen == SCREEN_DOMAIN_LIST) current_screen = DomainListLoop();
		else current_screen = GenericListLoop();
	}
	clear();
	move(0, 0);
	refresh();
	endwin();
	return 0;
}
