/*
 * syspol.c
 *
 * An editor for the current policy in the kernel.
 *
 * Copyright (C) 2005  NTT DATA CORPORATION
 *
 * Version: 1.0.1 2005/12/01
 *
 * This program handles policies that are unrelated to domains.
 * Policies that are related to domains are handled by poled.c .
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <curses.h>

#define ROOT_NAME "<kernel>"

#define MAXBUFSIZE          8192

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(1);
}

#define PAGE_SIZE  4096

static const char *SaveName(const char *name) {
	static char *buf = NULL;
	static int buf_used_len = PAGE_SIZE;
	int i, len;
	char *saved_name = NULL;
	char **new_ptr = NULL;
	static char **search_list = NULL;
	static int search_list_count = 0;
	if (!name) return NULL;
	len = strlen(name) + 1;
	if (len > PAGE_SIZE) {
		printf("ERROR: Name too long for SaveName().\n");
		return NULL;
	}
	for (i = 0; i < search_list_count; i++) {
		if (strcmp(name, search_list[i]) == 0) return search_list[i];
	}
	if (buf_used_len + len > PAGE_SIZE) {
		if ((buf = malloc(PAGE_SIZE)) == NULL) OutOfMemory();
		memset(buf, 0, PAGE_SIZE);
		buf_used_len = 0;
	}
	saved_name = buf + buf_used_len;
	memmove(saved_name, name, len);
	if ((new_ptr = (char **) realloc(search_list, (search_list_count + 1) * sizeof(char *))) == NULL) OutOfMemory();
	search_list = new_ptr;
	search_list[search_list_count++] = saved_name;
	buf_used_len += len;
	return (const char *) saved_name;
}

static inline int strendswith(const char *name, const char *tail) {
	int len;
	if (!name || !tail) return 0;
	len = strlen(name) - strlen(tail);
	return len >= 0 && strcmp(name + len, tail) == 0;
}

static const char **list = NULL;
static int list_count = 0;

///////////////////////////  UTILITY FUNCTIONS  //////////////////////////////

static void NormalizeLine(unsigned char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}

static void Sort(const char **list, int list_count) {
	int i, j;
	for (i = 0; i < list_count; i++) {
		for (j = i + 1; j < list_count; j++) {
			if (strcmp(list[i], list[j]) > 0) {
				const char *tmp = list[i]; list[i] = list[j]; list[j] = tmp;
			}
		}
	}
}

static void ReadPolicy(const char *filename) {
	static char buffer[MAXBUFSIZE];
	FILE *fp = fopen(filename, "r");
	list_count = 0;
	if (fp) {
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			const char **tmp;
			int i;
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			//if (!IsCorrectPath(buffer, 0)) continue;
			for (i = 0; i < list_count; i++) {
				if (strcmp(buffer, list[i]) == 0) break;
			}
			if (i < list_count) continue;
			if ((cp = (char *) SaveName(buffer)) == NULL) continue;
			if ((tmp = (const char **) realloc(list, (list_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
			list = tmp;
			list[list_count++] = cp;
		}
		fclose(fp);
		Sort(list, list_count);
	}
}

////////////////////////////  UI HANDLER  ////////////////////////////

static int MenuLoop(void);
static void EntryListLoop(void);
static void ShowMenuList(void);
static void ShowEntryList(void);
static void ShowList(const int screen);

static void ResizeWindow(const int screen);
static void UpArrowKey(const int screen);
static void DownArrowKey(const int screen);
static void PageUpKey(const int screen);
static void PageDownKey(const int screen);
static void ShowCurrent(const int screen);
static int GetCurrent(const int screen);
static void CommandEnter(const int screen);

#define MAXSCREEN 2

static int window_width = 0, window_height = 0;
static int current_y[MAXSCREEN], current_item_index[MAXSCREEN], list_item_count[MAXSCREEN];

static const int header_lines = 3;
static int body_lines = 0;

#define MAXMENUENTRY 2

#define MENU_EXCEPTION_POLICY 0
#define MENU_SYSTEM_POLICY    1

static const char *menu_list[MAXMENUENTRY] = {
	"Edit exception policy ",
	"Edit system policy    "
};

static const char *policy_file[MAXMENUENTRY] = {
	"/proc/ccs/policy/exception_policy",
	"/proc/ccs/policy/system_policy"
};

static void ShowMenuList(void) {
	int i;
	const int offset = current_item_index[0];
	static char buffer[MAXBUFSIZE];
	list_item_count[0] = MAXMENUENTRY;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	memset(buffer, 0, sizeof(buffer));
	snprintf(buffer, sizeof(buffer) - 1, "<<< Menu >>>");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(0, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "Commands =  Q:quit  Enter:select");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(1, 0, "%s", buffer);
	for (i = 0; i < body_lines; i++) {
		if (offset + i >= list_item_count[0]) break;
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "%s", menu_list[offset + i]);
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		mvprintw(header_lines + i, 0, "%s", buffer);
	}
	ShowCurrent(0);
}

static int MenuLoop(void) {
	ResizeWindow(0);
	ShowList(0);
	while (1) {
		int c = getch();
		if (c == 'q' || c == 'Q') break;
		if (c == ERR) continue; // Ignore invalid key.
		switch(c) {
		case KEY_RESIZE:
			ResizeWindow(0);
			ShowList(0);
			break;
		case KEY_UP:
			UpArrowKey(0);
			break;
		case KEY_DOWN:
			DownArrowKey(0);
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey(0);
			break;
		case KEY_NPAGE:
			PageDownKey(0);
			break;
		case '\r':
			CommandEnter(0);
			break;
		}
	}
	return 1;
}

static void ShowEntryList(void) {
	const int current = GetCurrent(0), offset = current_item_index[1];
	static char buffer[MAXBUFSIZE];
	int i;
	list_item_count[1] = list_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	memset(buffer, 0, sizeof(buffer));
	i = list_item_count[1];
	snprintf(buffer, sizeof(buffer) - 1, "<<< ACL Viewer >>>    %d acl%c", i, i > 1 ? 's' : ' ');
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(0, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "Commands =  Q:quit  A:append  D:delete");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(1, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "%s", menu_list[current]);
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(2, 0, "%s", buffer);
	for (i = 0; i < body_lines; i++) {
		if (offset + i >= list_item_count[1]) break;
		mvprintw(header_lines + i, 0, "%5d: %s", offset + i, list[offset + i]);
	}
	ShowCurrent(1);
}

static void EntryListLoop(void) {
	const int current = GetCurrent(0);
	current_y[1] = current_item_index[1] = 0;
 start:
	ReadPolicy(policy_file[current]);
	while (list_count > 0 && current_item_index[1] + current_y[1] >= list_count) UpArrowKey(1);
	ShowList(1);
	while (1) {
		int c = getch();
		//if (c == 0x1B) break;
		if (c == 'q' || c == 'Q') break;
		if (c == ERR) continue; // Ignore invalid key.
		switch(c) {
		case KEY_RESIZE:
			ResizeWindow(1);
			ShowList(1);
			break;
		case KEY_UP:
			UpArrowKey(1);
			break;
		case KEY_DOWN:
			DownArrowKey(1);
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey(1);
			break;
		case KEY_NPAGE:
			PageDownKey(1);
			break;
		case 'd':
		case 'D':
			{
				const int index = GetCurrent(1);
				if (index >= 0) {
					int fd = open(policy_file[current], O_WRONLY);
					if (fd != EOF) {
						write(fd, "delete ", 7); write(fd, list[index], strlen(list[index])); write(fd, "\n", 1); close(fd);
						close(fd);
					}
					goto start;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				static char buffer[4096];
				mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
				memset(buffer, 0, sizeof(buffer));
				getnstr(buffer, sizeof(buffer) - 1); noecho();
				NormalizeLine(buffer);
				{
					int fd = open(policy_file[current], O_WRONLY);
					if (fd != EOF) {
						write(fd, buffer, strlen(buffer)); write(fd, "\n", 1); close(fd);
					}
				}
				goto start;
			}
			break;
		}
	}
}

static void ShowList(const int screen) {
	if (screen == 0) ShowMenuList();
	else ShowEntryList();
}

static void ResizeWindow(const int screen) {
	getmaxyx(stdscr, window_height, window_width);
	body_lines = window_height - header_lines;
	if (body_lines <= current_y[screen]) current_y[screen] = body_lines - 1;
	if (current_y[screen] < 0) current_y[screen] = 0;
}

static void UpArrowKey(const int screen) {
	if (current_y[screen] > 0) {
		current_y[screen]--;
		ShowCurrent(screen);
	} else if (current_item_index[screen] > 0) {
		current_item_index[screen]--;
		ShowList(screen);
	}
}

static void DownArrowKey(const int screen) {
	if (current_y[screen] < body_lines - 1) {
		if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
			current_y[screen]++;
			ShowCurrent(screen);
		}
	} else if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
		current_item_index[screen]++;
		ShowList(screen);
	}
}

static void PageUpKey(const int screen) {
	if (current_item_index[screen] + current_y[screen] > body_lines) {
		current_item_index[screen] -= body_lines;
		if (current_item_index[screen] < 0) current_item_index[screen] = 0;
		ShowList(screen);
	} else if (current_item_index[screen] + current_y[screen] > 0) {
		current_item_index[screen] = 0;
		current_y[screen] = 0;
		ShowList(screen);
	}
}

static void PageDownKey(const int screen) {
	if (list_item_count[screen] - current_item_index[screen] > body_lines) {
		current_item_index[screen] += body_lines;
		if (current_item_index[screen] + current_y[screen] > list_item_count[screen] - 1) current_y[screen] = list_item_count[screen] - 1 - current_item_index[screen];
		ShowList(screen);
	} else if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
		current_y[screen] = list_item_count[screen] - current_item_index[screen] - 1;
		ShowCurrent(screen);
	}
}

static int GetCurrent(const int screen) {
	if (list_item_count[screen] == 0) return EOF;
	if (current_item_index[screen] + current_y[screen] < 0 || current_item_index[screen] + current_y[screen] >= list_item_count[screen]) {
		fprintf(stderr, "ERROR: current_item_index=%d current_y=%d\n", current_item_index[screen], current_y[screen]);
		exit(1);
	}
	return current_item_index[screen] + current_y[screen];
}

static void ShowCurrent(const int screen) {
	move(header_lines + current_y[screen], 0);
	refresh();
}

static void CommandEnter(const int screen) {
	//const int current = GetCurrent(screen);
	if (screen == 0) {
		EntryListLoop();
		ShowList(0);
	}
}

int main(int argc, char *argv[]) {
	int current_x, i;
	memset(current_y, 0, sizeof(current_y));
	memset(current_item_index, 0, sizeof(current_item_index));
	memset(list_item_count, 0, sizeof(list_item_count));
	if (access("/proc/ccs/", F_OK)) {
		fprintf(stderr, "You can't use this editor for this kernel.\n");
		return 1;
	}
	for (i = 0; i < MAXMENUENTRY; i++) {
		int fd;
		if ((fd = open(policy_file[i], O_RDWR)) == EOF) {
			fprintf(stderr, "Can't open %s\n", policy_file[i]);
			return 1;
		} else if (write(fd, "", 0) != 0) {
			fprintf(stderr, "You need to register this program to /root/security/manager.txt and reboot to run this program.\n");
			return 1;
		}
		close(fd);
	}
	setenv("TERM", "linux", 1);
	initscr();
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);
	getyx(stdscr, current_y[0], current_x);
	getmaxyx(stdscr, window_height, window_width);
	while (1) {
		int status = 0;
		switch(fork()) {
		case 0:
			_exit(MenuLoop());
		case -1:
			break;
		}
		while (wait(&status) == EOF && errno == EINTR);
		if (!WIFEXITED(status) || WEXITSTATUS(status)) break;
	}
	clear();
	move(0, 0);
	refresh();
	endwin();
	return 0;
}
