/*
 * savepolicy.c
 *
 * Save the current policy in the kernel onto the disk.
 *
 * Copyright (C) 2005  NTT DATA CORPORATION
 *
 * Version: 1.0 2005/11/11
 *
 * This program saves policies that are related to domains.
 * You can call this program to save policies just before the system halts
 * (i.e. at the end of "halt" script).
 * This program remounts root fs as read/write if necessary.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mount.h>
#include <errno.h>
#include <time.h>

#define ROOT_NAME "<kernel>"

typedef struct {
	char *ptr;
} SEGMENT;

static SEGMENT *SplitAndSortSegment(const char *data, int *count) {
	const char *cp, *start = data;
	SEGMENT *segment = NULL;
	int i, j, segment_count = 0;
	while ((cp = strchr(data, '\n')) != NULL) {
		if (!*(cp + 1) || strncmp(cp + 1, ROOT_NAME, strlen(ROOT_NAME)) == 0) {
			const int len = cp - start;
			char *ptr = malloc(len + 1);
			if (!ptr) goto out;
			memmove(ptr, start, len);
			ptr[len] = '\0';
			segment = (SEGMENT *) realloc(segment, sizeof(SEGMENT) * (segment_count + 1));
			if (!segment) goto out;
			segment[segment_count].ptr = ptr;
			segment_count++;
			start = cp + 1;
		}
		data = cp + 1;
	}
	for (i = 0; i < segment_count; i++) {
		for (j = i + 1; j < segment_count; j++) {
			if (strcmp(segment[i].ptr, segment[j].ptr) > 0) {
				SEGMENT tmp_segment = segment[i]; segment[i] = segment[j]; segment[j] = tmp_segment;
			}
		}
	}
	*count = segment_count;
	return segment;
 out:
	fprintf(stderr, "Out of memory.\n");
	*count = -1;
	return NULL;
}

typedef struct {
	char *ptr;
} ENTRY;

static ENTRY *SplitEntry(SEGMENT *segment, int *count) {
	char *cp, *data = segment->ptr;
	ENTRY *entry = NULL;
	int entry_count = 0;
	while ((cp = strchr(data, '\n')) != NULL) {
		*cp = '\0';
		entry = (ENTRY *) realloc(entry, sizeof(ENTRY) * (entry_count + 1));
		if (!entry) goto out;
		entry[entry_count].ptr = data;
		entry_count++;
		data = cp + 1;
	}
	*count = entry_count;
	return entry;
 out:
	fprintf(stderr, "Out of memory.\n");
	*count = -1;
	return NULL;
}

static void SortEntry(ENTRY *entry, const int entry_count) {
	int i, j;
	for (i = 1; i < entry_count; i++) {
		for (j = i + 1; j < entry_count; j++) {
			char *ptr0 = entry[i].ptr;
			char *ptr1 = entry[j].ptr;
			int perm0, perm1;
			if (sscanf(ptr0, "%d", &perm0) == 1 && sscanf(ptr1, "%d", &perm1) == 1) {
				char *p0 = strchr(ptr0, '/');
				char *p1 = strchr(ptr1, '/');
				if (p0 && p1 && strcmp(p0, p1) > 0) {
					ENTRY tmp_entry = entry[i]; entry[i] = entry[j]; entry[j] = tmp_entry;
				}
			} else if (strncmp(ptr0, "allow_signal ", 13) == 0 && strncmp(ptr1, "allow_signal ", 13) == 0) {
				char *p0 = strstr(ptr0, ROOT_NAME);
				char *p1 = strstr(ptr1, ROOT_NAME);
				if (p0 && p1 && strcmp(p0, p1) > 0) {
					ENTRY tmp_entry = entry[i]; entry[i] = entry[j]; entry[j] = tmp_entry;
				}
			} else if ((strncmp(ptr0, "allow_bind ", 11) == 0 && strncmp(ptr1, "allow_bind ", 11) == 0) ||
					   (strncmp(ptr0, "allow_connect ", 14) == 0 && strncmp(ptr1, "allow_connect ", 14) == 0) ||
					   (strncmp(ptr0, "allow_capability ", 17) == 0 && strncmp(ptr1, "allow_capability ", 17) == 0)) {
				if (strcmp(ptr0, ptr1) > 0) {
					ENTRY tmp_entry = entry[i]; entry[i] = entry[j]; entry[j] = tmp_entry;
				}
			}
		}
	}
}

static char *ReadFile(const char *filename) {
	char *buffer = NULL;
	int fd;
	if ((fd = open(filename, O_RDONLY)) != EOF) {
		int buffer_len = 0;
		while (1) {
			char *cp = realloc(buffer, buffer_len + 4096);
			int len;
			if (!cp) {
				free(buffer);
				return NULL;
			}
			buffer = cp;
			len = read(fd, buffer + buffer_len, 4095);
			if (len <= 0) break;
			buffer_len += len;
		}
		close(fd);
		buffer[buffer_len] = '\0';
	}
	return buffer;
}

int main(int argc, char *argv[]) {
	int fd;
	int remount_root = 0;
	static const char *proc_file = "/proc/ccs/policy/domain_policy";
	static const char *policy_filename = "domain_policy.txt";
	static char filename[1024];
	time_t now = time(NULL);
	struct tm *tm = localtime(&now);
	memset(filename, 0, sizeof(filename));
	{
		char c;
		if ((fd = open(proc_file, O_RDONLY)) == EOF || read(fd, &c, 1) < 0) return 0;
		close(fd);
	}
	if (chdir("/root/security/") == EOF) goto failed;
	if (access(".", W_OK) == EOF) {
		if (errno != EROFS || mount("/", "/", "rootfs", MS_REMOUNT, NULL) == EOF) goto failed;
		remount_root = 1;
	}
	snprintf(filename, sizeof(filename) - 1, "domain_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
	if ((fd = open(filename, O_WRONLY | O_CREAT | O_EXCL, 0600)) != EOF) {
		FILE *fp = fdopen(fd, "w");
		char *buffer = ReadFile(proc_file);
		if (buffer) {
			int segment_count = 0;
			SEGMENT *segment = SplitAndSortSegment(buffer, &segment_count);
			int i, j;
			for (i = 0; i < segment_count; i++) {
				int entry_count = 0;
				ENTRY *entry = SplitEntry(&segment[i], &entry_count);
				SortEntry(entry, entry_count);
				for (j = 0; j < entry_count; j++) fprintf(fp, "%s\n", entry[j].ptr);
				free(entry);
				fprintf(fp, "\n");
			}
			for (i = 0; i < segment_count; i++) free(segment[i].ptr);
			free(segment);
			free(buffer);
		}
		fclose(fp);
		close(fd);
		unlink(policy_filename);
		symlink(filename, policy_filename);
	}
	if (remount_root) mount("/", "/", "rootfs", MS_REMOUNT | MS_RDONLY, NULL);
	return 0;
 failed: ;
	printf("Failed to save policy. (%s)\n", strerror(errno));
	return 1;
}
