/*
 * remount.c
 *
 * Unmount /initrd, remount root fs as read/write, and enable swap partitions.
 *
 * Copyright (C) 2005  NTT DATA CORPORATION
 *
 * Version: 1.0 2005/11/11
 *
 * As with older versions, the rc.sysinit assumes root fs is writable and
 * it unconditionally remounts root fs as read/write mode.
 * This program do so only if /proc/cmdline doesn't include the word "readonly".
 * This program stops if unmounting /initrd failed, to ensure that files in the
 * initial ramdisk become invisible.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mount.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/swap.h>
#include <fstab.h>
#define MNT_DETACH 0x00000002

static int error = 0;

static void WaitKey(void) {
	int c;
	error = 1;
	fprintf(stderr, "Press 'C' to continue, 'A' to abort.\n");
	setbuf(stdin, NULL);
	while (1) {
		if ((c = getchar()) != EOF) {
			if (c == 'C' || c == 'c') break;
			else if (c == 'A' || c == 'a') exit(1);
		} else {
			sleep(10);
		}
	}
}

int main(int argc, char *argv[]) {
	if (getuid() != 0 || geteuid() != 0) {
		fprintf(stderr, "This program needs to be run by 'root' user.\n");
		return 1;
	}
	if (umount2("/initrd/loopfs", MNT_DETACH) == EOF) {
		if (errno != EINVAL && errno != ENOENT) {
			fprintf(stderr, "FATAL: 'umount -l /initrd/loopfs' failed. (%s)\n", strerror(errno));
			WaitKey();
		}
	}
	if (umount2("/initrd", MNT_DETACH) == EOF) {
		if (errno != EINVAL) {
			fprintf(stderr, "FATAL: 'umount -l /initrd' failed. (%s)\n", strerror(errno));
			WaitKey();
		}
	}
	{
		int fd;
		if ((fd = open("/proc/cmdline", O_RDONLY)) == EOF) {
			fprintf(stderr, "WARNING: Can't open /proc/cmdline (%s).\n", strerror(errno));
		} else {
			char buffer[1024];
			memset(buffer, 0, sizeof(buffer));
			read(fd, buffer, sizeof(buffer) - 1);
			close(fd);
			if (strstr(buffer, "readonly") == NULL) {
				if (mount(NULL, "/", NULL, MS_REMOUNT, NULL) == EOF) {
					fprintf(stderr, "ERROR: 'mount -o remount,rw /' failed. (%s)\n", strerror(errno));
				}
			}
			if (strstr(buffer, "noswap") == NULL) {
				FILE *fp = popen("LANG=C fdisk -l | grep 'Linux swap' | awk ' { print $1 } '", "r");
				if (fp != NULL) {
					while (fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
						char *cp;
						if ((cp = strchr(buffer, '\n')) != NULL) *cp = '\0';
						if (swapon(buffer, 0) == EOF) {
							fprintf(stderr, "ERROR: 'swapon %s' failed. (%s)\n", buffer, strerror(errno));
						}
					}
					pclose(fp);
				}
			}
		}
	}
	return error;
}
