/*
 * poled_old.c
 *
 * An editor for the policy on the disk.
 *
 * Copyright (C) 2005  NTT DATA CORPORATION
 *
 * Version: 1.0.1 2005/12/06
 *
 * This program handles policies that are related to domains.
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <curses.h>

#define ROOT_NAME "<kernel>"

#define BYTE unsigned char
#define WORD unsigned short int

#define MAXBUFSIZE  8192

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(1);
}

#define PAGE_SIZE  4096

static const char *SaveName(const char *name) {
	static char *buf = NULL;
	static int buf_used_len = PAGE_SIZE;
	int i, len;
	char *saved_name = NULL;
	char **new_ptr = NULL;
	static char **search_list = NULL;
	static int search_list_count = 0;
	if (!name) return NULL;
	len = strlen(name) + 1;
	if (len > PAGE_SIZE) {
		printf("ERROR: Name too long for SaveName().\n");
		return NULL;
	}
	for (i = 0; i < search_list_count; i++) {
		if (strcmp(name, search_list[i]) == 0) return search_list[i];
	}
	if (buf_used_len + len > PAGE_SIZE) {
		if ((buf = malloc(PAGE_SIZE)) == NULL) OutOfMemory();
		memset(buf, 0, PAGE_SIZE);
		buf_used_len = 0;
	}
	saved_name = buf + buf_used_len;
	memmove(saved_name, name, len);
	if ((new_ptr = (char **) realloc(search_list, (search_list_count + 1) * sizeof(char *))) == NULL) OutOfMemory();
	search_list = new_ptr;
	search_list[search_list_count++] = saved_name;
	buf_used_len += len;
	return (const char *) saved_name;
}

static int StrStr(const char *source, const char *target) {
	const char *cp;
	if ((cp = strstr(source, target)) != NULL) {
		const int len = strlen(target);
		if (cp[len] == ' ' || cp[len] == '\0') {
			if (cp == source || *(cp - 1) == ' ') return 1;
		}
	}
	return 0;
}

/*
 * Check whether the given filename is patterened.
 * Returns nonzero if patterned, zero otherwise.
 */
static int PathContainsPattern(const char *filename) {
	if (filename) {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c != '\\') continue;
			switch (c = *filename++) {
			case '\\':  /* "\\" */
				continue;
			case '0':   /* "\ooo" */
			case '1':
			case '2':
			case '3':
				if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7'
					&& (c != '0' || d != '0' || e != '0')) continue; /* pattern is not \000 */
			}
			return 1;
		}
	}
	return 0;
}

/*
 * Check whether the given filename follows the naming rules.
 * Returns nonzero if follows, zero otherwise.
 */
static int IsCorrectPath(const char *filename, const int may_contain_pattern) {
	if (filename && *filename == '/') {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c == '\\') {
				switch ((c = *filename++)) {
				case '\\':  /* "\\" */
					continue;
				case '$':   /* "\$" */
				case '+':   /* "\+" */
				case '?':   /* "\?" */
					if (may_contain_pattern) continue;
					break;
				case '*':   /* "\*" */
					if (may_contain_pattern) {
						while (*filename && *filename != '/') filename++;
						if (!*filename || *filename == '/') continue;
					}
					break;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				return 0;
			} else if (c <= ' ' || c >= 127) {
				return 0;
			}
		}
		return 1;
	}
	return 0;
}

static int IsCorrectDomain(const unsigned char *domainname) {
	unsigned char c, d, e;
	if (!domainname || strncmp(domainname, ROOT_NAME, strlen(ROOT_NAME))) goto out;
	domainname += strlen(ROOT_NAME);
	if (!*domainname) return 1;
	do {
		if (*domainname++ != ' ') goto out;
		if (*domainname++ != '/') goto out;
		while ((c = *domainname) != '\0' && c != ' ') {
			domainname++;
			if (c == '\\') {
				switch ((c = *domainname++)) {
				case '\\':  /* "\\" */
					continue;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *domainname++) >= '0' && d <= '7' && (e = *domainname++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				goto out;
			} else if (c < ' ' || c >= 127) {
				goto out;
			}
		}
	} while (*domainname);
	return 1;
 out:
	return 0;
}

static inline int strendswith(const char *name, const char *tail) {
	int len;
	if (!name || !tail) return 0;
	len = strlen(name) - strlen(tail);
	return len >= 0 && strcmp(name + len, tail) == 0;
}

static void LoadPolicy(void);
static void SortPolicy(void);
static void SavePolicy(void);
static int AddFileACL(const char *filename, WORD perm, const int index);
static int DelACL(const char *filename, const int index);
static int FindOrAssignNewDomain(const char *domainname);
static int FindDomain(const char *domainname);
static void NormalizeLine(unsigned char *buffer);
static const char *DomainName(const int index);
static int FindParent(const char *key);
static void DeleteDomain(const int index);
static int IsTrusted(const int index);
static int IsInitializer(const int index);
static int IsVirtual(const int index);
static int IsGloballyReadableFile(const char *filename);

typedef struct {
	const char *name;
	WORD perm;
} ACL_RECORD;

typedef struct domain_info {
	ACL_RECORD *acl_ptr;
	int acl_count;
	const char *domainname;
	const char **string_ptr;
	int string_count;
	int is_virtual;
} DOMAIN_INFO;

static DOMAIN_INFO *domain_list = NULL;
static int domain_list_count = 0;

static const char **trusted_domainnames = NULL;
static int trusted_domainnames_count = 0;

static const char **initializers = NULL;
static int initializers_count = 0;

static const char **globally_readable_list = NULL;
static int globally_readable_list_count = 0;

static int acl_updated = 0;

///////////////////////////  ACL HANDLER  //////////////////////////////

static int AddFileACL(const char *filename, WORD perm, const int index) {
	ACL_RECORD *acl_ptr, *new_aclptr;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count) {
		printf("AddFileACL: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!IsCorrectPath(filename, 1)) {
		fprintf(stderr, "AddFileACL: Invalid pathname '%s'\n", filename);
		return -EINVAL;
	}
	if (perm > 7) { /* Should I allow perm == 0, for somebody may wish to give '4 /etc/\*' and '0 /etc/shadow' ? */
		fprintf(stderr, "AddFileACL: Invalid permission '%d %s'\n", perm, filename);
		return -EINVAL;
	}
	if (strendswith(filename, "/")) {
		perm |= 5;  /* Always allow read and execute for dir. */
	} else if ((perm & 1) == 1 && PathContainsPattern(filename)) {
		perm ^= 1;  /* Never allow execute permission with patterns. */
		fprintf(stderr, "AddFileACL: Dropping execute permission for '%s'\n", filename);
	} else if (perm == 4 && IsGloballyReadableFile(filename)) {
		return 0;   /* Don't add if the file is globally readable files. */
	}
	
	acl_ptr = domain_list[index].acl_ptr;
	
	// Check if the file was accessed before.
	for (i = 0; i < domain_list[index].acl_count; i++) {
		const char *cp = acl_ptr[i].name;
		if (strcmp(filename, cp) == 0) {
			acl_ptr[i].perm |= perm;
			return 0;
		}
	}
	if ((cp = SaveName(filename)) == NULL) OutOfMemory();
	
	// Allocate space.
	i = domain_list[index].acl_count;
	if ((new_aclptr = (ACL_RECORD *) realloc(acl_ptr, (i + 1) * sizeof(ACL_RECORD))) == NULL) OutOfMemory();
	domain_list[index].acl_ptr = acl_ptr = new_aclptr;
	
	// Record the access log.
	acl_ptr[i].perm = perm;
	acl_ptr[i].name = cp;
	domain_list[index].acl_count++;
	return 0;
}

static int DelACL(const char *filename, const int index) {
	ACL_RECORD *acl_ptr;
	int i;
	if (index < 0 || index >= domain_list_count) {
		printf("DelACL: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	acl_ptr = domain_list[index].acl_ptr;
	for (i = 0; i < domain_list[index].acl_count; i++) {
		const char *cp = acl_ptr[i].name;
		if (strcmp(filename, cp) == 0) {
			int j;
			domain_list[index].acl_count--;
			for (j = i; j < domain_list[index].acl_count; j++) {
				acl_ptr[j] = acl_ptr[j + 1];
			}
			return 0;
		}
	}
	return -ENOENT;
}

static int AddStringEntry(const char *entry, const int index) {
	const char **acl_ptr;
	int acl_count;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count) {
		printf("AddStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	acl_ptr = domain_list[index].string_ptr;
	acl_count = domain_list[index].string_count;
	
	// Check for the same entry.
	for (i = 0; i < acl_count; i++) {
		if (strcmp(entry, acl_ptr[i]) == 0) return 0;
	}

	if ((cp = SaveName(entry)) == NULL) OutOfMemory();
	if ((acl_ptr = (const char **) realloc(acl_ptr, (acl_count + 1) * sizeof(char *))) == NULL) OutOfMemory();
	acl_ptr[acl_count++] = cp;
	domain_list[index].string_ptr = acl_ptr;
	domain_list[index].string_count = acl_count;
	return 0;
}

static int DelStringEntry(const char *entry, const int index) {
	const char **acl_ptr;
	int acl_count;
	int i;
	if (index < 0 || index >= domain_list_count) {
		printf("DelStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	acl_ptr = domain_list[index].string_ptr;
	acl_count = domain_list[index].string_count;
	for (i = 0; i < acl_count; i++) {
		int j;
		if (strcmp(entry, acl_ptr[i])) continue;
		acl_count--;
		for (j = i; j < acl_count; j++) {
			acl_ptr[j] = acl_ptr[j + 1];
		}
		domain_list[index].string_count--;
		return 0;
	}
	return -ENOENT;
}

static int IsTrusted(const int index) {
	int i;
	for (i = 0; i < trusted_domainnames_count; i++) {
		if (StrStr(domain_list[index].domainname, trusted_domainnames[i])) return 1;
	}
	return 0;
}

static int IsInitializer(const int index) {
	char *cp = strchr(domain_list[index].domainname, ' ');
	if (cp) {
		int i;
		cp++;
		for (i = 0; i < initializers_count; i++) {
			if (strcmp(cp, initializers[i]) == 0) return 1;
		}
	}
	return 0;
}

static int IsVirtual(const int index) {
	return domain_list[index].is_virtual;
}

static int IsGloballyReadableFile(const char *filename) {
	if (filename) {
		int i;
		for (i = 0; i < globally_readable_list_count; i++) {
			if (strcmp(filename, globally_readable_list[i]) == 0) return 1;
		}
	}
	return 0;
}

////////////////////////////  DOMAIN HANDLER  ////////////////////////////

static int FindDomain(const char *domainname) {
	int i;
	for (i = 0; i < domain_list_count; i++) {
		if (strcmp(domainname, domain_list[i].domainname) == 0) {
			return i;
		}
	}
	return EOF;
}

static int FindOrAssignNewDomain(const char *domainname) {
	const char *saved_domainname;
	int index;
	if ((index = FindDomain(domainname)) == EOF) {
		if (IsCorrectDomain(domainname)) {
			if ((domain_list = (DOMAIN_INFO *) realloc(domain_list, (domain_list_count + 1) * sizeof(DOMAIN_INFO))) == NULL) OutOfMemory();
			memset(&domain_list[domain_list_count], 0, sizeof(DOMAIN_INFO));
			if ((saved_domainname = SaveName(domainname)) == NULL) OutOfMemory();
			domain_list[domain_list_count].domainname = saved_domainname;
			index = domain_list_count++;
		} else {
			fprintf(stderr, "FindOrAssignNewDomain: Invalid domainname '%s'\n", domainname);
		}
	}
	return index;
}

static const char *DomainName(const int index) {
	return domain_list[index].domainname;
}

static int FindParent(const char *key) {
	int index = EOF;
	char *parent = strdup(key);
	char *cp;
	if (!parent) OutOfMemory();
	cp = strrchr(parent, ' ');
	if (cp != NULL) {
		int i;
		*cp++ = '\0';
		for (i = 0; i < domain_list_count; i++) {
			if (strcmp(domain_list[i].domainname, parent) == 0) {
				index = i;
				break;
			}
		}
	}
	free(parent);
	return index;
}

static void DeleteDomain(const int index) {
	if (index > 0 && index < domain_list_count) {
		int i;
		for (i = index; i < domain_list_count - 1; i++) domain_list[i] = domain_list[i + 1];
		domain_list_count--;
	}
}

///////////////////////////  UTILITY FUNCTIONS  //////////////////////////////

static int IsDomainDef(const unsigned char *buffer) {
	while (*buffer && (*buffer <= 32 || 127 <= *buffer)) buffer++;
	return strncmp(buffer, ROOT_NAME, strlen(ROOT_NAME)) == 0;
}

static void NormalizeLine(unsigned char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}

static void SortPolicy(void) {
	int i, j;
	for (i = 0; i < domain_list_count; i++) {
		for (j = i + 1; j < domain_list_count; j++) {
			if (strcmp(domain_list[i].domainname, domain_list[j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[i]; domain_list[i] = domain_list[j]; domain_list[j] = tmp;
			}
		}
	}
}

static int SortAndUniq(const char **list, int list_count) {
	int i, j;
	for (i = 0; i < list_count; i++) {
		for (j = i + 1; j < list_count; j++) {
			if (strcmp(list[i], list[j]) > 0) {
				const char *tmp = list[i];
				list[i] = list[j];
				list[j] = tmp;
			}
		}
	}
	for (i = 0; i < list_count - 1; i++) {
		if (strcmp(list[i], list[i + 1])) continue;
		for (j = i + 1; j < list_count - 1; j++) list[j] = list[j + 1];
		list_count--; i--;
	}
	return list_count;
}

static void LoadPolicy(void) {
	static char buffer[MAXBUFSIZE];
	FILE *fp;
	int index;
	FindOrAssignNewDomain(ROOT_NAME);

	// Load initializer list, trusted domain list, globally readable file list.
	initializers_count = 0;
	trusted_domainnames_count = 0;
	globally_readable_list = 0;

	if ((fp = fopen("exception_policy.txt", "r")) != NULL) {
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			const char **tmp;
			int i;
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			if (strncmp(buffer, "initializer ", 12) == 0) {
				memmove(buffer, buffer + 12, strlen(buffer + 12) + 1);
				if (!IsCorrectPath(buffer, 0)) continue;
				for (i = 0; i < initializers_count; i++) {
					if (strcmp(buffer, initializers[i]) == 0) break;
				}
				if (i < initializers_count) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) continue;
				if ((tmp = (const char **) realloc(initializers, (initializers_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				initializers = tmp;
				initializers[initializers_count++] = cp;
			} else if (strncmp(buffer, "trust_domain ", 13) == 0) {
				memmove(buffer, buffer + 13, strlen(buffer + 13) + 1);
				if (!IsCorrectDomain(buffer)) continue;
				for (i = 0; i < trusted_domainnames_count; i++) {
					if (strcmp(buffer, trusted_domainnames[i]) == 0) break;
				}
				if (i < trusted_domainnames_count) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) continue;
				if ((tmp = (const char **) realloc(trusted_domainnames, (trusted_domainnames_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				trusted_domainnames = tmp;
				trusted_domainnames[trusted_domainnames_count++] = cp;
			} else if (strncmp(buffer, "allow_read ", 11) == 0) {
				memmove(buffer, buffer + 11, strlen(buffer + 11) + 1);
				if (!IsCorrectPath(buffer, 0)) continue;
				for (i = 0; i < globally_readable_list_count; i++) {
					if (strcmp(buffer, globally_readable_list[i]) == 0) break;
				}
				if (i < globally_readable_list_count) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) continue;
				if ((tmp = (const char **) realloc(globally_readable_list, (globally_readable_list_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				globally_readable_list = tmp;
				globally_readable_list[globally_readable_list_count++] = cp;
			}
		}
		fclose(fp);
		initializers_count = SortAndUniq(initializers, initializers_count);
		trusted_domainnames_count = SortAndUniq(trusted_domainnames, trusted_domainnames_count);
		globally_readable_list_count = SortAndUniq(globally_readable_list, globally_readable_list_count);
	}
	
	// Load all domain list.
	if ((fp = fopen("domain_policy.txt", "r")) == NULL) {
		fprintf(stderr, "Can't open domain_policy.txt\n"); sleep(3); exit(1);
	}
	index = EOF;
	while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
		char *cp = strchr(buffer, '\n');
		int perm;
		if (cp) *cp = '\0';
		else if (!feof(fp)) break;
		NormalizeLine(buffer);
		if (IsDomainDef(buffer)) {
			index = FindOrAssignNewDomain(buffer);
		} else if (index >= 0 && sscanf(buffer, "%d", &perm) == 1 && (cp = strchr(buffer, ' ')) != NULL) {
			AddFileACL(cp + 1, (WORD) perm, index);
		} else if (index >= 0 && strncmp(buffer, "allow_", 6) == 0) {
			AddStringEntry(buffer, index);
		}
	}
	fclose(fp);
	
	SortPolicy();

	scrollok(stdscr, 1);
	
	// Check if a domain contains initializers.
	for (index = 1; index < domain_list_count; index++) {
		static char buffer2[MAXBUFSIZE];
		char *exe;
		if ((exe = strchr(domain_list[index].domainname, ' ')) == NULL) continue;
		if ((exe = strchr(exe + 1, ' ')) == NULL) continue;
		memset(buffer2, 0, sizeof(buffer2));
		strncpy(buffer2, exe, sizeof(buffer2) - 1);
		while ((exe = strrchr(buffer2, ' ')) != NULL) {
			int i, c;
			*exe++ = '\0';
			for (i = 0; i < initializers_count; i++) {
				if (strcmp(exe, initializers[i]) == 0) break;
			}
			if (i == initializers_count) continue;
			snprintf(buffer, sizeof(buffer) - 1, ROOT_NAME " %s", exe);
			printw("Domain '%s' contains initializer '%s'.\n", domain_list[index].domainname, exe);
			printw("Rename/Delete/Ignore? ");
			do { c = getch(); } while (!(c == 'R' || c == 'r' || c == 'D' || c == 'd' || c == 'I' || c == 'i' || c == EOF));
			printw("%c\n", c);
			if (c == 'R' || c == 'r') {
				printw("'%s' -> '%s' renamed.\n", domain_list[index].domainname, buffer);
				domain_list[index].domainname = SaveName(buffer);
				acl_updated = 1;
			} else if (c == 'D' || c == 'd') {
				printw("'%s' deleted.\n", domain_list[index].domainname);
				DeleteDomain(index); index--;
				acl_updated = 1;
			} else {
				printw("Ignored.\n");
			}
			buffer2[0] = '\0';
			printw("\n");
			break;
		}
	}
	SortPolicy();
	
	// Check if a domain contains a trusted pattern.
	for (index = 1; index < domain_list_count; index++) {
		const char *domainname = DomainName(index);
		int i;
		for (i = 0; i < trusted_domainnames_count; i++) {
			const char *p = trusted_domainnames[i];
			const int len = strlen(p);
			int c;
			if (strncmp(domainname, p, len) != 0 || domainname[len] != ' ') continue;
			printw("Domain '%s' is unreachable because it contains trusted pattern '%s'.\n", domainname, p);
			printw("Delete/Ignore? ");
			do { c = getch(); } while (!(c == 'D' || c == 'd' || c == 'I' || c == 'i' || c == EOF));
			printw("%c\n", c);
			if (c == 'D' || c == 'd') {
				printw("'%s' deleted.\n", domainname);
				DeleteDomain(index); index--;
				acl_updated = 1;
			}
			printw("\n");
			break;
		}
	}
	
	SortPolicy();

	// Check if a domain has a parent domain.
	for (index = 1; index < domain_list_count; index++) {
		const char *domainname = domain_list[index].domainname;
		char *cp;
		int c;
		memset(buffer, 0, sizeof(buffer));
		strncpy(buffer, domainname, sizeof(buffer) - 1);
		cp = strrchr(buffer, ' ');
		if (!cp) continue;
		*cp = '\0';
		if (FindDomain(buffer) != EOF) continue;
		printw("Domain '%s' doesn't have the parent domain.\n", domainname);
		printw("Create/Ignore? ");
		do { c = getch(); } while (!(c == 'C' || c == 'c' || c == 'I' || c == 'i' || c == EOF));
		printw("%c\n", c);
		if (c == 'C' || c == 'c') {
			FindOrAssignNewDomain(buffer);
			printw("Created parent domain '%s'.\n", buffer);
			index = 0;
			acl_updated = 1;
		}
		printw("\n");
	}

	// Check if the parent domain has a execute permission to transfer into a child domain.
	for (index = 1; index < domain_list_count; index++) {
		int parent_index;
		char *exe;
		int c;
		memset(buffer, 0, sizeof(buffer));
		strncpy(buffer, domain_list[index].domainname, sizeof(buffer) - 1);
		exe = strrchr(buffer, ' ');
		if (!exe) continue;
		*exe++ = '\0';
		if ((parent_index = FindDomain(buffer)) == EOF) continue;
		// If parent is <kernel>, check if exe is a initializer.
		if (strcmp(buffer, ROOT_NAME) == 0) {
			int i;
			for (i = 0; i < initializers_count; i++) {
				if (strcmp(exe, initializers[i]) == 0) break;
			}
			if (i < initializers_count) continue;
		}
		// Try to find execute permission in parent domain.
		{
			ACL_RECORD *acl_ptr = domain_list[parent_index].acl_ptr;
			int i;
			for (i = 0; i < domain_list[parent_index].acl_count; i++) {
				const char *cp = acl_ptr[i].name;
				if (PathContainsPattern(cp) == 0 && strcmp(cp, exe) == 0 && (acl_ptr[i].perm & 1) == 1) break;
			}
			if (i < domain_list[parent_index].acl_count) continue;
		}
		printw("Domain '%s' doesn't have 'execute %s' permission.\n", buffer, exe);
		printw("Append/Ignore? ");
		do { c = getch(); } while (!(c == 'A' || c == 'a' || c == 'I' || c == 'i' || c == EOF));
		printw("%c\n", c);
		if (c == 'A' || c == 'a') {
			AddFileACL(exe, 1, parent_index);
			printw("Added 'execute %s' permission to '%s'.\n", exe, buffer);
			acl_updated = 1;
		}
		printw("\n");
	}
		
	// Create virtual domains.
	for (index = 1; index < domain_list_count; index++) {
		ACL_RECORD *acl_ptr = domain_list[index].acl_ptr;
		int i, j;
		for (i = 0; i < domain_list[index].acl_count; i++) {
			const char *cp = acl_ptr[i].name;
			if ((acl_ptr[i].perm & 1) == 0 || PathContainsPattern(cp) || strendswith(cp, "/")) continue;
			for (j = 0; j < initializers_count; j++) {
				if (strcmp(cp, initializers[j])) continue;
				snprintf(buffer, sizeof(buffer) - 1, "%s %s", domain_list[index].domainname, initializers[j]);
				NormalizeLine(buffer);
				if (FindDomain(buffer) == EOF) {
					const int virtual_index = FindOrAssignNewDomain(buffer);
					if (virtual_index == EOF) {
						fprintf(stderr, "ERROR: Can't create domain for '%s'.\n", buffer);
						exit(1);
					}
					domain_list[virtual_index].is_virtual = 1;
				}
			}
		}
	}
	
	SortPolicy();

	// Merge ACLs for same domains.
	for (index = 1; index < domain_list_count; index++) {
		ACL_RECORD *acl_ptr;
		const char **string_ptr;
		int i;
		if (strcmp(domain_list[index - 1].domainname, domain_list[index].domainname)) continue;
		// Merge file policy.
		acl_ptr = domain_list[index].acl_ptr;
		for (i = 0; i < domain_list[index].acl_count; i++) AddFileACL(acl_ptr[i].name, acl_ptr[i].perm, index - 1);
		// Merge string policy.
		string_ptr = domain_list[index].string_ptr;
		for (i = 0; i < domain_list[index].string_count; i++) AddStringEntry(string_ptr[i], index - 1);
		// Delete duplicated domain.
		DeleteDomain(index);
		index = 0;
	}

	scrollok(stdscr, 0);

	SortPolicy();

	if (acl_updated) {
		int c;
		printw("Policy has changed. Save?(Y/N)");
		do { c = getch(); } while (!(c == 'Y' || c == 'y' || c == 'N' || c == 'n' || c == EOF));
		printw("%c\n", c);
		if (c == 'Y' || c == 'y') {
			SavePolicy();
			acl_updated = 0;
		}
	}
}

static void SavePolicy(void) {
	FILE *fp;
	int index;
	static char filename[1024];
	time_t now = time(NULL);
	struct tm *tm = localtime(&now);
	memset(filename, 0, sizeof(filename));
	snprintf(filename, sizeof(filename) - 1, "domain_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
	if ((fp = fopen(filename, "w")) == NULL) {
		fprintf(stderr, "Can't open %s\n", filename); return;
	}
	for (index = 0; index < domain_list_count; index++) {
		int i;
		ACL_RECORD *acl_ptr = domain_list[index].acl_ptr;
		const char **string_ptr = domain_list[index].string_ptr;
		if (domain_list[index].is_virtual == 1) continue;
		fprintf(fp, "%s\n\n", domain_list[index].domainname);
		for (i = 0; i < domain_list[index].acl_count; i++) fprintf(fp, "%d %s \n", acl_ptr[i].perm, acl_ptr[i].name);
		for (i = 0; i < domain_list[index].string_count; i++) fprintf(fp, "%s\n", string_ptr[i]);
		fprintf(fp, "\n");
	}
	fclose(fp);
	unlink("domain_policy.txt");
	symlink(filename, "domain_policy.txt");
}

////////////////////////////  UI HANDLER  ////////////////////////////

static int DomainTreeLoop(void);
static int ACLTreeLoop(void);
static void ShowDomainList(void);
static void ShowACLList(void);
static void ShowList(const int screen);

static void ResizeWindow(const int screen);
static void UpArrowKey(const int screen);
static void DownArrowKey(const int screen);
static void PageUpKey(const int screen);
static void PageDownKey(const int screen);
static void ShowCurrent(const int screen);
static void ShowPrompt(const int screen, const char *prompt);
static int GetKey(void);
static void UpdateCursorPos(const int screen, const int item_count);
static int GetCurrent(const int screen);
static void CommandEnter(const int screen);

#define MAXSCREEN 2

static int window_width = 0, window_height = 0;
static int current_y[MAXSCREEN], current_item_index[MAXSCREEN], list_item_count[MAXSCREEN];
static int pipe_fd[2] = { EOF, EOF };
static char search_buffer[4096];

static const int header_lines = 3;
static int body_lines = 0;

static void ShowDomainList(void) {
	int i;
	const int offset = current_item_index[0];
	static char buffer[MAXBUFSIZE];
	list_item_count[0] = domain_list_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	memset(buffer, 0, sizeof(buffer));
	i = list_item_count[0];
	snprintf(buffer, sizeof(buffer) - 1, "<<< Domain Transition Editor >>>      %d domain%c", i, i > 1 ? 's' : ' ');
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(0, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "Commands =  Q:quit  D:delete  R:refresh  Enter:select  @:search  %s", acl_updated ? "S:save" : "");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(1, 0, "%s", buffer);
	for (i = 0; i < body_lines; i++) {
		const char *sp;
		char *dp, *cp;
		char flag1, flag2;
		int count = 0;
		if (offset + i >= list_item_count[0]) break;
		flag1 = IsTrusted(offset + i) ? '#' : ' ';
		flag2 = IsInitializer(offset + i) ? '*' : ' ';
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "%5d: %c%c ", offset + i, flag1, flag2);
		sp = DomainName(offset + i);
		while ((cp = strchr(sp, ' ')) != NULL) {
			count += 4; sp = cp + 1;
		}
		dp = strchr(buffer, '\0');
		while (count) {
			if (dp - buffer >= sizeof(buffer)) exit(1);
			*dp++ = ' '; count--;
		}
		while (*sp) {
			if (dp - buffer >= sizeof(buffer)) exit(1);
			*dp++ = *sp++;
		}
		if (IsVirtual(offset + i)) {
			static char buf[MAXBUFSIZE];
			char *sp = buf;
			int redirect_index;
			memset(buf, 0, sizeof(buf));
			snprintf(buf, sizeof(buf) - 1, ROOT_NAME "%s", strrchr(DomainName(offset + i), ' '));
			redirect_index = FindDomain(buf);
			snprintf(buf, sizeof(buf) - 1, " ( -> %d )", redirect_index);
			while (*sp) {
				if (dp - buffer >= sizeof(buffer)) exit(1);
				*dp++ = *sp++;
			}
		}
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		mvprintw(header_lines + i, 0, "%s", buffer);
	}
	ShowCurrent(0);
}

static int DomainTreeLoop(void) {
	ShowList(0);
	while (1) {
		int c = getch();
		if (c == 'q' || c == 'Q') break;
		if (c == ERR) continue; // Ignore invalid key.
		switch(c) {
		case KEY_RESIZE:
			ResizeWindow(0);
			ShowList(0);
			break;
		case KEY_UP:
			UpArrowKey(0);
			break;
		case KEY_DOWN:
			DownArrowKey(0);
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey(0);
			break;
		case KEY_NPAGE:
			PageDownKey(0);
			break;
		case '@':
			{
				int i;
			input_path:
				mvprintw(window_height - 1, 0, "Search> "); clrtoeol(); refresh(); echo();
				memset(search_buffer, 0, sizeof(search_buffer));
				getnstr(search_buffer, sizeof(search_buffer) - 1); noecho();
				NormalizeLine(search_buffer);
				if (!search_buffer[0]) break;
				write(pipe_fd[1], search_buffer, sizeof(search_buffer));
				for (i = 0; i < list_item_count[0]; i++) {
					const char *cp = DomainName(i);
					if (strchr(search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, search_buffer)) continue;
					while (i < current_y[0] + current_item_index[0]) {
						if (current_y[0] > 0) current_y[0]--;
						else current_item_index[0]--;
					}
					while (i > current_y[0] + current_item_index[0]) {
						if (current_y[0] < body_lines - 1) current_y[0]++;
						else current_item_index[0]++;
					}
					ShowList(0);
					break;
				}
			}
			break;
		case KEY_LEFT:
			{
				int i;
				if (!search_buffer[0]) goto input_path;
				for (i = GetCurrent(0) - 1; i >= 0; i--) {
					const char *cp = DomainName(i);
					if (strchr(search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, search_buffer)) continue;
					while (i < current_y[0] + current_item_index[0]) {
						if (current_y[0] > 0) current_y[0]--;
						else current_item_index[0]--;
					}
					ShowList(0);
					break;
				}
			}
			break;
		case KEY_RIGHT:
			{
				int i;
				if (!search_buffer[0]) goto input_path;
				for (i = GetCurrent(0) + 1; i < list_item_count[0]; i++) {
					const char *cp = DomainName(i);
					if (strchr(search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, search_buffer)) continue;
					while (i > current_y[0] + current_item_index[0]) {
						if (current_y[0] < body_lines - 1) current_y[0]++;
						else current_item_index[0]++;
					}
					ShowList(0);
					break;
				}
			}
			break;
		case 'd':
		case 'D':
			{
				const int current = GetCurrent(0);
				if (current > 0) { // Never delete ROOT_NAME
					ShowPrompt(0, "DELETE?");
					if (GetKey()) {
						const char *key = DomainName(current);
						const int key_len = strlen(key);
						const char *exe = strrchr(key, ' ') + 1;
						const int parent = FindParent(key);
						if (IsVirtual(current)) {
							if (parent >= 0 && DelACL(exe, parent) == 0) {
								DeleteDomain(current);
								acl_updated = 1;
							}
						} else {
							int index;
							for (index = 0; index < domain_list_count; index++) {
								const char *cp = DomainName(index);
								if (strncmp(cp, key, key_len) || (cp[key_len] != '\0' && cp[key_len] != ' ')) continue;
								DeleteDomain(index);
								index = 0;
								acl_updated = 1;
							}
							if (parent >= 0 && DelACL(exe, parent) == 0) acl_updated = 1;
						}
						UpdateCursorPos(0, domain_list_count);
					}
					ShowList(0);
				}
			}
			break;
		case 's':
		case 'S':
			if (!acl_updated) break;
			SavePolicy();
			acl_updated = 0;
			ShowList(0);
			break;
		case 'r':
		case 'R':
			_exit(0);
			break;
		case '\r':
			CommandEnter(0);
			break;
		}
	}
	if (acl_updated) {
		int c;
		move(window_height - 1, 0);
		clrtoeol();
		printw("Policy has changed. Save?(Y/N)");
		do { c = getch(); } while (!(c == 'Y' || c == 'y' || c == 'N' || c == 'n' || c == EOF));
		printw("%c\n", c);
		if (c == 'Y' || c == 'y') SavePolicy();
	}
	return 1;
}

static void ShowACLList(void) {
	const int current = GetCurrent(0), offset = current_item_index[1];
	static char buffer[MAXBUFSIZE];
	DOMAIN_INFO *ptr = &domain_list[current];
	ACL_RECORD *acl_ptr = ptr->acl_ptr;
	const char **string_ptr = ptr->string_ptr;
	int i;
	list_item_count[1] = ptr->acl_count + ptr->string_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	memset(buffer, 0, sizeof(buffer));
	i = list_item_count[1];
	snprintf(buffer, sizeof(buffer) - 1, "<<< ACL Viewer >>>      %d acl%c", i, i > 1 ? 's' : ' ');
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(0, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "Commands =  Q:quit  A:append  D:delete  %s", acl_updated ? "S:save" : "");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(1, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "%s", DomainName(current));
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(2, 0, "%s", buffer);
	for (i = 0; i < body_lines; i++) {
		int index;
		if (offset + i >= list_item_count[1]) break;
		if (offset + i < ptr->acl_count) {
			index = offset + i;
			mvprintw(header_lines + i, 0, "%5d: %d %s", offset + i, acl_ptr[index].perm, acl_ptr[index].name);
		} else {
			index = offset + i - ptr->acl_count;
			mvprintw(header_lines + i, 0, "%5d: %s", offset + i, string_ptr[index]);
		}
	}
	ShowCurrent(1);
}

static int ACLTreeLoop(void) {
	current_y[1] = current_item_index[1] = 0;
	ShowList(1);
	while (1) {
		int c = getch();
		//if (c == 0x1B) break;
		if (c == 'q' || c == 'Q') break;
		if (c == ERR) continue; // Ignore invalid key.
		switch(c) {
		case KEY_RESIZE:
			ResizeWindow(1);
			ShowList(1);
			break;
		case KEY_UP:
			UpArrowKey(1);
			break;
		case KEY_DOWN:
			DownArrowKey(1);
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey(1);
			break;
		case KEY_NPAGE:
			PageDownKey(1);
			break;
		case 'a':
		case 'A':
			{
				static char buffer[MAXBUFSIZE];
				const int domain = GetCurrent(0);
				int perm = 0;
				char *cp;
				mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
				memset(buffer, 0, sizeof(buffer));
				getnstr(buffer, sizeof(buffer) - 1); noecho();
				NormalizeLine(buffer);
				if (sscanf(buffer, "%d", &perm) == 1 && (cp = strchr(buffer, ' ')) != NULL && IsCorrectPath(cp + 1, 1)) {
					cp++;
					if (AddFileACL(cp, (WORD) perm, domain) == 0) {
						acl_updated = 1;
						if ((perm & 1) == 1 && PathContainsPattern(cp) == 0 && strendswith(cp, "/") == 0) {
							int i;
							for (i = 0; i < initializers_count; i++) {
								if (strcmp(cp, initializers[i])) continue;
								memset(buffer, 0, sizeof(buffer));
								snprintf(buffer, sizeof(buffer) - 1, "%s %s", domain_list[domain].domainname, initializers[i]);
								NormalizeLine(buffer);
								if (FindDomain(buffer) == EOF) {
									int virtual_index = FindOrAssignNewDomain(buffer);
									if (virtual_index == EOF) {
										fprintf(stderr, "ERROR: Can't create domain for '%s'.\n", buffer);
										exit(1);
									}
									domain_list[virtual_index].is_virtual = 1;
								}
								break;
							}
						}
					}
				} else if (strncmp(buffer, "allow_", 6) == 0) {
					if (AddStringEntry(buffer, domain) == 0) acl_updated = 1;
				}
				ShowList(1);
			}
			break;
		case 'd':
		case 'D':
			{
				const int current = GetCurrent(1);
				if (current >= 0) {
					ShowPrompt(1, "DELETE?");
					if (GetKey()) {
						const int domain = GetCurrent(0);
						DOMAIN_INFO *ptr = &domain_list[domain];
						int index;
						if (current < ptr->acl_count) {
							index = current;
							if (DelACL(ptr->acl_ptr[current].name, domain) == 0) acl_updated = 1;
						} else {
							index = current - ptr->acl_count;
							if (DelStringEntry(ptr->string_ptr[index], domain) == 0) acl_updated = 1;
						}
						UpdateCursorPos(1, ptr->acl_count + ptr->string_count);
					}
					ShowList(1);
				}
			}
			break;
		case 's':
		case 'S':
			if (!acl_updated) break;
			SavePolicy();
			acl_updated = 0;
			ShowList(1);
			break;
		case '\r':
			CommandEnter(1);
			break;
		}
	}
	return 0;
}

static void ShowList(const int screen) {
	if (screen == 0) ShowDomainList();
	else ShowACLList();
}

static void ResizeWindow(const int screen) {
	getmaxyx(stdscr, window_height, window_width);
	body_lines = window_height - header_lines;
	if (body_lines <= current_y[screen]) current_y[screen] = body_lines - 1;
	if (current_y[screen] < 0) current_y[screen] = 0;
}

static void UpArrowKey(const int screen) {
	if (current_y[screen] > 0) {
		current_y[screen]--;
		ShowCurrent(screen);
	} else if (current_item_index[screen] > 0) {
		current_item_index[screen]--;
		ShowList(screen);
	}
}

static void DownArrowKey(const int screen) {
	if (current_y[screen] < body_lines - 1) {
		if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
			current_y[screen]++;
			ShowCurrent(screen);
		}
	} else if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
		current_item_index[screen]++;
		ShowList(screen);
	}
}

static void PageUpKey(const int screen) {
	if (current_item_index[screen] + current_y[screen] > body_lines) {
		current_item_index[screen] -= body_lines;
		if (current_item_index[screen] < 0) current_item_index[screen] = 0;
		ShowList(screen);
	} else if (current_item_index[screen] + current_y[screen] > 0) {
		current_item_index[screen] = 0;
		current_y[screen] = 0;
		ShowList(screen);
	}
}

static void PageDownKey(const int screen) {
	if (list_item_count[screen] - current_item_index[screen] > body_lines) {
		current_item_index[screen] += body_lines;
		if (current_item_index[screen] + current_y[screen] > list_item_count[screen] - 1) current_y[screen] = list_item_count[screen] - 1 - current_item_index[screen];
		ShowList(screen);
	} else if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
		current_y[screen] = list_item_count[screen] - current_item_index[screen] - 1;
		ShowCurrent(screen);
	}
}

static int GetCurrent(const int screen) {
	if (list_item_count[screen] == 0) return EOF;
	if (current_item_index[screen] + current_y[screen] < 0 || current_item_index[screen] + current_y[screen] >= list_item_count[screen]) {
		fprintf(stderr, "ERROR: current_item_index=%d current_y=%d\n", current_item_index[screen], current_y[screen]);
		exit(1);
	}
	return current_item_index[screen] + current_y[screen];
}

static void ShowCurrent(const int screen) {
	if (screen == 0) {
		static char buffer[MAXBUFSIZE];
		int current = GetCurrent(screen);
		memset(buffer, 0, sizeof(buffer));
		//if (IsVirtual(current)) {
		//snprintf(buffer, sizeof(buffer) - 1, ROOT_NAME "%s", strrchr(DomainName(current), ' '));
		//} else {
		snprintf(buffer, sizeof(buffer) - 1, "%s", DomainName(current));
		//}
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		move(2, 0);
		clrtoeol();
		printw("%s", buffer);
	}
	move(header_lines + current_y[screen], 0);
	refresh();
}

static void ShowPrompt(const int screen, const char *prompt) {
	static char buffer[MAXBUFSIZE];
	memset(buffer, 0, sizeof(buffer));
	snprintf(buffer, sizeof(buffer) - 1, "%s", prompt);
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(header_lines + current_y[screen], 0, "%s", buffer);
	move(body_lines + current_y[screen], 0);
	refresh();
}

static int GetKey(void) {
	int c = getch();
	if (c == 'Y' || c == 'y' || c == '\r') return 1;
	if (c == KEY_RESIZE) ungetch(c);
	return 0;
}

static void UpdateCursorPos(const int screen, const int item_count) {
	list_item_count[screen] = item_count;
	if (list_item_count[screen] > 0) {
		if (current_item_index[screen] + current_y[screen] > list_item_count[screen] - 1) current_y[screen] = list_item_count[screen] - 1 - current_item_index[screen];
		if (current_y[screen] < 0) {
			current_item_index[screen] += current_y[screen]; current_y[screen] = 0;
		}
	}
}

static void CommandEnter(const int screen) {
	int current = GetCurrent(screen);
	if (screen == 0) {
		if (IsVirtual(current)) {
			static char buf[MAXBUFSIZE];
			int redirect_index;
			memset(buf, 0, sizeof(buf));
			snprintf(buf, sizeof(buf) - 1, ROOT_NAME "%s", strrchr(DomainName(current), ' '));
			redirect_index = FindDomain(buf);
			if (redirect_index != EOF) {
				current_item_index[0] = redirect_index - current_y[0];
				ShowList(0);
			}
		} else {
			ACLTreeLoop();
			ShowList(0);
		}
	} else {
		// Call pathname editor.
	}
}

int main(int argc, char *argv[]) {
	int current_x;
	memset(search_buffer, 0, sizeof(search_buffer));
	memset(current_y, 0, sizeof(current_y));
	memset(current_item_index, 0, sizeof(current_item_index));
	memset(list_item_count, 0, sizeof(list_item_count));
	if (argc > 1) {
		if (chdir(argv[1]) != 0) {
			fprintf(stderr, "Can't chdir %s\n", argv[1]);
			return 1;
		}
	} else {
		if (chdir("/root/security") != 0) {
			fprintf(stderr, "Can't chdir /root/security\n");
			return 1;
		}
	}
	setenv("TERM", "linux", 1);
	initscr();
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);
	getyx(stdscr, current_y[0], current_x);
	getmaxyx(stdscr, window_height, window_width);
	while (1) {
		int status = 0;
		pipe(pipe_fd);
		switch(fork()) {
		case 0:
			close(pipe_fd[0]);
			LoadPolicy();
			ResizeWindow(0);
			_exit(DomainTreeLoop());
		case -1:
			close(pipe_fd[0]);
			break;
		}
		close(pipe_fd[1]);
		while (read(pipe_fd[0], search_buffer, sizeof(search_buffer)) > 0);
		close(pipe_fd[0]);
		while (wait(&status) == EOF && errno == EINTR);
		if (!WIFEXITED(status) || WEXITSTATUS(status)) break;
	}
	clear();
	move(0, 0);
	refresh();
	endwin();
	return 0;
}
