/*
    orzmdi
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "orzmdi.h"
#include "misc/fileio.h"
#include "misc/misc.h"


enum
{
  ADD_WINDOW_SIGNAL,
  CHANGE_WINDOW_SIGNAL,
  CLOSE_WINDOW_SIGNAL,
  REMOVE_WINDOW_SIGNAL,
  REORDER_WINDOW_SIGNAL,
  SWITCH_WINDOW_SIGNAL,
  LAST_SIGNAL
};
enum
{
  TARGET_ORZ_MDI_TAB
};


struct _OrzMdiTab
{
  gpointer page_data;
  gboolean created;     /* ja:TRUE:新規作成,FALSE:既存ファイル */
  gboolean edited;      /* ja:TRUE:編集済み,FALSE:未編集 */
  gchar *file;
  gint same;            /* ja:-2:名前のみ,-1:フルパス,0...:フルパス+数字 */
  GtkWidget *child, *image, *label;
  OrzMdi *mdi;
};


static gint orz_mdi_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
static void
orz_mdi_label_markup (OrzMdiTab *tab)
{
  gchar *label;

  label = tab->same < -1 ? g_filename_display_basename (tab->file)
                         : g_filename_display_name (tab->file);
  if (tab->edited)
    {
      gchar *str;

      str = g_strconcat ("<b><i>", label, "</i></b>", NULL);
      g_free (label);
      label = str;
    }
  if (tab->same >= 0)
    {
      gchar *str;

      str = g_strdup_printf ("%s:%d", label, tab->same);
      g_free (label);
      label = str;
    }
  gtk_label_set_markup (GTK_LABEL (tab->label), label);
  g_free (label);
}


static void
file_add_edit (OrzMdi    *mdi,
               OrzMdiTab *tab0)
{
  GList *gl, *glist;
  OrzMdiTab *tab1;

  tab0->same = -2;
  /* ja:ファイル名を調べる */
  glist = gtk_container_get_children (GTK_CONTAINER (mdi));
  /* ja:同名同パスを調べる */
  for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
    {
      tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
                                (GTK_NOTEBOOK (mdi), gl->data)), "user_data");
      if (tab1 && tab0 != tab1 && g_strfilecmp (tab0->file, tab1->file) == 0)
        {
          if (tab1->same < 0)
            {
              tab1->same = 0;
              orz_mdi_label_markup (tab1);
            }
          if (tab0->same <= tab1->same)
            tab0->same = tab1->same + 1;
        }
    }
  if (tab0->same < 0)
    {
      gchar *name0;

      /* ja:同名同パスはないとき */
      /* ja:同名異パスを調べる */
      name0 = g_path_get_basename (tab0->file);
      for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
        {
          tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
                                (GTK_NOTEBOOK (mdi), gl->data)), "user_data");
          if (tab1 && tab0 != tab1)
            {
              gchar *name1;

              name1 = g_path_get_basename (tab1->file);
              if (g_strfilecmp (name0, name1) == 0)
                {
                  if (tab1->same < -1)
                    {
                      tab1->same = -1;
                      orz_mdi_label_markup (tab1);
                    }
                  g_free (name1);
                  break;
                }
              g_free (name1);
            }
        }
      g_free (name0);
      if (gl)
        tab0->same = -1;
    }
  g_list_free (glist);
}


static void
file_remove_edit (OrzMdi      *mdi,
                  OrzMdiTab   *tab0,
                  const gchar *file)
{
  gint count0 = 0, count1 = 0;
  gchar *name0, *name1;
  GList *gl, *glist;
  OrzMdiTab *tab1;

  /* ja:ファイル名を調べる */
  glist = gtk_container_get_children (GTK_CONTAINER (mdi));
  name0 = g_path_get_basename (tab0->file);
  for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
    {
      tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
                                (GTK_NOTEBOOK (mdi), gl->data)), "user_data");
      if (tab1 && tab0 != tab1)
        {
          /* ja:同名同パスを調べる */
          if (g_strfilecmp (tab0->file, tab1->file) == 0)
            count0++;
          /* ja:同名異パスを調べる */
          name1 = g_path_get_basename (tab1->file);
          if (g_strfilecmp (name0, name1) == 0)
            count1++;
          g_free (name1);
        }
    }
  if (count1 == 1)/* ja:同名のファイルが1つだけあるとき */
    {
      gchar *name2;

      name2 = file ? g_path_get_basename (file) : NULL;
      if (!name2 || g_strfilecmp (name0, name2) != 0)
        /* 名前が変わっているとき */
        for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
          {
            /* ja:同名のファイルを通常表示に変更する */
            tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
                                (GTK_NOTEBOOK (mdi), gl->data)), "user_data");
            if (tab1 && tab0 != tab1)
              {
                /* ja:同名を調べる */
                name1 = g_path_get_basename (tab1->file);
                if (g_strfilecmp (name0, name1) == 0)
                  {
                    tab1->same = -2;
                    orz_mdi_label_markup (tab1);
                  }
                g_free (name1);
              }
          }
      g_free (name2);
    }
  else if (count0 == 1)/* ja:同名同パスが1つだけある(異パスは複数)とき */
    {
      for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
        {
          /* ja:同名同パスのファイルをフルパスに変更する */
          tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
                                (GTK_NOTEBOOK (mdi), gl->data)), "user_data");
          /* ja:同名同パスを調べる */
          if (tab1 && tab0 != tab1
                                && g_strfilecmp (tab0->file, tab1->file) == 0)
            {
              tab1->same = -1;
              orz_mdi_label_markup (tab1);
            }
        }
    }
  g_free (name0);
  g_list_free (glist);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
G_DEFINE_TYPE (OrzMdi, orz_mdi, GTK_TYPE_NOTEBOOK);


/* ja:子ウィジェットが破棄されるとき */
static void
orz_mdi_remove (GtkContainer *container,
                GtkWidget    *widget)
{
  GtkWidget *tab_label;

  tab_label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (container), widget);
  if (tab_label)
    {
      OrzMdiTab *tab;

      tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
      if (tab)
        {
          g_signal_emit (container,
                    orz_mdi_signals[REMOVE_WINDOW_SIGNAL],
                    0,
                    gtk_notebook_page_num (GTK_NOTEBOOK (container), widget),
                    tab->page_data);
          file_remove_edit (ORZ_MDI (container), tab, NULL);
        }
    }
  GTK_CONTAINER_CLASS (orz_mdi_parent_class)->remove (container, widget);
}


static void
orz_mdi_switch_page (GtkNotebook     *notebook,
#if GTK_CHECK_VERSION(2,22,0)
                     gpointer         page,
#else /* not GTK_CHECK_VERSION(2,22,0) */
                     GtkNotebookPage *page,
#endif /* not GTK_CHECK_VERSION(2,22,0) */
                     guint            page_num)
{
  GtkWidget *child;

  GTK_NOTEBOOK_CLASS (orz_mdi_parent_class)->switch_page
                                                    (notebook, page, page_num);
  child = gtk_notebook_get_nth_page (notebook, page_num);
  if (child)
    {
      GtkWidget *tab_label;

      tab_label = gtk_notebook_get_tab_label (notebook, child);
      if (tab_label)
        {
          OrzMdiTab *tab;

          tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
          if (tab)
            g_signal_emit (ORZ_MDI (notebook),
                                        orz_mdi_signals[SWITCH_WINDOW_SIGNAL],
                                        0,
                                        page_num,
                                        tab->page_data);
        }
    }
}


static void
orz_mdi_class_init (OrzMdiClass *klass)
{
  GtkContainerClass *container_class;
  GtkNotebookClass *notebook_class;

  container_class = GTK_CONTAINER_CLASS (klass);
  container_class->remove = orz_mdi_remove;
  notebook_class = GTK_NOTEBOOK_CLASS (klass);
  notebook_class->switch_page = orz_mdi_switch_page;

  klass->add_window = NULL;
  klass->change_window = NULL;
  klass->close_window = NULL;
  klass->remove_window = NULL;
  klass->reorder_window = NULL;
  klass->switch_window = NULL;

  orz_mdi_signals[ADD_WINDOW_SIGNAL]
                = g_signal_new ("add_window",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (OrzMdiClass, add_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  orz_mdi_signals[CHANGE_WINDOW_SIGNAL]
                = g_signal_new ("change_window",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (OrzMdiClass, change_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  orz_mdi_signals[CLOSE_WINDOW_SIGNAL]
                = g_signal_new ("close_window",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (OrzMdiClass, close_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  orz_mdi_signals[REMOVE_WINDOW_SIGNAL]
                = g_signal_new ("remove_window",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (OrzMdiClass, remove_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  orz_mdi_signals[REORDER_WINDOW_SIGNAL]
                = g_signal_new ("reorder_window",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (OrzMdiClass, reorder_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  orz_mdi_signals[SWITCH_WINDOW_SIGNAL]
                = g_signal_new ("switch_window",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (OrzMdiClass, switch_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
}


static void
orz_mdi_init (OrzMdi *mdi)
{
#if ! GTK_CHECK_VERSION(2,12,0)
  mdi->tooltips = gtk_tooltips_new ();
#endif /* not GTK_CHECK_VERSION(2,12,0) */
}


/******************************************************************************
*                                                                             *
* ja:更新関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget *
orz_mdi_new (void)
{
  return GTK_WIDGET (g_object_new (ORZ_TYPE_MDI, NULL));
}


/* ja:ボタンが押されたとき */
static void
orz_mdi_clicked (GtkWidget *widget,
                 GtkWidget *tab_label)
{
  OrzMdiTab *tab;

  tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
  if (tab)
    g_signal_emit (tab->mdi,
           orz_mdi_signals[CLOSE_WINDOW_SIGNAL],
           0,
           gtk_notebook_page_num (GTK_NOTEBOOK (tab->mdi), tab->child),
           tab->page_data);
}


/* ja:子ウィジェットが破棄されるとき */
static void
orz_mdi_destroy_child (GtkWidget *widget,
                       OrzMdi    *mdi)
{
  GtkWidget *tab_label;

  tab_label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (mdi), widget);
  if (tab_label)
    {
      OrzMdiTab *tab;

      tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
      if (tab)
        {
          g_signal_emit (mdi,
                         orz_mdi_signals[REMOVE_WINDOW_SIGNAL],
                         0,
                         gtk_notebook_page_num (GTK_NOTEBOOK (mdi), widget),
                         tab->page_data);
          file_remove_edit (ORZ_MDI (mdi), tab, NULL);
        }
    }
}


/* ja:タブが破棄されるとき */
static void
orz_mdi_destroy_tab (GtkWidget *widget,
                     OrzMdi    *mdi)
{
  OrzMdiTab *tab;

  tab = g_object_get_data (G_OBJECT (widget), "user_data");
  if (tab)
    {
      g_free (tab->file);
      g_free (tab);
      g_object_set_data (G_OBJECT (widget), "user_data", NULL);
    }
}


/* ja:ドロップされたとき */
static void
orz_mdi_tab_drag_begin (GtkWidget      *widget,
                        GdkDragContext *context,
                        gpointer        user_data)
{
  GdkPixbuf *pixbuf;
  OrzMdiTab *tab;

  tab = g_object_get_data (G_OBJECT (widget), "user_data");
  pixbuf = fileio_extract_icon (tab->file, GTK_ICON_SIZE_DND);
  if (pixbuf)
    {
      gtk_drag_source_set_icon_pixbuf (widget, pixbuf);
      g_object_unref (pixbuf);
    }
}
static void
orz_mdi_tab_drag_data_get (GtkWidget        *widget,
                           GdkDragContext   *context,
                           GtkSelectionData *data,
                           guint             info,
                           guint             time,
                           gpointer          user_data)
{
  const static gchar *text = "dummy";

  if (info == TARGET_ORZ_MDI_TAB)
    gtk_selection_data_set (data, gtk_selection_data_get_selection (data), 8,
                                        (const guchar *)text, g_strlen (text));
}
static void
orz_mdi_tab_drag_data_received (GtkWidget        *widget,
                                GdkDragContext   *context,
                                gint              x,
                                gint              y,
                                GtkSelectionData *data,
                                guint             info,
                                guint             time,
                                gpointer          user_data)
{
  if (info == TARGET_ORZ_MDI_TAB)
    {
      gint page_num;
      OrzMdiTab *tab;

      tab = g_object_get_data (G_OBJECT (widget), "user_data");
      page_num = gtk_notebook_page_num (GTK_NOTEBOOK (tab->mdi), tab->child);
      tab = g_object_get_data
                (G_OBJECT (gtk_drag_get_source_widget (context)), "user_data");
      gtk_notebook_reorder_child (GTK_NOTEBOOK (tab->mdi), tab->child,
                                                                    page_num);
      g_signal_emit (tab->mdi,
                     orz_mdi_signals[REORDER_WINDOW_SIGNAL],
                     0,
                     page_num,
                     tab->page_data);
    }
}


/*  ja:ページを最後に追加する
          mdi,ウィジェット
        child,内容
         file,ファイル名
      created,TRUE:新規作成,FALSE:既存ファイル
    page_data,ページデータ
          RET,ページ(0...,-1:エラー)                                        */
gint
orz_mdi_append_page (OrzMdi         *mdi,
                     GtkWidget      *child,
                     const gchar    *file,
                     const gboolean  created,
                     gpointer        page_data)
{
  return orz_mdi_insert_page (mdi, child, -1, file, created, page_data);
}


/*  ja:ページを最初に追加する
          mdi,ウィジェット
        child,内容
         file,ファイル名
      created,TRUE:新規作成,FALSE:既存ファイル
    page_data,ページデータ
          RET,ページ(0...,-1:エラー)                                        */
gint
orz_mdi_prepend_page (OrzMdi         *mdi,
                      GtkWidget      *child,
                      const gchar    *file,
                      const gboolean  created,
                      gpointer        page_data)
{
  return orz_mdi_insert_page (mdi, child, 0, file, created, page_data);
}


/*  ja:ページを挿入する
          mdi,ウィジェット
        child,内容
     position,位置(0...)
         file,ファイル名
      created,TRUE:新規作成,FALSE:既存ファイル
    page_data,ページデータ
       RET,ページ(0...,-1:エラー)                                           */
gint
orz_mdi_insert_page (OrzMdi         *mdi,
                     GtkWidget      *child,
                     const gint      position,
                     const gchar    *file,
                     const gboolean  created,
                     gpointer        page_data)
{
  gint page_num;
  GdkPixbuf *pixbuf;
  GtkTargetEntry dnd_entries[1] = {{"ORZ_MDI_TAB",
                                    GTK_TARGET_SAME_APP, TARGET_ORZ_MDI_TAB}};
  GtkWidget *button, *hbox, *event;
  OrzMdiTab *tab;

  if (!mdi || !child)
    return -1;
  tab = g_malloc0 (sizeof (OrzMdiTab));
  tab->page_data = page_data;
  tab->created = created;
  tab->file = file ? created ? g_strdup (file) : fileio_get_full_path (file)
                                                            : g_strdup ("");
  tab->same = -2;
  tab->mdi = mdi;
  tab->child = child;
  file_add_edit (mdi, tab);
  /* ja:ボタン */
  button = misc_close_button ();
  gtk_widget_show (button);
  /* ja:ラベル */
  tab->label = gtk_label_new ("");
  orz_mdi_label_markup (tab);
  gtk_widget_show (tab->label);
  /* ja:アイコン */
  pixbuf = fileio_extract_icon (tab->file, GTK_ICON_SIZE_BUTTON);
  if (pixbuf)
    {
      tab->image = gtk_image_new_from_pixbuf (pixbuf);
      g_object_unref (pixbuf);
      gtk_widget_show (tab->image);
    }
  /* ja:表示 */
  hbox = gtk_hbox_new (FALSE, 0);
  if (tab->image)
    gtk_box_pack_start (GTK_BOX (hbox), tab->image, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), tab->label, TRUE, TRUE, 0);
  gtk_widget_show (hbox);
  /* ja:ツールチップス */
  event = gtk_event_box_new ();
#if GLIB_CHECK_VERSION(2,4,0)
  gtk_event_box_set_visible_window (GTK_EVENT_BOX (event), FALSE);
#endif /* GLIB_CHECK_VERSION(2,4,0) */
  gtk_container_add (GTK_CONTAINER (event), hbox);
#if GTK_CHECK_VERSION(2,12,0)
  gtk_widget_set_has_tooltip (event, TRUE);
#else /* not GTK_CHECK_VERSION(2,12,0) */
  gtk_tooltips_set_tip (mdi->tooltips, event, tab->file, tab->file);
#endif /* not GTK_CHECK_VERSION(2,12,0) */
  g_object_set_data (G_OBJECT (event), "user_data", tab);

  g_signal_connect (G_OBJECT (button), "clicked",
                                    G_CALLBACK (orz_mdi_clicked), event);
  g_signal_connect (G_OBJECT (child), "destroy",
                                    G_CALLBACK (orz_mdi_destroy_child), mdi);
  g_signal_connect (G_OBJECT (event), "destroy",
                                    G_CALLBACK (orz_mdi_destroy_tab), mdi);
  /* en:Drag & Drop */
  gtk_drag_source_set (event, GDK_BUTTON1_MASK,
                    dnd_entries, G_N_ELEMENTS (dnd_entries), GDK_ACTION_MOVE);
  gtk_drag_dest_set (event, GTK_DEST_DEFAULT_ALL,
                    dnd_entries, G_N_ELEMENTS (dnd_entries), GDK_ACTION_MOVE);
  g_signal_connect (G_OBJECT (event), "drag_begin",
                            G_CALLBACK (orz_mdi_tab_drag_begin), NULL);
  g_signal_connect (G_OBJECT (event), "drag_data_get",
                            G_CALLBACK (orz_mdi_tab_drag_data_get), NULL);
  g_signal_connect (G_OBJECT (event), "drag_data_received",
                            G_CALLBACK (orz_mdi_tab_drag_data_received), NULL);
  page_num = gtk_notebook_insert_page (GTK_NOTEBOOK (mdi), child,
                                                            event, position);
  g_signal_emit (mdi,
                 orz_mdi_signals[ADD_WINDOW_SIGNAL],
                 0,
                 page_num,
                 page_data);
  return page_num;
}


/*  ja:ページ数を取得する
    mdi,ウィジェット
    RET,ページ数                                                            */
gint
orz_mdi_get_n_pages (OrzMdi *mdi)
{
#if GTK_CHECK_VERSION(2,2,0)
  return gtk_notebook_get_n_pages (GTK_NOTEBOOK (mdi));
#else /* not GTK_CHECK_VERSION(2,2,0) */
  gint n_pages = 0;

  if (mdi)
    {
      GList *glist;

      glist = gtk_container_get_children (GTK_CONTAINER (mdi));
      if (glist)
        {
          n_pages = g_list_length (glist);
          g_list_free (glist);
        }
    }
  return n_pages;
#endif /* not GTK_CHECK_VERSION(2,2,0) */
}


/*  ja:ページをデータから取得する
          mdi,ウィジェット
    page_data,ページデータ
          RET,ページ(0...),-1:なし                                          */
gint
orz_mdi_get_page_from_data (OrzMdi  *mdi,
                            gpointer page_data)
{
  if (mdi)
    {
      gint i;

      for (i = orz_mdi_get_n_pages (ORZ_MDI (mdi)) - 1; i >= 0; i--)
        if (orz_mdi_get_data (ORZ_MDI (mdi), i) == page_data)
          return i;
    }
  return -1;
}


/*  ja:ページの内部を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,ページ内部データ                                               */
static OrzMdiTab *
orz_mdi_get_tab (OrzMdi     *mdi,
                 const gint  page_num)
{
  if (mdi)
    {
      GtkWidget *child;

      child = gtk_notebook_get_nth_page (GTK_NOTEBOOK (mdi), page_num >= 0
            ? page_num : gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi)));
      if (child)
        {
          GtkWidget *tab_label;

          tab_label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (mdi), child);
          if (tab_label)
            return g_object_get_data (G_OBJECT (tab_label), "user_data");
        }
    }
  return NULL;
}


/*  ja:ページデータを取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,ページデータ                                                   */
gpointer
orz_mdi_get_data (OrzMdi     *mdi,
                  const gint  page_num)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num);
  return tab ? tab->page_data : NULL;
}


/*  ja:ページデータを設定する
           mdi,ウィジェット
      page_num,ページ(0...),-1:カレント
     page_data,ページデータ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
orz_mdi_set_data (OrzMdi     *mdi,
                  const gint  page_num,
                  gpointer    page_data)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num);
  if (tab)
    {
      tab->page_data = page_data;
      return TRUE;
    }
  return FALSE;
}


/*  ja:ページのファイル名を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,ファイル名,NULL:エラー                                         */
const gchar *
orz_mdi_get_file (OrzMdi     *mdi,
                  const gint  page_num)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num);
  return tab ? tab->file : NULL;
}


/*  ja:ページのファイル名を設定する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
        file,ファイル名
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
orz_mdi_set_file (OrzMdi      *mdi,
                  const gint   page_num,
                  const gchar *file)
{
  if (mdi && file)
    {
      GtkWidget *child;

      child = gtk_notebook_get_nth_page (GTK_NOTEBOOK (mdi), page_num >= 0
            ? page_num : gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi)));
      if (child)
        {
          GtkWidget *tab_label;

          tab_label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (mdi), child);
          if (tab_label)
            {
              OrzMdiTab *tab;

              tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
              if (tab)
                {
                  GdkPixbuf *pixbuf;
                  GtkWidget *hbox;

                  /* ja:ラベル */
                  file_remove_edit (mdi, tab, file);
                  g_free (tab->file);
                  tab->file = tab->created ? g_strdup (file)
                                           : fileio_get_full_path (file);
                  tab->same = -2;
                  file_add_edit (mdi, tab);
                  orz_mdi_label_markup (tab);
                  /* ja:ツールチップス */
# if GTK_CHECK_VERSION(2,12,0)
                  gtk_widget_set_tooltip_text (tab_label, tab->file);
# else /* not GTK_CHECK_VERSION(2,12,0) */
                  gtk_tooltips_set_tip (mdi->tooltips, tab_label,
                                                        tab->file, tab->file);
# endif /* not GTK_CHECK_VERSION(2,12,0) */
                  /* ja:アイコン */
                  hbox = gtk_bin_get_child (GTK_BIN (tab_label));
                  if (tab->image)
                    gtk_container_remove (GTK_CONTAINER (hbox), tab->image);
                  tab->image = NULL;
                  pixbuf = fileio_extract_icon (tab->file,
                                                        GTK_ICON_SIZE_BUTTON);
                  if (pixbuf)
                    {
                      tab->image = gtk_image_new_from_pixbuf (pixbuf);
                      g_object_unref (pixbuf);
                      gtk_widget_show (tab->image);
                    }
                  if (tab->image)
                    gtk_box_pack_start (GTK_BOX (hbox),
                                                tab->image, FALSE, FALSE, 0);
                  g_signal_emit (mdi,
                                 orz_mdi_signals[CHANGE_WINDOW_SIGNAL],
                                 0,
                                 page_num,
                                 tab->page_data);
                  return TRUE;
                }
            }
        }
    }
  return FALSE;
}


/*  ja:ページの作成状態を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,TRUE:新規作成,FALSE:既存ファイル                               */
gboolean
orz_mdi_get_created (OrzMdi     *mdi,
                     const gint  page_num)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num);
  return tab ? tab->created : FALSE;
}


/*  ja:ページの作成状態を設定する
          mdi,ウィジェット
     page_num,ページ(0...),-1:カレント
    is_create,TRUE:新規作成,FALSE:既存ファイル
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
orz_mdi_set_created (OrzMdi         *mdi,
                     const gint      page_num,
                     const gboolean  is_create)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num);
  if (tab)
    {
      if (is_create ^ tab->created)
        tab->created = is_create;
      return TRUE;
    }
  return FALSE;
}


/*  ja:ページの編集状態を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,TRUE:編集済み,FALSE:未編集                                     */
gboolean
orz_mdi_get_edited (OrzMdi     *mdi,
                    const gint  page_num)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num);
  return tab ? tab->edited : FALSE;
}


/*  ja:ページの編集状態を設定する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
     is_edit,TRUE:編集済み,FALSE:未編集
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
orz_mdi_set_edited (OrzMdi         *mdi,
                    const gint      page_num,
                    const gboolean  is_edit)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num);
  if (tab)
    {
      if (is_edit ^ tab->edited)
        {
          tab->edited = is_edit;
          orz_mdi_label_markup (tab);
        }
      return TRUE;
    }
  return FALSE;
}


/*  ja:ページを移動する
             mdi,ウィジェット
    page_num_src,移動元ページ(0...),-1:カレント
    page_num_dst,移動先ページ(0...),-1:末尾
             RET,TRUE:正常終了,FALSE:エラー                                 */
gboolean
orz_mdi_reorder (OrzMdi     *mdi,
                 const gint  page_num_src,
                 const gint  page_num_dst)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num_src);
  if (!tab)
    return FALSE;
  gtk_notebook_reorder_child (GTK_NOTEBOOK (tab->mdi), tab->child,
                                                                page_num_dst);
  g_signal_emit (tab->mdi,
                 orz_mdi_signals[REORDER_WINDOW_SIGNAL],
                 0,
                 page_num_dst,
                 tab->page_data);
  return TRUE;
}


/*  ja:ページの表示名を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,表示名                                                         */
const gchar *
orz_mdi_get_display_name (OrzMdi     *mdi,
                          const gint  page_num)
{
  OrzMdiTab *tab;

  tab = orz_mdi_get_tab (mdi, page_num);
  return tab ? gtk_label_get_text (GTK_LABEL (tab->label)) : NULL;
}
