/*
    profile
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "fileio.h"
#include "profile.h"


#define PROFILE_DATA_TYPE_UNKNOW  0
#define PROFILE_DATA_TYPE_SPACE   1
#define PROFILE_DATA_TYPE_COMMENT 2
#define PROFILE_DATA_TYPE_SECTION 3
#define PROFILE_DATA_TYPE_KEY     4


typedef struct _ProfileList
{
  gchar *data, *section, *key, *value;
  guint type;
  struct _ProfileList *prev, *next;
} ProfileList;
struct _Profile
{
  gboolean edit;
  gchar *file;
  ProfileList *list, *sublist;
};


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群(内部)                                               *
*                                                                             *
******************************************************************************/
/*  ja:リストをテキスト化する
    profile_list,リスト
             RET,テキストリスト                                             */
static gchar *
profile_list_to_text (ProfileList *profile_list)
{
  gchar *text = NULL;
  ProfileList *p;

  for (p = profile_list; p; p = p->next)
    if (text)
      {
        gchar *tmp;

        tmp = text;
        text = g_strconcat (text, "\n", p->data, NULL);
        g_free (tmp);
      }
    else
      {
        text = p->data ? g_strdup (p->data) : g_strdup ("");
      }
  return text;
}


/*  ja:テキストをリスト化する
    text,テキスト
     RET,リスト                                                             */
static ProfileList *
profile_list_from_text (const gchar *text)
{
  gchar *buf, **str, *section = NULL;
  gint i, j = 0;
  ProfileList *p = NULL, *q, *r;

  /* ja:CRLFをLFに変換する */
  buf = g_strdup (text);
  for (i = 0; buf[i] != '\0'; i++)
    if (buf[i] != '\r' || buf[i + 1] != '\n')
      buf[j++] = buf[i];
  buf[j] = '\0';
  /* ja:CRをLFに変換する */
  for (i = 0; buf[i] != '\0'; i++)
    if (buf[i] == '\r')
      buf[i] = '\n';
  str = g_strsplit (buf, "\n", 0);
  g_free (buf);
  for (i = 0; str[i]; i++)
    {
      q = g_malloc0 (sizeof (ProfileList));
      q->data = g_strdup (str[i]);
      q->prev = p;
      if (p)
        p->next = q;
      p = q;
    }
  g_strfreev (str);
  if (p)
    while (p->prev)
      p = p->prev;
  r = p;
  /* ja:リストを整理する */
  while (p)
    {
      gsize n;
      gchar *data, *eq;

      data = p->data ? g_strstrip (g_strdup (p->data)) : NULL;
      n = data ? g_strlen (data) : 0;
      if (n <= 0)
        {
          p->type = PROFILE_DATA_TYPE_SPACE;
        }
      else if (data[0] == '#' || data[0] == ';')
        {
          p->type = PROFILE_DATA_TYPE_COMMENT;
        }
      else if (data[0] == '[' && data[n - 1] == ']')
        {
          p->type = PROFILE_DATA_TYPE_SECTION;
          g_free (section);
          section = g_strndup (data + 1, n - 2);
          for (q = p->prev; q && q->type == PROFILE_DATA_TYPE_SPACE;
                                                                q = q->prev)
            {
              g_free (q->section);
              q->section = NULL;
            }
        }
      else if ((eq = g_strchr (p->data, '=')))
        {
          p->type = PROFILE_DATA_TYPE_KEY;
          p->key = g_strndup (p->data, eq - p->data);
          p->value = eq + 1;
        }
      else
        {
          p->type = PROFILE_DATA_TYPE_UNKNOW;
        }
      g_free (data);
      p->section = g_strdup (section);
      p = p->next;
    }
  g_free (section);
  return r;
}


/*  ja:ファイルをリスト化する
    file,ファイル名
     RET,リスト                                                             */
static ProfileList *
profile_list_from_file (const gchar *file)
{
  gchar *data;
  gssize length;
  ProfileList *p = NULL;

  data = fileio_load (file, &length);
  if (data)
    {
      data = g_realloc (data, length + 1);
      data[length] = '\0';
      p = profile_list_from_text (data);
      g_free (data);
    }
  return p;
}


/*  ja:初期化ファイルを解放する
    profile_list,リスト                                                     */
static void
profile_list_free (ProfileList *profile_list)
{
  ProfileList *p, *q;

  for (p = profile_list; p; p = q)
    {
      q = p->next;
      g_free (p->data);
      g_free (p->section);
      g_free (p->key);
      g_free (p);
    }
}


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群(独自)                                               *
*                                                                             *
******************************************************************************/
/*  ja:初期化ファイルを開く
    file,ファイル名
     RET,プロファイル                                                       */
Profile *
profile_open (const gchar *file)
{
  gchar *base = NULL, *name;
  gint n;
  Profile *profile;

  profile = g_malloc0 (sizeof (Profile));
  name = g_get_prgname ();
  if (file)
    {
      base = g_path_get_basename (file);
      if (g_strfilecmp (file, base) != 0)
        {
          g_free (base);
          base = NULL;
        }
    }
  else
    {
      base = g_path_get_basename (name);
      n = g_strlen (base);
      if (n >= 5 && g_strfilecmp (base + n - 4, ".exe") == 0)
        base[n -= 4] = '\0';
    }
  if (base)
    {
      gchar *tmp;

      tmp = g_strdup (base);
      n = g_strlen (tmp);
      while (tmp[0] == '.')
        g_memmove (tmp, tmp + 1, n--);
      if (n > 0)
        {
          gchar *sys;

#ifdef SYSCONFDIR
          sys = g_strconcat (SYSCONFDIR, G_DIR_SEPARATOR_S,
                                                        tmp, ".conf", NULL);
#else /* not SYSCONFDIR */
          gchar *dir;

          dir = g_path_get_dirname (name);
          sys = g_strconcat (dir, G_DIR_SEPARATOR_S, tmp, ".conf", NULL);
          g_free (dir);
#endif /* not SYSCONFDIR */
          profile->sublist = profile_list_from_file (sys);
          g_free (sys);
        }
      g_free (tmp);
    }
  profile->file = !base ? g_strdup (file)
#ifdef G_OS_WIN32
      : g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, base, ".ini", NULL);
#else /* not G_OS_WIN32 */
      : g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".", base, NULL);
#endif /* not G_OS_WIN32 */
  g_free (base);
  profile->list = profile_list_from_file (profile->file);
  return profile;
}


/*  ja:初期化ファイルを閉じる
    profile,プロファイル
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_close (Profile *profile)
{
  if (!profile)
    return TRUE;
  if (profile->edit)
    {
      /* ja:変更があったとき */
      gchar *text;

      text = profile_list_to_text (profile->list);
      if (text)
        {
          fileio_save (profile->file, text, g_strlen (text) * sizeof (gchar));
          g_free (text);
        }
    }
  profile_list_free (profile->list);
  profile_list_free (profile->sublist);
  g_free (profile->file);
  g_free (profile);
  return TRUE;
}


/*  ja:初期化ファイルから文字列を取得する
    profile,プロファイル
    section,セクション
        key,キー
        RET,文字列,NULL:エラー                                              */
gchar *
profile_get_string (Profile     *profile,
                    const gchar *section,
                    const gchar *key)
{
  ProfileList *p;

  if (!profile || !section || !key)
    return NULL;
  for (p = profile->list; p; p = p->next){
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      return g_strdup (p->value);}
  for (p = profile->sublist; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      return g_strdup (p->value);
  return NULL;
}


/*  ja:初期化ファイルから値のサイズ取得する
    profile,プロファイル
    section,セクション
        key,キー
       type,タイプ
        RET,バイト数,0:エラー                                               */
gsize
profile_get_size (Profile     *profile,
                  const gchar *section,
                  const gchar *key,
                  const guint  type)
{
  ProfileList *p;

  if (!profile || !section || !key)
    return 0;
  for (p = profile->list; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    for (p = profile->sublist; p; p = p->next)
      if (p->type == PROFILE_DATA_TYPE_KEY
        && g_strcmp (p->section, section) == 0 && g_strcmp (p->key, key) == 0)
        break;
  if (!p)
    return 0;
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        return g_ascii_strcasecmp (p->value, "true") == 0
                || g_ascii_strcasecmp (p->value, "false") == 0
                || g_ascii_strcasecmp (p->value, "yes") == 0
                || g_ascii_strcasecmp (p->value, "no") == 0
                || g_ascii_strcasecmp (p->value, "on") == 0
                || g_ascii_strcasecmp (p->value, "off") == 0
                || g_ascii_strcasecmp (p->value, "ok") == 0
                || g_ascii_strcasecmp (p->value, "cancel") == 0
                ? sizeof (gboolean) : 0;
      case PROFILE_VALUE_TYPE_INT:
        return sizeof (gint);
      case PROFILE_VALUE_TYPE_STRING:
        return g_strlen (p->value) + 1;
      case PROFILE_VALUE_TYPE_ARRAY:
        {
          gsize n = 0;
          gchar *nptr, *endptr;

          nptr = endptr = p->value;
          while (*endptr != '\0')
            {
              g_strtoul (nptr, &endptr, 0);
              if (nptr != endptr)
                n++;
              nptr = endptr + 1;
            }
          return n;
        }
    }
  return 0;
}


/*  ja:初期化ファイルから値を取得する
    profile,プロファイル
    section,セクション
        key,キー
      value,値を入れるバッファ
       size,値を入れるバッファのサイズ
       type,タイプ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_get_value (Profile     *profile,
                   const gchar *section,
                   const gchar *key,
                   gpointer     value,
                   const gsize  size,
                   const guint  type)
{
  ProfileList *p;

  if (!profile || !section || !key || !value)
    return FALSE;
  for (p = profile->list; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    for (p = profile->sublist; p; p = p->next)
      if (p->type == PROFILE_DATA_TYPE_KEY
        && g_strcmp (p->section, section) == 0 && g_strcmp (p->key, key) == 0)
        break;
  if (!p)
    return FALSE;
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        if (size < sizeof (gboolean))
          return FALSE;
        if (g_ascii_strcasecmp (p->value, "true") == 0
            || g_ascii_strcasecmp (p->value, "yes") == 0
            || g_ascii_strcasecmp (p->value, "on") == 0
            || g_ascii_strcasecmp (p->value, "ok") == 0)
          *((gboolean *)value) = TRUE;
        else if (g_ascii_strcasecmp (p->value, "false") == 0
            || g_ascii_strcasecmp (p->value, "no") == 0
            || g_ascii_strcasecmp (p->value, "off") == 0
            || g_ascii_strcasecmp (p->value, "cancel") == 0)
          *((gboolean *)value) = FALSE;
        else
          return FALSE;
        break;
      case PROFILE_VALUE_TYPE_INT:
        if (size < sizeof (gint))
          return FALSE;
        *((gint *)value) = g_strtol (p->value, NULL, 0);
        break;
      case PROFILE_VALUE_TYPE_STRING:
        if (size < g_strlen (p->value) + 1)
          return FALSE;
        g_strcpy ((gchar *)value, p->value);
        break;
      case PROFILE_VALUE_TYPE_ARRAY:
        {
          gchar *nptr, *endptr;
          guint8 *v;
          gint n = 0;

          v = value;
          nptr = endptr = p->value;
          while (*endptr != '\0' && n < size)
            {
              v[n] = g_strtoul (nptr, &endptr, 0);
              if (nptr != endptr)
                n++;
              nptr = endptr + 1;
            }
          if (n < size)
            return FALSE;
        }
        break;
      default:
        return FALSE;
    }
  return TRUE;
}


/*  ja:初期化ファイルに値を設定する
    profile,プロファイル
    section,セクション
        key,キー
      value,値が入っているバッファ
       size,値が入っているバッファのサイズ
       type,タイプ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_set_value (Profile       *profile,
                   const gchar   *section,
                   const gchar   *key,
                   gconstpointer  value,
                   const gsize    size,
                   const guint    type)
{
  gchar *data;
  gint i;
  ProfileList *p ,*q = NULL;

  if (!profile || !section || !key || !value)
    return FALSE;
  for (p = profile->list; p; q = p, p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    {
      for (p = q; p; p = p->prev)
        if (p->section && g_strcmp (p->section, section) == 0)
          break;
      if (!p)
        {
          if (q)
            {
              /* ja:前のデータとの間にスペースを入れる */
              p = g_malloc (sizeof (ProfileList));
              p->type = PROFILE_DATA_TYPE_SPACE;
              p->data = p->section = p->key = p->value = NULL;
              p->prev = q;
              p->next = q->next;
              q->next = p;
              q = p;
            }
          /* ja:セクションもキーもないときにはセクションを作る */
          p = g_malloc (sizeof (ProfileList));
          p->type = PROFILE_DATA_TYPE_SECTION;
          p->data = g_strdup_printf ("[%s]", section);
          p->section = g_strdup (section);
          p->key = p->value = NULL;
          p->prev = q;
          if (q)
            {
              p->next = q->next;
              q->next = p;
            }
          else
            {
              p->next = NULL;
              profile->list = p;
            }
        }
      q = p;
      while (q->type == PROFILE_DATA_TYPE_SPACE && q->section
                            && g_strcmp (p->section, section) == 0 && q->prev)
        q = q->prev;
      /* ja:セクションの最後にキーを作る */
      p = g_malloc (sizeof (ProfileList));
      p->type = PROFILE_DATA_TYPE_KEY;
      p->data = g_strdup_printf ("%s=", key);
      p->section = g_strdup (section);
      p->key = g_strdup (key);
      p->value = g_strchr (p->data, '=') + 1;
      p->prev = q;
      p->next = q->next;
      q->next = p;
      if (p->next)
        p->next->prev = p;
    }
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%s",
                            p->key, *((gboolean *)value) ? "true" : "false");
        break;
      case PROFILE_VALUE_TYPE_INT:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%d", p->key, *((gint *)value));
        break;
      case PROFILE_VALUE_TYPE_STRING:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%s", p->key, (gchar *)value);
        break;
      case PROFILE_VALUE_TYPE_ARRAY:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%u", p->key, ((guint8 *)value)[0]);
        for (i = 1; i < size; i++)
          {
            data = g_strdup_printf ("%s %u", p->data, ((guint8 *)value)[i]);
            g_free (p->data);
            p->data = data;
          }
        break;
      default:
        return FALSE;
    }
  p->value = g_strchr (p->data, '=') + 1;
  profile->edit = TRUE;
  return TRUE;
}


/*  ja:初期化ファイルのセクションを削除する
    profile,プロファイル
    section,セクション
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_delete_section (Profile     *profile,
                        const gchar *section)
{
  gboolean result = FALSE;
  ProfileList *p, *q;

  if (!profile || !section)
    return FALSE;
  for (p = profile->list; p; p = q)
    {
      q = p->next;
      if (p->section && g_strcmp (p->section, section) == 0)
        {
          if (p->prev)
            p->prev->next = p->next;
          if (p->next)
            p->next->prev = p->prev;
          g_free (p->data);
          g_free (p->section);
          g_free (p->key);
          g_free (p);
          profile->edit = TRUE;
          result = TRUE;
        }
    }
  return result;
}


/*  ja:初期化ファイルのキーを削除する
    profile,プロファイル
    section,セクション
        key,キー
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_delete_key (Profile     *profile,
                    const gchar *section,
                    const gchar *key)
{
  gboolean result = FALSE;
  ProfileList *p, *q;

  if (!profile || !section || !key)
    return FALSE;
  for (p = profile->list; p; p = q)
    {
      q = p->next;
      if (p->section && p->key && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
        {
          if (p->prev)
            p->prev->next = p->next;
          if (p->next)
            p->next->prev = p->prev;
          g_free (p->data);
          g_free (p->section);
          g_free (p->key);
          g_free (p);
          profile->edit = TRUE;
          result = TRUE;
        }
    }
  return result;
}


/*  ja:初期化ファイルのセクションを列挙する
    profile,プロファイル
        RET,セクションのリスト,NULL:エラー                                  */
GList *
profile_enum_section (Profile *profile)
{
  GList *glist = NULL;
  ProfileList *p;

  if (!profile)
    return NULL;
  for (p = profile->list; p; p = p->next)
    if (p->section && (!glist || !g_list_find_custom (glist, p->section,
                                                    (GCompareFunc)g_strcmp)))
      glist = g_list_insert_sorted (glist, p->section, (GCompareFunc)g_strcmp);
  for (p = profile->sublist; p; p = p->next)
    if (p->section && (!glist || !g_list_find_custom (glist, p->section,
                                                    (GCompareFunc)g_strcmp)))
      glist = g_list_insert_sorted (glist, p->section, (GCompareFunc)g_strcmp);
  return glist;
}


/*  ja:初期化ファイルのキーを列挙する
    profile,プロファイル
    section,セクション
        RET,セクションのリスト,NULL:エラー                                  */
GList *
profile_enum_key (Profile     *profile,
                  const gchar *section)
{
  GList *glist = NULL;
  ProfileList *p;

  if (!profile)
    return NULL;
  for (p = profile->list; p; p = p->next)
    if (p->section && p->key && g_strcmp (p->section, section) == 0
                            && (!glist || !g_list_find_custom (glist, p->key,
                                                    (GCompareFunc)g_strcmp)))
      glist = g_list_insert_sorted (glist, p->key, (GCompareFunc)g_strcmp);
  for (p = profile->sublist; p; p = p->next)
    if (p->section && p->key && g_strcmp (p->section, section) == 0
                            && (!glist || !g_list_find_custom (glist, p->key,
                                                    (GCompareFunc)g_strcmp)))
      glist = g_list_insert_sorted (glist, p->key, (GCompareFunc)g_strcmp);
  return glist;
}


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群(互換)                                               *
*                                                                             *
******************************************************************************/
Profile *
profile_keyfile_new (void)
{
  return g_malloc0 (sizeof (Profile));
}


void
profile_keyfile_free (Profile *profile)
{
  if (profile)
    {
      profile_list_free (profile->list);
      profile_list_free (profile->sublist);
      g_free (profile->file);
      g_free (profile);
    }
}


gboolean
profile_keyfile_load_from_file (Profile      *profile,
                                const gchar  *file,
                                guint         flags,
                                GError      **error)
{
  gboolean result = FALSE;

  if (error)
    *error = NULL;
  if (profile)
    {
      gchar *data;
      gssize length;

      data = fileio_load (file, &length);
      if (data)
        result = profile_keyfile_load_from_data (profile, data, length,
                                                                flags, error);
      g_free (data);
    }
  return result;
}


gboolean
profile_keyfile_load_from_data (Profile      *profile,
                                const gchar  *data,
                                gsize         length,
                                guint         flags,
                                GError      **error)
{
  if (error)
    *error = NULL;
  if (!profile)
    return FALSE;
  /* ja:リスト解放 */
  profile_list_free (profile->list);
  profile_list_free (profile->sublist);
  profile->list = profile->sublist = NULL;
  if (data && length != 0)
    {
      if (length == (gsize)-1)
        {
          profile->list = profile_list_from_text (data);
        }
      else
        {
          gchar *text;

          text = g_strndup (data, length);
          profile->list = profile_list_from_text (text);
          g_free (text);
        }
    }
  return TRUE;
}


gchar *
profile_keyfile_to_data (Profile  *profile,
                         gsize    *length,
                         GError  **error)
{
  gchar *data;

  if (error)
    *error = NULL;
  data = profile ? profile_list_to_text (profile->list) : NULL;
  if (data && length)
    *length = g_strlen (data);
  return data;
}


gchar **
profile_keyfile_get_groups (Profile *profile,
                            gsize   *length)
{
  gchar **str = NULL;
  GList *glist;

  glist = profile_enum_section (profile);
  if (glist)
    {
      guint i, leng;

      leng = g_list_length (glist);
      str = g_malloc ((leng + 1) * sizeof (gchar *));
      for (i = 0; i < leng; i++)
        str[i] = g_strdup (g_list_nth_data (glist, i));
      str[i] = NULL;
      g_list_free (glist);
      if (length)
        *length = leng;
    }
  return str;
}


gchar **
profile_keyfile_get_keys (Profile      *profile,
                          const gchar  *section,
                          gsize        *length,
                          GError      **error)
{
  gchar **str = NULL;
  GList *glist;

  if (error)
    *error = NULL;
  glist = profile_enum_key (profile, section);
  if (glist)
    {
      guint i, leng;

      leng = g_list_length (glist);
      str = g_malloc ((leng + 1) * sizeof (gchar *));
      for (i = 0; i < leng; i++)
        str[i] = g_strdup (g_list_nth_data (glist, i));
      str[i] = NULL;
      g_list_free (glist);
      if (length)
        *length = leng;
    }
  return str;
}


gboolean
profile_keyfile_has_group (Profile     *profile,
                           const gchar *section)
{
  if (profile && section)
    {
      ProfileList *p;

      for (p = profile->list; p; p = p->next)
        if (p->section && g_strcmp (p->section, section) == 0)
          return TRUE;
      for (p = profile->sublist; p; p = p->next)
        if (p->section && g_strcmp (p->section, section) == 0)
          return TRUE;
    }
  return FALSE;
}


gboolean
profile_keyfile_has_key (Profile      *profile,
                         const gchar  *section,
                         const gchar  *key,
                         GError      **error)
{
  if (error)
    *error = NULL;
  if (profile && section && key)
    {
      ProfileList *p;

      for (p = profile->list; p; p = p->next)
        if (p->section && p->key && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
          return TRUE;
      for (p = profile->sublist; p; p = p->next)
        if (p->section && p->key && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
          return TRUE;
    }
  return FALSE;
}


gchar *
profile_keyfile_get_value (Profile      *profile,
                           const gchar  *section,
                           const gchar  *key,
                           GError      **error)
{
  if (error)
    *error = NULL;
  return profile_get_string (profile, section, key);
}


gchar *
profile_keyfile_get_string (Profile      *profile,
                            const gchar  *section,
                            const gchar  *key,
                            GError      **error)
{
  gchar *str, *ret;
  gint i = 0, j = 0;

  if (error)
    *error = NULL;
  str = profile_get_string (profile, section, key);
  if (!str)
    return NULL;
  ret = g_malloc ((g_strlen (str) + 1) * sizeof (gchar));
  while (str[i] != '\0')
    if (str[i] == '\\' && str[i + 1] != '\0')
      switch (str[++i])
        {
          case 'a': ret[j++] = '\a'; i++; break;
          case 'b': ret[j++] = '\b'; i++; break;
          case 't': ret[j++] = '\t'; i++; break;
          case 'n': ret[j++] = '\n'; i++; break;
          case 'v': ret[j++] = '\v'; i++; break;
          case 'f': ret[j++] = '\f'; i++; break;
          case 'r': ret[j++] = '\r'; i++; break;
          case 's': ret[j++] = ' ';  i++; break;
          case 'u':
          case 'x':
          case 'U':
            {
              gunichar c = 0;
              gint k, len;

              len = str[i] == 'x' ? 2 : str[i] == 'u' ? 4 : 8;
              for (k = i + 1; k <= i + len; k++)
                {
                  gint n;

                  n = g_ascii_isxdigit (str[k]);
                  if (n < 0)
                    break;
                  c = c * 16 + n;
                }
              if (k > i + len)
                {
                  gchar utf8str[6];

                  i += len + 1;
                  len = g_unichar_to_utf8 (c, utf8str);
                  for (k = 0; k < len; k++)
                    ret[j++] = utf8str[k];
                }
              else
                {
                  ret[j++] = str[i++];
                }
            }
            break;
          default:
            if ('0' <= str[i] && str[i] <= '7'
                                    && '0' <= str[i + 1] && str[i + 1] <= '7'
                                    && '0' <= str[i + 2] && str[i + 2] <= '7')
              {
                gint k;

                ret[j] = 0;
                for (k = 0; k < 3; k++)
                  ret[j] = ret[j] * 8 + g_ascii_digit_value (str[i++]);
                j++;
              }
            else
              {
                ret[j++] = str[i++];
              }
        }
    else
      ret[j++] = str[i++];
  ret[j] = '\0';
  g_free (str);
  return ret;
}


gboolean
profile_keyfile_get_boolean (Profile      *profile,
                             const gchar  *section,
                             const gchar  *key,
                             GError      **error)
{
  gboolean value;

  if (error)
    *error = NULL;
  if (!profile_get_value (profile, section, key,
                (gpointer)&value, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
    value = FALSE;
  return value;
}


gint
profile_keyfile_get_integer (Profile      *profile,
                             const gchar  *section,
                             const gchar  *key,
                             GError      **error)
{
  gint value;

  if (error)
    *error = NULL;
  if (!profile_get_value (profile, section, key,
                    (gpointer)&value, sizeof (gint), PROFILE_VALUE_TYPE_INT))
    value = 0;
  return value;
}


gint *
profile_keyfile_get_integer_list (Profile      *profile,
                                  const gchar  *section,
                                  const gchar  *key,
                                  gsize        *length,
                                  GError      **error)
{
  gsize leng;
  gint *value = NULL;

  if (error)
    *error = NULL;
  leng = profile_get_size (profile, section, key, PROFILE_VALUE_TYPE_ARRAY);
  if (leng > 0)
    {
      value = g_malloc (leng);
      if (!profile_get_value (profile, section, key,
                            (gpointer)value, leng, PROFILE_VALUE_TYPE_ARRAY))
        {
          g_free (value);
          value = NULL;
          leng = 0;
        }
    }
  if (length)
    *length = leng / sizeof (gint);
  return value;
}


void
profile_keyfile_set_value (Profile     *profile,
                           const gchar *section,
                           const gchar *key,
                           const gchar *string)
{
  profile_set_value (profile, section, key, (gpointer)string,
        (g_strlen (string) + 1) * sizeof (gchar), PROFILE_VALUE_TYPE_STRING);
}


void
profile_keyfile_set_string (Profile     *profile,
                            const gchar *section,
                            const gchar *key,
                            const gchar *string)
{
  gchar *str;
  gint i = 0;

  if (!string)
    return;
  str = g_malloc ((g_strlen (string) * 4 + 1) * sizeof (gchar));
  str[0] = '\0';
  while (string[i] != '\0')
    {
      gchar tmp[7];

      switch (string[i])
        {
          case '\a': tmp[0] = '\\'; tmp[1] = 'a';  tmp[2] = '\0'; i++; break;
          case '\b': tmp[0] = '\\'; tmp[1] = 'b';  tmp[2] = '\0'; i++; break;
          case '\t': tmp[0] = '\\'; tmp[1] = 't';  tmp[2] = '\0'; i++; break;
          case '\n': tmp[0] = '\\'; tmp[1] = 'n';  tmp[2] = '\0'; i++; break;
          case '\v': tmp[0] = '\\'; tmp[1] = 'v';  tmp[2] = '\0'; i++; break;
          case '\f': tmp[0] = '\\'; tmp[1] = 'f';  tmp[2] = '\0'; i++; break;
          case '\r': tmp[0] = '\\'; tmp[1] = 'r';  tmp[2] = '\0'; i++; break;
          case '\\': tmp[0] = '\\'; tmp[1] = '\\'; tmp[2] = '\0'; i++; break;
          default:
            {
              gunichar c;

              c = g_utf8_get_char_validated (string + i, -1);
              if (c == (gunichar)-1 || c == (gunichar)-2
                                                    || !g_unichar_isprint (c))
                {
                  tmp[0] = '\\';
                  tmp[1] = '0' + ((string[i] >> 6) & 7);
                  tmp[2] = '0' + ((string[i] >> 3) & 7);
                  tmp[3] = '0' + (string[i] & 7);
                  tmp[4] = '\0';
                  i++;
                }
              else
                {
                  gint n;

                  n = g_unichar_to_utf8 (c, tmp);
                  tmp[n] = '\0';
                  i += n;
                }
            }
        }
      g_strcat (str, tmp);
    }
  profile_set_value (profile, section, key, (gpointer)str,
            (g_strlen (str) + 1) * sizeof (gchar), PROFILE_VALUE_TYPE_STRING);
  g_free (str);
}


void
profile_keyfile_set_boolean (Profile     *profile,
                             const gchar *section,
                             const gchar *key,
                             gboolean     value)
{
  profile_set_value (profile, section, key,
                (gpointer)&value, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
}


void
profile_keyfile_set_integer (Profile     *profile,
                             const gchar *section,
                             const gchar *key,
                             gint         value)
{
  profile_set_value (profile, section, key,
                    (gpointer)&value, sizeof (gint), PROFILE_VALUE_TYPE_INT);
}


void
profile_keyfile_set_integer_list (Profile     *profile,
                                  const gchar *section,
                                  const gchar *key,
                                  gint         list[],
                                  gsize        length)
{
  profile_set_value (profile, section, key,
            (gpointer)list, length * sizeof (gint), PROFILE_VALUE_TYPE_ARRAY);
}


void
profile_keyfile_remove_group (Profile      *profile,
                              const gchar  *section,
                              GError      **error)
{
  if (error)
    *error = NULL;
  profile_delete_section (profile, section);
}


void
profile_keyfile_remove_key (Profile      *profile,
                            const gchar  *section,
                            const gchar  *key,
                            GError      **error)
{
  if (error)
    *error = NULL;
  profile_delete_key (profile, section, key);
}
