/*
    gcommon
    copyright (c) 1998-2010 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"


/******************************************************************************
* Byte Order Macros                                                           *
******************************************************************************/
guint16
gcommon_swap_16 (const guint16 val)
{
  return (val >> 8) | (val << 8);
}


guint32
gcommon_swap_32 (const guint32 val)
{
  return ((val & 0x000000ff) << 24) | ((val & 0x0000ff00) << 8)
                    | ((val & 0x00ff0000) >> 8) | ((val & 0xff000000) >> 24);
}


guint64
gcommon_swap_64 (const guint64 val)
{
  guint8 *p, temp;
  guint64 v;

  v = val;
  p = (guint8 *)&v;
  temp = p[0]; p[0] = p[7]; p[7] = temp;
  temp = p[1]; p[1] = p[6]; p[6] = temp;
  temp = p[2]; p[2] = p[5]; p[5] = temp;
  temp = p[3]; p[3] = p[4]; p[4] = temp;
  return v;
}


/******************************************************************************
* String Utility Functions                                                    *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,2,0)
gboolean
g_str_has_prefix (const gchar *str,
                  const gchar *prefix)
{
  return str && prefix && g_strncmp (str, prefix, g_strlen (prefix)) == 0;
}


gboolean
g_str_has_suffix (const gchar *str,
                  const gchar *suffix)
{
  if (str && suffix)
    {
      gsize len_str, len_suffix;

      len_str = g_strlen (str);
      len_suffix = g_strlen (suffix);
      if (len_str > len_suffix)
        return strcmp (str + len_str - len_suffix, suffix) == 0;
    }
  return FALSE;
}


# ifndef HAVE_VASPRINTF
gint
g_vasprintf (gchar       **string,
             gchar const  *format,
             va_list       args)
{
  if (string)
    {
      gchar *str = NULL;
      gsize len, length = 0;

      do
        {
          length += 1024;
          if (length > 16416)
            {
              free (str);
              str = NULL;
              break;
            }
          str = g_realloc (str, length * sizeof (gchar));
# ifdef _MSC_VER
          len = _vsnprintf (str, length, format, args);
# else /* not _MSC_VER */
          len = vsnprintf (str, length, format, args);
# endif /* not _MSC_VER */
        }
      while (len < 0 || length <= len);
      *string = str;
    }
  return string && *string ? g_strlen (*string) + 1 : -1;
}
# endif /* not HAVE_VASPRINTF */
#endif /* not GLIB_CHECK_VERSION(2,2,0) */


/******************************************************************************
* Extended                                                                    *
******************************************************************************/
gchar *
g_strcpy (gchar       *dest,
          const gchar *src)
{
  return dest && src ? strcpy (dest, src) : dest;
}


gchar *
g_strncpy (gchar       *dest,
           const gchar *src,
           const gsize  n)
{
  return dest && src ? strncpy (dest, src, n) : dest;
}


gchar *
g_strcat (gchar       *dest,
          const gchar *src)
{
  return dest && src ? strcat (dest, src) : dest;
}


gchar *
g_strncat (gchar       *dest,
           const gchar *src,
           const gsize  n)
{
  return dest && src ? strncat (dest, src, n) : dest;
}


#if ! GLIB_CHECK_VERSION(2,16,0)
gint
g_strcmp (const gchar *s1,
          const gchar *s2)
{
  return s1 && s2 ? strcmp (s1, s2) : 0;
}
#endif /* not GLIB_CHECK_VERSION(2,16,0) */


gint
g_strncmp (const gchar *s1,
           const gchar *s2,
           const gsize  n)
{
  return s1 && s2 ? strncmp (s1, s2, n) : 0;
}


gchar *
g_strchr (const gchar *s,
          const gint   c)
{
  return s ? strchr (s, c) : NULL;
}


gchar *
g_strrchr (const gchar *s,
           const gint   c)
{
  return s ? strrchr (s, c) : NULL;
}


gsize
g_strspn (const gchar *s,
          const gchar *accept)
{
  return s && accept ? strspn (s, accept) : 0;
}


gsize
g_strcspn (const gchar *s,
           const gchar *reject)
{
  return s && reject ? strcspn (s, reject) : 0;
}


gchar *
g_strpbrk (const gchar *s,
           const gchar *accept)
{
  return s && accept ? strpbrk (s, accept) : NULL;
}


gchar *
g_strstr (const gchar *haystack,
          const gchar *needle)
{
  return g_strstr_len (haystack, g_strlen (haystack), needle);
}


gsize
g_strlen (const gchar *s)
{
  return s ? strlen (s) : 0;
}


gchar *
g_strtok (gchar       *str,
          const gchar *delim)
{
  return delim ? strtok (str, delim) : NULL;
}


gint
g_memcmp (gconstpointer s1,
          gconstpointer s2,
          const gsize   n)
{
  return s1 && s2 ? memcmp (s1, s2, n) : 0;
}


gpointer
g_memchr (gconstpointer s,
          const gint    c,
          const gsize   n)
{
  return s ? memchr (s, c, n) : NULL;
}


gpointer
g_memset (gpointer    s,
          const gint  c,
          const gsize n)
{
  return s ? memset (s, c, n) : s;
}


#ifndef HAVE_ASPRINTF
gint
g_asprintf (gchar       **string,
            const gchar  *format,
            ...)
{
  gsize len;
  va_list ap;

  va_start (ap, format);
  len = g_vasprintf (string, format, ap);
  va_end (ap);
  return len;
}
#endif /* not HAVE_ASPRINTF */
