/*
    Text maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "charset.h"
#include "edit.h"
#include "misc/misc.h"


/******************************************************************************
*                                                                             *
* ja:編集関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:ラインバッファを求める
    start,ラインバッファ
      off,現在のy座標
        y,y座標
      RET,ラインバッファ                                                    */
LineBuffer *
edit_get_line_buf (LineBuffer **start,
                   gint        *off,
                   const gint   y)
{
  while (y < *off && (*start)->prev)
    {
      *start = (*start)->prev;
      (*off)--;
    }
  while (*off < y && (*start)->next)
    {
      *start = (*start)->next;
      (*off)++;
    }
  return *off == y ? *start : NULL;
}


/*  ja:桁数を取得する
    tmaid,ウインドウ情報
        y,行
      RET,桁数                                                              */
gint
edit_get_width (TmaidWindow *tmaid,
                const gint   y)
{
  gint data_pos = 0, width = 0;
  LineBuffer *p;

  p = edit_get_line_buf (&tmaid->start, &tmaid->off, y);
  if (p->margin)
    while (data_pos < p->length - 1)
      if (p->text[data_pos] == '\t')
        {
          width = (width / tmaid->ft.tab + 1) * tmaid->ft.tab;
          data_pos++;
        }
      else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
        {
          if (data_pos + charset_length (p->text[data_pos]) == p->length)
            break;
          width += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
          data_pos += charset_length (p->text[data_pos]);
        }
      else
        {
          width++;
          data_pos++;
        }
  else
    while (data_pos < p->length)
      if (p->text[data_pos] == '\t')
        {
          width = (width / tmaid->ft.tab + 1) * tmaid->ft.tab;
            data_pos++;
        }
      else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
        {
          width += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
          data_pos += charset_length (p->text[data_pos]);
        }
      else
        {
          width++;
          data_pos++;
        }
  return width;
}


/*  ja:最大桁数を取得する
    tmaid,ウインドウ情報
      RET,桁数                                                              */
gint
edit_get_width_max (TmaidWindow *tmaid)
{
  gint y, data_pos, width, max = 0;
  LineBuffer *p;

  for (y = 0, p = edit_get_line_buf (&tmaid->start, &tmaid->off, tmaid->top.y);
                                    y < tmaid->drawing->allocation.height && p;
                                        y += tmaid->font_height, p = p->next)
    {
      width = 0;
      data_pos = 0;
      while (data_pos < p->length)
        if (p->text[data_pos] == '\t')
          {
            width = (width / tmaid->ft.tab + 1) * tmaid->ft.tab;
              data_pos++;
          }
        else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
          {
            width += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
            data_pos += charset_length (p->text[data_pos]);
          }
        else
          {
            width++;
            data_pos++;
          }
        if (width > max)
          max = width;
    }
    return max;
}


/*  ja:キャレットのアライメントされたx座標を求める
    tmaid,ウインドウ情報
        x,x座標
        y,y座標
    align,TRUE:右に寄せる,FALSE:左に寄せる
      RET,座標                                                              */
gint
edit_get_align_pos (TmaidWindow    *tmaid,
                    const gint      x,
                    const gint      y,
                    const gboolean  align)
{
  gint data_pos = 0, screen_pos = 0;
  LineBuffer *p;

  p = edit_get_line_buf (&tmaid->start, &tmaid->off, y);
  if (p->margin)
    while (screen_pos < x && data_pos < p->length - 1)
      if (p->text[data_pos] == '\t')
        {
          if (x < (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab)
            return align ? (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab
                         : screen_pos;
          screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
          data_pos++;
        }
      else if (data_pos + charset_length (p->text[data_pos]) > p->length)
        {
          screen_pos++;
          data_pos++;
        }
      else if (data_pos + charset_length (p->text[data_pos]) == p->length)
        {
          break;
        }
      else if (x < screen_pos + charset_width (tmaid->layout,
                    p->text + data_pos, tmaid->font_width, tmaid->font_buf))
        {
          return align ? screen_pos + charset_width (tmaid->layout,
                        p->text + data_pos, tmaid->font_width, tmaid->font_buf)
                       : screen_pos;
        }
      else
        {
          screen_pos += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
          data_pos += charset_length (p->text[data_pos]);
        }
  else
    while (screen_pos < x && data_pos < p->length)
      if (p->text[data_pos] == '\t')
        {
          if (x < (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab)
            return align ? (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab
                         : screen_pos;
            screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
            data_pos++;
        }
      else if (data_pos + charset_length (p->text[data_pos]) > p->length)
        {
          screen_pos++;
          data_pos++;
        }
      else if (x < screen_pos + charset_width (tmaid->layout,
                    p->text + data_pos, tmaid->font_width, tmaid->font_buf))
        {
          return align ? screen_pos + charset_width (tmaid->layout,
                        p->text + data_pos, tmaid->font_width, tmaid->font_buf)
                       : screen_pos;
        }
      else
        {
          screen_pos += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
          data_pos += charset_length (p->text[data_pos]);
        }
  return screen_pos;
}


/*  ja:画面上の座標からラインバッファ上のアドレスを求める
    tmaid,ウインドウ情報
        x,x座標
        y,y座標
    align,TRUE:右に寄せる,FALSE:左に寄せる
      RET,アドレス                                                          */
gint
edit_get_data_pos (TmaidWindow    *tmaid,
                   const gint      x,
                   const gint      y,
                   const gboolean  align)
{
  gint data_pos = 0, screen_pos = 0;
  LineBuffer *p;

  p = edit_get_line_buf (&tmaid->start, &tmaid->off, y);
  while (screen_pos < x && data_pos < p->length)
    if (p->text[data_pos] == '\t')
      {
        screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
        if (x < screen_pos)
          return align ? data_pos + 1 : data_pos;
        data_pos++;
      }
    else if (data_pos + charset_length (p->text[data_pos]) > p->length)
      {
        screen_pos++;
        data_pos++;
      }
    else if (x < screen_pos + charset_width (tmaid->layout,
                    p->text + data_pos, tmaid->font_width, tmaid->font_buf))
      {
        return align ? data_pos + charset_length (p->text[data_pos])
                     : data_pos;
      }
    else
      {
        screen_pos += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
        data_pos += charset_length (p->text[data_pos]);
      }
  return MIN (data_pos, p->length);
}


/*  ja:ラインバッファ上のアドレスから画面上の座標を求める
    tmaid,ウインドウ情報
        x,x座標
        y,y座標
      RET,座標                                                              */
gint
edit_get_screen_pos (TmaidWindow *tmaid,
                     const gint   x,
                     const gint   y)
{
  gint data_pos = 0, screen_pos = 0;
  LineBuffer *p;

  p = edit_get_line_buf (&tmaid->start, &tmaid->off, y);
  while (data_pos < x)
    if (p->text[data_pos] == '\t')
      {
        screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
        data_pos++;
      }
    else if (data_pos + charset_length (p->text[data_pos]) <= x)
      {
        screen_pos += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
        data_pos += charset_length (p->text[data_pos]);
      }
    else
      {
        break;
      }
  return screen_pos;
}


/*  ja:文字の分類
      text,文字列
    length,文字数
       RET,分類                                                             */
static gint
edit_get_char_block (const gchar *text, const gsize length)
{
  gint cb;

  if (length < charset_length (*text))
    return 0;
  cb = charset_block (text);
  switch (cb)
    {
      case CS_BASIC_LATIN:
      case CS_LATIN_1_SUPPLEMENT:
      case CS_LATIN_EXTENDED_A:
      case CS_LATIN_EXTENDED_B:
      case CS_SPACING_MODIFIER_LETTERS:
      case CS_COMBINING_DIACRITICAL_MARKS:
      case CS_LATIN_EXTENDED_ADDITIONAL:
      case CS_ALPHABETIC_PRESENTATION_FORMS:
        cb = CS_BASIC_LATIN;
        break;
      case CS_GREEK_AND_COPTIC:
      case CS_GREEK_EXTENDED:
        cb = CS_GREEK_AND_COPTIC;
        break;
      case CS_CYRILLIC:
      case CS_CYRILLIC_SUPPLEMENTARY:
        cb = CS_CYRILLIC;
        break;
      case CS_ARABIC:
      case CS_ARABIC_PRESENTATION_FORMS_A:
        cb = CS_ARABIC;
        break;
      case CS_HANGUL_JAMO:
      case CS_HANGUL_COMPATIBILITY_JAMO:
      case CS_HANGUL_SYLLABLES:
        cb = CS_HANGUL_SYLLABLES;
        break;
      case CS_CJK_RADICALS_SUPPLEMENT:
      case CS_KANGXI_RADICALS:
      case CS_IDEOGRAPHIC_DESCRIPTION_CHARACTERS:
      case CS_CJK_COMPATIBILITY:
      case CS_CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A:
      case CS_CJK_UNIFIED_IDEOGRAPHS:
      case CS_CJK_COMPATIBILITY_IDEOGRAPHS:
      case CS_CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B:
      case CS_CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT:
        cb = CS_CJK_UNIFIED_IDEOGRAPHS;
        break;
      case CS_YI_SYLLABLES:
      case CS_YI_RADICALS:
        cb = CS_YI_SYLLABLES;
    }
  return cb;
}


/*  ja:文字の種類の配列を求める
      p,ラインバッファ
    RET,配列                                                                */
static gint *
edit_make_block (LineBuffer *p)
{
  gint i = 0, j, length, *block;
  GUnicodeType ct;

  block = g_malloc (p->length * sizeof (gint));
  while (i < p->length)
    {
      length = charset_length (p->text[i]);
      if (i + length <= p->length)
        {
          ct = g_unichar_type (g_utf8_get_char (p->text + i));
          switch (ct)
            {
              case G_UNICODE_MODIFIER_LETTER:
              case G_UNICODE_MODIFIER_SYMBOL:
                block[i] = i > 0 && block[i - 1] > 0 ? block[i - 1] : 0;
                break;
              case G_UNICODE_LOWERCASE_LETTER:
              case G_UNICODE_OTHER_LETTER:
              case G_UNICODE_TITLECASE_LETTER:
              case G_UNICODE_UPPERCASE_LETTER:
              case G_UNICODE_COMBINING_MARK:
              case G_UNICODE_ENCLOSING_MARK:
              case G_UNICODE_NON_SPACING_MARK:
              case G_UNICODE_DECIMAL_NUMBER:
                block[i] = edit_get_char_block (p->text + i, length);
                if (ct == G_UNICODE_DECIMAL_NUMBER)
                    block[i] *= -1;
                break;
              default:
                block[i] = 0;
            }
          for (j = 1; j < length; j++)
            block[i + j] = block[i];
          i += length;
        }
      else
        {
          block[i++] = 0;
        }
    }
  return block;
}

/*  ja:区切りを求める
    tmaid,ウインドウ情報
        x,x座標
        y,y座標
     move,TRUE:右に移動する,FALSE:左に移動する
      RET,座標                                                              */
gint
edit_get_move_pos (TmaidWindow    *tmaid,
                   const gint      x,
                   const gint      y,
                   const gboolean  move)
{
  gint cb, cblock, data_pos, *block;
  LineBuffer *p;

  p = edit_get_line_buf (&tmaid->start, &tmaid->off, y);
  block = edit_make_block (p);
  data_pos = edit_get_data_pos (tmaid, x, y, FALSE);
  if (move)
    {
      /* en:Move to Right */
      /* ja:右へ移動する */
      cblock = block[data_pos];
      while (++data_pos < p->length)
        {
          cb = block[data_pos];
          if (cblock != cb)
            {
              if (cb)
                break;
              cblock = 0;
            }
        }
      if (data_pos > p->length)
        data_pos = p->length;
    }
  else
    {
      /* en:Move to Left */
      /* ja:左へ移動する */
      data_pos--;
      if (data_pos <= 0)
        {
          g_free (block);
          return 0;
        }
      cblock = block[data_pos];
      while (--data_pos >= 0)
        {
          cb = block[data_pos];
          if (cblock != cb)
            {
              if (cblock)
                {
                  data_pos++;
                  break;
                }
              cblock = cb;
            }
        }
      if (data_pos < 0)
        data_pos = 0;
    }
  g_free (block);
  return edit_get_screen_pos (tmaid, data_pos, y);
}


/*  ja:単語を選択する
    tmaid,ウインドウ情報
      RET,TRUE:単語あり,FALSE:単語なし                                      */
gboolean
edit_select_word (TmaidWindow *tmaid)
{
  gint st, ed, cblock, data_pos, *block;
  LineBuffer *p;

  if (tmaid->select.x >= 0)
    return FALSE;
  p = edit_get_line_buf (&tmaid->start, &tmaid->off, tmaid->cursor.y);
  if (p->length <= 0)
    return FALSE;
  block = edit_make_block (p);
  data_pos = edit_get_data_pos (tmaid, tmaid->cursor.x, tmaid->cursor.y,
                                                                        FALSE);
  if (data_pos >= p->length)
    data_pos--;
  cblock = block[data_pos];
  /* en:Move to Right */
  /* ja:右へ移動する */
  for (ed = data_pos + 1; ed < p->length; ed++)
    if (cblock != block[ed])
      break;
  /* en:Move to Left */
  /* ja:左へ移動する */
  for (st = data_pos; st > 0; st--)
    if (cblock != block[st - 1])
      break;
  g_free (block);
  tmaid->cursor.x = edit_get_screen_pos (tmaid, ed, tmaid->cursor.y);
  tmaid->select.x = edit_get_screen_pos (tmaid, st, tmaid->cursor.y);
  tmaid->select.y = tmaid->cursor.y;
  if (tmaid->cursor.x == tmaid->select.x)
    {
      tmaid->select.x = -1;
      return FALSE;
    }
  return TRUE;
}


/*  ja:指定範囲のバイト数を求める
       tmaid,ウインドウ情報
     start_p,選択範囲
       end_p,選択範囲
         RET,バイト数                                                       */
gint
edit_get_sel_bytes (TmaidWindow *tmaid,
                    GdkPoint    *start_p,
                    GdkPoint     *end_p)
{
  gint i, st_pos, ed_pos, length;
  GdkPoint st, ed;
  LineBuffer *p;

  if (start_p->y < end_p->y
                        || (start_p->y == end_p->y && start_p->x < end_p->x))
    {
      st=*start_p;
      ed=*end_p;
    }
  else
    {
      st = *end_p;
      ed = *start_p;
    }
  st_pos = edit_get_data_pos (tmaid, st.x, st.y, FALSE);
  ed_pos = edit_get_data_pos (tmaid, ed.x, ed.y, FALSE);
  if (st.y == ed.y)
    {
      /* ja:同じ行 */
      length = ed_pos - st_pos;
    }
  else
    {
      /* ja:違う行 */
      p = edit_get_line_buf (&tmaid->start, &tmaid->off, st.y);
      length = p->length - st_pos + ed_pos + (p->margin ? 0 : 1);
      for (i = st.y + 1, p = p->next; i < ed.y && p; i++, p = p->next)
        length += p->length + (p->margin ? 0 : 1);
    }
  return length;
}


/*  ja:指定範囲をメモリにコピーする
      tmaid,ウインドウ情報
    start_p,選択範囲
      end_p,選択範囲
       text,メモリ                                                          */
void
edit_cpy_sel_mem (TmaidWindow *tmaid,
                  GdkPoint    *start_p,
                  GdkPoint    *end_p,
                  gchar       *text)
{
  gint i, st_pos, ed_pos, length;
  GdkPoint st, ed;
  LineBuffer *p;

  if (start_p->y < end_p->y
                        || (start_p->y == end_p->y && start_p->x < end_p->x))
    {
      st = *start_p;
      ed = *end_p;
    }
  else
    {
      st = *end_p;
      ed = *start_p;
    }
  st_pos = edit_get_data_pos (tmaid, st.x, st.y, FALSE);
  ed_pos = edit_get_data_pos (tmaid, ed.x, ed.y, FALSE);
  p = edit_get_line_buf (&tmaid->start, &tmaid->off, st.y);
  if (st.y == ed.y)
    {
      /* ja:同じ行 */
      g_memmove (text, p->text + st_pos, (ed_pos - st_pos) * sizeof (gchar));
    }
  else
    {
      /* ja:違う行 */
      length = p->length - st_pos;
      g_memmove (text, p->text + st_pos, length * sizeof (gchar));
      if (!p->margin)
        {
          g_memmove (text + length, "\n", sizeof (gchar));
          length++;
        }
      for (i = st.y+  1, p = p->next; i < ed.y && p; i++, p = p->next)
        {
          g_memmove (text + length, p->text, p->length * sizeof (gchar));
          length += p->length;
          if (!p->margin)
            {
              /* ja:改行 */
              g_memmove (text + length, "\n", sizeof (gchar));
              length++;
            }
        }
      g_memmove (text + length, p->text, ed_pos * sizeof (gchar));
    }
}


/*  ja:メモリの内容を貼り付ける
     tmaid,ウインドウ情報
       put,貼り付ける座標
       new,新しい座標
      text,メモリ
    length,バイト数
       RET,挿入された行数                                                   */
gint
edit_put_mem (TmaidWindow *tmaid,
              GdkPoint    *put,
              GdkPoint    *new,
              const gchar *text,
              const gint   length)
{
  gchar *temp;
  gint i = 0, temp_leng, insert = 0, data_pos, screen_pos;
  LineBuffer *p, *q;

  data_pos = edit_get_data_pos (tmaid,put->x, put->y, FALSE);
  p = edit_get_line_buf (&tmaid->start, &tmaid->off, put->y);
  if (data_pos <= 0 && p->prev && p->prev->margin && length >= 1
                                    && !g_memcmp (text, "\n", sizeof(gchar)))
    {
      /* ja:行頭への挿入で前の行が折り返されており、挿入する文字が改行のとき */
      p = p->prev;
      data_pos = p->length;
      put->x = edit_get_screen_pos (tmaid, p->length, --put->y);
      /* ja:前の行の右端へ移動 */
    }
  if (data_pos<p->length)
    {
      /* ja:挿入位置以降の文字列をバッファに保存する */
      temp_leng = p->length-data_pos;
      temp = g_malloc (temp_leng * sizeof (gchar));
      g_memmove (temp, p->text + data_pos, temp_leng * sizeof (gchar));
      p->length = data_pos;
    }
  else
    {
      temp_leng = 0;
      temp = NULL;
    }
  if (tmaid->ft.limit)
    {
      /* ja:右マージンで折り返す */
      screen_pos = 0;
      for (data_pos = 0; data_pos < p->length; data_pos++)
        if (p->text[data_pos] == '\t')
          screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
        else if (data_pos + charset_length (text[data_pos]) <= length)
          screen_pos += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
        else
          screen_pos++;
      while (i < length)
        {
          data_pos = i;
          while (data_pos < length && text[data_pos] != '\n')
            if (text[data_pos] == '\t')
              {
                screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos++;
              }
            else if (data_pos + charset_length (text[data_pos]) <= length)
              {
                screen_pos += charset_width (tmaid->layout,
                                        p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos+=charset_length(text[data_pos]);
              }
            else
              {
                screen_pos++;
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos++;
              }
          if (p->length > 0 || i != data_pos)
            p->text = g_realloc (p->text,
                                (p->length + data_pos - i) * sizeof (gchar));
          g_memmove (p->text + p->length, text + i,
                                            (data_pos - i) * sizeof (gchar));
          p->length += data_pos - i;
          if (length <= data_pos)
            {
              i=data_pos;
            }
          else
            {
              q = g_malloc (sizeof (LineBuffer));
              q->length = 0;
              q->margin = p->margin;
              q->text = NULL;
              q->prev = p;
              q->next = p->next;
              p->next = q;
              if (q->next)
                q->next->prev=q;
              if (data_pos <= length - 1 && text[data_pos] == '\n')
                {
                  /* ja:改行 */
                  i = data_pos + 1;
                  p->margin = FALSE;
                }
              else
                {
                  /* ja:擬似改行 */
                  i = data_pos;
                  p->margin = TRUE;
                }
              p = q;
              screen_pos = 0;
              insert++;
            }
        }
    }
  else
    {
      /* ja:右マージンで折り返さない */
      while (i < length)
        {
          for (data_pos = i; data_pos < length && text[data_pos] != '\n';
                                                                data_pos++);
          if (p->length > 0 || i != data_pos)
            p->text = g_realloc (p->text,
                                (p->length + data_pos - i) * sizeof (gchar));
          g_memmove (p->text + p->length, text + i,
                                            (data_pos - i) * sizeof (gchar));
          p->length += data_pos - i;
          if (length <= data_pos)
            {
              i=data_pos;
            }
          else
            {
              i = data_pos + 1;
              q = g_malloc (sizeof (LineBuffer));
              q->length = 0;
              q->margin = FALSE;
              q->text = NULL;
              q->prev = p;
              q->next = p->next;
              p->next = q;
              if (q->next)
                q->next->prev = q;
              p = q;
              insert++;
            }
        }
    }
  if (new)
    {
      new->x = 0;
      data_pos = 0;
      while (data_pos < p->length)
        if (p->text[data_pos] == '\t')
          {
            new->x = (new->x / tmaid->ft.tab + 1) * tmaid->ft.tab;
            data_pos++;
          }
        else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
          {
            new->x += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
            data_pos += charset_length (p->text[data_pos]);
          }
        else
          {
            new->x++;
            data_pos++;
          }
      new->y = put->y+insert;
      if (tmaid->ft.limit && p->margin && (tmaid->ft.margin <= new->x
                || (temp && tmaid->ft.margin <= new->x
                                        + charset_width (tmaid->layout, temp,
                                        tmaid->font_width, tmaid->font_buf))))
        {
          new->x = 0;
          new->y++;
        }
    }
  if (temp)
    {
      /* ja:挿入位置以降の文字列を戻す */
      p->text = g_realloc (p->text, (p->length + temp_leng) * sizeof (gchar));
      g_memmove (p->text + p->length, temp, temp_leng * sizeof (gchar));
      p->length += temp_leng;
      g_free (temp);
    }
  if (tmaid->ft.limit)
      /* ja:右マージンで折り返す処理 */
      while (p)
        {
          data_pos = screen_pos = 0;
          while (data_pos < p->length)
            if (p->text[data_pos] == '\t')
              {
                screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos++;
              }
            else if (data_pos + charset_length (text[data_pos]) <= length)
              {
                screen_pos += charset_width (tmaid->layout,
                        text + data_pos, tmaid->font_width, tmaid->font_buf);
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos += charset_length (text[data_pos]);
              }
            else
              {
                screen_pos++;
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos++;
              }
          if (tmaid->ft.margin < screen_pos)
            {
              /* ja:右マージンを超えているとき、擬似改行する */
              q = g_malloc (sizeof (LineBuffer));
              q->length = p->length - data_pos;
              q->margin = p->margin;
              q->text = g_malloc (q->length * sizeof (gchar));
              q->prev = p;
              q->next = p->next;
              p->next = q;
              if (q->next)
                q->next->prev = q;
              g_memmove (q->text, p->text + data_pos,
                                                q->length * sizeof (gchar));
              p->length = data_pos;
              p->margin = TRUE;
              p->text = g_realloc (p->text, p->length * sizeof (gchar));
              p = q;
              insert++;
            }
          else if (screen_pos < tmaid->ft.margin && p->margin && p->next)
            {
              /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
              data_pos = p->length;
              q = p->next;
              if (q->next)
                q->next->prev = p;
              p->next = q->next;
              p->length += q->length;
              p->margin = q->margin;
              p->text = g_realloc (p->text, p->length * sizeof (gchar));
              g_memmove (p->text + data_pos, q->text,
                                                q->length * sizeof (gchar));
              g_free (q->text);
              g_free (q);
              insert--;
            }
          else
            {
              break;
            }
        }
  return insert;
}


/*  ja:指定範囲を削除する
      tmaid,ウインドウ情報
    start_p,選択範囲
      end_p,選択範囲
        RET,削除された行数                                                  */
gint
edit_del_sel_mem (TmaidWindow *tmaid,
                  GdkPoint    *start_p,
                  GdkPoint     *end_p)
{
  gint i, st_pos, ed_pos, result, data_pos, screen_pos;
  GdkPoint st, ed;
  LineBuffer *p, *q, *r;

  if (start_p->y < end_p->y
                        || (start_p->y == end_p->y && start_p->x < end_p->x))
    {
      st = *start_p;
      ed = *end_p;
    }
  else
    {
      st = *end_p;
      ed = *start_p;
    }
  st_pos = edit_get_data_pos (tmaid, st.x, st.y, FALSE);
  ed_pos = edit_get_data_pos (tmaid, ed.x, ed.y, FALSE);
  p = edit_get_line_buf (&tmaid->start, &tmaid->off, st.y);
  if (st.y == ed.y)
    {
      /* ja:同じ行 */
      g_memmove (p->text + st_pos,p->text + ed_pos,
                                        (p->length - ed_pos) * sizeof (gchar));
      p->length -= ed_pos - st_pos;
      p->text = g_realloc (p->text, p->length);
      result = 0;
    }
  else
    {
      /* ja:違う行 */
      for (i = st.y + 1, q = p->next; i < ed.y && q; i++, q = r)
        {
          g_free (q->text);
          r = q->next;
          g_free (q);
        }
      if (q->next)
        q->next->prev = p;
      p->next = q->next;
      p->length = st_pos + q->length - ed_pos;
      p->margin = q->margin;
      p->text = g_realloc (p->text, p->length);
      g_memmove (p->text + st_pos, q->text + ed_pos,
                                    (q->length - ed_pos) * sizeof (gchar));
      g_free (q->text);
      g_free (q);
      result = ed.y - st.y;
    }
  q = p->prev;
  if (q && q->margin && st.x <= 0)
    {
      /* ja:前の行が擬似改行で削除開始が行頭のとき */
      if (p->length > 0)
        {
          data_pos = screen_pos = 0;
          while (data_pos < q->length)
            if (q->text[data_pos] == '\t')
              {
                screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
                data_pos++;
              }
            else if (data_pos + charset_length (q->text[data_pos])
                                                                <= q->length)
              {
                screen_pos += charset_width (tmaid->layout, 
                                        q->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
                data_pos+=charset_length(q->text[data_pos]);
              }
            else
              {
                screen_pos++;
                data_pos++;
              }
          if (p->text[0] == '\t')
            screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
          else if (charset_length (p->text[0]) <= p->length)
            screen_pos += charset_width (tmaid->layout, p->text,
                                        tmaid->font_width, tmaid->font_buf);
          else
            screen_pos++;
        }
      else
        {
          screen_pos = 0;
        }
      if (p->length <= 0 || screen_pos <= tmaid->ft.margin)
        {
          /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
          data_pos = q->length;
          if (p->next)
            p->next->prev=q;
          q->next = p->next;
          q->length += p->length;
          q->margin = p->margin;
          q->text = g_realloc (q->text, q->length * sizeof (gchar));
          g_memmove (q->text + data_pos, p->text, p->length * sizeof (gchar));
          g_free (p->text);
          g_free (p);
          result++;
          p = q;
          if (start_p->y < end_p->y
                        || (start_p->y == end_p->y && start_p->x < end_p->x))
            {
              start_p->x = screen_pos;
              start_p->y--;
            }
          else
            {
              end_p->x = screen_pos;
              end_p->y--;
            }
        }
    }
    if (tmaid->ft.limit)/* ja:右マージンで折り返す処理 */
      while (p)
        {
          data_pos = screen_pos = 0;
          while (data_pos < p->length)
            if (p->text[data_pos] == '\t')
              {
                screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos++;
              }
            else if (data_pos + charset_length (p->text[data_pos])
                                                                <= p->length)
              {
                screen_pos += charset_width (tmaid->layout,
                                        p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos += charset_length (p->text[data_pos]);
              }
            else
              {
                screen_pos++;
                if (tmaid->ft.margin < screen_pos)
                  break;
                data_pos++;
              }
          if (tmaid->ft.margin < screen_pos)
            {
              /* ja:右マージンを超えているとき、擬似改行する */
              q = g_malloc (sizeof (LineBuffer));
              q->length = p->length - data_pos;
              q->margin = p->margin;
              q->text = g_malloc (q->length * sizeof (gchar));
              q->prev = p;
              q->next = p->next;
              p->next = q;
              if (q->next)
                q->next->prev=q;
              g_memmove (q->text, p->text + data_pos,
                                                q->length * sizeof (gchar));
              p->length = data_pos;
              p->margin = TRUE;
              p->text = g_realloc (p->text, p->length * sizeof (gchar));
              p = q;
              result--;
            }
          else if (screen_pos < tmaid->ft.margin && p->margin && p->next)
            {
              /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
              data_pos = p->length;
              q = p->next;
              if (q->next)
                q->next->prev = p;
              p->next = q->next;
              p->length += q->length;
              p->margin = q->margin;
              p->text = g_realloc (p->text, p->length * sizeof (gchar));
              g_memmove (p->text + data_pos, q->text,
                                                q->length * sizeof (gchar));
              g_free (q->text);
              g_free (q);
              result++;
            }
          else if (!p->margin)
            {
              break;
            }
          else
            {
              p=p->next;
            }
        }
  return result;
}
