/*
    Text maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <locale.h>
#include "tmaid.h"
#include "command.h"
#include "file.h"
#include "find.h"
#include "general.h"
#include "jump.h"
#include "print.h"
#include "replace.h"
#include "sigfile.h"
#include "sigmain.h"
#include "toolbar.h"
#include "misc/argument.h"
#include "misc/charuty.h"
#include "misc/fileio.h"
#include "misc/license.h"
#include "misc/profile.h"
#include "orz/orzaccel.h"
#include "orz/orzcomm.h"
#include "orz/orzhistory.h"
#include "orz/orzmdi.h"
#include "orz/orzwinmenu.h"


gboolean caret = FALSE;             /* ja:キャレットの形 */
gboolean ins = TRUE;                /* ja:TRUE:挿入,FALSE:上書き */
gboolean newfile = FALSE;           /* ja:TRUE:新規ファイル,FALSE:なし */
gboolean second = FALSE;            /* ja:TRUE:新規,FALSE:既存 */
gchar *open_path = NULL;            /* ja:開くのパス */
gchar *save_path = NULL;            /* ja:名前を付けて保存のパス */
gchar *clipboard_text = NULL;       /* ja:クリップボードのテキスト */
gint def_width, def_height;         /* ja:デフォルトのウインドウのサイズ */
gint ftnum = 0;                     /* ja:ファイルタイプの数 */
#if GTK_CHECK_VERSION(2,4,0) || defined G_OS_WIN32
gint findex = 0;                    /* ja:ファイルタイプのフィルタ */
#endif /* GTK_CHECK_VERSION(2,4,0) or G_OS_WIN32 */
gint tcount = 0;                    /* ja:ファイルタイプのカウンタ */
guint timer_id = 0;                 /* ja:タイマID */
FileType *ftype = NULL;             /* ja:ファイルタイプ */
GObject *history;                   /* ja:ファイルの履歴 */
GdkAtom atom_targets;               /* ja:ターゲット */
GdkAtom atom_text;                  /* ja:テキスト */
GdkAtom atom_compound;              /* ja:コンパウンド */
GdkAtom atom_utf8;                  /* ja:UTF-8 */
GtkWidget *window;                  /* ja:ウインドウ */
GtkWidget *mdi;                     /* en:MDI */
GtkWidget *status;                  /* ja:ステータスウインドウ */
GtkWidget *label_cursor;            /* ja:カーソル座標 */
GtkPositionType n_pos = GTK_POS_TOP;/* ja:ノートブックの位置 */
GtkItemFactory *ifactory_menu;      /* ja:メニュー */
GtkItemFactory *ifactory_popup;     /* ja:ポップアップ */
PangoFontDescription *system_font = NULL;
GdkColor system_color[12] = {
    {0, 0x0000, 0x0000, 0x0000}, {0, 0xffff, 0xffff, 0xffff},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0x8000, 0x8000, 0x8000},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0x8000, 0x8000, 0x8000},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0xa000, 0xa000, 0xa000},
    {0, 0xffff, 0xffff, 0xffff}, {0, 0x0000, 0x0000, 0x8000},
    {0, 0x0000, 0x0000, 0x0000}, {0, 0xffff, 0xffff, 0xffff}};
GtkTargetEntry targets[4] = {       /* ja:Text maidが提供可能なセレクション */
    {"STRING",        0, TARGET_STRING},
    {"TEXT",          0, TARGET_TEXT}, 
    {"COMPOUND_TEXT", 0, TARGET_COMPOUND_TEXT},
    {"UTF8_STRING",   0, TARGET_UTF8}};
MiscCreateMenuEntry menu_entries[] = {
{"/",                     NULL,                          NULL,         MISC_CREATE_MENU_BAR,       NULL,             NULL,            NULL},
{"/file",                 N_("_File"),                   NULL,         MISC_CREATE_MENU_ITEM,      NULL,             NULL,            NULL},
{"/file/",                NULL,                          NULL,         MISC_CREATE_MENU_SHELL,     NULL,             NULL,            NULL},
{"/file/new",             N_("_New"),                    "<control>N", MISC_CREATE_MENU_ITEM,      command_new,      NULL,            NULL},
{"/file/open",            N_("_Open..."),                "<control>O", MISC_CREATE_MENU_ITEM,      command_open,     NULL,            NULL},
{"/file/close",           N_("_Close"),                  "<control>W", MISC_CREATE_MENU_ITEM,      command_close,    NULL,            NULL},
{"/file/save",            N_("_Save"),                   "<control>S", MISC_CREATE_MENU_ITEM,      command_save,     NULL,            NULL},
{"/file/saveas",          N_("Save _As..."),             NULL,         MISC_CREATE_MENU_ITEM,      command_saveas,   NULL,            NULL},
{"/file/reload",          N_("_Reload..."),              "<control>R", MISC_CREATE_MENU_ITEM,      command_reload,   NULL,            NULL},
{"/file/sep1",            NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/file/print",           N_("_Print..."),               "<control>P", MISC_CREATE_MENU_ITEM,      command_print,    NULL,            NULL},
{"/file/sep2",            NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/file/properties",      N_("Propert_ies..."),          "<alt>Return",MISC_CREATE_MENU_ITEM,      command_property, NULL,            NULL},
{"/file/sep3",            NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/file/exit",            N_("E_xit"),                   "<control>Q", MISC_CREATE_MENU_ITEM,      command_exit,     NULL,            NULL},
{"/edit",                 N_("_Edit"),                   NULL,         MISC_CREATE_MENU_ITEM,      NULL,             NULL,            NULL},
{"/edit/",                NULL,                          NULL,         MISC_CREATE_MENU_SHELL,     NULL,             NULL,            NULL},
{"/edit/undo",            N_("_Undo"),                   "<control>Z", MISC_CREATE_MENU_ITEM,      command_undo,     NULL,            NULL},
{"/edit/redo",            N_("_Redo"),                   "F4",         MISC_CREATE_MENU_ITEM,      command_redo,     NULL,            NULL},
{"/edit/sep1",            NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/edit/cut",             N_("Cu_t"),                    "<control>X", MISC_CREATE_MENU_ITEM,      command_cut,      NULL,            NULL},
{"/edit/copy",            N_("_Copy"),                   "<control>C", MISC_CREATE_MENU_ITEM,      command_copy,     NULL,            NULL},
{"/edit/paste",           N_("_Paste"),                  "<control>V", MISC_CREATE_MENU_ITEM,      command_paste,    NULL,            NULL},
{"/edit/delete",          N_("De_lete"),                 NULL,         MISC_CREATE_MENU_ITEM,      command_delete,   NULL,            NULL},
{"/edit/sep2",            NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/edit/return",          N_("Return at Right _Margin"), NULL,         MISC_CREATE_MENU_ITEM,      command_margin,   NULL,            NULL},
{"/edit/convert",         N_("Convert Ta_b to Space"),   NULL,         MISC_CREATE_MENU_ITEM,      command_tab,      NULL,            NULL},
{"/edit/sep3",            NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/edit/character",       N_("_Character by _Value..."), NULL,         MISC_CREATE_MENU_ITEM,      command_valchr,   NULL,            NULL},
{"/edit/jump",            N_("_Jump..."),                "<control>J", MISC_CREATE_MENU_ITEM,      command_jump,     NULL,            NULL},
{"/edit/sep4",            NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/edit/all",             N_("_All"),                    "<control>A", MISC_CREATE_MENU_ITEM,      command_all,      NULL,            NULL},
{"/search",               N_("_Search"),                 NULL,         MISC_CREATE_MENU_ITEM,      NULL,             NULL,            NULL},
{"/search/",              NULL,                          NULL,         MISC_CREATE_MENU_SHELL,     NULL,             NULL,            NULL},
{"/search/find",          N_("_Find..."),                "<control>F", MISC_CREATE_MENU_ITEM,      command_find,     NULL,            NULL},
{"/search/next",          N_("_Next"),                   "F3",         MISC_CREATE_MENU_ITEM,      command_next,     (gpointer)TRUE,  NULL},
{"/search/previous",      N_("_Previous"),               "<shift>F3",  MISC_CREATE_MENU_ITEM,      command_next,     (gpointer)FALSE, NULL},
{"/search/next",          N_("_Replace..."),             "<control>H", MISC_CREATE_MENU_ITEM,      command_replace,  NULL,            NULL},
{"/option",               N_("_Option"),                 NULL,         MISC_CREATE_MENU_ITEM,      NULL,             NULL,            NULL},
{"/option/",              NULL,                          NULL,         MISC_CREATE_MENU_SHELL,     NULL,             NULL,            NULL},
{"/option/configuration", N_("_Configuration..."),       NULL,         MISC_CREATE_MENU_ITEM,      command_conf,     NULL,            NULL},
{"/option/file",          N_("File _Type..."),           NULL,         MISC_CREATE_MENU_ITEM,      command_format,   NULL,            NULL},
{"/window",               N_("_Window"),                 NULL,         MISC_CREATE_MENU_ITEM,      NULL,             NULL,            NULL},
{"/window/",              NULL,                          NULL,         MISC_CREATE_MENU_SHELL,     NULL,             NULL,            NULL},
{"/help",                 N_("_Help"),                   NULL,         MISC_CREATE_MENU_ITEM,      NULL,             NULL,            NULL},
{"/help/",                NULL,                          NULL,         MISC_CREATE_MENU_SHELL,     NULL,             NULL,            NULL},
{"/help/about",           N_("_About..."),               NULL,         MISC_CREATE_MENU_ITEM,      command_version,  NULL,            NULL},
{NULL,                    NULL,                          NULL,         MISC_CREATE_MENU_TERMINAL,  NULL,             NULL,            NULL}};
MiscCreateMenuEntry popup_entries[] = {
{"/",                     NULL,                          NULL,         MISC_CREATE_MENU_SHELL,     NULL,             NULL,            NULL},
{"/undo",                 N_("_Undo"),                   "<control>Z", MISC_CREATE_MENU_ITEM,      command_undo,     NULL,            NULL},
{"/sep1",                 NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/cut",                  N_("Cu_t"),                    "<control>X", MISC_CREATE_MENU_ITEM,      command_cut,      NULL,            NULL},
{"/copy",                 N_("_Copy"),                   "<control>C", MISC_CREATE_MENU_ITEM,      command_copy,     NULL,            NULL},
{"/paste",                N_("_Paste"),                  "<control>V", MISC_CREATE_MENU_ITEM,      command_paste,    NULL,            NULL},
{"/delete",               N_("De_lete"),                 NULL,         MISC_CREATE_MENU_ITEM,      command_delete,   NULL,            NULL},
{"/sep2",                 NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/all",                  N_("_All"),                    "<control>A", MISC_CREATE_MENU_ITEM,      command_all,      NULL,            NULL},
{"/sep3",                 NULL,                          NULL,         MISC_CREATE_MENU_SEPARATOR, NULL,             NULL,            NULL},
{"/input",                N_("_Input Methods"),          NULL,         MISC_CREATE_MENU_ITEM,      NULL,             NULL,            NULL},
{NULL,                    NULL,                          NULL,         MISC_CREATE_MENU_TERMINAL,  NULL,             NULL,            NULL}};


/******************************************************************************
*                                                                             *
* ja:メイン関数                                                               *
*                                                                             *
******************************************************************************/
int
main (int   argc,
      char *argv[])
{
  gboolean *arg_files, license = FALSE;
  gchar *section, *key, *tmp, *utf8str;
  const gchar *charset;
  gint i, count = 0, init_line = 0;
  Profile *profile;
  TmaidWindow *tmaid;
  GObject *accel, *winmenu;
  GtkAccelGroup *accel_group;
  GtkStyle *style;
  GtkWidget *handle, *label, *table, *vbox, *frame;
  /* en:Drag&Drop */
  GtkTargetEntry drag_types[1] = {{"text/uri-list", 0, 0}};

  /* ja:初期化 */
  setlocale (LC_ALL, "");
#ifdef LOCALEDIR
  bindtextdomain (PACKAGE, LOCALEDIR);
#else /* not LOCALEDIR */
  {
      gchar *dirname, *path, *locale;
# if defined (G_OS_WIN32) && GLIB_CHECK_VERSION(2,6,0)
      gchar *temp;
# endif /* defined (G_OS_WIN32) && GLIB_CHECK_VERSION(2,6,0) */

      path = fileio_get_full_path (argv[0]);
      dirname = g_path_get_dirname (path);
      g_free (path);
      locale = g_strconcat (dirname, G_DIR_SEPARATOR_S, "locale", NULL);
      g_free (dirname);
# ifdef G_OS_WIN32
      if (!fileio_isfile (locale))
        {
          g_free (locale);
          locale = g_win32_get_package_installation_subdirectory
                            (NULL, argv[0], "lib" G_DIR_SEPARATOR_S "locale");
        }
# endif /* G_OS_WIN32 */
# if defined (G_OS_WIN32) && GLIB_CHECK_VERSION(2,6,0)
#  if GLIB_CHECK_VERSION(2,8,0)
      temp = g_win32_locale_filename_from_utf8 (locale);
#  else /* not GLIB_CHECK_VERSION(2,8,0) */
      temp = g_locale_from_utf8 (locale, -1, NULL, NULL, NULL);
#  endif /* not GLIB_CHECK_VERSION(2,8,0) */
      g_free (locale);
      locale = temp;
# endif /* defined (G_OS_WIN32) && GLIB_CHECK_VERSION(2,6,0) */
      bindtextdomain (PACKAGE, locale);
      g_free (locale);
  }
#endif /* not LOCALEDIR */
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  textdomain (PACKAGE);
  gtk_set_locale ();
  gtk_init (&argc, &argv);

  def_width = gdk_screen_width () * 2 / 3;
  def_height = gdk_screen_height () * 2 / 3;

  /* ja:ファイルの履歴 */
  history = orz_history_new ();

  /* ja:レジストリから読み出す */
  profile = profile_open (NULL);
  if (profile)
    {
      gint num;

      if (!profile_get_value (profile, "tmaid", "license",
            (gpointer)&license, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
        license = FALSE;
      if (!profile_get_value(profile, "tmaid", "ftnum",
            (gpointer)&ftnum, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        ftnum = 0;
#if GTK_CHECK_VERSION(2,4,0) || defined G_OS_WIN32
      if (!profile_get_value(profile, "tmaid", "findex",
            (gpointer)&findex, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        findex = 0;
#endif /* GTK_CHECK_VERSION(2,4,0) or defined G_OS_WIN32 */
      if (profile_get_value (profile, "tmaid", "history",
            (gpointer)&num, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        orz_history_set_num (ORZ_HISTORY (history), num);
      if (!profile_get_value (profile, "tmaid", "newfile",
            (gpointer)&newfile, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
        newfile = FALSE;
      if (!profile_get_value (profile, "tmaid", "second",
            (gpointer)&second, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
        second = FALSE;
      if (!profile_get_value (profile, "tmaid", "def_width",
            (gpointer)&def_width, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        def_width = gdk_screen_width () * 2 / 3;
      if (!profile_get_value (profile, "tmaid", "def_height",
            (gpointer)&def_height, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        def_height = gdk_screen_height () * 2 / 3;
      if (!profile_get_value (profile, "tmaid", "n_pos",
            (gpointer)&n_pos, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        n_pos = GTK_POS_TOP;
      utf8str = profile_get_string (profile, "tmaid", "openpath");
      open_path = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
      utf8str = profile_get_string (profile, "tmaid", "savepath");
      save_path = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
      /* ja:ファイルの履歴 */
      for (i = orz_history_get_num (ORZ_HISTORY (history)) - 1; i >= 0; i--)
        {
          key = g_strdup_printf ("history%d", i);
          utf8str = profile_get_string (profile, "tmaid", key);
          g_free (key);
          if (utf8str)
            {
              gchar *file;

              file = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
              orz_history_add_file (ORZ_HISTORY (history), file);
              g_free (file);
            }
        }
      /* ja:検索 */
      if (!profile_get_value (profile, "tmaid", "find_arrow",
            (gpointer)&find_arrow, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
        find_arrow = TRUE;
      if (!profile_get_value (profile, "tmaid", "find_ignorecase",
            (gpointer)&find_ignorecase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
        find_ignorecase = FALSE;
      if (!profile_get_value (profile, "tmaid", "find_num",
            (gpointer)&find_num, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        find_num = 0;
      for (i = 0; i < find_num; i++)
        {
          key = g_strdup_printf ("find%02d", i);
          find_text[i] = profile_get_string (profile, "tmaid", key);
          g_free (key);
          if (!find_text[i])
            break;
        }
      find_num = i;
      /* ja:置換 */
      if (!profile_get_value (profile, "tmaid", "replace_arrow",
            (gpointer)&replace_arrow, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
        replace_arrow = TRUE;
      if (!profile_get_value (profile, "tmaid", "replace_ignorecase",
            (gpointer)&replace_ignorecase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
        replace_ignorecase = FALSE;
      if (!profile_get_value (profile, "tmaid", "replace_num",
            (gpointer)&replace_num, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        replace_num = 0;
      for (i = 0; i < replace_num; i++)
        {
          key = g_strdup_printf ("replace%02d", i);
          replace_text[i] = profile_get_string (profile, "tmaid", key);
          g_free (key);
          if (!replace_text[i])
            break;
        }
      replace_num = i;
      /* ja:印刷 */
#ifdef G_OS_UNIX
      if (!profile_get_value (profile, "printer", "mode",
            (gpointer)&printer.mode, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
        printer.mode = TRUE;
      utf8str = profile_get_string (profile, "printer", "printer");
      if (utf8str)
        {
          printer.printer
                        = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      if (!printer.printer)
        printer.printer = g_strdup ("lpr");
#endif /* G_OS_UNIX */
#if defined G_OS_UNIX || defined G_OS_WIN32
      if (!profile_get_value (profile, "printer", "copies",
            (gpointer)&printer.copies, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.copies = 1;
#endif /* G_OS_UNIX or G_OS_WIN32 */
#ifndef G_OS_WIN32
      utf8str = profile_get_string (profile, "printer", "file");
      if (utf8str)
        {
          printer.file = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      if (!printer.file)
        printer.file = g_strdup ("tmaid.ps");
      if (!profile_get_value (profile, "printer", "width",
            (gpointer)&printer.width, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.width = 594;
      if (!profile_get_value (profile, "printer", "height",
            (gpointer)&printer.height, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.height = 840;
      if (!profile_get_value (profile, "printer", "left",
            (gpointer)&printer.left, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.left = 20;
      if (!profile_get_value (profile, "printer", "top",
            (gpointer)&printer.top, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.top = 20;
      if (!profile_get_value (profile, "printer", "right",
            (gpointer)&printer.right, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.right = 20;
      if (!profile_get_value (profile, "printer", "bottom",
            (gpointer)&printer.bottom, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.bottom = 20;
#endif /* not G_OS_WIN32 */
      /* ja:ファイルタイプ */
      if (ftnum > 0)
        ftype = g_malloc (ftnum * sizeof (FileType));
      for (i = 0; i < ftnum; i++)
        {
          section = g_strdup_printf ("filetype%04d", i);
          ftype[i].ft_id = tcount++;
          ftype[i].text = profile_get_string (profile, section, "text");
          if (!ftype[i].text)
            ftype[i].text = g_strdup("UNKNOW");
          ftype[i].ext = profile_get_string (profile, section, "ext");
          if (!ftype[i].ext)
            ftype[i].ext = g_strdup ("*");
          if (!profile_get_value (profile, section, "crlf",
                (gpointer)&ftype[i].crlf, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT))
            ftype[i].crlf = 3;
          if (!profile_get_value (profile, section, "margin",
                (gpointer)&ftype[i]. margin, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT))
            ftype[i].margin=80;
          if (!profile_get_value(profile,section,"tab",
                (gpointer)&ftype[i].tab, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT))
            ftype[i].tab=8;
          if (!profile_get_value (profile, section, "writemode",
                (gpointer)&ftype[i].writemode, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT))
            ftype[i].writemode = 0;
          if (!profile_get_value (profile, section, "associate",
                (gpointer)&ftype[i].associate, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].associate = TRUE;
          if (!profile_get_value (profile, section, "autoindent",
                (gpointer)&ftype[i].autoindent, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].autoindent = FALSE;
          if (!profile_get_value (profile, section, "code",
                (gpointer)&ftype[i].code, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].code = FALSE;
          if (!profile_get_value(profile,section,"eof",
                (gpointer)&ftype[i].eof, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].eof = FALSE;
          if (!profile_get_value (profile, section, "hexcase",
                (gpointer)&ftype[i].hexcase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].hexcase = FALSE;
          if (!profile_get_value (profile, section, "html",
                (gpointer)&ftype[i].html, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].html = FALSE;
          if (!profile_get_value (profile, section, "esc",
                (gpointer)&ftype[i].esc, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].esc = FALSE;
          if (!profile_get_value (profile, section, "limit",
                (gpointer)&ftype[i].limit, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].limit = FALSE;
          if (!profile_get_value(profile,section,"overwrite",
                (gpointer)&ftype[i].overwrite, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].overwrite = FALSE;
          if (!profile_get_value(profile,section,"space",
                (gpointer)&ftype[i].space, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].space = FALSE;
          if (!profile_get_value(profile,section,"syscol",
                (gpointer)&ftype[i].syscol, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].syscol = TRUE;
          if (!profile_get_value (profile, section, "tabconv",
                (gpointer)&ftype[i].tabconv, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].tabconv = FALSE;
          if (!profile_get_value (profile, section, "gline",
                (gpointer)&ftype[i].gline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].gline = FALSE;
          if (!profile_get_value(profile,section,"mline",
                (gpointer)&ftype[i].mline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].mline = FALSE;
          if (!profile_get_value(profile,section,"nline",
                (gpointer)&ftype[i].nline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].nline = FALSE;
          if (!profile_get_value (profile, section, "uline",
                (gpointer)&ftype[i].uline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].uline = FALSE;
          if (!profile_get_value (profile, section, "vline",
                (gpointer)&ftype[i].vline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].vline = FALSE;
          /* ja:キャラクターセット */
          ftype[i].charset = profile_get_string (profile, section, "charset");
          if (ftype[i].charset)
            {
              tmp = charuty_strip (ftype[i].charset);
              g_free (ftype[i].charset);
              ftype[i].charset = tmp;
            }
          if (!ftype[i].charset)
            ftype[i].charset = g_get_charset (&charset)
                                        ? g_strdup_printf ("%s,UTF-8", charset)
                                                        : g_strdup ("UTF-8");
          /* ja:フォント */
          ftype[i].font_name = profile_get_string (profile, section, "font");
          /* ja:色 */
          if (!profile_get_value (profile, section, "color",
                    (gpointer)&ftype[i].color, sizeof (GdkColor) * 12,
                                                    PROFILE_VALUE_TYPE_ARRAY))
            g_memmove(ftype[i].color,system_color,sizeof(GdkColor)*12);
          g_free (section);
        }
      profile_close(profile);
    }
  else
    {
#ifdef G_OS_UNIX
      printer.mode = TRUE;
      printer.printer = g_strdup ("lpr");
#endif /* G_OS_UNIX */
#ifndef G_OS_WIN32
      printer.file = g_strdup ("tmaid.ps");
#if defined G_OS_UNIX || defined G_OS_WIN32
      printer.copies = 1;
#endif /* G_OS_UNIX or G_OS_WIN32 */
      printer.width = 594;
      printer.height = 840;
      printer.left = printer.top = printer.right = printer.bottom = 20;
#endif /* not G_OS_WIN32 */
    }
  if (!license)
    license = license_dialog ();
  if (!license)
    gtk_exit (0);

  /* ja:コマンドラインの解析 */
  {
    gint array[2], def[2];
    ArgumentBool arg_n, arg_x;
    ArgumentValue arg_h, arg_l;
    ArgumentArray arg_s;
    ArgumentExit arg_v;
    ArgumentHelp arg_q;
    ArgumentList arg_opts[] = {
{"history", 'h',  "NUM",          N_("History"),        ARGUMENT_TYPE_VALUE, &arg_h},
{"line",    'l',  "NUM",          N_("Caret Position"), ARGUMENT_TYPE_VALUE, &arg_l},
{"new",     'n',  "ON/OFF",       N_("New File"),       ARGUMENT_TYPE_BOOL,  &arg_n},
{"size",    's',  "WIDTH,HEIGHT", N_("Window Size"),    ARGUMENT_TYPE_ARRAY, &arg_s},
{"second",  'x',  "ON/OFF",       N_("Open Window"),    ARGUMENT_TYPE_BOOL,  &arg_x},
{"version", 'v',  NULL,           N_("About"),          ARGUMENT_TYPE_EXIT,  &arg_v},
{"help",    '?',  NULL,           N_("Help"),           ARGUMENT_TYPE_HELP,  &arg_q},
{NULL,      '\0', NULL,           NULL,                 0,                   NULL}};

    /* ja:ファイルの履歴 */
    arg_h.val = orz_history_get_num (ORZ_HISTORY (history));
    arg_h.def = ORZ_HISTORY_NUM_DEF;
    /* ja:初期のキャレット位置 */
    arg_l.val = init_line;
    arg_l.def = 0;
    /* ja:新規ファイル */
    arg_n.flag = newfile;
    arg_n.def = FALSE;
    /* ja:ウインドウサイズ */
    array[0] = def_width;
    array[1] = def_height;
    def[0] = gdk_screen_width ()  * 2 / 3;
    def[1] = gdk_screen_height () * 2 / 3;
    arg_s.leng = 2;
    arg_s.array = array;
    arg_s.def = def;
    /* ja:新規ウインドウ */
    arg_x.flag = second;
    arg_x.def = FALSE;
    /* ja:バージョン */
    arg_v.str =
TMAID_VERSION_NOTICE"\n"
TMAID_COPYRIGHT_NOTICE"\n"
"This is free software; see the source for copying conditions.  There is NO\n"
"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n";
    /* ja:ヘルプ */
    arg_q.str = TMAID_COPYRIGHT_NOTICE"\n\n";
    arg_files = arg_analyse (argv, arg_opts);
    orz_history_set_num (ORZ_HISTORY (history), arg_h.val);
    if (arg_l.val >= 0)
      init_line = arg_l.val;
    def_width = array[0];
    def_height = array[1];
    second = arg_x.flag;
  }

  /* ja:ファイルタイプ */
  if (ftnum <= 0)
    {
      ftnum = 1;
      ftype = g_malloc0 (ftnum * sizeof (FileType));
      ftype[0].text = g_strdup (_("All Files"));
      ftype[0].ext = g_strdup ("*");
      ftype[0].crlf = 3;
      ftype[0].margin = 80;
      ftype[0].tab = 8;
      ftype[0].associate = TRUE;
      ftype[0].syscol = TRUE;
      /* ja:キャラクターセット */
      ftype[0].charset = !g_get_charset (&charset) && charset
                ? g_strdup_printf ("%s,UTF-8", charset) : g_strdup ("UTF-8");
      /* ja:色 */
      g_memmove (ftype[0].color, system_color, sizeof (GdkColor) * 12);
    }

  /* ja:メインウインドウ */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Text maid");
  gtk_window_set_default_size (GTK_WINDOW (window), def_width, def_height);
  g_signal_connect (G_OBJECT(window), "delete-event",
                                            G_CALLBACK (signal_delete), NULL);
  g_signal_connect (G_OBJECT (window), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  gtk_widget_realize (window);
  /* en:Drag and Drop */
  gtk_drag_dest_set (window, GTK_DEST_DEFAULT_MOTION
                        | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
                                            drag_types, 1, GDK_ACTION_COPY);
  g_signal_connect (G_OBJECT (window), "drag-data-received",
                                G_CALLBACK (signal_drag_data_received), NULL);
  /* ja:クリップボード */
  atom_targets = gdk_atom_intern ("TARGETS", FALSE);
  atom_compound = gdk_atom_intern ("COMPOUND_TEXT", FALSE);
  atom_text = gdk_atom_intern ("TEXT", FALSE);
  atom_utf8 = gdk_atom_intern ("UTF8_STRING", FALSE);
  gtk_selection_add_targets (window, GDK_SELECTION_PRIMARY,
                        targets, sizeof (targets) / sizeof (GtkTargetEntry));
  gtk_selection_add_targets (window, GDK_SELECTION_CLIPBOARD,
                        targets, sizeof (targets) / sizeof (GtkTargetEntry));
  g_signal_connect (G_OBJECT (window), "selection-clear-event",
                                    G_CALLBACK (signal_selection_clear), NULL);
  g_signal_connect (G_OBJECT (window), "selection-get",
                                    G_CALLBACK (signal_selection_get), NULL);
  g_signal_connect (G_OBJECT (window), "selection-received",
                                G_CALLBACK (signal_selection_received), NULL);
  /* en:MDI */
  mdi = orz_mdi_new ();
  gtk_notebook_set_scrollable (GTK_NOTEBOOK (mdi), TRUE);
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (mdi), n_pos);
  g_signal_connect (G_OBJECT (mdi), "close-window",
                                G_CALLBACK (signal_close_window), NULL);
  g_signal_connect (G_OBJECT (mdi), "switch-window",
                                G_CALLBACK (signal_switch_window), NULL);
  g_signal_connect (GTK_OBJECT (mdi), "destroy",
                                G_CALLBACK (signal_destroy_mdi), NULL);
  /* ja:ポップアップ */
  accel_group = gtk_accel_group_new ();
  misc_create_menu (popup_entries, accel_group);
  /* ja:メニュー */
  accel_group = gtk_accel_group_new ();
  misc_create_menu (menu_entries, accel_group);
  /* ja:ファイルの履歴 */
  orz_history_set_menu (ORZ_HISTORY (history),
                        misc_find_menu (menu_entries, "/file/"), MENUFILE - 2);
  g_signal_connect (ORZ_HISTORY (history), "activate",
                            G_CALLBACK (signal_activate_menu_history), NULL);
  /* ja:ウインドウメニュー */
  winmenu = orz_winmenu_new ();
  orz_winmenu_set_mdi (ORZ_WINMENU (winmenu), ORZ_MDI (mdi));
  orz_winmenu_set_menu (ORZ_WINMENU (winmenu),
                                misc_find_menu (menu_entries, "/window/"), 0);
  /* ja:アクセルグループ */
  accel = orz_accel_new ();
  orz_accel_set_window (ORZ_ACCEL (accel), GTK_WINDOW (window));
  orz_accel_add_accel_group (ORZ_ACCEL (accel), accel_group);
  /* ja:テーブル */
  table = gtk_table_new (1, 5, FALSE);
  /* ja:ステータスバー */
  status = gtk_statusbar_new ();
  /* ja:カーソル座標 */
  label_cursor = gtk_label_new ("");
  gtk_label_set_justify (GTK_LABEL (label_cursor), GTK_JUSTIFY_CENTER);
  frame = gtk_frame_new (NULL);
  gtk_container_add( GTK_CONTAINER (frame), label_cursor);
  /* ja:テキスト */
  label = gtk_label_new ("");
  g_signal_connect (G_OBJECT (label), "style-set",
                                        G_CALLBACK (signal_style_set), NULL);
  /* ja:垂直ボックス/ハンドルボックス */
  vbox = gtk_vbox_new (FALSE, 0);
  handle = gtk_handle_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (handle), 2);
  gtk_container_add (GTK_CONTAINER (handle),
                                        misc_find_menu (menu_entries, "/"));
  gtk_box_pack_start (GTK_BOX (vbox), handle, FALSE, FALSE, 0);
  handle = gtk_handle_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (handle), 2);
  gtk_container_add (GTK_CONTAINER (handle),
                                        misc_create_toolbar (toolbar_entries));
  gtk_box_pack_start (GTK_BOX (vbox), handle, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), mdi, TRUE, TRUE, 0);
  gtk_table_attach (GTK_TABLE (table), status, 0, 4, 0, 1,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), frame, 4, 5, 0, 1,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_box_pack_end (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  /* ja:タイマ */
  timer_id = g_timeout_add (500, signal_timeout, NULL);

  /* ja:メニュー表示 */
  set_menu_bar (NULL);

  /* ja:既に起動しているかチェックする */
  if (!second)
    {
      GObject *comm;

      comm = orz_comm_new (window, "Text maid InterProcess communication");
      if (!orz_comm_is_first (ORZ_COMM (comm)))
        {
          orz_comm_send_with_files (ORZ_COMM (comm), argc, argv, arg_files);
          gtk_exit (0);
        }
      g_signal_connect (G_OBJECT (comm), "recv",
                                    G_CALLBACK (signal_selection_recv), NULL);
    }

  /* ja:表示 */
  gtk_widget_show_all (window);
  gtk_widget_hide_all (label);
  gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, TRUE);

  /* ja:システム色 */
  style = gtk_widget_get_style (label);
  system_color[0] = style->text[0];
  system_color[1] = style->base[0];
  system_color[2] = style->dark[0];
  system_color[3] = style->dark[0];
  system_color[4] = style->dark[0];
  system_color[5] = style->dark[0];
  system_color[6] = style->dark[0];
  system_color[7] = style->mid[0];
  system_color[8] = style->fg[3];
  system_color[9] = style->bg[3];
  /* ja:システムフォント */
  system_font = pango_font_description_copy
                                    (pango_context_get_font_description
                                    (gtk_widget_get_pango_context (label)));

  /* ja:新規ウインドウ */
  for (i = 1; i < argc; i++)
    if (arg_files[i])
      {
        tmaid = file_open_edit (argv[i]);
        if (init_line != 0)/* ja:指定行に移動 */
          jump_operation (tmaid, init_line);
        count++;
      }
  g_free (arg_files);
  if (newfile && count <= 0)
    file_open_edit (NULL);

  gtk_main ();

  /* ja:レジストリに書き込む */
  profile = profile_open (NULL);
  if (profile)
    {
      gint num;
      GList *glist;

      num = orz_history_get_num (ORZ_HISTORY (history));
      profile_set_value (profile, "tmaid", "license",
            (gpointer)&license, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "ftnum",
            (gpointer)&ftnum, sizeof (gint), PROFILE_VALUE_TYPE_INT);
#if GTK_CHECK_VERSION(2,4,0) || defined G_OS_WIN32
      profile_set_value (profile, "tmaid", "findex",
            (gpointer)&findex, sizeof (gint), PROFILE_VALUE_TYPE_INT);
#endif /* GTK_CHECK_VERSION(2,4,0) or G_OS_WIN32 */
      profile_set_value (profile, "tmaid", "history",
            (gpointer)&num, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "tmaid", "newfile",
            (gpointer)&newfile, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "second",
            (gpointer)&second, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "def_width",
            (gpointer)&def_width, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "tmaid", "def_height",
            (gpointer)&def_height, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "tmaid", "n_pos",
            (gpointer)&n_pos, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      if (open_path)
        {
          utf8str = g_filename_to_utf8 (open_path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              profile_set_value (profile, "tmaid", "openpath",
                                    (gpointer)utf8str,
                                    (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
              g_free (utf8str);
            }
        }
      if (save_path)
        {
          utf8str = g_filename_to_utf8 (save_path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              profile_set_value (profile, "tmaid", "savepath",
                                    (gpointer)utf8str,
                                    (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
              g_free (utf8str);
            }
        }
      /* ja:ファイルの履歴 */
      glist = orz_history_get_file (ORZ_HISTORY (history));
      for (i = 0; i < g_list_length (glist); i++)
        {
          key = g_strdup_printf ("history%d", i);
          utf8str = g_filename_to_utf8 (g_list_nth_data (glist, i), -1,
                                                            NULL, NULL, NULL);
          if (utf8str)
            {
              profile_set_value (profile, "tmaid", key, (gpointer)utf8str,
                                    (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
              g_free (utf8str);
            }
          g_free (key);
        }
      g_list_free (glist);
      while (i <= ORZ_HISTORY_NUM_MAX)
        {
          key = g_strdup_printf ("history%d", i++);
          profile_delete_key (profile, "tmaid", key);
          g_free (key);
        }
      /* ja:検索 */
      profile_set_value (profile, "tmaid", "find_arrow",
            (gpointer)&find_arrow, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "find_ignorecase",
            (gpointer)&find_ignorecase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "find_num",
            (gpointer)&find_num, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      for (i = 0; i < find_num; i++)
        {
          key = g_strdup_printf ("find%02d", i);
          profile_set_value (profile, "tmaid", key, (gpointer)find_text[i],
                            (g_strlen (find_text[i]) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (key);
        }
      while (i < 32)
        {
          key = g_strdup_printf ("find%02d", i++);
          profile_set_value (profile, "tmaid", key,
                            (gpointer)"", sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (key);
        }
      /* ja:置換 */
      profile_set_value (profile, "tmaid", "replace_arrow",
            (gpointer)&replace_arrow, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "replace_ignorecase",
            (gpointer)&replace_ignorecase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "replace_num",
            (gpointer)&replace_num, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      for (i = 0; i < replace_num; i++)
        {
          key = g_strdup_printf ("replace%02d", i);
          profile_set_value (profile, "tmaid", key, (gpointer)replace_text[i],
                            (g_strlen (replace_text[i]) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free(key);
        }
      while (i < 32)
        {
          key = g_strdup_printf ("replace%02d", i++);
          profile_set_value (profile, "tmaid", key,
                            (gpointer)"", sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (key);
        }
      /* ja:印刷 */
#ifdef G_OS_UNIX
      profile_set_value (profile, "printer", "mode",
            (gpointer)&printer.mode, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
      utf8str = g_filename_to_utf8 (printer.printer, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          profile_set_value (profile, "printer", "printer", (gpointer)utf8str,
                                    (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (utf8str);
        }
#endif /* not G_OS_UNIX */
#if defined G_OS_UNIX || defined G_OS_WIN32
      profile_set_value (profile, "printer", "copies",
                                    (gpointer)&printer.copies, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
#endif /* G_OS_UNIX or G_OS_WIN32 */
#ifndef G_OS_WIN32
      utf8str = g_filename_to_utf8 (printer.file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          profile_set_value (profile, "printer", "file", (gpointer)utf8str,
                                  (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (utf8str);
        }
      profile_set_value (profile, "printer", "width",
            (gpointer)&printer.width, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "height",
            (gpointer)&printer.height, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "left",
            (gpointer)&printer.left, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "top",
            (gpointer)&printer.top, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "right",
            (gpointer)&printer.right, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "bottom",
            (gpointer)&printer.bottom, sizeof (gint), PROFILE_VALUE_TYPE_INT);
#endif /* not G_OS_WIN32 */
      /* ja:ファイルタイプ */
      section = NULL;
      for (i = 0; i < ftnum; i++)
        {
          section = g_strdup_printf ("filetype%04d", i);
          profile_set_value (profile, section, "text", (gpointer)ftype[i].text,
                (g_strlen (ftype[i].text) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          profile_set_value (profile, section, "ext", (gpointer)ftype[i].ext,
                (g_strlen (ftype[i].ext) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          profile_set_value (profile, section, "crlf",
                (gpointer)&ftype[i].crlf, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
          profile_set_value (profile,section, "margin",
                (gpointer)&ftype[i].margin, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
          profile_set_value (profile, section, "tab",
                (gpointer)&ftype[i].tab, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
          profile_set_value (profile,section, "writemode",
                (gpointer)&ftype[i].writemode, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
          profile_set_value (profile,section, "associate",
                (gpointer)&ftype[i].associate, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile,section, "autoindent",
                (gpointer)&ftype[i].autoindent, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "code",
                (gpointer)&ftype[i].code, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "eof",
                (gpointer)&ftype[i].eof, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "hexcase",
                (gpointer)&ftype[i].hexcase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "html",
                (gpointer)&ftype[i].html, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "esc",
                (gpointer)&ftype[i].esc, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "limit",
                (gpointer)&ftype[i].limit, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "overwrite",
                (gpointer)&ftype[i].overwrite, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile,  section,"space",
                (gpointer)&ftype[i].space, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "syscol",
                (gpointer)&ftype[i].syscol, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "tabconv",
                (gpointer)&ftype[i].tabconv, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "gline",
                (gpointer)&ftype[i].gline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "mline",
                (gpointer)&ftype[i].mline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "nline",
                (gpointer)&ftype[i].nline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "uline",
                (gpointer)&ftype[i].uline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "vline",
                (gpointer)&ftype[i].vline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          /* ja:キャラクターセット */
          profile_set_value (profile, section, "charset",
                            (gpointer)ftype[i].charset,
                            (g_strlen (ftype[i].charset) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          /* ja:フォント */
          if (ftype[i].font_name)
            profile_set_value (profile, section, "font",
                        (gpointer)ftype[i].font_name,
                        (g_strlen (ftype[i].font_name) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          /* ja:色 */
          profile_set_value (profile, section, "color",
                        (gpointer)&ftype[i].color, sizeof (GdkColor) * 12,
                                                    PROFILE_VALUE_TYPE_ARRAY);
        }
      do
        {
          g_free (section);
          section = g_strdup_printf ("filetype%04d", i++);
        }
      while (profile_delete_section (profile, section));
      g_free (section);
      profile_close (profile);
    }
  pango_font_description_free (system_font);
  g_free (open_path);
  g_free (save_path);
  g_free (clipboard_text);
#ifdef G_OS_UNIX
  g_free (printer.printer);
#endif /* G_OS_UNIX */
#ifndef G_OS_WIN32
  g_free (printer.file);
#endif /* not G_OS_WIN32 */
  for (i = 0; i < find_num; i++)
    g_free (find_text[i]);
  for (i = 0; i < replace_num; i++)
    g_free (replace_text[i]);
  for (i = 0; i < ftnum; i++)
    {
      g_free (ftype[i].text);
      g_free (ftype[i].ext);
      g_free (ftype[i].font_name);
    }
  g_free (ftype);
  return 0;
}
