/*
    Text maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "charset.h"
#include "edit.h"
#include "file.h"
#include "general.h"
#include "sigfile.h"
#include "misc/misc.h"
#include "orz/orzmdi.h"


/******************************************************************************
*                                                                             *
* シグナル/イベント関数群(タイマ)                                             *
*                                                                             *
******************************************************************************/
gboolean
signal_timeout (gpointer data)
{
  gint page;

  caret = !caret;
  page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  if (page >= 0)
    {
      GdkRectangle rc;
      TmaidWindow *tmaid;

      tmaid = orz_mdi_get_data (ORZ_MDI (mdi), page);
      rc.x = (edit_get_align_pos (tmaid, tmaid->cursor.x,
                tmaid->cursor.y, FALSE) - tmaid->top.x) * tmaid->font_width;
      rc.y = (tmaid->cursor.y - tmaid->top.y) * tmaid->font_height;
      rc.width = tmaid->font_width;
      rc.height = tmaid->font_height;
      gtk_widget_draw (tmaid->drawing, &rc);
    }
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(テキスト)                                        *
*                                                                             *
******************************************************************************/
gboolean
signal_focus_in (GtkWidget     *widget,
                 GdkEventFocus *event,
                 TmaidWindow   *tmaid)
{
#ifndef G_OS_WIN32
  gchar *str;
#endif /* not G_OS_WIN32 */

  GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);
#ifndef G_OS_WIN32
  gtk_im_context_get_preedit_string (tmaid->im_context, &str, NULL, NULL);
  if (g_strlen (str) > 0)
    gtk_widget_show (gtk_widget_get_parent (tmaid->preedit));
  else
    gtk_widget_hide (gtk_widget_get_parent (tmaid->preedit));
  g_free (str);
#endif /* not G_OS_WIN32 */
  gtk_im_context_focus_in (tmaid->im_context);
  draw_caret (tmaid, NULL);
  return FALSE;
}


gboolean
signal_focus_out (GtkWidget     *widget,
                  GdkEventFocus *event,
                  TmaidWindow   *tmaid)
{
  GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);
#ifndef G_OS_WIN32
  gtk_widget_hide (gtk_widget_get_parent (tmaid->preedit));
#endif /* not G_OS_WIN32 */
  gtk_im_context_focus_out (tmaid->im_context);
  draw_caret (tmaid, NULL);
  return FALSE;
}


void
signal_realize (GtkWidget   *widget,
                TmaidWindow *tmaid)
{
  GdkCursor *cursor;

  /* ja:カーソル */
  cursor = gdk_cursor_new (GDK_XTERM);
  gdk_window_set_cursor (widget->window, cursor);
  gdk_cursor_unref (cursor);
  gtk_im_context_set_client_window (tmaid->im_context, widget->window);
  /* ja:フォント */
  g_object_unref (tmaid->layout);
  if (tmaid->ft.font_name)
    {
      PangoFontDescription *desc;

      desc = pango_font_description_from_string (tmaid->ft.font_name);
      gtk_widget_modify_font (widget, desc);
      pango_font_description_free (desc);
    }
  else
    {
      gtk_widget_modify_font (widget, system_font);
    }
  tmaid->layout = gtk_widget_create_pango_layout (widget, NULL);
}


void
signal_unrealize (GtkWidget   *widget,
                  TmaidWindow *tmaid)
{
  gtk_im_context_set_client_window (tmaid->im_context, NULL);
}


void
signal_value_changed_hscroll (GtkAdjustment *adjust,
                              TmaidWindow   *tmaid)
{
  GdkPoint top;

  top = tmaid->top;
  tmaid->top.x = adjust->value;
  move_edit_window (tmaid, &top);
  draw_caret (tmaid, NULL);
}


void
signal_value_changed_vscroll (GtkAdjustment *adjust,
                              TmaidWindow   *tmaid)
{
  gint max, sx;
  GdkPoint top;

  top = tmaid->top;
  tmaid->top.y = adjust->value + 0.5;
  sx = MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1);
  max = edit_get_width_max (tmaid);
  if (tmaid->top.x > max - sx + 1)
    tmaid->top.x = MAX (max - sx + 1, 0);
  move_edit_window (tmaid, &top);
  if (tmaid->top.x != top.x
        || gtk_range_get_adjustment (GTK_RANGE (tmaid->hscroll))->upper != max)
        misc_set_scroll_bar (tmaid->hscroll,
                            G_CALLBACK (signal_value_changed_hscroll), tmaid,
                                                0, max + 1, sx, tmaid->top.x);
  draw_caret (tmaid, NULL);
}


gboolean
signal_config (GtkWidget         *widget,
               GdkEventConfigure *config,
               TmaidWindow       *tmaid)
{
  gint max, sx, sy;

  sx = MAX (config->width / tmaid->font_width, 1);
  sy = MAX (config->height / tmaid->font_height, 1);
  if (tmaid->top.y > tmaid->max - sy)
    tmaid->top.y = MAX (tmaid->max - sy, 0);
  max = edit_get_width_max (tmaid);
  if (tmaid->top.x > max - sx + 1)
    tmaid->top.x = MAX (max - sx + 1, 0);
  misc_set_scroll_bar (tmaid->hscroll,
                            G_CALLBACK (signal_value_changed_hscroll), tmaid,
                                            0, max + 1, sx, tmaid->top.x);
  misc_set_scroll_bar (tmaid->vscroll,
                            G_CALLBACK (signal_value_changed_vscroll), tmaid,
                                            0, tmaid->max, sy, tmaid->top.y);
  return TRUE;
}


void
signal_destroy_draw (GtkWidget   *widget,
                     TmaidWindow *tmaid)
{
  /* ja:メニュー */
  if (orz_mdi_get_n_pages (ORZ_MDI (mdi)) <= 1)
    {
      set_menu_bar (NULL);
      gtk_statusbar_pop (GTK_STATUSBAR (status),
            gtk_statusbar_get_context_id (GTK_STATUSBAR (status), "Status"));
      gtk_label_set_text (GTK_LABEL (label_cursor), "");
    }
  /* ja:メモリを解放する */
  g_object_unref (tmaid->layout);
  if (tmaid->timer_id != 0)
    {
      gtk_timeout_remove (tmaid->timer_id);
      tmaid->timer_id = 0;
    }
  while (tmaid->start->prev)
    tmaid->start = tmaid->start->prev;
  while (tmaid->start)
    {
      LineBuffer *p;

      p = tmaid->start->next;
      g_free (tmaid->start->text);
      g_free (tmaid->start);
      tmaid->start = p;
    }
  delete_list (&tmaid->undo);
  delete_list (&tmaid->redo);
  /* en:Input Method */
#ifndef G_OS_WIN32
  gtk_widget_destroy (gtk_widget_get_parent (tmaid->preedit));
#endif /* G_OS_WIN32 */
  g_signal_connect_swapped (G_OBJECT (window), "destroy",
                    G_CALLBACK (g_object_unref), G_OBJECT (tmaid->im_context));
  g_free (tmaid);
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(Input Method)                                    *
*                                                                             *
******************************************************************************/
void
signal_commit (GtkIMContext *im_context,
               gchar        *arg1, 
               TmaidWindow  *tmaid)
{
  gchar *str;
  gint i, leng, length;

  length = g_strlen (arg1);
  for (i = 0; i < length; i += leng)
    {
      leng = charset_length (arg1[i]);
      if (i + leng > length || g_unichar_iscntrl (g_utf8_get_char (arg1 + i)))
        break;
    }
  if (i == length)
    {
      GdkPoint select;
      TmaidHistory *d;

      select = tmaid->select;
      tmaid->cursor.x = edit_get_align_pos
                            (tmaid, tmaid->cursor.x, tmaid->cursor.y, FALSE);
      if (tmaid->select.x < 0 && !ins)
        {
          tmaid->select.x = edit_get_align_pos
                        (tmaid, tmaid->cursor.x + 1, tmaid->cursor.y, TRUE);
          if (tmaid->select.x <= tmaid->cursor.x)
            tmaid->select.x = -1;
          tmaid->select.y = tmaid->cursor.y;
        }
      d = edit_operation (tmaid, arg1, length, TRUE, FALSE);
      d->next = tmaid->undo;
      tmaid->undo = d;
      if (delete_list (&tmaid->redo) > 0 || !d->next || select.x >= 0)
        set_menu_bar (tmaid);
      orz_mdi_set_edited (ORZ_MDI (mdi),
                    orz_mdi_get_page_from_data (ORZ_MDI (mdi), tmaid), TRUE);
    }
#ifndef G_OS_WIN32
  gtk_im_context_get_preedit_string (im_context, &str, NULL, NULL);
  if (g_strlen (str) <= 0)
    gtk_widget_hide (gtk_widget_get_parent (tmaid->preedit));
#endif /* not G_OS_WIN32 */
}


#ifndef G_OS_WIN32
void
signal_preedit_changed (GtkIMContext *im_context,
                        TmaidWindow  *tmaid)
{
  gchar *str;
  GtkWidget *widget;
  PangoAttrList *attrs;

  gtk_im_context_get_preedit_string (im_context, &str, &attrs, NULL);
  widget = gtk_widget_get_parent (tmaid->preedit);
  if (g_strlen (str) > 0)
    {
      int width;

      pango_layout_set_text (tmaid->layout, str, -1);
      pango_layout_set_attributes (tmaid->layout, attrs);
      pango_layout_get_pixel_size (tmaid->layout, &width, NULL);
      pango_layout_set_attributes (tmaid->layout, NULL);
      gtk_window_resize (GTK_WINDOW (widget), width, tmaid->font_height);
      gtk_widget_show (widget);
      gtk_widget_queue_draw_area (tmaid->preedit,
                                            0, 0, width, tmaid->font_height);
    }
  else
    {
      gtk_widget_hide (widget);
    }
  g_free (str);
  pango_attr_list_unref (attrs);
}
#endif /* not G_OS_WIN32 */


gboolean
signal_retrieve_surrounding (GtkIMContext *imcontext,
                             TmaidWindow  *tmaid)
{
  LineBuffer *p;

  p = edit_get_line_buf (&tmaid->start, &tmaid->off, tmaid->cursor.y);
  gtk_im_context_set_surrounding (imcontext, p->text, p->length,
        edit_get_data_pos (tmaid, tmaid->cursor.x, tmaid->cursor.y, FALSE));
  return TRUE;
}


gboolean
signal_delete_surrounding (GtkIMContext *imcontext,
                           gint          offset,
                           gint          n_chars,
                           TmaidWindow  *tmaid)
{
  return TRUE;
}


#ifndef G_OS_WIN32
gboolean
signal_expose_preedit (GtkWidget      *widget,
                       GdkEventExpose *event,
                       TmaidWindow    *tmaid)
{
  gchar *str;
  GdkGC *gc;
  PangoAttrList *attrs;

  gtk_im_context_get_preedit_string (tmaid->im_context, &str, &attrs, NULL);
  pango_layout_set_text (tmaid->layout, str, -1);
  g_free (str);
  pango_layout_set_attributes (tmaid->layout, attrs);
  pango_attr_list_unref (attrs);

  gc = gdk_gc_new (widget->window);
  gdk_color_alloc (gdk_colormap_get_system (),
                    (tmaid->ft.syscol ? system_color : tmaid->ft.color));
  gdk_color_alloc (gdk_colormap_get_system (),
                    (tmaid->ft.syscol ? system_color : tmaid->ft.color) + 1);
  /* ja:背景 */
  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 1);
  gdk_draw_rectangle(widget->window, gc, TRUE, event->area.x, event->area.y,
                                        event->area.width, event->area.height);
  /* ja:文字 */
  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color));
  gdk_gc_set_background (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 1);
  gdk_draw_layout (widget->window, gc, 0, 0, tmaid->layout);
  g_object_unref (gc);
  pango_layout_set_attributes (tmaid->layout, NULL);
  return TRUE;
}
#endif /* not G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(描画)                                            *
*                                                                             *
******************************************************************************/
gboolean
signal_expose (GtkWidget      *widget,
               GdkEventExpose *event,
               TmaidWindow    *tmaid)
{
  gboolean select, space = FALSE;
  gint i, x, y, data_pos, length, width;
  GdkGC *gc;
  GdkPoint start, end, cursor, pt[4];
  LineBuffer *p;

  gc = gdk_gc_new (widget->window);
  gdk_gc_set_line_attributes (gc, 1,
                                GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
  for (i = 0; i < 10; i++)
    gdk_color_alloc (gdk_colormap_get_system (),
                    (tmaid->ft.syscol ? system_color : tmaid->ft.color) + i);
    cursor.x = edit_get_align_pos (tmaid,
                                   tmaid->cursor.x, tmaid->cursor.y, FALSE);
    cursor.y = tmaid->cursor.y;
  if (tmaid->select.x == -1)
    {
      /* ja:選択範囲がないとき */
      start.x = -1;
    }
  else if (tmaid->select.y < tmaid->cursor.y
                                        || (tmaid->select.y == tmaid->cursor.y
                                        && tmaid->select.x < tmaid->cursor.x))
    {
      /* ja:選択範囲があるとき */
      start = tmaid->select;
      end = cursor;
    }
  else
    {
      /* ja:選択範囲があるとき */
      start = cursor;
      end = tmaid->select;
    }
  /* ja:背景 */
  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 1);
  gdk_draw_rectangle (widget->window, gc, TRUE, event->area.x, event->area.y,
                                        event->area.width, event->area.height);
  if (tmaid->ft.gline)
    {
      /* ja:グリッド */
      gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 7);
      for (x = tmaid->font_width - 1; x < widget->allocation.width;
                                                    x += tmaid->font_width)
        gdk_draw_line (widget->window, gc,
                    x, event->area.y, x, event->area. y + event->area.height);
      for (y = tmaid->font_height - 1; y < widget->allocation.height;
                                                    y += tmaid->font_height)
        gdk_draw_line (widget->window, gc,
                    event->area.x, y, event->area.x + event->area.width, y);
    }
  if (tmaid->ft.vline)
    {
      /* ja:縦線 */
      gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 7);
      for (x = (tmaid->ft.tab - tmaid->top.x % tmaid->ft.tab)
                                                    * tmaid->font_width - 1;
                                        x < widget->allocation.width;
                                        x += tmaid->font_width * tmaid->ft.tab)
        gdk_draw_line (widget->window, gc,
                    x, event->area.y, x, event->area.y + event->area.height);
    }
  if (tmaid->ft.mline)
    {
      /* ja:右マージン */
      gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 6);
      x = (tmaid->ft.margin - tmaid->top.x) * tmaid->font_width;
      gdk_draw_line (widget->window, gc,
                    x, event->area.y, x, event->area.y + event->area.height);
    }

  for (y = 0, p = edit_get_line_buf (&tmaid->start, &tmaid->off, tmaid->top.y);
                                y < event->area.y + event->area.height && p;
                                        y += tmaid->font_height, p = p->next)
    {
      if (y + tmaid->font_height < event->area.y)
        continue;
      x = -tmaid->top.x * tmaid->font_width;
      data_pos = 0;
      while (x <= event->area.x + event->area.width && data_pos < p->length)
        {
          if (x + MAX (tmaid->ft.tab, 2) * tmaid->font_width < event->area.x)
            {
              /* ja:左側の表示されない部分を計算する */
              if (p->text[data_pos] == '\t')
                {
                  x = (((x / tmaid->font_width + tmaid->top.x)
                                                        / tmaid->ft.tab + 1)
                        * tmaid->ft.tab - tmaid->top.x) * tmaid->font_width;
                  data_pos++;
                }
              else if (data_pos + charset_length(p->text[data_pos])
                                                                <= p->length)
                {
                  /* ja:半角/全角 */
                  x += tmaid->font_width * charset_width (tmaid->layout,
                                        p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
                  data_pos += charset_length (p->text[data_pos]);
                }
              else
                {
                  /* ja:強制半角 */
                  x += tmaid->font_width;
                  data_pos++;
                }
              continue;
            }
          /* ja:選択範囲 */
          select = start.x != -1 && (start.y == end.y ?
                    start.y == tmaid->top.y + y / tmaid->font_height
                            && start.x <= tmaid->top.x + x / tmaid->font_width
                            && tmaid->top.x + x / tmaid->font_width < end.x
                    : (start.y < tmaid->top.y + y / tmaid->font_height
                        && tmaid->top.y + y / tmaid->font_height < end.y)
                        || (start.y == tmaid->top.y + y / tmaid->font_height
                            && start.x <= tmaid->top.x + x / tmaid->font_width)
                        || (end.y == tmaid->top.y + y / tmaid->font_height
                            && tmaid->top.x + x / tmaid->font_width < end.x));
          if (p->text[data_pos] == '\t')
            {
              /* ja:タブ */
              width = (((x / tmaid->font_width + tmaid->top.x)
                                                        / tmaid->ft.tab + 1)
                    * tmaid->ft.tab - tmaid->top.x) * tmaid->font_width - x;
              if (select)
                {
                  /* ja:選択 */
                  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 9);
                  gdk_draw_rectangle (widget->window, gc, TRUE, x, y,
                                                    width, tmaid->font_height);
                }
              if (tmaid->ft.uline)
                {
                  /* ja:下線 */
                  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 5);
                  gdk_draw_line (widget->window, gc,
                                        x, y + tmaid->font_height - 1, 
                                        x + width, y + tmaid->font_height - 1);
                }
              if (tmaid->ft.code)
                {
                  /* ja:コード文字 */
                  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 2);
                  gdk_draw_rectangle (widget->window, gc, FALSE,
                        x + tmaid->font_width / 6, y + tmaid->font_height / 3,
                        tmaid->font_height / 3, tmaid->font_height / 3);
                }
              data_pos++;
            }
          else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
            {
              /* ja:半角/全角 */
              width = tmaid->font_width * charset_width (tmaid->layout,
                                        p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
              if (select)
                {
                  /* ja:選択 */
                  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 9);
                  gdk_draw_rectangle (widget->window, gc, TRUE, x, y,
                                                    width, tmaid->font_height);
                  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 8);
                }
              else
                {
                  gdk_gc_set_foreground (gc, tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color);
                }
              if (tmaid->ft.space)
                {
                  space = FALSE;/* ja:全角スペース判定 */
                  for (i = 0; charset_space[i] != 0; i += charset_space[i] + 1)
                    if (charset_length (p->text[data_pos]) == charset_space[i]
                            && g_memcmp (p->text + data_pos,
                                charset_space + i + 1, charset_space[i]) == 0)
                      space = TRUE;
                }
              if (space)
                {
                  /* ja:全角スペース */
                  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 3);
                  gdk_draw_rectangle (widget->window, gc, TRUE, x + 1, y + 1,
                                            width - 1, tmaid->font_height - 1);
                }
              else
                {
                  /* ja:文字 */
                  pango_layout_set_text (tmaid->layout, p->text + data_pos,
                                        charset_length (p->text[data_pos]));
                  gdk_draw_layout (widget->window, gc, x, y, tmaid->layout);
                }
              data_pos += charset_length (p->text[data_pos]);
            }
          else
            {
              /* ja:強制半角 */
              width = tmaid->font_width;
              if (select)
                {
                  /* ja:選択 */
                  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 9);
                  gdk_draw_rectangle (widget->window, gc, TRUE, x, y,
                                                    width, tmaid->font_height);
                  gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 8);
                }
              else
                {
                  gdk_gc_set_foreground (gc, tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color);
                }
              pango_layout_set_text (tmaid->layout, p->text + data_pos, 1);
              gdk_draw_layout (widget->window, gc, x, y, tmaid->layout);
              data_pos++;
            }
          /* ja:キャレットの描画 */
          if (caret && GTK_WIDGET_HAS_FOCUS (widget)
                        && cursor.x == tmaid->top.x + x / tmaid->font_width
                        && cursor.y == tmaid->top.y + y / tmaid->font_height)
            {
              gdk_gc_set_function (gc, GDK_INVERT);
              gdk_draw_rectangle (widget->window, gc, TRUE, x, y,
                            ins ? 2 : tmaid->font_width, tmaid->font_height);
              gdk_gc_set_function (gc, GDK_COPY);
            }
          x += width;
        }
      if (data_pos == p->length && tmaid->ft.nline)
        {
          if (start.x >= 0 && start.y <= tmaid->top.y + y / tmaid->font_height
                            && tmaid->top.y + y / tmaid->font_height < end.y)
            {
              /* ja:選択 */
              gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 9);
              gdk_draw_rectangle (widget->window, gc, TRUE, x, y,
                                        tmaid->font_width, tmaid->font_height);
            }
          gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + 4);
          if (!p->next)
            {
              /* ja:ファイルの最後 */
              gdk_draw_arc (widget->window, gc, TRUE,
                x + tmaid->font_width / 6, y + tmaid->font_height / 3,
                tmaid->font_height / 3, tmaid->font_height / 3, 0, 360 * 64);
            }
          else
            {
              if (p->margin)
                {
                  /* ja:右マージンによる改行 */
                  pt[0].x = pt[3].x = x + tmaid->font_width / 6;
                  pt[0].y = pt[1].y = y + tmaid->font_height / 3;
                  pt[1].x = pt[2].x = x + tmaid->font_width * 5 / 6;
                  pt[2].y = pt[3].y = y + tmaid->font_height * 2 / 3;
                  length = 4;
                }
              else
                {
                  /* ja:通常の改行 */
                  pt[0].x = x + tmaid->font_width / 6;
                  pt[1].x = x + tmaid->font_width * 5 / 6;
                  pt[0].y = pt[1].y = y + tmaid->font_height / 3;
                  pt[2].x = x + tmaid->font_width / 2;
                  pt[2].y = y + tmaid->font_height * 2 / 3;
                  length = 3;
                }
              gdk_draw_polygon (widget->window, gc, TRUE, pt, length);
            }
        }
      /* ja:キャレットの描画 */
      if (caret && GTK_WIDGET_HAS_FOCUS (widget)
                        && cursor.x == tmaid->top.x + x / tmaid->font_width
                        && cursor.y == tmaid->top.y + y / tmaid->font_height)
        {
          gdk_gc_set_function (gc, GDK_INVERT);
          gdk_draw_rectangle (widget->window, gc, TRUE, x, y,
                            ins ? 2 : tmaid->font_width, tmaid->font_height);
          gdk_gc_set_function (gc, GDK_COPY);
        }
    }

    g_object_unref (gc);
    return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(マウス)                                          *
*                                                                             *
******************************************************************************/
gboolean
signal_button_press (GtkWidget      *widget,
                     GdkEventButton *event,
                     TmaidWindow    *tmaid)
{
  if (!GTK_WIDGET_HAS_FOCUS (widget))
    gtk_widget_grab_focus (widget);
  switch (event->type)
    {
      case GDK_BUTTON_PRESS:
        switch (event->button)
          {
            case 1:/* ja:左クリック */
              {
                gboolean shift;
                GdkPoint cursor, select;

                cursor = tmaid->cursor;
                select = tmaid->select;
                tmaid->cursor.x = tmaid->top.x+event->x / tmaid->font_width;
                tmaid->cursor.y = tmaid->top.y+event->y / tmaid->font_height;
                shift = event->state & GDK_SHIFT_MASK;
                if (tmaid->cursor.y < 0)
                    tmaid->cursor.y = 0;
                else if (tmaid->cursor.y > tmaid->max - 1)
                    tmaid->cursor.y = tmaid->max - 1;
                if (tmaid->cursor.x < edit_get_width (tmaid,tmaid->cursor.y))
                    tmaid->cursor.x = edit_get_align_pos (tmaid,
                                    tmaid->cursor.x, tmaid->cursor.y, FALSE);
                if (tmaid->select.x >= 0
                            && (!shift || (tmaid->cursor.x == tmaid->select.x
                                    && tmaid->cursor.y == tmaid->select.y)))
                  {
                    /* ja:選択を解除する */
                    tmaid->select.x = -1;
                    clear_sel (tmaid, &cursor, &select);
                    set_menu_bar (tmaid);
                  }
                else if (shift && (tmaid->cursor.x != cursor.x
                                            || tmaid->cursor.y != cursor.y))
                  {
                    if (tmaid->select.x < 0)
                      {
                        /* ja:新たに選択する */
                        tmaid->select.x = edit_get_align_pos (tmaid,
                                                    cursor.x, cursor.y, FALSE);
                        tmaid->select.y = cursor.y;
                        gtk_selection_owner_set (widget, GDK_SELECTION_PRIMARY,
                                                            GDK_CURRENT_TIME);
                      }
                    clear_sel (tmaid, &cursor, &tmaid->cursor);
                    set_menu_bar (tmaid);
                  }
                draw_caret (tmaid, &cursor);
              }
              break;
            case 2:/* ja:中クリック */
              {
                gchar *text = NULL;
                gint length;
                GdkPoint select;
                TmaidHistory *d;

                if (gdk_selection_owner_get (GDK_SELECTION_PRIMARY)
                                    == widget->window && tmaid->select.x >= 0)
                  {
                    /* ja:セレクションのオーナーが自分のとき */
                    /* ja:選択範囲を取得 */
                    length = edit_get_sel_bytes
                                    (tmaid, &tmaid->select, &tmaid->cursor);
                    text = g_malloc ((length + 1) * sizeof (gchar));
                    edit_cpy_sel_mem (tmaid,
                                        &tmaid->select, &tmaid->cursor, text);
                    text[length] = '\0';
                  }
                /* ja:選択を解除する */
                if (tmaid->select.x >= 0)
                  {
                    select = tmaid->select;
                    tmaid->select.x = -1;
                    clear_sel (tmaid, &tmaid->cursor, &select);
                    if (!text)
                        set_menu_bar (tmaid);
                  }
                /* ja:キャレット */
                draw_caret (tmaid, &tmaid->cursor);
                if (text)
                  {
                    /* ja:貼り付け */
                    d = edit_operation (tmaid,
                                        text, g_strlen (text), FALSE, TRUE);
                    g_free (text);
                    d->next = tmaid->undo;
                    tmaid->undo = d;
                    if (delete_list(&tmaid->redo) > 0 || !d->next
                                                                || select.x<0)
                      set_menu_bar (tmaid);
                    orz_mdi_set_edited (ORZ_MDI (mdi),
                            orz_mdi_get_page_from_data (ORZ_MDI (mdi), tmaid),
                                                                        TRUE);
                  }
                else
                  {
                    /* ja:セレクションのオーナーが自分ではないとき */
                    gtk_selection_convert (window, GDK_SELECTION_PRIMARY,
                                            atom_targets, GDK_CURRENT_TIME);
                  }
              }
              break;
            case 3:/* ja:右クリック */
              {
                GtkWidget *menu_shell, *menu_item;

                menu_item = misc_find_menu (popup_entries, "/input");
                gtk_menu_item_remove_submenu (GTK_MENU_ITEM (menu_item));
                menu_shell = gtk_menu_new ();
                gtk_im_multicontext_append_menuitems (GTK_IM_MULTICONTEXT
                            (tmaid->im_context), GTK_MENU_SHELL (menu_shell));
                gtk_menu_item_set_submenu (GTK_MENU_ITEM (menu_item),
                                                                menu_shell);
                gtk_widget_set_sensitive (misc_find_menu (popup_entries,
                                            "/undo"), tmaid->undo != NULL);
                gtk_widget_set_sensitive (misc_find_menu (popup_entries,
                                            "/cut"), tmaid->select.x >= 0);
                gtk_widget_set_sensitive (misc_find_menu (popup_entries,
                                            "/copy"), tmaid->select.x >= 0);
                gtk_widget_set_sensitive (misc_find_menu (popup_entries,
                                            "/delete"), tmaid->select.x >= 0);
                gtk_menu_popup (GTK_MENU (misc_find_menu (popup_entries, "/")),
                        NULL, NULL, NULL, NULL, event->button, event->time);
              }
          }
          break;
      case GDK_2BUTTON_PRESS:
        {
          GdkPoint cursor;

          /* ja:ダブルクリックしたとき単語を選択する */
          if (event->button != 1)
            return FALSE;
          cursor = tmaid->cursor;
          if (edit_select_word (tmaid))
            {
              /* ja:新たに選択する */
              set_menu_bar (tmaid);
              draw_caret (tmaid, &cursor);
              clear_sel (tmaid, &tmaid->select, &tmaid->cursor);
              gtk_selection_owner_set (widget, GDK_SELECTION_PRIMARY,
                                                            GDK_CURRENT_TIME);
            }
        }
        break;
      default:
        return FALSE;
    }
  return TRUE;
}


/*  ja:マウスの移動に関する処理
    tmaid,ウインドウ情報
        x,X座標
        y,Y座標
    state,ステータス
      RET,TRUE:スクロールあり,FALSE:スクロールなし                          */
static gboolean
signal_motion_notify_draw (TmaidWindow     *tmaid,
                           const gint       x,
                           const gint       y,
                           GdkModifierType  state)
{
  gint s, sx, sy;
  GdkPoint cursor, select, top, ps;

  cursor = tmaid->cursor;
  select = tmaid->select;
  top = tmaid->top;
  sx = MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1);
  sy = MAX (tmaid->drawing->allocation.height / tmaid->font_height, 1);
  ps.x = x;
  ps.y = y;
  if (ps.x < 0)
    ps.x = 0;
  else if (ps.x > tmaid->drawing->allocation.width)
    ps.x = tmaid->drawing->allocation.width;
  if (ps.y < 0)
    ps.y = 0;
  else if (ps.y > tmaid->drawing->allocation.height)
    ps.y = tmaid->drawing->allocation.height;
  /* ja:カーソルが外にあるときスクロールする */
  if (sx > 1)
    {
      if (x < 0 && tmaid->top.x > 0)
        tmaid->top.x--;
      else if (x > tmaid->drawing->allocation.width
            && tmaid->top.x < edit_get_width (tmaid, tmaid->cursor.y) + 1 - sx)
        tmaid->top.x++;
    }
  if (sy > 1)
    {
      if (y < 0 && tmaid->top.y > 0)
        tmaid->top.y--;
      else if (y > tmaid->drawing->allocation.height
                                            && tmaid->top.y < tmaid->max - sy)
        tmaid->top.y++;
    }
  s = ps.x - ((tmaid->select.x < 0 ? tmaid->cursor.x : tmaid->select.x)
                                        - tmaid->top.x) * tmaid->font_width;
  if (s < tmaid->font_width / 2 || tmaid->font_width < s
                || (tmaid->select.x < 0 ? tmaid->cursor.y : tmaid->select.y)
                                != ps.y / tmaid->font_height + tmaid->top.y)
    {
      if (tmaid->select.x < 0)
        {
          /* ja:新たに選択する */
          tmaid->select.x = edit_get_align_pos (tmaid,
                                    tmaid->cursor.x, tmaid->cursor.y, FALSE);
          tmaid->select.y = tmaid->cursor.y;
        }
      tmaid->cursor.x = ps.x / tmaid->font_width + tmaid->top.x;
      tmaid->cursor.y = ps.y / tmaid->font_height + tmaid->top.y;
      if (tmaid->cursor.y < 0)
        tmaid->cursor.y = 0;
      else if (tmaid->cursor.y > tmaid->max - 1)
        tmaid->cursor.y = tmaid->max - 1;
      tmaid->cursor.x = edit_get_align_pos (tmaid,
                                    tmaid->cursor.x, tmaid->cursor.y, FALSE);
    }
  else if (tmaid->select.x >= 0)
    {
      /* ja:選択を解除する */
      tmaid->cursor = tmaid->select;
      tmaid->select.x = -1;
    }
  if (tmaid->select.x == tmaid->cursor.x && tmaid->select.y == tmaid->cursor.y)
    tmaid->select.x = -1;/* ja:選択の初めと終わりが同じとき */
  else if (select.x == -1)/* ja:新たに選択する */
    gtk_selection_owner_set (tmaid->drawing, GDK_SELECTION_PRIMARY,
                                                            GDK_CURRENT_TIME);
  if (tmaid->cursor.x != cursor.x || tmaid->cursor.y != cursor.y
                            || tmaid->top.x != top.x || tmaid->top.y != top.y)
    {
      set_menu_bar (tmaid);
      move_edit_window (tmaid, &top);
      if (tmaid->top.x != top.x)
        misc_set_scroll_bar (tmaid->hscroll,
                        G_CALLBACK (signal_value_changed_hscroll), tmaid,
                        0, edit_get_width_max (tmaid) + 1, sx, tmaid->top.x);
      if (tmaid->top.y != top.y)
        misc_set_scroll_bar (tmaid->vscroll,
                        G_CALLBACK (signal_value_changed_vscroll), tmaid,
                        0, tmaid->max, sy, tmaid->top.y);
      draw_caret (tmaid, &cursor);
      clear_sel (tmaid, &tmaid->cursor, &cursor);
    }
  return tmaid->top.x != top.x || tmaid->top.y != top.y;
}


static gboolean
signal_timeout_draw (TmaidWindow *tmaid)
{
  gboolean result;
  gint x, y;
  GdkModifierType state;

  gdk_window_get_pointer (tmaid->drawing->window, &x, &y, &state);
  if (state&GDK_BUTTON1_MASK)
    {
      result = signal_motion_notify_draw (tmaid, x, y, state);
      if (!result)
        {
          /* ja:スクロールが終わったらタイマは削除 */
          gtk_timeout_remove (tmaid->timer_id);
          tmaid->timer_id = 0;
        }
    }
  else
    {
      result = FALSE;
    }
  return result;
}


gboolean
signal_motion_notify (GtkWidget      *widget,
                      GdkEventMotion *event,
                      TmaidWindow    *tmaid)
{
  gint x, y;
  GdkModifierType state;

  if (event->is_hint)
    {
      gdk_window_get_pointer (event->window, &x, &y, &state);
    }
  else
    {
      x = event->x;
      y = event->y;
      state = event->state;
    }
  if ((state&GDK_BUTTON1_MASK)
                            && signal_motion_notify_draw (tmaid, x, y, state)
                                                    && tmaid->timer_id == 0)
    /* ja:はじめてスクロールしたとき */
    tmaid->timer_id = gtk_timeout_add (1, (GtkFunction)signal_timeout_draw,
                                                                        tmaid);
  return TRUE;
}


gboolean
signal_button_release (GtkWidget      *widget,
                       GdkEventButton *event,
                       TmaidWindow    *tmaid)
{
  if (tmaid->timer_id!=0)
    {
      gtk_timeout_remove (tmaid->timer_id);
      tmaid->timer_id = 0;
    }
  return TRUE;
}


gboolean
signal_scroll (GtkWidget      *widget,
               GdkEventScroll *event,
               TmaidWindow    *tmaid)
{
  gint max, sx, sy;
  GdkPoint top;

  top = tmaid->top;
  switch (event->direction)
    {
      case GDK_SCROLL_UP: tmaid->top.y -= 4; break;
      case GDK_SCROLL_DOWN: tmaid->top.y += 4; break;
      case GDK_SCROLL_LEFT: tmaid->top.x--; break;
      case GDK_SCROLL_RIGHT: tmaid->top.x++;
    }
  sx = MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1);
  sy = MAX (tmaid->drawing->allocation.height / tmaid->font_height, 1);
  if (tmaid->top.y < 0)
    tmaid->top.y = 0;
  else if (tmaid->top.y > tmaid->max - sy)
    tmaid->top.y = MAX (tmaid->max - sy, 0);
  max = edit_get_width_max (tmaid);
  if (tmaid->top.x > max - sx + 1)
    tmaid->top.x = MAX (max - sx + 1, 0);
  move_edit_window (tmaid, &top);
  if (tmaid->top.y != top.y)
    misc_set_scroll_bar (tmaid->vscroll,
                            G_CALLBACK (signal_value_changed_vscroll), tmaid,
                                            0, tmaid->max, sy, tmaid->top.y);
  if (tmaid->top.x != top.x
        || gtk_range_get_adjustment (GTK_RANGE (tmaid->hscroll))->upper != max)
    misc_set_scroll_bar (tmaid->hscroll,
                            G_CALLBACK (signal_value_changed_hscroll), tmaid,
                                            0, max + 1, sx, tmaid->top.x);
  draw_caret (tmaid, NULL);
  return TRUE;
}
