/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 * All rights reserved.
 */

#include <core/exint.h>
#include "asm.h"
#include "apic.h"
#include "config.h"
#include "constants.h"
#include "current.h"
#include "exint_pass.h"
#include "exint_8259a.h"
#include "initfunc.h"
#include "int.h"
#include "string.h"

/* #define EXINT_DEBUG 2 */
#ifdef EXINT_DEBUG
#include <core/printf.h>
#define EXINT_DBG(level, ...) \
	do {							\
		if (level <= EXINT_DEBUG) {			\
			printf("EXINT: "__VA_ARGS__);		\
		}						\
	} while (0)
#else
#define EXINT_DBG(...)
#endif

void
exint_send_apic_int(vector_t vector, apic_id_t dest)
{
	apic_send_ipi(vector, dest);
}

void
exint_reset_apic(void)
{
	EXINT_DBG(1, "Reset APIC\n");

	long vector;
	int i;

	for (i = 0; i< 255; i++) {
		vector = do_externalint_enable();
		if (vector < 0) {
			break;
		}
		apic_write_eoi();
	}
}

static void exint_pass_int_enabled (void);
static void exint_pass_default (int num);
static void exint_pass_hlt (void);

static struct exint_func func = {
	exint_pass_int_enabled,
	exint_pass_default,
	exint_pass_hlt,
};

static void
exint_pass_int_enabled (void)
{
	current->vmctl.exint_pending (false);
#ifdef NO_INTR_INTERCEPT
	current->vmctl.exint_pass (true);
#else
	current->vmctl.exint_pass (!!config.vmm.no_intr_intercept);
#endif
}

static void
exint_pass_default (int num)
{
	current->vmctl.generate_external_int (num);
}

/*
 * Called when VM-exit occured due to interrupt-window.
 */
static void
exint_pass_hlt (void)
{
	ulong rflags;
	int num;

	current->vmctl.read_flags (&rflags);
	if (rflags & RFLAGS_IF_BIT) { /* if interrupts are enabled */
		num = exint_receive_8259a_int();
		if (num >= 0) {
			EXINT_DBG(2, "R%d 8259a %d\n", get_apic_id(),
				  num);
			current->exint.exintfunc_default (num);
			current->vmctl.exint_pending (true);
		} else {
			num = do_externalint_enable ();
			if (num >= 0) {
				EXINT_DBG(2, "1 R%d %d\n", get_apic_id(),
					  num);
				current->exint.exintfunc_default (num);
			}
			current->vmctl.exint_pending (false);
		}
#ifdef NO_INTR_INTERCEPT
		current->vmctl.exint_pass (true);
#else
		current->vmctl.exint_pass (!!config.vmm.no_intr_intercept);
#endif
	} else {
		current->vmctl.exint_pending (true);
		current->vmctl.exint_pass (true);
	}
}

/*
 * Called when VM-exit occured due to external int.
 */
void
do_exint_pass (void)
{
	ulong rflags;
	int num;

	current->vmctl.read_flags (&rflags);
	if (rflags & RFLAGS_IF_BIT) { /* if interrupts are enabled */
		num = do_externalint_enable ();
		if (num >= 0) {
			EXINT_DBG(2, "2 R%d %d\n", get_apic_id(),
				  num);
			current->exint.exintfunc_default (num);
		}
		num = exint_check_8259a_int();
		if (num >= 0) {
			EXINT_DBG(2, "C%d 8259a %d\n", get_apic_id(),
				  num);
			current->vmctl.exint_pending (true);
		} else {
			current->vmctl.exint_pending (false);
		}
#ifdef NO_INTR_INTERCEPT
		current->vmctl.exint_pass (true);
#else
		current->vmctl.exint_pass (!!config.vmm.no_intr_intercept);
#endif
	} else {
		current->vmctl.exint_pending (true);
		current->vmctl.exint_pass (true);
	}
}

static void
exint_pass_init (void)
{
	memcpy ((void *)&current->exint, (void *)&func, sizeof func);
}

INITFUNC ("pass0", exint_pass_init);
