/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.web.codelist;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import junit.framework.TestCase;

/**
 * {@link jp.terasoluna.fw.web.codelist.AbstractMultilingualCodeListLoader} NX̃ubN{bNXeXgB
 * 
 * <p>
 * <h4>yNX̊Tvz</h4>
 * <br>
 * EO<br>
 * <p>
 * 
 * @see jp.terasoluna.fw.web.codelist.AbstractMultilingualCodeListLoader
 */
public class AbstractMultilingualCodeListLoaderTest extends TestCase {
    
    private static Map<Locale, List<CodeBean>> CODE_LIST01;
    static {
        CODE_LIST01 = new HashMap<Locale, List<CodeBean>>();
        
        List<CodeBean> list;
        
        CodeBean b;
        // ja
        list = new ArrayList<CodeBean>();
        b = new CodeBean();
        b.setId("001");
        b.setName("ja_1");
        list.add(b);
        
        b = new CodeBean();
        b.setId("002");
        b.setName("ja_2");
        list.add(b);
        
        b = new CodeBean();
        b.setId("003");
        b.setName("ja_3");
        list.add(b);
        
        CODE_LIST01.put(Locale.JAPANESE, list);
        
        // en
        list = new ArrayList<CodeBean>();
        b = new CodeBean();
        b.setId("001");
        b.setName("en_1");
        list.add(b);
        
        b = new CodeBean();
        b.setId("002");
        b.setName("en_2");
        list.add(b);
        
        b = new CodeBean();
        b.setId("003");
        b.setName("en_3");
        list.add(b);
        
        CODE_LIST01.put(Locale.ENGLISH, list);
        
        // en_GB
        list = new ArrayList<CodeBean>();
        b = new CodeBean();
        b.setId("001");
        b.setName("en_GB_1");
        list.add(b);
        
        b = new CodeBean();
        b.setId("002");
        b.setName("en_GB_2");
        list.add(b);
        
        b = new CodeBean();
        b.setId("003");
        b.setName("en_GB_3");
        list.add(b);
        
        CODE_LIST01.put(Locale.UK, list);
        
        // en_GB_gb01
        list = new ArrayList<CodeBean>();
        b = new CodeBean();
        b.setId("001");
        b.setName("en_GB_gb01_1");
        list.add(b);
        
        b = new CodeBean();
        b.setId("002");
        b.setName("en_GB_gb01_2");
        list.add(b);
        
        b = new CodeBean();
        b.setId("003");
        b.setName("en_GB_gb01_3");
        list.add(b);
        
        CODE_LIST01.put(new Locale("en", "GB", "gb01"), list);
        
        // de_AT_at01
        list = new ArrayList<CodeBean>();
        b = new CodeBean();
        b.setId("001");
        b.setName("de_AT_at01_1");
        list.add(b);
        
        b = new CodeBean();
        b.setId("002");
        b.setName("de_AT_at01_2");
        list.add(b);
        
        b = new CodeBean();
        b.setId("003");
        b.setName("de_AT_at01_3");
        list.add(b);
        
        CODE_LIST01.put(new Locale("de", "AT", "at01"), list);
    }
    
    /**
     * ̃eXgP[Xsׂ GUI AvP[VNB
     * 
     * @param args
     *            java R}hɐݒ肳ꂽp[^
     */
    public static void main(String[] args) {
        junit.swingui.TestRunner.run(
                AbstractMultilingualCodeListLoaderTest.class);
    }

    /* (non-Javadoc)
     * @see junit.framework.TestCase#setUp()
     */
    @Override
    protected void setUp() throws Exception {
        super.setUp();
    }

    /* (non-Javadoc)
     * @see junit.framework.TestCase#tearDown()
     */
    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
    }
    
    /**
     * testDefaultLocale01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFȂ<br>
     *         
     * <br>
     * ҒlFVM݂̌̂̃ftHgP[
     *         
     * <br>
     * ftHgP[ݒ肵ȂꍇVM݂̌̂
     * ftHgP[ƂȂ邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testDefaultLocale01() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        assertEquals(new Locale(Locale.getDefault().getLanguage()), 
                loader.defaultLocale);
    }
    
    /**
     * testSetDefaultLocale01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnew Locale("ja", "JP", "tokyo")<br>
     *         
     * <br>
     * ҒlF͒lƓP[
     *         
     * <br>
     * Ŏw肵P[ݒ肳邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testSetDefaultLocale01() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        Locale locale = new Locale("ja", "JP", "tokyo");
        loader.setDefaultLocale(locale);
        assertEquals(locale, loader.defaultLocale);
    }
    
    /**
     * testSetDefaultLocale01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFLocale.getDefault()<br>
     *         
     * <br>
     * ҒlF0
     *         
     * <br>
     * ۉΉR[hXgnull̏ꍇA0̃R[hXg擾邱
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateCodeBeans01() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = null;
        CodeBean[] codeList = loader.createCodeBeans(Locale.getDefault());
        
        assertEquals(codeList.length, 0);
    }
    
    /**
     * testCreateCodeBeans02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFLocale.getDefault()<br>
     *         
     * <br>
     * ҒlF0
     *         
     * <br>
     * ۉΉR[hXgɑ݂ȂP[̃R[hXgw肵ꍇA
     * 0̃R[hXg擾邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateCodeBeans02() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        CodeBean[] codeList = loader.createCodeBeans(Locale.CHINESE);
        
        assertEquals(codeList.length, 0);
    }
    
    /**
     * testCreateCodeBeans03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnew Locale("en" , "GB", "gb01")<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'en_GB_gb01_XX'ł邱
     *         
     * <br>
     * ȉ̏ŃR[hXg擾ł邱ƂmFB<br>
     * ẼP[ɃoAg܂ŎwB<br>
     * ECODE_LIST01ɃoAg܂ňvR[hXg񂪑݂B
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateCodeBeans03() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        CodeBean[] codeList = 
            loader.createCodeBeans(new Locale("en" , "GB", "gb01"));
        
        assertEquals(3, codeList.length);
        for (int i = 0; i < codeList.length; i++) {
            CodeBean b = codeList[i];
            assertEquals(String.format("%1$03d", i + 1), b.getId());
            assertEquals("en_GB_gb01_" + (i + 1), b.getName());
        }
    }
    
    /**
     * testCreateCodeBeans04()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnew Locale("en" , "GB", "gb02")<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'en_GB_XX'ł邱
     *         
     * <br>
     * ȉ̏ŃR[hXg擾ł邱ƂmFB<br>
     * ẼP[ɃoAg܂ŎwB<br>
     * ECODE_LIST01ɃoAg܂ňvR[hXg񂪑݂ȂB
     * ECODE_LIST01ɍR[h܂ňvR[hXg񂪑݂B
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateCodeBeans04() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        CodeBean[] codeList = 
            loader.createCodeBeans(new Locale("en" , "GB", "gb02"));
        
        assertEquals(3, codeList.length);
        for (int i = 0; i < codeList.length; i++) {
            CodeBean b = codeList[i];
            assertEquals(String.format("%1$03d", i + 1), b.getId());
            assertEquals("en_GB_" + (i + 1), b.getName());
        }
    }
    
    /**
     * testCreateCodeBeans05()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnew Locale("en" , "US", "us01")<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'en_XX'ł邱
     *         
     * <br>
     * ȉ̏ŃR[hXg擾ł邱ƂmFB<br>
     * ẼP[ɃoAg܂ŎwB<br>
     * ECODE_LIST01ɍR[hAoAg܂ňvR[hXg񂪑݂ȂB
     * ECODE_LIST01ɌR[hvR[hXg񂪑݂B
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateCodeBeans05() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        CodeBean[] codeList = 
            loader.createCodeBeans(new Locale("en" , "US", "us01"));
        
        assertEquals(3, codeList.length);
        for (int i = 0; i < codeList.length; i++) {
            CodeBean b = codeList[i];
            assertEquals(String.format("%1$03d", i + 1), b.getId());
            assertEquals("en_" + (i + 1), b.getName());
        }
    }
    
    /**
     * testCreateCodeBeans06()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnull<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'ja_x'ł邱
     *         
     * <br>
     * null̏ꍇAftHgP[̃R[hXg擾邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateCodeBeans06() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.defaultLocale = Locale.JAPANESE;
        loader.localeMap = CODE_LIST01;
        CodeBean[] codeList = 
            loader.createCodeBeans(null);
        
        assertEquals(3, codeList.length);
        for (int i = 0; i < codeList.length; i++) {
            CodeBean b = codeList[i];
            assertEquals(String.format("%1$03d", i + 1), b.getId());
            assertEquals("ja_" + (i + 1), b.getName());
        }
    }
    
    /**
     * testCreateCodeBeans07()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnull<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'ja_x'ł邱
     *         
     * <br>
     * ftHgP[nullnull̏ꍇA
     * IllegalStateException邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateCodeBeans07() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.defaultLocale = null;
        loader.localeMap = CODE_LIST01;
        
        try {
            loader.createCodeBeans(null);
        } catch (IllegalStateException e) {
            assertTrue(true);
            return;
        }
        
        fail();
    }
    
    /**
     * testGetCodeBeansByLocale01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnew Locale("en" , "GB", "gb01")<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'en_GB_gb01_XX'ł邱
     *         
     * <br>
     * ȉ̏ŃR[hXg擾ł邱ƂmFB<br>
     * ECODE_LIST01ɃP[ɈvR[hXg񂪑݂B
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetCodeBeansByLocale01() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        loader.defaultLocale = Locale.JAPANESE;
        CodeBean[] codeList = 
            loader.getCodeBeans(new Locale("en" , "GB", "gb01"));
        
        assertEquals(3, codeList.length);
        for (int i = 0; i < codeList.length; i++) {
            CodeBean b = codeList[i];
            assertEquals(String.format("%1$03d", i + 1), b.getId());
            assertEquals("en_GB_gb01_" + (i + 1), b.getName());
        }
    }
    
    /**
     * testGetCodeBeansByLocale02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFLocale.CHINESE<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'ja_XX'ł邱
     *         
     * <br>
     * P[ɈvR[hXg݂ȂꍇAftHgP[
     * R[hXg擾邱ƊmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetCodeBeansByLocale02() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        loader.defaultLocale = Locale.JAPANESE;
        CodeBean[] codeList = 
            loader.getCodeBeans(Locale.CHINESE);
        
        assertEquals(3, codeList.length);
        for (int i = 0; i < codeList.length; i++) {
            CodeBean b = codeList[i];
            assertEquals(String.format("%1$03d", i + 1), b.getId());
            assertEquals("ja_" + (i + 1), b.getName());
        }
    }
    
    /**
     * testGetCodeBeansByLocale03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnew Locale("de", "AT", "at01")<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'de_AT_at01_XX'ł邱
     *         
     * <br>
     * P[ɈvR[hXg݂ȂꍇAftHgP[
     * R[hXg擾邱ƊmFB
     * ftHgP[̕ʂ̃P[̏ꍇB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetCodeBeansByLocale03() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        loader.defaultLocale = new Locale("de", "AT", "at01");
        CodeBean[] codeList = 
            loader.getCodeBeans(new Locale("de"));
        
        assertEquals(3, codeList.length);
        for (int i = 0; i < codeList.length; i++) {
            CodeBean b = codeList[i];
            assertEquals(String.format("%1$03d", i + 1), b.getId());
            assertEquals("de_AT_at01_" + (i + 1), b.getName());
        }
    }
    
    /**
     * testGetCodeBeansByLocale04()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFnew Locale(de", "AT", "XX")<br>
     *         
     * <br>
     * ҒlF 0B
     *         
     * <br>
     * ftHgP[ɂR[hXg`ĂȂꍇ
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetCodeBeansByLocale04() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        loader.defaultLocale = new Locale("de");
        CodeBean[] codeList = 
            loader.getCodeBeans(new Locale("de", "AT", "XX"));
        
        assertEquals(0, codeList.length);
    }
    
    
    /**
     * testGetCodeBeans01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FC
     * <br><br>
     * ͒lFȂ<br>
     *         
     * <br>
     * ҒlF 3B擾R[hXg̃R[hl'ja_XX'ł邱
     *         
     * <br>
     * P[ɈvR[hXg݂ȂꍇAftHgP[
     * R[hXg擾邱ƊmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetCodeBeans01() throws Exception {
        MultilingualCodeListLoaderImpl01 loader =
                                        new MultilingualCodeListLoaderImpl01();
        loader.localeMap = CODE_LIST01;
        loader.defaultLocale = Locale.JAPAN;
        CodeBean[] codeList = 
            loader.getCodeBeans();

        
        assertEquals(3, codeList.length);
        for (int i = 0; i < codeList.length; i++) {
            CodeBean b = codeList[i];
            assertEquals(String.format("%1$03d", i + 1), b.getId());
            assertEquals("ja_" + (i + 1), b.getName());
        }
    }
}
