// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.IO;
using TERASOLUNA.Fw.Web.Controller;
using TERASOLUNA.Fw.WebUT.Controller.Multipart;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;

namespace TERASOLUNA.Fw.WebUT.Controller
{
    ///<summary>
    /// MultipartFileItem のテストクラスです。
    ///</summary>
 
    [TestClass()]
    public class MultipartItemTest
    {
        private TestContext testContextInstance;

        private static readonly string FILE_PATH
            = @"MultipartItem.txt";

        private static readonly string BOUNDARY = "boundary";

        private static readonly Encoding ENCODING = new UTF8Encoding();

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数が全て正常な値の時、正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest01.txt";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=" + name);
                headerList.Add(
                    "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor accessor =
                    new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor(target);

                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(3, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; name=" + name, accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", accessor._headerList["content-type"]);
                Assert.AreEqual("MultipartItem", accessor._headerList["content-part"]);
                Assert.AreSame(ENCODING, accessor._encoding);

            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数boundaryがnullの時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string boundary = null;
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=ConstructorTest02.txt");
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(boundary, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (ArgumentNullException ex)
                {
                    // 成功
                    Assert.AreEqual("boundary", ex.ParamName);
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "boundary"));

        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数boundaryが空文字の時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string boundary = "";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=ConstructorTest03");
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(boundary, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (ArgumentException ex)
                {
                    // 成功
                    Assert.AreEqual(typeof(ArgumentException).Name, ex.GetType().Name);
                    Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "boundary"), ex.Message);
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(ArgumentException), "E_EMPTY_STRING", "boundary"));

        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数partStreamがnullの時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest04()
        {
            // 準備
            IDictionary<string, string> headerList = new Dictionary<string, string>();
            headerList.Add(
                "CONTENT-DISPOSITION", "form-data; name=ConstructorTest03");
            headerList.Add("content-type", "application/x-www-form-urlencoded");
            headerList.Add("content-part", "MultipartItem");
            Stream pastStream = null;

            try
            {
                // 失敗
                new MultipartItemStub01(BOUNDARY, pastStream, headerList, ENCODING);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                // 成功
                Assert.AreEqual("partStream", ex.ParamName);
            }
            catch (Exception ex)
            {
                // 失敗
                Assert.Fail("予期せぬエラーです。" + ex.ToString());
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "partStream"));

        }


        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListがnullの時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest05()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList = null;

                try
                {
                    // 失敗
                    new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (ArgumentNullException ex)
                {
                    // 成功
                    Assert.AreEqual("headerList", ex.ParamName);
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "headerList"));

        }


        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListの要素数が0の時、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest06()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException ex)
                {
                    // 成功
                    Assert.AreEqual(typeof(InvalidRequestException).Name, ex.GetType().Name);
                    Assert.AreEqual(MessageManager.GetMessage("E_NOT_FOUND_CONTENTDISPOSITION"), ex.Message);

                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(InvalidRequestException), "E_NOT_FOUND_CONTENTDISPOSITION"));

        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListの要素数が1の時、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest07()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest07";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=ConstructorTest07");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor accessor =
                    new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor(target);

                //検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(1, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; name=ConstructorTest07",
                    accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreSame(ENCODING, accessor._encoding);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListの要素に"Content-Disposition"が存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest08()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT", "form-data; name=ConstructorTest08");
                headerList.Add(
                    "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException ex)
                {
                    // 成功
                    Assert.AreEqual(typeof(InvalidRequestException).Name, ex.GetType().Name);
                    Assert.AreEqual(MessageManager.GetMessage("E_NOT_FOUND_CONTENTDISPOSITION"), ex.Message);

                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(InvalidRequestException), "E_NOT_FOUND_CONTENTDISPOSITION"));

        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListの要素に"Cotent-Disposition"(大文字)が存在し、nameが存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest09()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data;");
                headerList.Add(
                     "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException ex)
                {
                    // 成功
                    Assert.AreEqual(typeof(InvalidRequestException).Name, ex.GetType().Name);
                    Assert.AreEqual(MessageManager.GetMessage("E_NOT_FOUND_NAME"), ex.Message);

                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(InvalidRequestException), "E_NOT_FOUND_NAME"));
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)の値"name"(小文字+大文字)が定義されている時、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest10()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest10.txt";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; Name=" + name);
                headerList.Add(
                     "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor accessor =
                    new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor(target);


                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(3, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; Name=" + name, accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", accessor._headerList["content-type"]);
                Assert.AreEqual("MultipartItem", accessor._headerList["content-part"]);
                Assert.AreSame(ENCODING, accessor._encoding);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)の値"name"(大文字)が定義されている時、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest11()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest10.txt";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; NAME=" + name);
                headerList.Add(
                     "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor accessor =
                    new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor(target);

                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(3, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; NAME=" + name, accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", accessor._headerList["content-type"]);
                Assert.AreEqual("MultipartItem", accessor._headerList["content-part"]);
                Assert.AreSame(ENCODING, accessor._encoding);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)の値は"name"を含む文字列で定義されている時、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest12()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; names=ConstructorTest11");
                headerList.Add(
                    "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException ex)
                {
                    // 成功
                    Assert.AreEqual(typeof(InvalidRequestException).Name, ex.GetType().Name);
                    Assert.AreEqual(MessageManager.GetMessage("E_NOT_FOUND_NAME"), ex.Message);

                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(InvalidRequestException), "E_NOT_FOUND_NAME"));

        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)の値"name"が２つ定義されている時、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest13()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION",
                    "form-data; name=ConstructorTest12; name=ConstructorTest12");
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException ex)
                {
                    // 成功
                    Assert.AreEqual(typeof(InvalidRequestException).Name, ex.GetType().Name);
                    Assert.AreEqual(MessageManager.GetMessage("E_NOT_ACCEPTED_HEADER"), ex.Message);

                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(InvalidRequestException), "E_NOT_ACCEPTED_HEADER"));

        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)を含む別の文字列で定義されている時、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest14()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITIONS", "form-data; name=ConstructorTest13;");
                headerList.Add(
                    "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException ex)
                {
                    // 成功
                    Assert.AreEqual(typeof(InvalidRequestException).Name, ex.GetType().Name);
                    Assert.AreEqual(MessageManager.GetMessage("E_NOT_FOUND_CONTENTDISPOSITION"), ex.Message);

                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(InvalidRequestException), "E_NOT_FOUND_CONTENTDISPOSITION"));

        }


        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数encodingがnullの時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest15()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=ConstructorTest14");
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");
                Encoding encoding = null;

                try
                {
                    // 失敗
                    new MultipartItemStub01(BOUNDARY, partStream, headerList, encoding);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (ArgumentNullException ex)
                {
                    // 成功
                    Assert.AreEqual("encoding", ex.ParamName);
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(MultipartItem), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "encoding"));

        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数encodingがShift-JISの時、正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void ConstructorTest16()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest15";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=" + name);
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");
                Encoding encoding = Encoding.GetEncoding("SHIFT-JIS");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, encoding);
                TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor accessor =
                    new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_MultipartItemAccessor(target);

                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(3, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; name=" + name, accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", accessor._headerList["content-type"]);
                Assert.AreEqual("MultipartItem", accessor._headerList["content-part"]);
                Assert.AreSame(encoding, accessor._encoding);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// HeaderList のテスト
        ///</summary>
        ///<remarks>
        /// HeaderListにnullを設定し、
        /// HeaderListでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void HeaderListTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=HeaderList");
                MultipartItem target = new MultipartItemStub01(
                    BOUNDARY, partStream, headerList, ENCODING);
                target.HeaderList = null;

                // 実行 & 検証
                Assert.IsNull(target.HeaderList);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// HeaderList のテスト
        ///</summary>
        ///<remarks>
        /// HeaderListに要素数0を設定し、
        /// HeaderListで要素数0を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void HeaderListTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=HeaderList");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                headerList = new Dictionary<string, string>();
                target.HeaderList = headerList;

                // 実行 & 検証
                Assert.AreSame(headerList, target.HeaderList);
                Assert.AreEqual(0, target.HeaderList.Count);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// HeaderList のテスト
        ///</summary>
        ///<remarks>
        /// HeaderListに要素数1を設定し、
        /// HeaderListで要素数1を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void HeaderListTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=HeaderList");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                headerList = new Dictionary<string, string>();
                headerList.Add("key01", "value01");
                target.HeaderList = headerList;

                // 実行 & 検証
                Assert.AreSame(headerList, target.HeaderList);
                Assert.AreEqual(1, target.HeaderList.Count);
                Assert.AreEqual("value01", target.HeaderList["key01"]);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// HeaderList のテスト
        ///</summary>
        ///<remarks>
        /// HeaderListに要素数3を設定し、
        /// HeaderListで要素数3を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void HeaderListTest04()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=HeaderList");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                headerList = new Dictionary<string, string>();
                headerList.Add("key01", "value01");
                headerList.Add("key02", "value02");
                headerList.Add("key03", "value03");
                target.HeaderList = headerList;

                // 実行 & 検証
                Assert.AreSame(headerList, target.HeaderList);
                Assert.AreEqual(3, target.HeaderList.Count);
                Assert.AreEqual("value01", target.HeaderList["key01"]);
                Assert.AreEqual("value02", target.HeaderList["key02"]);
                Assert.AreEqual("value03", target.HeaderList["key03"]);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// Name のテスト
        ///</summary>
        ///<remarks>
        /// Nameにnullを設定し、
        /// Nameでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void NameTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Name");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Name = null;

                // 実行 & 検証
                Assert.IsNull(target.Name);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// Name のテスト
        ///</summary>
        ///<remarks>
        /// Nameに空文字を設定し、
        /// Nameで空文字を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void NameTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Name");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Name = "";

                // 実行 & 検証
                Assert.AreEqual("", target.Name);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// Name のテスト
        ///</summary>
        ///<remarks>
        /// Nameに任意の文字列"abcde"を設定し、
        /// Nameで任意の文字列"abcde"を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void NameTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Name");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Name = "abcde";

                // 実行 & 検証
                Assert.AreEqual("abcde", target.Name);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// Encoding のテスト
        ///</summary>
        ///<remarks>
        /// Encodingにnullを設定し、
        /// Encodingでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void EncodingTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Encoding");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Encoding = null;

                // 実行 & 検証
                Assert.IsNull(target.Encoding);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// Encoding のテスト
        ///</summary>
        ///<remarks>
        /// EncodingにUTF-8を設定し、
        /// EncodingでUTF-8を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void EncodingTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Encoding");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Encoding = new UTF8Encoding();

                // 実行 & 検証
                Assert.AreEqual(new UTF8Encoding(), target.Encoding);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// Encoding のテスト
        ///</summary>
        ///<remarks>
        /// EncodingにShift-JISを設定し、
        /// EncodingでShift-JISを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void EncodingTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Encoding");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Encoding = Encoding.GetEncoding("SHIFT-JIS");

                // 実行 & 検証
                Assert.AreEqual(Encoding.GetEncoding("SHIFT-JIS"), target.Encoding);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// PositionStart のテスト
        ///</summary>
        ///<remarks>
        /// PositionStartに0を設定し、
        /// PositionStartで0を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void PositionStartTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionStart");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionStart = 0;

                // 実行 & 検証
                long num = 0;
                Assert.AreEqual(num, target.PositionStart);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// PositionStart のテスト
        ///</summary>
        ///<remarks>
        /// PositionStartに100を設定し、
        /// PositionStartで100を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void PositionStartTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionStart");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionStart = 100;

                // 実行 & 検証
                long num = 100;
                Assert.AreEqual(num, target.PositionStart);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// PositionStart のテスト
        ///</summary>
        ///<remarks>
        /// PositionStartに-100を設定し、
        /// PositionStartで-100を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void PositionStartTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionStart");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionStart = -100;

                // 実行 & 検証
                long num = -100;
                Assert.AreEqual(num, target.PositionStart);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// PositionEnd のテスト
        ///</summary>
        ///<remarks>
        /// PositionEndに0を設定し、
        /// PositionEndで0を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void PositionEndTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionEnd");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionEnd = 0;

                // 実行 & 検証
                long num = 0;
                Assert.AreEqual(num, target.PositionEnd);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// PositionEnd のテスト
        ///</summary>
        ///<remarks>
        /// PositionEndに100を設定し、
        /// PositionEndで100を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void PositionEndTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionEnd");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionEnd = 100;

                // 実行 & 検証
                long num = 100;
                Assert.AreEqual(num, target.PositionEnd);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// PositionEnd のテスト
        ///</summary>
        ///<remarks>
        /// PositionEndに-100を設定し、
        /// PositionEndで-100を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void PositionEndTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionEnd");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionEnd = -100;

                // 実行 & 検証
                long num = -100;
                Assert.AreEqual(num, target.PositionEnd);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// Filename のテスト
        ///</summary>
        ///<remarks>
        /// Filenameにnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void FilenameTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Filename");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);

                // 実行 & 検証
                Assert.IsNull(target.Filename);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// OutputStream のテスト
        ///</summary>
        ///<remarks>
        /// OutputStreamにnullを設定し、
        /// OutputStreamでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void OutputStreamTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);

                // 試験実施及び判定
                Assert.IsNull(target.OutputStream);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }

        ///<summary>
        /// Text のテスト
        ///</summary>
        ///<remarks>
        /// Textにnullを設定し、
        /// Textでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Controller\Files\MultipartItem.txt")]
        public void TextTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);

                // 試験実施及び判定
                Assert.IsNull(target.Text);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartItem), LogLevel.INFO));

        }
    }
}
