// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Web.Configuration.ExceptionCode;
using System.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.WebUT.Configuration.ExceptionCode
{
   /// <summary>
    /// TERASOLUNA.Fw.Configuration.ExceptionCodeConfiguration のテスト クラスです。すべての
    /// TERASOLUNA.Fw.Configuration.ExceptionCodeConfiguration 単体テストをここに含めます
    /// </summary>

    [TestClass()]
    public class ExceptionCodeConfigurationTest
    {

        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        /// <summary>
        /// GetCode (Type) のテスト
        /// </summary>
        /// <TestTarget>GetCode (Type)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に設定ファイルに存在する例外が指定されていること
        /// </TestCondition>
        /// <TestAssertion>引数に対応するcode属性の設定情報が返却されること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetCode01()
        {
            Type type = typeof(ArgumentException);

            string expected = "E0001";
            string actual;

            actual = ExceptionCodeConfiguration.GetCode(type);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetCode (Type) のテスト
        /// </summary>
        /// <TestTarget>GetCode (Type)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に設定ファイルに存在しない例外が指定されていること
        /// </TestCondition>
        /// <TestAssertion>nullが返却されること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetCode02()
        {
            Type type = typeof(InvalidOperationException);

            string actual;

            actual = ExceptionCodeConfiguration.GetCode(type);

            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetCode (Type) のテスト
        /// </summary>
        /// <TestTarget>GetCode (Type)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数にnullが指定されていること
        /// </TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetCode03()
        {
            Type type = null;
            string actual;

            try
            {
                actual = ExceptionCodeConfiguration.GetCode(type);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("type", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                   typeof(ExceptionCodeConfiguration), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "type"));

        }

        /// <summary>
        /// GetCode (Type) のテスト
        /// </summary>
        /// <TestTarget>GetCode (Type)</TestTarget>
        /// <TestCondition>exceptionTypeが空文字設定の場合</TestCondition>
        /// <TestAssertion>例外が発生しないこと</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetCode04()
        {
            Type type = typeof(InvalidOperationException);

            string actual;

            actual = ExceptionCodeConfiguration.GetCode(type);

            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }
        
        /// <summary>
        /// GetCode (Type) のテスト
        /// </summary>
        /// <TestTarget>GetCode (Type)</TestTarget>
        /// <TestCondition>一致したexceptionTypeのcodeが空文字設定の場合</TestCondition>
        /// <TestAssertion></TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetCode05()
        {
            Type type = typeof(InvalidCastException);
            string actual;
            string expected = "";
            actual = ExceptionCodeConfiguration.GetCode(type);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetExceptinoCode (Exception) のテスト
        /// </summary>
        /// <TestTarget>GetExceptinoCode (Exception)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に設定ファイルに存在する例外が指定されていること
        /// 引数に設定ファイルにベースクラスの例外が設定されていない例外が指定されていること
        /// 引数に設定ファイルにサブクラスの例外が設定されていない例外が指定されていること
        /// </TestCondition>
        /// <TestAssertion>引数に対応するcode属性の設定情報が返却されること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetExceptionCode01()
        {
            Exception exception = new ConfigurationErrorsException();

            string expected = "E0003";
            string actual;

            actual = ExceptionCodeConfiguration.GetExceptionCode(exception);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetExceptinoCode (Exception) のテスト
        /// </summary>
        /// <TestTarget>GetExceptinoCode (Exception)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に設定ファイルに存在しない例外が指定されていること
        /// 引数に設定ファイルにベースクラスの例外が設定されていない例外が指定されていること
        /// 引数に設定ファイルにサブクラスの例外が設定されていない例外が指定されていること
        /// </TestCondition>
        /// <TestAssertion>nullが返却されること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetExceptionCode02()
        {
            Exception exception = new TypeLoadException();

            string expected = null;
            string actual;

            actual = ExceptionCodeConfiguration.GetExceptionCode(exception);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetExceptinoCode (Exception) のテスト
        /// </summary>
        /// <TestTarget>GetExceptinoCode (Exception)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に設定ファイルに存在しない例外が指定されていること
        /// 引数に設定ファイルにベースクラスの例外が設定されていない例外が指定されていること
        /// 引数に設定ファイルにサブクラスの例外が設定されている例外が指定されていること
        /// </TestCondition>
        /// <TestAssertion>nullが返却されること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetExceptionCode03()
        {
            Exception exception = new Exception();

            string expected = null;
            string actual;

            actual = ExceptionCodeConfiguration.GetExceptionCode(exception);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetExceptinoCode (Exception) のテスト
        /// </summary>
        /// <TestTarget>GetExceptinoCode (Exception)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に設定ファイルに存在する例外が指定されていること
        /// 引数に設定ファイルにベースクラスの例外が設定されている例外が指定されていること
        /// 引数に設定ファイルにサブクラスの例外が設定されていない例外が指定されていること
        /// </TestCondition>
        /// <TestAssertion>引数に対応するcode属性の設定情報が返却されること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetExceptionCode04()
        {
            Exception exception = new ArgumentNullException();

            string expected = "E0002";
            string actual;

            actual = ExceptionCodeConfiguration.GetExceptionCode(exception);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetExceptinoCode (Exception) のテスト
        /// </summary>
        /// <TestTarget>GetExceptinoCode (Exception)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に設定ファイルに存在しない例外が指定されていること
        /// 引数に設定ファイルにベースクラスの例外が設定されている例外が指定されていること
        /// 引数に設定ファイルにサブクラスの例外が設定されていない例外が指定されていること
        /// </TestCondition>
        /// <TestAssertion>引数のベースクラスに対応するcode属性の設定情報が返却されること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetExceptionCode05()
        {
            Exception exception = new ArgumentOutOfRangeException();

            string expected = "E0001";
            string actual;

            actual = ExceptionCodeConfiguration.GetExceptionCode(exception);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetExceptinoCode (Exception) のテスト
        /// </summary>
        /// <TestTarget>GetExceptinoCode (Exception)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数にnullを設定すること
        /// </TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal01.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal02.config")]
        [DeploymentItem(@"Configuration\ExceptionCode\TestItem_ExceptionCodeExternal11.config")]
        public void TestGetExceptionCode06()
        {
            Exception exception = null;

            string actual;

            try
            {
                actual = ExceptionCodeConfiguration.GetExceptionCode(exception);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("ex", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ExceptionCodeConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                   typeof(ExceptionCodeConfiguration), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "ex"));
        }
    }
}
