﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License. 

using System.Collections.Generic;
using System.IO;
using System.Text;

namespace TERASOLUNA.Fw.Web.Controller
{
    /// <summary>
    /// アップロードされたテキストを格納するための <see cref="IMultipartItem"/> です。
    /// </summary>
    public class MultipartTextItem : MultipartItem
    {
        /// <summary>
        /// アップロードデータを <seealso cref="string"/> 型にて保持します。
        /// </summary>
        private string _text = null;

        /// <summary>
        /// 格納されているデータがテキストかどうかを取得します。
        /// </summary>
        /// <value>
        /// 格納されているデータがテキストかどうかのフラグ。
        /// </value>
        /// <remarks>
        /// 必ず true を返却します。
        /// </remarks>
        public override bool IsText
        {
            get
            {
                return true;
            }
        }

        /// <summary>
        /// アップロードデータを <seealso cref="string"/> 型にて取得します。
        /// </summary>
        /// <value>
        /// コンストラクタで設定された値。
        /// </value>
        public override string Text
        {
            get 
            {
                return _text;
            }
        }

        /// <summary>
        /// <see cref="MultipartTextItem"/> クラスの新しいインスタンスを初期化します
        /// </summary>
        /// <param name="boundary">アップロードデータを区切るための文字列。</param>
        /// <param name="partStream">アップロードされたデータ。</param>
        /// <param name="headerList">アップロードデータのヘッダ情報。</param>
        /// <param name="encoding">テキストをエンコードするための <seealso cref="Encoding"/> 。</param>
        /// <exception cref="InvalidRequestException">
        /// ボディ部に終端文字列が存在しません。
        /// </exception>
        public MultipartTextItem(string boundary, 
                                 Stream partStream, 
                                 IDictionary<string, string> headerList, 
                                 Encoding encoding)
            : base(boundary, partStream, headerList, encoding)
        {
            // ボディ部の終わり(区切り文字列)を見つける。
            FindBody(boundary, partStream);
            _text = encoding.GetString(CreateBodyByte(partStream));
        }
        
        /// <summary>
        /// リソースを解放します。
        /// </summary>
        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Design", "CA1063:ImplementIDisposableCorrectly")]
        public override void Dispose()
        {
        }

        
        /// <summary>
        /// データを <seealso cref="Stream"/> から取り出し、 <seealso cref="byte"/> 型配列に格納します。
        /// </summary>
        /// <param name="partStream">アップロードされたデータ。</param>
        /// <returns>データを <seealso cref="byte"/> 型配列に格納した情報。</returns>
        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Design", "CA1062:ValidateArgumentsOfPublicMethods")]
        protected byte[] CreateBodyByte(Stream partStream)
        {
            int length = (int)(PositionEnd - PositionStart);
            byte[] textByte = new byte[length];

            // ボディ部をバイト配列に格納する。
            partStream.Position = PositionStart;
            partStream.Read(textByte, 0, length);
            // 次の区切り文字までPositionを進める。
            partStream.Position += CRLF_CODE.Length;
            return textByte;
        }
    }
}
