﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Configuration;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Common.Configuration
{
    /// <summary>
    /// file 要素のコレクションを格納する構成要素を表します。
    /// </summary>
    /// <remarks>
    /// <see cref="FilesCollection"/> は、アプリケーション構成ファイル内の file 要素のコレクションを表します。 
    /// 詳細については <see cref="ConfigurationElementCollection"/> を参照してください。
    /// </remarks>
    public class FilesCollection : ConfigurationElementCollection
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(FilesCollection));

        /// <summary>
        /// file 要素の要素名です。
        /// </summary>
        /// <remarks>
        /// この定数の値は、 "file" です。
        /// </remarks>
        private static readonly string FILE_ELEMENT_NAME = "file";

        /// <summary>
        /// <see cref="FilesCollection"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>デフォルトコンストラクタです。</remarks>
        public FilesCollection()
        {
        }

        /// <summary>
        /// <see cref="FilesCollection"/> の型を取得します。 
        /// </summary>
        /// <remarks>
        /// このコレクションの <see cref="ConfigurationElementCollectionType"/> は BasicMap です。
        /// </remarks>
        /// <value>
        /// このコレクションの <see cref="ConfigurationElementCollectionType"/> 。
        /// </value>
        public override ConfigurationElementCollectionType CollectionType
        {
            get
            {
                return ConfigurationElementCollectionType.BasicMap;
            }
        }

        /// <summary>
        /// 指定したインデックス位置にある file 要素を取得します。
        /// </summary>
        /// <param name="index">取得する <see cref="FileElement"/> のインデックス位置。</param>
        /// <returns>指定したインデックス位置にある <see cref="FileElement"/> 。</returns>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseGet(int)"/> メソッドを使用して、
        /// <see cref="FilesCollection"/> から <see cref="FileElement"/> を取得します。
        /// </remarks>
        public FileElement this[int index]
        {
            get
            {
                return (FileElement)BaseGet(index);
            }
        }

        /// <summary>
        /// 指定したキーを持つ file 要素を取得します。
        /// </summary>
        /// <param name="path">取得する file 要素のキー。</param>
        /// <returns>指定したキーを持つ <see cref="FileElement"/> 。それ以外の場合は null 参照です。</returns>
        /// <remarks>
        /// <seealso cref="ConfigurationElementCollection.BaseGet(object)"/> メソッドを使用して、 <see cref="FilesCollection"/> から <see cref="FileElement"/> を取得します。
        /// 指定したキーを持つ <see cref="FileElement"/> がコレクションに存在しない場合、null 参照を返します。
        /// </remarks>
        public new FileElement this[string path]
        {
            get
            {
                return (FileElement)BaseGet(path);
            }
        }

        /// <summary>
        /// <see cref="FilesCollection"/> の要素数を取得します。
        /// </summary>
        public new int Count
        {
            get
            {
                return base.Count;
            }
        }

        /// <summary>
        /// アプリケーション構成ファイル内のこの要素のコレクションを識別するために使用する名前を取得します。
        /// </summary>
        /// <remarks>
        /// デフォルト値は file です。
        /// </remarks>
        /// <value>
        /// FileCollection の要素名。
        /// </value>
        protected override string ElementName
        {
            get
            {
                return FILE_ELEMENT_NAME;
            }
        }

        /// <summary>
        /// 新しい file 要素を作成します。
        /// </summary>
        /// <returns>新しい file 要素。</returns>
        /// <remarks>
        /// 新しい <see cref="FileElement"/> クラスのインスタンスを作成します。
        /// </remarks>
        protected override ConfigurationElement CreateNewElement()
        {
            return new FileElement();
        }

        /// <summary>
        /// file 要素の要素キーを取得します。
        /// </summary>
        /// <param name="element">キーを返す <see cref="FileElement"/> 。</param>
        /// <returns>指定した <see cref="FileElement"/> のキーとして機能する <see cref="Object"/> 。</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="element"/> が null 参照です。
        /// </exception>
        /// <exception cref="ArgumentException">
        /// <paramref name="element"/> が <see cref="FileElement"/> にキャストできません。
        /// </exception>
        protected override Object GetElementKey(ConfigurationElement element)
        {
            if (element == null)
            {
                ArgumentNullException exception = new ArgumentNullException("element");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Common.Properties.Resources.E_NULL_ARGUMENT, "element"), exception);
                }
                throw exception;
            }

            FileElement fileElement = element as FileElement;
            if (fileElement == null)
            {
                string message = string.Format(Common.Properties.Resources.E_INVALID_CAST, element.GetType().FullName, typeof(FileElement).FullName);
                ArgumentException exception = new ArgumentException(message);
                if(_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            return fileElement.Path;
        }
    }
}
