/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.oxm.xsd.xerces;

import java.io.BufferedReader;
import java.io.StringReader;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Stack;

import jp.terasoluna.fw.oxm.xsd.exception.IllegalSchemaDefinitionException;
import jp.terasoluna.fw.oxm.xsd.exception.UnknownXMLDataException;
import jp.terasoluna.fw.oxm.xsd.message.ErrorMessage;
import jp.terasoluna.fw.oxm.xsd.message.ErrorMessages;
import jp.terasoluna.utlib.LogUTUtil;
import jp.terasoluna.utlib.UTUtil;
import junit.framework.TestCase;

import org.apache.xerces.impl.xs.util.SimpleLocator;
import org.apache.xerces.util.DefaultErrorHandler;
import org.apache.xerces.xni.XMLLocator;

/**
 * {@link jp.terasoluna.fw.oxm.xsd.xerces.XMLErrorReporterEx} NX̃ubN{bNXeXgB
 * 
 * <p>
 * <h4>yNX̊Tvz</h4>
 * XMLf[^̃p[X̏ڍׂȃG[ƂāAtB[h߁AXMLErrorReporter̊gsNXB<br>
 * EO<br>
 * reportError\bh̓p[TĂяo邱ƂOłAlocationAdomainAkeyAseveritynullɂȂ邱Ƃ͂ȂB<br>
 * addErrorMessage\bhreportError\bhĂяo邱ƂOłAkeyAoptionsAoptions[1]nullA󕶎ɂȂ邱Ƃ͂ȂBtagStackɂ͕Kȏ̗vfi[ĂB<br>
 * errorLog\bhreportError\bhĂяo邱ƂOłAkeyAoptionsnullɂȂ邱Ƃ͂ȂAtagStackɂ͕Kȏ̗vfi[ĂB<br>
 * indexResolve\bh́AXMLSchemaValidatorExstartElement܂endElementĂяo邱ƂOłAelementnullɂȂ邱Ƃ͂ȂB<br>
 * getKey\bh̃eXgP[X́AindexResolve\bh̃eXgP[XŖԗB<br>
 * getMessage\bh̃eXgP[X́AerrorLog\bhŖԗB
 * </p>
 * 
 * @see jp.terasoluna.fw.oxm.xsd.xerces.XMLErrorReporterEx
 */
public class XMLErrorReporterExTest extends TestCase {

    /**
     * ̃eXgP[Xsׂ GUI AvP[VNB
     * 
     * @param args java R}hɐݒ肳ꂽp[^
     */
    public static void main(String[] args) {
        junit.swingui.TestRunner.run(XMLErrorReporterExTest.class);
    }

    /**
     * XMLErrorReporterExCX^X
     */
    private XMLErrorReporterEx errorReporter = null;

    /**
     * sB
     * 
     * @throws Exception ̃\bhŔO
     * @see junit.framework.TestCase#setUp()
     */
    @Override
    protected void setUp() throws Exception {
        super.setUp();

        // XMLErrorReporterExCX^X
        ErrorMessages errorMessages = new ErrorMessages();
        this.errorReporter = new XMLErrorReporterEx(errorMessages);

    }

    /**
     * IsB
     * 
     * @throws Exception ̃\bhŔO
     * @see junit.framework.TestCase#tearDown()
     */
    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
     * RXgN^B
     * 
     * @param name ̃eXgP[X̖OB
     */
    public XMLErrorReporterExTest(String name) {
        super(name);
    }

    /**
     * testXMLErrorReporterEx01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() errorMessages:not null<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:not null<br>
     *         
     * <br>
     * ̃CX^Xɐݒ肳邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testXMLErrorReporterEx01() throws Exception {
        // O
        ErrorMessages errorMessages = new ErrorMessages();

        // eXg{
        XMLErrorReporterEx reporter = new XMLErrorReporterEx(errorMessages);

        // 
        assertSame(errorMessages, UTUtil.getPrivateField(reporter,
                "errorMessages"));
    }

    /**
     * testXMLErrorReporterEx02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,C
     * <br><br>
     * ͒lF() errorMessages:null<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:null<br>
     *         
     * <br>
     * ̃CX^Xɐݒ肳邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testXMLErrorReporterEx02() throws Exception {
        // O
        ErrorMessages errorMessages = null;

        // eXg{
        XMLErrorReporterEx reporter = new XMLErrorReporterEx(errorMessages);

        // 
        assertNull(UTUtil.getPrivateField(reporter, "errorMessages"));

    }

    /**
     * testGetErrorMessages01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() this.errorMessages:not null<br>
     *         
     * <br>
     * ҒlF(߂l) ErrorMessages:not null<br>
     *         
     * <br>
     * ̒lԋp邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetErrorMessages01() throws Exception {
        // O
        ErrorMessages errorMessages = new ErrorMessages();
        XMLErrorReporterEx reporter = new XMLErrorReporterEx(errorMessages);

        // eXg{
        ErrorMessages result = reporter.getErrorMessages();

        // 
        assertSame(errorMessages, result);

    }

    /**
     * testReportError01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() location:not null<br>
     *         () domain:not null<br>
     *         () key:not null<br>
     *         () arguments:̃IuWFNg<br>
     *         () severity:not null<br>
     *         () super.reportError():ĂяoꂽƂmFB<br>
     *                nꂽƂmFB<br>
     *         () errorLog():ĂяoꂽƂmFB<br>
     *         () addErrorMessage():ĂяoꂽƂmFB<br>
     *                nꂽƂmFB<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * arguments̏ꍇAsuper.reportError\bhAerrorLog\bhAaddErrorMessages\bhĂяoꂽƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testReportError01() throws Exception {
        // O
        ErrorMessages errorMessages = new ErrorMessages();
        XMLErrorReporterEx_XMLErrorReporterExStub01 reporter = new XMLErrorReporterEx_XMLErrorReporterExStub01(
                errorMessages);

        // super.reportError\bȟĂяomF̂߂̑O
        UTUtil.setPrivateField(reporter, "fErrorHandler", null);
        DefaultErrorHandler defaultErrorHandler = new DefaultErrorHandler();
        UTUtil.setPrivateField(reporter, "fDefaultErrorHandler",
                defaultErrorHandler);
        
        // 
        XMLLocator location = new SimpleLocator();
        String domain = "abc";
        String key = "def";
        Object[] arguments = new Object[] {};
        short severity = 123;

        // eXg{
        reporter.reportError(location, domain, key, arguments, severity);

        // 
        // super.reportError\bȟĂяomF
        assertSame(defaultErrorHandler, UTUtil
                .getPrivateField(reporter, "fDefaultErrorHandler"));

        // errorLog\bȟĂяomF
        assertTrue(reporter.isErrorLog);

        // addErrorMessage\bhɈnꂽƂ̊mF
        assertEquals("def", reporter.key);
        assertEquals(0, reporter.options.length);

    }

    /**
     * testReportError02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() location:not null<br>
     *         () domain:not null<br>
     *         () key:not null<br>
     *         () arguments:Object[0]{"xyz"}<br>
     *         () severity:not null<br>
     *         () super.reportError():ĂяoꂽƂmFB<br>
     *                nꂽƂmFB<br>
     *         () errorLog():ĂяoꂽƂmFB<br>
     *         () addErrorMessage():ĂяoꂽƂmFB<br>
     *                nꂽƂmFB<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * arguments̗vfP̏ꍇAsuper.reportError\bhAerrorLog\bhAaddErrorMessages\bhĂяoꂽƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testReportError02() throws Exception {
        // O
        ErrorMessages errorMessages = new ErrorMessages();
        XMLErrorReporterEx_XMLErrorReporterExStub01 reporter = new XMLErrorReporterEx_XMLErrorReporterExStub01(
                errorMessages);

        // super.reportError\bȟĂяomF̂߂̑O
        UTUtil.setPrivateField(reporter, "fErrorHandler", null);
        DefaultErrorHandler defaultErrorHandler = new DefaultErrorHandler();
        UTUtil.setPrivateField(reporter, "fDefaultErrorHandler",
                defaultErrorHandler);

        // 
        XMLLocator location = new SimpleLocator();
        String domain = "abc";
        String key = "def";
        Object[] arguments = new Object[] { "xyz" };
        short severity = 123;

        // eXg{
        reporter.reportError(location, domain, key, arguments, severity);

        // 
        // super.reportErrořĂяomF
        assertSame(defaultErrorHandler, UTUtil
                .getPrivateField(reporter, "fDefaultErrorHandler"));

        // errorLog\bȟĂяomF
        assertTrue(reporter.isErrorLog);

        // addErrorMessage\bhɈnꂽƂ̊mF
        assertEquals("def", reporter.key);
        assertEquals(1, reporter.options.length);
        assertEquals("xyz", reporter.options[0]);


    }

    /**
     * testReportError03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() location:not null<br>
     *         () domain:not null<br>
     *         () key:not null<br>
     *         () arguments:Object[0]{"xyz"}<br>
     *                Object[1]{null}<br>
     *                Object[1]{456}<br>
     *         () severity:not null<br>
     *         () super.reportError():ĂяoꂽƂmFB<br>
     *                nꂽƂmFB<br>
     *         () errorLog():ĂяoꂽƂmFB<br>
     *         () addErrorMessage():ĂяoꂽƂmFB<br>
     *                nꂽƂmFB<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * arguments̗vfR̏ꍇAsuper.reportError\bhAerrorLog\bhAaddErrorMessages\bhĂяoꂽƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testReportError03() throws Exception {
        // O
        ErrorMessages errorMessages = new ErrorMessages();
        XMLErrorReporterEx_XMLErrorReporterExStub01 reporter = new XMLErrorReporterEx_XMLErrorReporterExStub01(
                errorMessages);

        // super.reportError\bȟĂяomF̂߂̑O
        UTUtil.setPrivateField(reporter, "fErrorHandler", null);
        DefaultErrorHandler defaultErrorHandler = new DefaultErrorHandler();
        UTUtil.setPrivateField(reporter, "fDefaultErrorHandler",
                defaultErrorHandler);

        // 
        XMLLocator location = new SimpleLocator();
        String domain = "abc";
        String key = "def";
        Object[] arguments = new Object[] { "xyz", null, 456 };
        short severity = 123;

        // eXg{
        reporter.reportError(location, domain, key, arguments, severity);

        // 
        // super.reportErrořĂяomF        
        assertSame(defaultErrorHandler, UTUtil
                .getPrivateField(reporter, "fDefaultErrorHandler"));

        // errorLog\bȟĂяomF
        assertTrue(reporter.isErrorLog);

        // addErrorMessage\bhɈnꂽƂ̊mF
        assertEquals("def", reporter.key);
        assertEquals(3, reporter.options.length);
        assertEquals("xyz", reporter.options[0]);
        assertNull(reporter.options[1]);
        assertEquals("456", reporter.options[2]);

    }


    /**
     * testReportError01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() location:not null<br>
     *         () domain:not null<br>
     *         () key:not null<br>
     *         () arguments:null<br>
     *         () severity:not null<br>
     *         () super.reportError():ĂяoꂽƂmFB<br>
     *                nꂽƂmFB<br>
     *         () errorLog():ĂяoꂽƂmFB<br>
     *         () addErrorMessage():ĂяoꂽƂmFB<br>
     *                nꂽƂmFB<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * argumentsnull̏ꍇAsuper.reportError\bhAerrorLog\bhAaddErrorMessages\bhĂяoꂽƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testReportError04() throws Exception {
        // O
        ErrorMessages errorMessages = new ErrorMessages();
        XMLErrorReporterEx_XMLErrorReporterExStub01 reporter = new XMLErrorReporterEx_XMLErrorReporterExStub01(
                errorMessages);

        // super.reportError\bȟĂяomF̂߂̑O
        UTUtil.setPrivateField(reporter, "fErrorHandler", null);
        DefaultErrorHandler defaultErrorHandler = new DefaultErrorHandler();
        UTUtil.setPrivateField(reporter, "fDefaultErrorHandler",
                defaultErrorHandler);
        
        // 
        XMLLocator location = new SimpleLocator();
        String domain = "abc";
        String key = "def";
        Object[] arguments = null;
        short severity = 123;

        // eXg{
        reporter.reportError(location, domain, key, arguments, severity);

        // 
        // super.reportError\bȟĂяomF
        assertSame(defaultErrorHandler, UTUtil
                .getPrivateField(reporter, "fDefaultErrorHandler"));

        // errorLog\bȟĂяomF
        assertTrue(reporter.isErrorLog);

        // addErrorMessage\bhɈnꂽƂ̊mF
        assertEquals("def", reporter.key);
        assertEquals(0, reporter.options.length);

    }
    
    /**
     * testIndexResolve01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() element:"abc"<br>
     *         () this.tagIndex:HashMap<br>
     *         () this.tagStack:Stack<br>
     *         
     * <br>
     * ҒlF(߂l) String:"abc[0]"<br>
     *         (ԕω) this.tagIndex:HashMap{<br>
     *                      "abc"=0<br>
     *                    }<br>
     *         
     * <br>
     * getField()̖߂lAL[tagIndexɑ݂ȂꍇA<br>
     * 0̃CfbNXt񂪕ԋp邱ƂmFB<br>
     * tagIndexɃL[Al0̃Zbgi[邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testIndexResolve01() throws Exception {
        // O
        // 
        String element = "abc";

        // O
        HashMap<String, Integer> tagIndex = new HashMap<String, Integer>();
        UTUtil.setPrivateField(this.errorReporter, "tagIndex", tagIndex);
        Stack<String> stack = new Stack<String>();
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        String result = this.errorReporter.indexResolve(element);

        // 
        // ߂l
        assertEquals("abc[0]", result);
        
        // ԕω
        HashMap map = (HashMap) UTUtil.getPrivateField(this.errorReporter,
                "tagIndex");
        assertEquals(1, map.size());
        assertEquals(0, map.get("abc"));

    }

    /**
     * testIndexResolve02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,C
     * <br><br>
     * ͒lF() element:""<br>
     *         () this.tagIndex:HashMap<br>
     *         () this.tagStack:Stack<br>
     *         
     * <br>
     * ҒlF(߂l) String:"[0]"<br>
     *         (ԕω) this.tagIndex:HashMap{<br>
     *                      ""=0<br>
     *                    }<br>
     *         
     * <br>
     * getField()̖߂lAL[tagIndexɑ݂ȂꍇA<br>
     * 0̃CfbNXt񂪕ԋp邱ƂmFB<br>
     * tagIndexɃL[Al0̃Zbgi[邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testIndexResolve02() throws Exception {
        // O
        // 
        String element = "";

        // O
        HashMap<String, Integer> tagIndex = new HashMap<String, Integer>();
        UTUtil.setPrivateField(this.errorReporter, "tagIndex", tagIndex);
        Stack<String> stack = new Stack<String>();
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        String result = this.errorReporter.indexResolve(element);

        // 
        // ߂l
        assertEquals("[0]", result);
        
        // ԕω
        HashMap resultTagIndex = (HashMap) UTUtil.getPrivateField(this.errorReporter,
                "tagIndex");
        assertEquals(1, resultTagIndex.size());
        assertEquals(0, resultTagIndex.get(""));

    }

    /**
     * testIndexResolve03()
     * <br><br>
     * 
     * (n) 
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() element:"def"<br>
     *         () this.tagIndex:HashMap{<br>
     *                  "abc"=0,<br>
     *                  "abc[0].def"=0<br>
     *                }<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         
     * <br>
     * ҒlF(߂l) String:"def[1]"<br>
     *         (ԕω) this.tagIndex:HashMap{<br>
     *                      "abc"=0,<br>
     *                      "abc[0].def"=1<br>
     *                    }<br>
     *         
     * <br>
     * getField()̖߂lAL[tagIndexɑ݂ꍇA<br>
     * ɃCfbNXt񂪕ԋp邱ƂmFB<br>
     * ߂l̃CfbNXYL[̒lCNglł邱ƂmFB<br>
     * tagIndex̊YL[̒lCNg邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testIndexResolve03() throws Exception {
        // O
        // 
        String element = "def";

        // O
        HashMap<String, Integer> tagIndex = new HashMap<String, Integer>();
        tagIndex.put("abc", 0);
        tagIndex.put("abc[0].def", 0);
        UTUtil.setPrivateField(this.errorReporter, "tagIndex", tagIndex);
        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        String result = this.errorReporter.indexResolve(element);

        // 
        assertEquals("def[1]", result);
        HashMap map = (HashMap) UTUtil.getPrivateField(this.errorReporter,
                "tagIndex");
        assertEquals(2, map.size());
        assertEquals(0, map.get("abc"));
        assertEquals(1, map.get("abc[0].def"));

        HashMap resultTagIndex = (HashMap) UTUtil.getPrivateField(
                this.errorReporter, "tagIndex");
        assertEquals(2, resultTagIndex.size());
        assertEquals(0, resultTagIndex.get("abc"));
        assertEquals(1, resultTagIndex.get("abc[0].def"));

    }

    /**
     * testIndexResolve04()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() element:"ghi"<br>
     *         () this.tagIndex:HashMap{<br>
     *                  "abc"=0,<br>
     *                  "abc[0].def"=0<br>
     *                  "abc[0].def[0].ghi"=1<br>
     *                }<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *                Stack[1]{"def[0]"}<br>
     *         
     * <br>
     * ҒlF(߂l) String:"ghi[2]"<br>
     *         (ԕω) this.tagIndex:HashMap{<br>
     *                      "abc"=0,<br>
     *                      "abc[0].def"=0<br>
     *                      "abc[0].def[0].ghi"=2<br>
     *                    }<br>
     *         
     * <br>
     * getField()̖߂lAL[tagIndexɑ݂ꍇA<br>
     * ɃCfbNXt񂪕ԋp邱ƂmFB<br>
     * ߂l̃CfbNXYL[̒lCNglł邱ƂmFB <br>
     * tagIndex̊YL[̒lCNg邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testIndexResolve04() throws Exception {
        // O
        // 
        String element = "ghi";

        // O
        HashMap<String, Integer> tagIndex = new HashMap<String, Integer>();
        tagIndex.put("abc", 0);
        tagIndex.put("abc[0].def", 1);
        tagIndex.put("abc[0].def[0].ghi", 2);
        UTUtil.setPrivateField(this.errorReporter, "tagIndex", tagIndex);
        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        stack.push("def[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        String result = this.errorReporter.indexResolve(element);

        // 
        assertEquals("ghi[3]", result);
        HashMap map = (HashMap) UTUtil.getPrivateField(this.errorReporter,
                "tagIndex");
        assertEquals(3, map.size());
        assertEquals(0, map.get("abc"));
        assertEquals(1, map.get("abc[0].def"));
        assertEquals(3, map.get("abc[0].def[0].ghi"));

        HashMap resultTagIndex = (HashMap) UTUtil.getPrivateField(
                this.errorReporter, "tagIndex");
        assertEquals(3, resultTagIndex.size());
        assertEquals(0, resultTagIndex.get("abc"));
        assertEquals(1, resultTagIndex.get("abc[0].def"));
        assertEquals(3, resultTagIndex.get("abc[0].def[0].ghi"));
    
    }

    /**
     * testAddErrorMessage01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() key:"cvc-type.3.1.3"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"def"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:not null<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessagesCX^X<br>
     *         
     * <br>
     * key"cvc-type.3.1.3"̏ꍇAsȂƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage01() throws Exception {
        // O
        // 
        String key = "cvc-type.3.1.3";
        String[] options = new String[] { "abc", "def", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        assertFalse(resultErrorMessages.hasErrorMessage());

    }

    /**
     * testAddErrorMessage02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,C
     * <br><br>
     * ͒lF() key:"cvc-attribute.3"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"def"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:null<br>
     *         () this.tagStack:not null<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessagesCX^X<br>
     *         
     * <br>
     * key"cvc-attribute.3"̏ꍇAsȂƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage02() throws Exception {
        // O
        // 
        String key = "cvc-attribute.3";
        String[] options = new String[] { "abc", "def", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = null;
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        assertFalse(resultErrorMessages.hasErrorMessage());

    }

    /**
     * testAddErrorMessage03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() key:"cvc-attribute.3"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"def"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:ErrorMessageCX^X{<br>
     *                  field= abc[0]<br>
     *                }<br>
     *         () this.tagStack:not null<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessagesCX^X<br>
     *         (ԕω) this.tmpErrorMessage:ErrorMessageCX^X{<br>
     *                      field = "abc[0].def"<br>
     *                    }<br>
     *         
     * <br>
     * key"cvc-attribute.3"̏ꍇAsȂƂmFB<br>
     * tmpErrorMessageCX^XĂꍇAtmpErrorMessagefield̖options[1]̒lt邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage03() throws Exception {
        // O
        // 
        String key = "cvc-attribute.3";
        String[] options = new String[] { "abc", "def", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        // sĂȂiG[i[ĂȂjƂ̊mF
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        assertFalse(resultErrorMessages.hasErrorMessage());

        // tmpErrorMessage̊mF
        ErrorMessage resultTmpErrorMessage = (ErrorMessage) UTUtil
                .getPrivateField(this.errorReporter, "tmpErrorMessage");
        assertEquals("abc[0].def", resultTmpErrorMessage.getField());

    }

    /**
     * testAddErrorMessage04()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() key:"cvc-datatype-valid.1.2.1"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"boolean"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessageCX^X{<br>
     *                      key = "typeMismatch.boolean"<br>
     *                      field = "abc[0]"<br>
     *                      replaceValues[0]{"abc"}<br>
     *                      replaceValues[1]{"boolean"}<br>
     *                      replaceValues[2]{"ghi"}<br>
     *                    }<br>
     *         
     * <br>
     * key"cvc-datatype-valid.1.2.1"̏ꍇAkey"typeMismatch.boolean"AreplaceValuesɈoptionsi[ꂽErrorMessageCX^X邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage04() throws Exception {
        // O
        // 
        String key = "cvc-datatype-valid.1.2.1";
        String[] options = new String[] { "abc", "boolean", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        // ErrorMessageǉĂ邱Ƃ̊mF
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        List<ErrorMessage> resultErrorMessageList = resultErrorMessages
                .getErrorMessages();
        assertEquals(1, resultErrorMessageList.size());
        ErrorMessage resultErrorMessage = resultErrorMessageList.get(0);
        assertEquals("typeMismatch.boolean", resultErrorMessage.getKey());
        assertEquals("abc[0]", resultErrorMessage.getField());
        assertEquals(3, resultErrorMessage.getReplaceValues().length);
        assertEquals("abc", resultErrorMessage.getReplaceValues()[0]);
        assertEquals("boolean", resultErrorMessage.getReplaceValues()[1]);
        assertEquals("ghi", resultErrorMessage.getReplaceValues()[2]);

    }

    /**
     * testAddErrorMessage05()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() key:"cvc-datatype-valid.1.2.1"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"dateTime"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessageCX^X{<br>
     *                      key = "typeMismatch.date"<br>
     *                      field = "abc[0]"<br>
     *                      replaceValues[0]{"abc"}<br>
     *                      replaceValues[1]{"date"}<br>
     *                      replaceValues[2]{"ghi"}<br>
     *                    }<br>
     *         
     * <br>
     * key"cvc-datatype-valid.1.2.1"̏ꍇAkey"typeMismatch.date"AreplaceValuesɈoptionsi[ꂽErrorMessageCX^X邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage05() throws Exception {
        // O
        // 
        String key = "cvc-datatype-valid.1.2.1";
        String[] options = new String[] { "abc", "dateTime", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        // ErrorMessageǉĂ邱Ƃ̊mF
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        List<ErrorMessage> resultErrorMessageList = resultErrorMessages
                .getErrorMessages();
        assertEquals(1, resultErrorMessageList.size());
        ErrorMessage resultErrorMessage = resultErrorMessageList.get(0);
        assertEquals("typeMismatch.date", resultErrorMessage.getKey());
        assertEquals("abc[0]", resultErrorMessage.getField());
        assertEquals(3, resultErrorMessage.getReplaceValues().length);
        assertEquals("abc", resultErrorMessage.getReplaceValues()[0]);
        assertEquals("dateTime", resultErrorMessage.getReplaceValues()[1]);
        assertEquals("ghi", resultErrorMessage.getReplaceValues()[2]);

    }

    /**
     * testAddErrorMessage06()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() key:"cvc-datatype-valid.1.2.1"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"integer"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessageCX^X{<br>
     *                      key = "typeMismatch.number"<br>
     *                      field = "abc[0]"<br>
     *                      replaceValues[0]{"abc"}<br>
     *                      replaceValues[1]{"integer"}<br>
     *                      replaceValues[2]{"ghi"}<br>
     *                    }<br>
     *         
     * <br>
     * key"cvc-datatype-valid.1.2.1"̏ꍇAkey"typeMismatch.integer"AreplaceValuesɈoptionsi[ꂽErrorMessageCX^X邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage06() throws Exception {
        // O
        // 
        String key = "cvc-datatype-valid.1.2.1";
        String[] options = new String[] { "abc", "integer", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        // ErrorMessageǉĂ邱Ƃ̊mF
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        List<ErrorMessage> resultErrorMessageList = resultErrorMessages
                .getErrorMessages();
        assertEquals(1, resultErrorMessageList.size());
        ErrorMessage resultErrorMessage = resultErrorMessageList.get(0);
        assertEquals("typeMismatch.number", resultErrorMessage.getKey());
        assertEquals("abc[0]", resultErrorMessage.getField());
        assertEquals(3, resultErrorMessage.getReplaceValues().length);
        assertEquals("abc", resultErrorMessage.getReplaceValues()[0]);
        assertEquals("integer", resultErrorMessage.getReplaceValues()[1]);
        assertEquals("ghi", resultErrorMessage.getReplaceValues()[2]);

    }

    /**
     * testAddErrorMessage07()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() key:"cvc-datatype-valid.1.2.3"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"integerAllowEmpty"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessageCX^X{<br>
     *                      key = "typeMismatch.number"<br>
     *                      field = "abc[0]"<br>
     *                      replaceValues[0]{"abc"}<br>
     *                      replaceValues[1]{"integer"}<br>
     *                    }<br>
     *         
     * <br>
     * key"cvc-datatype-valid.1.2.3"Aoptions[1]"integerAllowEmpty"̏ꍇAkey"typeMismatch.integer"AreplaceValuesɈoptions̃CfbNX2ȍ~폜ꂽz񂪊i[ꂽAErrorMessageCX^X邱ƂmFB<br>
     * ErrorMessagesreplaceValuesž2ł邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage07() throws Exception {
        // O
        // 
        String key = "cvc-datatype-valid.1.2.3";
        String[] options = new String[] { "abc", "integerAllowEmpty", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        // ErrorMessageǉĂ邱Ƃ̊mF
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        List<ErrorMessage> resultErrorMessageList = resultErrorMessages
                .getErrorMessages();
        assertEquals(1, resultErrorMessageList.size());
        ErrorMessage resultErrorMessage = resultErrorMessageList.get(0);
        assertEquals("typeMismatch.number", resultErrorMessage.getKey());
        assertEquals("abc[0]", resultErrorMessage.getField());
        assertEquals(2, resultErrorMessage.getReplaceValues().length);
        assertEquals("abc", resultErrorMessage.getReplaceValues()[0]);
        assertEquals("integer", resultErrorMessage.getReplaceValues()[1]);

    }

    /**
     * testAddErrorMessage08()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() key:"cvc-maxInclusive-valid"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"integer"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessageCX^X{<br>
     *                      key = "typeMismatch.numberMaxRange"<br>
     *                      field = "abc[0]"<br>
     *                      replaceValues[0]{"abc"}<br>
     *                      replaceValues[1]{"integer"}<br>
     *                      replaceValues[2]{"ghi"}<br>
     *                    }<br>
     *         
     * <br>
     * key"cvc-maxInclusive-valid"̏ꍇAkey"typeMismatch.numberMaxRange"i[ꂽErrorMessageCX^X邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage08() throws Exception {
        // O
        // 
        String key = "cvc-maxInclusive-valid";
        String[] options = new String[] { "abc", "integer", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        // ErrorMessageǉĂ邱Ƃ̊mF
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        List<ErrorMessage> resultErrorMessageList = resultErrorMessages
                .getErrorMessages();
        assertEquals(1, resultErrorMessageList.size());
        ErrorMessage resultErrorMessage = resultErrorMessageList.get(0);
        assertEquals("typeMismatch.numberMaxRange", resultErrorMessage.getKey());
        assertEquals("abc[0]", resultErrorMessage.getField());
        assertEquals(3, resultErrorMessage.getReplaceValues().length);
        assertEquals("abc", resultErrorMessage.getReplaceValues()[0]);
        assertEquals("integer", resultErrorMessage.getReplaceValues()[1]);
        assertEquals("ghi", resultErrorMessage.getReplaceValues()[2]);

    }

    /**
     * testAddErrorMessage09()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() key:"cvc-minInclusive-valid"<br>
     *         () options:options[0]{"abc"}<br>
     *                options[1]{"integer"}<br>
     *                options[2]{"ghi"}<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) this.errorMessages:ErrorMessageCX^X{<br>
     *                      key = "typeMismatch.numberMinRange"<br>
     *                      field = "abc[0]"<br>
     *                      replaceValues[0]{"abc"}<br>
     *                      replaceValues[1]{"integer"}<br>
     *                      replaceValues[2]{"ghi"}<br>
     *                    }<br>
     *         
     * <br>
     * key"cvc-minInclusive-valid"̏ꍇAkey"typeMismatch.numberMinRange"i[ꂽErrorMessageCX^X邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage09() throws Exception {
        // O
        // 
        String key = "cvc-minInclusive-valid";
        String[] options = new String[] { "abc", "integer", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.addErrorMessage(key, options);

        // 
        // ErrorMessageǉĂ邱Ƃ̊mF
        ErrorMessages resultErrorMessages = 
            (ErrorMessages)UTUtil.getPrivateField(this.errorReporter, "errorMessages");
        List<ErrorMessage> resultErrorMessageList = resultErrorMessages
                .getErrorMessages();
        assertEquals(1, resultErrorMessageList.size());
        ErrorMessage resultErrorMessage = resultErrorMessageList.get(0);
        assertEquals("typeMismatch.numberMinRange", resultErrorMessage.getKey());
        assertEquals("abc[0]", resultErrorMessage.getField());
        assertEquals(3, resultErrorMessage.getReplaceValues().length);
        assertEquals("abc", resultErrorMessage.getReplaceValues()[0]);
        assertEquals("integer", resultErrorMessage.getReplaceValues()[1]);
        assertEquals("ghi", resultErrorMessage.getReplaceValues()[2]);

    }

    /**
     * testAddErrorMessage10()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,G
     * <br><br>
     * ͒lF() key:"cvc-abc"<br>
     *         () options:not null<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:not null<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) O:UnknownXMLDataException<br>
     *         (ԕω) Ox:Ox:error<br>
     *                    xml data is invalid.<br>
     *         
     * <br>
     * key̕"cvc-"n܂邪AL̕łȂꍇAOX[AOo͂邱ƂmFB<br>
     * Ecvc-datatype-valid.1.2.1<br>
     * Ecvc-datatype-valid.1.2.3<br>
     * Ecvc-maxInclusive-valid<br>
     * Ecvc-minInclusive-valid
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage10() throws Exception {
        // O
        // 
        String key = "cvc-abc";
        String[] options = new String[] { "abc", "integer", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        try {
            this.errorReporter.addErrorMessage(key, options);
            fail();
        } catch (UnknownXMLDataException e) {
            // 
            // OmF
            String message = "xml data is invalid.";
            assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testAddErrorMessage11()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,G
     * <br><br>
     * ͒lF() key:"abc"<br>
     *         () options:not null<br>
     *         () this.errorMessages:ErrorMessagesCX^X<br>
     *         () tmpErrorMessage:not null<br>
     *         () this.tagStack:not null<br>
     *         () this.DATATYPE_DATE:String[]{"date","time","dateTime"}<br>
     *         
     * <br>
     * ҒlF(ԕω) O:IllegalSchemaDefinitionException<br>
     *         (ԕω) Ox:Ox:error<br>
     *                    schema is invalid.<br>
     *         
     * <br>
     * key̕񂪃nhOG[R[hɊYȂꍇAOX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testAddErrorMessage11() throws Exception {
        // O
        // 
        String key = "abc";
        String[] options = new String[] { "abc", "integer", "ghi" };

        // O
        ErrorMessage tmpErrorMessage = new ErrorMessage("error", "abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tmpErrorMessage",
                tmpErrorMessage);

        Stack<String> stack = new Stack<String>();
        stack.push("abc[0]");
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        try {
            this.errorReporter.addErrorMessage(key, options);
            fail();
        } catch (IllegalSchemaDefinitionException e) {
            // 
            // OmF
            String message = "schema is invalid.";
            assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testErrorLog01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() key:cvc-type.3.1.3<br>
     *         () options:Object[]<br>
     *         () XMLSchemaMessages.properties:XMLSchemaMessages.propertiest@C擾ł<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         
     * <br>
     * ҒlF(ԕω) O:XL[}G[[]------------------------<br>
     *                    xpath=abc[0]<br>
     *                    getMessage=cvc-type.3.1.3: The value '00a1' of element abc' is not valid.<br>
     *                    key=cvc-type.3.1.3<br>
     *                    -----------------------------------------<br>
     *         
     * <br>
     * key,optionsƁAOtagStack̒lfꂽOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testErrorLog01() throws Exception {
        // O
        // 
        String key = "cvc-type.3.1.3";
        Object[] arg = new Object[] {};

        // O
        Stack<String> stack = new Stack<String>();
        String xpath = "abc[0]";
        stack.push(xpath);
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.errorLog(key, arg);

        // 
        CheckLogObj resultLog = getErrorLog();
        assertNotNull(resultLog);        
        assertEquals(xpath, resultLog.xpath);
        String message = "cvc-type.3.1.3: The value '{1}' of element '{0}' is not valid.";
        assertEquals(message, resultLog.message);
        assertEquals(key, resultLog.key);
        
    }

    /**
     * testErrorLog02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() key:cvc-type.3.1.3<br>
     *         () options:Object[0]{"123"}<br>
     *         () XMLSchemaMessages.properties:XMLSchemaMessages.propertiest@C擾ł<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         
     * <br>
     * ҒlF(ԕω) O:XL[}G[[]------------------------<br>
     *                    xpath=abc[0]<br>
     *                    getMessage=cvc-type.3.1.3: The value '123' of element '{1}' is not valid.<br>
     *                    key=cvc-type.3.1.3<br>
     *                    arg[0]=123<br>
     *                    -----------------------------------------<br>
     *         
     * <br>
     * key,optionsƁAOtagStack̒lfꂽOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testErrorLog02() throws Exception {
        // O
        // 
        String key = "cvc-type.3.1.3";
        Object[] arg = new Object[] { "123" };

        // O
        Stack<String> stack = new Stack<String>();
        String xpath = "abc[0]";
        stack.push(xpath);
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.errorLog(key, arg);

        // 
        CheckLogObj resultLog = getErrorLog();
        assertNotNull(resultLog);
        assertEquals(xpath, resultLog.xpath);
        String message = "cvc-type.3.1.3: The value '{1}' of element '123' is not valid.";
        assertEquals(message, resultLog.message);
        assertEquals(key, resultLog.key);
        assertEquals(arg[0], resultLog.arg0);

    }

    /**
     * testErrorLog03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() key:cvc-type.3.1.3<br>
     *         () options:Object[0]{"123"}<br>
     *                Object[1]{"456"}<br>
     *                Object[2]{"789"}<br>
     *         () XMLSchemaMessages.properties:XMLSchemaMessages.propertiest@C擾ł<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         
     * <br>
     * ҒlF(ԕω) O:XL[}G[[]------------------------<br>
     *                    xpath=abc[0]<br>
     *                    getMessage=cvc-type.3.1.3: The value '123' of element '456' is not valid.<br>
     *                    key=cvc-type.3.1.3<br>
     *                    arg[0]=123<br>
     *                    arg[1]=456<br>
     *                    arg[1]=789<br>
     *                    -----------------------------------------<br>
     *         
     * <br>
     * key,optionsƁAOtagStack̒lfꂽOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testErrorLog03() throws Exception {
        // O
        // 
        String key = "cvc-type.3.1.3";
        Object[] arg = new Object[] { "123", "456", "789" };

        // O
        Stack<String> stack = new Stack<String>();
        String xpath = "abc[0]";
        stack.push(xpath);
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.errorLog(key, arg);

        // 
        CheckLogObj resultLog = getErrorLog();
        assertNotNull(resultLog);                
        assertEquals(xpath, resultLog.xpath);
        String message = "cvc-type.3.1.3: The value '456' of element '123' is not valid.";
        assertEquals(message, resultLog.message);
        assertEquals(key, resultLog.key);
        assertEquals(arg[0], resultLog.arg0);
        assertEquals(arg[1], resultLog.arg1);
        assertEquals(arg[2], resultLog.arg2);

    }

    /**
     * testErrorLog04()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,C,G
     * <br><br>
     * ͒lF() key:݂ȂR[h<br>
     *         () options:Object[0]{null}<br>
     *         () XMLSchemaMessages.properties:XMLSchemaMessages.propertiest@C擾ł<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         
     * <br>
     * ҒlF(ԕω) O:XL[}G[[]------------------------<br>
     *                    xpath=abc[0]<br>
     *                    getMessage=Can't find resource for bundle java.util.PropertyResourceBundle, key abc<br>
     *                    key=abc<br>
     *                    arg[0]=123<br>
     *                    arg[1]=456<br>
     *                    arg[2]=789<br>
     *                    -----------------------------------------<br>
     *         
     * <br>
     * keyvpeBɑ݂ȂꍇAOɃvpeBȂƂbZ[Wo͂邱ƂmFB<br>
     * key,optionsƁAOtagStack̒lfꂽOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testErrorLog04() throws Exception {
        // O
        // 
        String key = "abc";
        Object[] arg = new Object[] {null};

        // O
        Stack<String> stack = new Stack<String>();
        String xpath = "abc[0]";
        stack.push(xpath);
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.errorLog(key, arg);

        // 
        CheckLogObj resultLog = getErrorLog();
        assertNotNull(resultLog);    
        assertEquals(xpath, resultLog.xpath);
        String message = "[[Can't find resource for bundle java.util.PropertyResourceBundle, key abc]]";
        assertEquals(message, resultLog.message);
        assertEquals(key, resultLog.key);
        assertEquals("null", resultLog.arg0);

    }

    /**
     * testErrorLog05()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,G
     * <br><br>
     * ͒lF() key:not null<br>
     *         () options:Object[0]{null}<br>
     *         () XMLSchemaMessages.properties:XMLSchemaMessages.propertiest@C擾łȂ<br>
     *         () this.tagStack:Stack[0]{"abc[0]"}<br>
     *         
     * <br>
     * ҒlF(ԕω) O:XL[}G[[]------------------------<br>
     *                    xpath=abc[0]<br>
     *                    getMessage=Can't find bundle for base name XMLSchemaMessages, locale ja_JP<br>
     *                    key=cvc-type.3.1.3<br>
     *                    arg[0]=123<br>
     *                    arg[1]=456<br>
     *                    arg[2]=789<br>
     *                    -----------------------------------------<br>
     *         
     * <br>
     * vpeBt@C擾łȂꍇAOɃ\[X擾łȂƂbZ[Wo͂邱ƂmFB<br>
     * key,optionsƁAOtagStack̒lfꂽOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testErrorLog05() throws Exception {
        
        // @̃eXgP[X͕svȂ߁A폜@
        // xercesŎgpvpeBt@CxercesŒ񋟂Ăjart@C
        // Ă邽߁AvpeBt@C擾łȂP[X݂͑ȂB
        
        
//        // O
//        // 
//        String key = "cvc-type.3.1.3";
//        Object[] arg = new Object[] { "123", "456", "789" };
//
//        // vpeBt@C̃t@CύXivpeBt@C̎擾s邽߁j
//        URL url = Thread.currentThread().getContextClassLoader().getResource(
//                "XMLSchemaMessages.properties");
//        File xercesProperties = new File(url.getFile());
//        File tmp = new File(url.getFile() + ".tmp");
//        boolean bol = xercesProperties.renameTo(tmp);
//
//        // ResourceBundlẽLbV폜ivpeBt@C̎擾s邽߁jB
//        ArrayList<Object> deleteCacheKeyList = new ArrayList<Object>();
//        SoftCache cache = (SoftCache) UTUtil.getPrivateField(
//                ResourceBundle.class, "cacheList");
//        for (Object cacheKey : cache.keySet()) {
//            if ("XMLSchemaMessages".equals((String) UTUtil.getPrivateField(
//                    cacheKey, "searchName"))
//                    || "XMLSchemaMessages_ja".equals((String) UTUtil
//                            .getPrivateField(cacheKey, "searchName"))
//                    || "XMLSchemaMessages_ja_JP".equals((String) UTUtil
//                            .getPrivateField(cacheKey, "searchName"))) {
//                deleteCacheKeyList.add(cacheKey);
//            }
//        }
//        for (Object deleteCacheKey : deleteCacheKeyList) {
//            cache.remove(deleteCacheKey);
//        }
//        
//        // O
//        String xpath = "abc[0]";
//        Stack<String> stack = new Stack<String>();
//        stack.push(xpath);
//        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);
//
//        // eXg{
//        this.errorReporter.errorLog(key, arg);
//        
//        // 
//        CheckLogObj resultLog = getErrorLog();
//        assertNotNull(resultLog);    
//        assertEquals(xpath, resultLog.xpath);
//        String message = "[[Can't find bundle for base name XMLSchemaMessages, locale ja_JP]]";
//        assertEquals(message, resultLog.message);
//        assertEquals(key, resultLog.key);
//        assertEquals(arg[0], resultLog.arg0);
//        assertEquals(arg[1], resultLog.arg1);
//        assertEquals(arg[2], resultLog.arg2);
//
//        // vpeBt@C̃t@Cɖ߂
//        tmp.renameTo(xercesProperties);
    }

    /**
     * testErrorLog06()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() key:cvc-type.3.1.3<br>
     *         () options:Object[]<br>
     *         () XMLSchemaMessages.properties:XMLSchemaMessages.propertiest@C擾ł<br>
     *         () this.tagStack:̃X^bN<br>
     *         
     * <br>
     * ҒlF(ԕω) O:XL[}G[[]------------------------<br>
     *                    xpath=<br>
     *                    getMessage=cvc-type.3.1.3: The value '00a1' of element abc' is not valid.<br>
     *                    key=cvc-type.3.1.3<br>
     *                    -----------------------------------------<br>
     *         
     * <br>
     * key,optionsƁAOtagStack̒lfꂽOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testErrorLog06() throws Exception {
        // O
        // 
        String key = "cvc-type.3.1.3";
        Object[] arg = new Object[] {};

        // O
        Stack<String> stack = new Stack<String>();
        UTUtil.setPrivateField(this.errorReporter, "tagStack", stack);

        // eXg{
        this.errorReporter.errorLog(key, arg);

        // 
        CheckLogObj resultLog = getErrorLog();
        assertNotNull(resultLog);        
        assertNull(resultLog.xpath);
        String message = "cvc-type.3.1.3: The value '{1}' of element '{0}' is not valid.";
        assertEquals(message, resultLog.message);
        assertEquals(key, resultLog.key);
        
    }

    /**
     * O̒g`FbN郁\bhB
     * <br>
     * Ǒ`sȏꍇAnulԋpB
     * <br>
     */
    private CheckLogObj getErrorLog() throws Exception{
        //LogUTUtil.checkError("");
        List logObject = (List) UTUtil.getPrivateField(LogUTUtil.class, "__logObjects");
        Class me = this.getClass();
        String me2 = this.getName();
        
        String logStr = null;
        boolean flag = false ;
        for(Object obj : logObject){
            Method method = (Method)UTUtil.getPrivateField(obj, "_calledMethod");
            if(me.equals(method.getClass()) || me2.equals(method.getName())){
                logStr = (String)UTUtil.getPrivateField(obj, "_message");
                flag = true;
                break;
            }
        }
        if(!flag){
            return null;
        }
        
        StringReader reader = new StringReader(logStr);
        BufferedReader b = new BufferedReader(reader);
        
        CheckLogObj co = new CheckLogObj();        
        String st = null;

        while((st = b.readLine()) != null){
            String[] str = st.split("=");
            if(str.length == 1){
               co.otherStr.add(str[0]);
            }else if (str.length == 2){
                if ("xpath".equals(str[0])) {
                    co.xpath = str[1];
                }
                else if ("getMessage".equals(str[0])) {
                    co.message = str[1];
                }
                else if ("key".equals(str[0])) {
                    co.key = str[1];
                }
                else if ("arg[0]".equals(str[0])) {
                    co.arg0 = str[1];
                }
                else if ("arg[1]".equals(str[0])) {
                    co.arg1 = str[1];
                }
                else if ("arg[2]".equals(str[0])) {
                    co.arg2 = str[1];
                }
                else{
                    return null;
                }
            }else{
                return null;
            }
        }
        return co;
    }

    /**
     * O͂ʂێIuWFNg
     *
     */
    private class CheckLogObj {
        ArrayList<String> otherStr = new ArrayList<String>();
        protected String xpath = null;
        protected String message = null;
        protected String key = null;
        protected String arg0 = null;
        protected String arg1 = null;
        protected String arg2 = null;
    }
    
    /**
     * testGetTagStack01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() this.tagStack:not null<br>
     *         
     * <br>
     * ҒlF(߂l) Stack:not null<br>
     *         
     * <br>
     * ̒lԋp邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetTagStack01() throws Exception {
        // O 
        Stack<String> tagStack = new Stack<String>();
        UTUtil.setPrivateField(this.errorReporter, "tagStack", tagStack);

        // eXg{
        Stack result = this.errorReporter.getTagStack();

        // 
        assertSame(tagStack, result);

    }

}
