package jp.terasoluna.toursample.server.interfaces.tour.ws;

import java.util.List;

import javax.jws.WebMethod;
import javax.jws.WebService;

import jp.terasoluna.toursample.server.application.tour.TourBizLogic;
import jp.terasoluna.toursample.server.interfaces.tour.assembler.DeptPlaceAssembler;
import jp.terasoluna.toursample.server.interfaces.tour.assembler.TourAssembler;
import jp.terasoluna.toursample.server.interfaces.tour.assembler.TourCategoryAssembler;
import jp.terasoluna.toursample.server.interfaces.tour.dto.DeptPlaceDto;
import jp.terasoluna.toursample.server.interfaces.tour.dto.TourCategoryDto;
import jp.terasoluna.toursample.server.interfaces.tour.dto.TourDto;
import jp.terasoluna.toursample.server.interfaces.tour.dto.TourSearchKeyDto;
import jp.terasoluna.toursample.server.interfaces.tour.dto.TourSummaryListDto;
import jp.terasoluna.toursample.server.interfaces.validation.SampleValidator;
import jp.terasoluna.toursample.server.interfaces.ws.exception.SampleSoapFaultException;
import jp.terasoluna.toursample.server.interfaces.ws.exception.handler.SampleSoapFaultExceptionHandler;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.context.support.SpringBeanAutowiringSupport;

/**
 *
 * 通常のWebサービスのサンプル SpringBeanAutowiringSupportを継承し作成する
 *
 */
@WebService(serviceName = "TourService")
public class TourService extends SpringBeanAutowiringSupport {

    @Autowired
    private TourBizLogic bizLogic;

    @Autowired
    private SampleSoapFaultExceptionHandler exceptionHandler;

    @Autowired
    @Qualifier("defaultValidator")
    private SampleValidator defaultValidator;
    @Autowired
    @Qualifier("addTourValidator")
    private SampleValidator addTourValidator;

    private TourAssembler tourAssembler = new TourAssembler();
    private TourCategoryAssembler categoryAssembler = new TourCategoryAssembler();
    private DeptPlaceAssembler deptPlaceAssembler = new DeptPlaceAssembler();

    /**
     * ツアー情報を登録する
     * @param dto
     * @throws SampleSoapFaultException
     */
    @WebMethod(operationName = "AddTour")
    public void addTour(TourDto dto) throws SampleSoapFaultException {
        try {
            // 入力チェック処理の実行
            addTourValidator.validate(dto);
            // ビジネスロジックの実行
            bizLogic.addTour(tourAssembler.fromDto(dto));
        } catch (Exception e) {
            // 例外が発生した場合に、必要に応じてSampleSoapFaultExceptionをスロー
            exceptionHandler.handleException(e);
        }
    }

    /**
     * ツアー情報を検索する
     * @param dto
     * @return
     * @throws SampleSoapFaultException
     */
    @WebMethod(operationName = "FindTour")
    public TourSummaryListDto findTour(TourSearchKeyDto dto)
            throws SampleSoapFaultException {
        try {
            // 入力チェック処理の実行
            defaultValidator.validate(dto);
            // ビジネスロジックの実行
            return tourAssembler.toSummaryListDto(bizLogic.findTourByName(dto
                    .getTourName()));
        } catch (Exception e) {
            // 例外が発生した場合に、必要に応じてSampleSoapFaultExceptionをスロー
            exceptionHandler.handleException(e);
        }
        return null;
    }

    /**
     * 出発地リストを取得する
     * @return
     * @throws SampleSoapFaultException
     */
    @WebMethod(operationName = "FindAllDeptPlaces")
    public List<DeptPlaceDto> findAllDeptPlaces()
            throws SampleSoapFaultException {
        try {
            // ビジネスロジックの実行
            return deptPlaceAssembler.toDtoList(bizLogic.findAllDeptPlaces());
        } catch (Exception e) {
            // 例外が発生した場合に、必要に応じてSampleSoapFaultExceptionをスロー
            exceptionHandler.handleException(e);
        }
        return null;
    }

    /**
     * ツアー分類リストを取得する
     * @return
     * @throws SampleSoapFaultException
     */
    @WebMethod(operationName = "FindAllTourCategories")
    public List<TourCategoryDto> findAllTourCategories()
            throws SampleSoapFaultException {
        try {
            // ビジネスロジックの実行
            return categoryAssembler
                    .toDtoList(bizLogic.findAllTourCategories());
        } catch (Exception e) {
            // 例外が発生した場合に、必要に応じてSampleSoapFaultExceptionをスロー
            exceptionHandler.handleException(e);
        }
        return null;
    }

}
