﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Terasoluna.Windows.ViewModel.Validation;
using System.ComponentModel;
using Terasoluna.Validation.Validators;
using Microsoft.Practices.EnterpriseLibrary.Validation.Validators;
using Microsoft.Practices.EnterpriseLibrary.Validation;
using System.Globalization;
using Terasoluna.TourSample.Client.Common.Code;
using Terasoluna.TourSample.Client.Properties;


namespace Terasoluna.TourSample.Client.A01.ViewData
{
    /// <summary>
    /// ツアー情報の画面データクラス（ネスト）
    /// </summary>
    [HasSelfValidation] 
    public class SC_A01_01_01Tour : ValidatableNestedViewData
    {
        public SC_A01_01_01Tour()
        {
            //クリアボタン押下時に、ラジオボタンを初期化
            TourType = TourTypeEnum.Unknown.ToString();
            //クリアボタン押下時に、チェックボタンを初期化
            HasOptionalTour = false.ToString();
        }

        [DisplayName("ツアーコード")]
        [RequiredValidator(Tag = "ツアーコード", Ruleset = "RS01")]
        [NumericStringValidator(Tag = "ツアーコード", Ruleset = "RS01")]
        ///固定文字列長チェックは、デフォルトメッセージがないためメッセージを設定
        [StringLengthRangeValidator(Tag = "ツアーコード", Ruleset = "RS01",
           LowerBound = 7, LowerBoundType = RangeBoundaryType.Inclusive,
           UpperBound = 7, UpperBoundType = RangeBoundaryType.Inclusive,
           MessageTemplateResourceType = typeof(Terasoluna.TourSample.Client.Common.Properties.Resources),
           MessageTemplateResourceName = "ERROR_COMMON_0001")]
        public virtual string TourCode { get; set; }

        [DisplayName("ツアー区分")]
        public virtual string TourType { get; set; }

        [DisplayName("交通機関")]
        public virtual string Transportation { get; set; }

 
        [DisplayName("ツアー名称")]
        [RequiredValidator(Tag = "ツアー名称", Ruleset = "RS01")]
        ///TextBox.MaxLengthで制御するのでコメントアウト
        ///[StringLengthRangeValidator(Tag = "ツアー名称", Ruleset = "RS01",
        ///    UpperBound = 30, UpperBoundType = RangeBoundaryType.Inclusive)]
        public virtual string TourName { get; set; }

        /// 出発地    
        public virtual ValidatableNestedViewDataList<SC_A01_01_01DeptPlace> DeptPlaceList { get; set; }

        [DisplayName("目的地")]
        [RequiredValidator(Tag = "目的地", Ruleset = "RS01")]
        ///TextBox.MaxLengthで制御するのでコメントアウト
        ///[StringLengthRangeValidator(Tag = "目的地", Ruleset = "RS01",
        ///    UpperBound = 30, UpperBoundType = RangeBoundaryType.Inclusive)]
        public virtual string DestName { get; set; }

        [DisplayName("出発日")]
        [DateTimeFormatValidator("yyyy/MM/dd", Tag = "出発日", Ruleset = "RS01")]
        public virtual string DeptDate { get; set; }

        [DisplayName("ツアー日数")]
        [IntRangeValidator(Tag = "ツアー日数", Ruleset = "RS01",
            LowerBound = 1, LowerBoundType = RangeBoundaryType.Inclusive,
            UpperBound = 99, UpperBoundType = RangeBoundaryType.Inclusive)]
        public virtual string TourDay { get; set; }
        
        [DisplayName("オプショナルツアー")]
        public virtual string HasOptionalTour { get; set; }

        /// ホテルランク
        public virtual SC_A01_01_01HotelRank HotelRank { get; set; }

        /// ツアーカテゴリ
        public virtual ValidatableNestedViewDataList<SC_A01_01_01TourCategory> TourCategoryList { get; set; }


        /// <summary>
        /// カスタム入力チェックの例
        /// </summary>
        [SelfValidation(Ruleset = "RS01")]
        private void CustomValidate01(ValidationResults results)
        {
            bool deptDateInputed = !string.IsNullOrEmpty(DeptDate);
            bool tourDayInputed = !string.IsNullOrEmpty(TourDay);
            if (deptDateInputed ^ tourDayInputed)
            {
                ///出発地とツアー日数のどちらが入力されていたらもう片方も入力されていない場合エラー
                ///出発地とツアー日数のフィールドの両方にエラー表示するように、ValidationResultとそれぞれ追加
                results.AddResult(
                   new ValidationResult(Resources.ERROR_A01_01_0001, this, "DeptDate", null, EventSpecificValidator.DefaultInstance));
                results.AddResult(
                    new ValidationResult(Resources.ERROR_A01_01_0001, this, "TourDay", null, EventSpecificValidator.DefaultInstance));
            }
        }

        /// <summary>
        /// カスタム入力チェックの例
        /// </summary>
        [SelfValidation(Ruleset = "RS01")]
        private void CustomValidate02(ValidationResults results)
        {
            if (string.IsNullOrEmpty(DeptDate))
            {
                ///未入力時はチェックしない
                return;
            }
            DateTime target = new DateTime();
            bool canParse = DateTime.TryParseExact(DeptDate, "yyyy/MM/dd", CultureInfo.CurrentCulture,
                DateTimeStyles.None, out target);
            if (canParse && target.CompareTo(DateTime.Today) < 0)
            {
                ///出発日付が今日の日付より前の場合はエラー
                results.AddResult(
                    new ValidationResult(Resources.ERROR_A01_01_0002, this, "DeptDate", null, EventSpecificValidator.DefaultInstance));
            }
        }

        /// <summary>
        /// カスタム入力チェックの例
        /// <param name="results"></param>
        [SelfValidation(Ruleset = "RS01")]
        private void CustomValidate03(ValidationResults results)
        {
            if (DeptPlaceList.Count < 1)
            {
                ///出発地が1件以上でない場合はエラー
                results.AddResult(
                    new ValidationResult(Resources.ERROR_A01_01_0003, this, "DeptPlaceList", null, EventSpecificValidator.DefaultInstance));
            }
        }

    }
}
