/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.standard;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import jp.terasoluna.fw.batch.core.JobException;
import jp.terasoluna.fw.batch.core.JobStatus;
import jp.terasoluna.fw.batch.core.QueueingException;
import jp.terasoluna.fw.batch.core.WorkQueue;
import jp.terasoluna.fw.batch.core.WorkUnit;
import jp.terasoluna.fw.batch.openapi.JobContext;

/**
 * <code>WorkQueue</code> C^tF[X̕WNXB
 * 
 * <p>̎ł́AJDK <code>BLockingQueue</code> ɂăL[CO
 * B</p>
 * 
 * <p>L[ւ̗vfǉAL[̗vf擾̍ۂɂ́AubN̂ł
 * ȂA^CAEgԂsƂ <code>JobStatus</code> `FbN
 * <code>QueueStoppedException</code> OX[B
 * L[ւ̗vfǉAL[̗vf擾̍ۂɁAXbhC^vgꂽ
 * ۂɂ<code>QueueStoppedException</code> OX[B
 * </p>
 *
 */
public class StandardWorkQueue implements WorkQueue {

    /**
     * L[`FbNۂ̃^CAEgBi~bj
     */
    private int queueCheckTimeout = 10000;
    
    /**
     * L[B
     */
    private BlockingQueue<WorkUnit> queue;

    /**
     * WuXe[^XB
     */
    private JobStatus jobStatus;

    /**
     * L[̗vf̏ʂ̃XgB
     */
    private List<Future< ? >> futureResultList = new ArrayList<Future< ? >>();

    /**
     * L[̏I[}[NB
     */
    public static final WorkUnit END_MARK = new WorkUnit() {
        public boolean isEndMark() {
            return true;
        }

        public JobContext getJobContext() {
            throw new UnsupportedOperationException();
        }

        public void setJobContext(JobContext jobContext) {
            throw new UnsupportedOperationException();
        }
    };

    /**
     * RXgN^B
     *
     * @param queueLength L[̒
     * @param jobStatus WuXe[^X
     */
    public StandardWorkQueue(int queueLength, JobStatus jobStatus) {
        queue = new ArrayBlockingQueue<WorkUnit>(queueLength);
        this.jobStatus = jobStatus;
    }

    /**
     * Ώۃf[^L[ɒǉB
     *
     * @param element L[ɒǉvf
     */
    public void put(WorkUnit element) {
        try {
            while (!queue.offer(element, queueCheckTimeout, 
                    TimeUnit.MILLISECONDS)) {
                checkJobStatus();
            }
        } catch (InterruptedException e) {
            throw new QueueingException(e);
        }
    }

    /**
     * WuXe[^X`FbNB
     *
     * <p>Vbg_EvĂƂɂ́A<code>JobStatus<code>
     * Vbg_EɍXVŁA<code>QueueingStoppedException</code>
     * OX[B</p>
     * 
     * <p>ɁAWuXe[^X <code>JobStatus.STATE.STARTED</code>A
     * 邢 <code>JobStatus.STATE.RESTARTED</code>
     * ̂ǂłȂƂɂ́A<code>QueueingStoppedException</code>
     * OX[B</p>
     */
    private void checkJobStatus() {
        if (jobStatus.isShutdownGraceful() || jobStatus.isShutdownImmediate()) {
            jobStatus.suspend();
            throw new QueueingException();
        }
        if (!jobStatus.isExecuting()) {
            throw new QueueingException();
        }
    }

    /**
     * L[IB
     * 
     * <p>L[̏I[łCX^XL[ɒǉB</p>
     */
    public void close() {
        put(END_MARK);
    }

    /**
     * L[̗vf擾B
     *
     * @return L[̗vf
     */
    public WorkUnit take() {

        //SXbhҋ@ԎAEND_MARKtakeꂽꍇ
        //END_MARKqueue폜邽ߍēKv
        try {
            while (true) {
                WorkUnit workUnit = 
                    queue.poll(queueCheckTimeout, TimeUnit.MILLISECONDS);
                if (workUnit == END_MARK) {
                    put(END_MARK);
                    return END_MARK;
                }
                if (workUnit != null) {
                    return workUnit;
                }
                checkJobStatus();
            }
        } catch (InterruptedException e) {
            throw new QueueingException(e);
        }
    }

    /**
     * L[̗vf邷ׂẴ[J[̏I҂킹B
     */
    public void waitForAllWorkers() {
        for (Future< ? > futureResult : futureResultList) {
            try {
                futureResult.get();
            } catch (ExecutionException e) {
                throw new JobException(e);
            } catch (InterruptedException e) {
                throw new JobException(e);
            }
        }
    }
    
    /**
     * [J̏ʂɔ񓯊ŃANZX邽߂̃t[`[ǉB
     * 
     * @param futureResult [J̏ʂɔ񓯊ŃANZX邽߂̃t[`
     * [
     */
    public void addFutureJobResult(Future< ? > futureResult) {
        futureResultList.add(futureResult);
    }

    /**
     * ^CAEgݒ肷B
     * 
     * @param timeout ^CAEg(~b)
     */
    public void setQueueCheckTimeout(int timeout) {
        queueCheckTimeout = timeout;
    }
}
