/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.standard;

import java.util.LinkedHashMap;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import jp.terasoluna.fw.batch.core.ExceptionHandlerUtil;
import jp.terasoluna.fw.batch.core.JobException;
import jp.terasoluna.fw.batch.core.JobExceptionHandler;
import jp.terasoluna.fw.batch.core.JobStatus;
import jp.terasoluna.fw.batch.core.SupportLogicException;
import jp.terasoluna.fw.batch.core.SupportLogicResultHandler;
import jp.terasoluna.fw.batch.core.SupportProcessor;
import jp.terasoluna.fw.batch.core.ThrowableHandler;
import jp.terasoluna.fw.batch.openapi.BLogicResult;
import jp.terasoluna.fw.batch.openapi.JobContext;
import jp.terasoluna.fw.batch.openapi.SupportLogic;


/**
 * <code>SupportProcessor</code> C^tF[X̕WNXB
 *
 * <p>T|[gWbÑXgɃT|[gWbNNB
 * NT|[gWbN <code>ERROR_END</code>A邢 
 * <code>NORMAL_END</code>ԋpꍇɂ͂̎_ŏ𒆎~B</p>
 * 
 * <p><code>canSkip()</code>  <code>true</code> ԂƂɂ́AT|[gWb
 * N͋NȂB܂Aœnꂽ <code>jobStatus</code> pԂłȂ
 * ꍇiIԂłꍇjɂ́AT|[gWbNNɒɃ^[
 * B</p>
 * 
 */
public class StandardSupportProcessor implements SupportProcessor {

    /**
     * OCX^XB
     */
    private static Log log = LogFactory.getLog(StandardSupportProcessor.class);

    /**
     * T|[gWbÑXgB
     */
    private List<SupportLogic<JobContext>> supportLogicList = null;
    
    /**
     * T|[gWbN̏ʃnhB
     */
    private SupportLogicResultHandler supportLogicResultHandler = null;

    /**
     * Onh}bvB
     */
    private LinkedHashMap<JobException, JobExceptionHandler> 
        exceptionHandlerMap = null;

    /**
     * ftHgOnhB
     */
    private JobExceptionHandler defaultJobExceptionHandler = null;

    /**
     * ThrowablenhB
     */
    private ThrowableHandler throwableHandler = new ThrowBatchSystemErrorThrowableHandler();

    /**
     * B
     */
    private String supportProcessorName = null;

    /**
     * T|[gsB
     * 
     * <p>ɐݒ肳ꂽT|[gWbÑXgsB</p>
     * 
     * @param jobContext WuReNXg
     * @param jobStatus WuXe[^X
     */
    public void process(JobContext jobContext, JobStatus jobStatus) {
        if (canSkip()) {
            return;
        }
        
        if (!jobStatus.isContinue()) {
            return;
        }
        
        BLogicResult supportLogicResult = null;
        try {
            for (SupportLogic<JobContext> supportLogic : supportLogicList) {
                // ȈꍇSUSPEND
                if (jobStatus.isShutdownImmediate()) {
                    jobStatus.suspend();
                    return;
                }

                try {
                    supportLogicResult = supportLogic.execute(jobContext);
                    supportLogicResultHandler.handle(supportLogicResult,
                            jobStatus, supportProcessorName);
                } catch (RuntimeException e) {
                    // PreLogicŔȌ
                    SupportLogicException wrappingException = createSupportLogicException(
                            supportLogicResult, e);

                    JobExceptionHandler handler = ExceptionHandlerUtil
                            .getJobExceptionHandler(wrappingException,
                                    exceptionHandlerMap,
                                    defaultJobExceptionHandler);

                    try {
                        handler.handlException(jobContext, wrappingException,
                                jobStatus);
                    } catch (RuntimeException handlerException) {
                        log.error(
                                "[Exception in ExceptionHandler] Error JobID: "
                                        + jobStatus.getJobId(),
                                handlerException);
                        jobStatus
                                .setJobState(JobStatus.STATE.ENDING_ABNORMALLY);
                    } 
                }

                if (!jobStatus.isContinue()) {
                    break;
                }
            }
        } catch (Throwable throwable) {
            throwableHandler.handle(jobContext, throwable, jobStatus);
        }
    }

    /**
     * T|[gO𐶐B
     * @param supportLogicResult T|[g
     * @param e O
     * @return T|[gOCX^X
     */
    private SupportLogicException createSupportLogicException(
        BLogicResult supportLogicResult, RuntimeException e) {
        SupportLogicException wrappingException = new SupportLogicException(
                e, supportLogicResult, supportProcessorName);
        return wrappingException;
    }

    /**
     * T|[gXLbvłꍇɂ́A<code>true</code> ԂB 
     * 
     * @return T|[gWbNЂƂݒ肳ĂȂꍇɂ
     *  <code>true</code>
     */
    public boolean canSkip() {
        return supportLogicList == null || supportLogicList.size() == 0;
    }

    /**
     * T|[gWbÑXgݒ肷B
     * 
     * @param supportLogicList T|[gWbÑXg
     */
    public void setSupportLogicList(
            List<SupportLogic<JobContext>> supportLogicList) {
        this.supportLogicList = supportLogicList;
    }

    /**
     * ftHgOnhݒ肷B
     * 
     * @param defaultJobExceptionHandler ftHgOnh
     */
    public void setDefaultJobExceptionHandler(
            JobExceptionHandler defaultJobExceptionHandler) {
        this.defaultJobExceptionHandler = defaultJobExceptionHandler;
    }

    /**
     * Onh}bvݒ肷B
     * 
     * @param exceptionHandlerMap Onh}bv
     */
    public void setExceptionHandlerMap(
            LinkedHashMap<JobException,
            JobExceptionHandler> exceptionHandlerMap) {
        this.exceptionHandlerMap = exceptionHandlerMap;
    }

    /**
     * T|[gWbN̏ʃnhݒ肷B
     * 
     * @param supportLogicResultHandler T|[gWbN̏ʃnh
     */
    public void setSupportLogicResultHandler(
            SupportLogicResultHandler supportLogicResultHandler) {
        this.supportLogicResultHandler = supportLogicResultHandler;
    }

    /**
     * ݒ肷B
     * 
     * @param supportProcessorName 
     */
    public void setSupportProcessorName(String supportProcessorName) {
        this.supportProcessorName = supportProcessorName;
    }

    /**
     * Throwablenhݒ肷B
     * 
     * @param throwableHandler Throwablenh
     */
    public void setThrowableHandler(ThrowableHandler throwableHandler) {
        this.throwableHandler = throwableHandler;
    }

}
