// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections;
using System.Collections.Generic;
using System.Configuration;
using System.Reflection;
using System.Web;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.BLogic;
using TERASOLUNA.Fw.Common.Configuration.BLogic;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Web.Configuration.RequestController;
using TERASOLUNA.Fw.Web.Validation;

namespace TERASOLUNA.Fw.Web.Controller
{
    /// <summary>
    /// NGXgɑΏۂƂȂ郊NGXgRg[
    /// t@NgNXłB
    /// </summary>
    /// <remarks>
    /// <para>̃NX𗘗pɂ́A
    /// AvP[V\t@C( Web.config )  &lt;httpHandlers&gt; 
    /// vnhƂāAǉݒ肷Kv܂B</para>
    /// <example>
    /// ̗ł́ARequestController.aspx  POST vꂽNGXgnhNX̐ɁA
    /// <see cref="RequestControllerFactory"/> 𗘗p悤Ɏw肷@܂B
    /// <code>
    /// <![CDATA[
    /// <configuration>
    ///   <system.web>
    ///     <httpHandlers>
    ///       <add verb="POST" path="RequestController.aspx" type="TERASOLUNA.Fw.Web.Controller.RequestControllerFactory,TERASOLUNA.Fw.Web"/>
    ///     </httpHandlers>
    ///   </system.web>
    /// </configuration>
    /// ]]>
    /// </code>
    /// </example>
    /// </remarks>
    public class RequestControllerFactory : IHttpHandlerFactory
    {
        /// <summary>
        /// <see cref="ILog"/> NX̃CX^XłB
        /// </summary>
        /// <remarks>
        /// Oo͂ɗp܂B
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(RequestControllerFactory));

        /// <summary>
        /// NGXĝi[HTTPwb_vfłB
        /// </summary>
        /// <remarks>
        /// <para>
        /// ̒萔̒l "RequestName" łB
        /// </para>
        /// </remarks>
        protected static readonly string REQUEST_NAME = "RequestName";

        /// <summary>
        /// ΏۃNGXgRg[łȂꍇɗp郊NGXg^Cv̂łB
        /// </summary>
        /// <remarks>
        /// <para>
        /// ̒萔̒l "Unknown" łB
        /// </para>
        /// </remarks>
        protected static readonly string REQUEST_CTRL_UNKNOWN = "Unknown";

        /// <summary>
        /// <see cref="IBLogic"/> NXɑΉ <see cref="Type"/> CX^X
        /// i[LbVłB
        /// </summary>
        private IDictionary<string, Type> _blogicTypes = new Dictionary<string, Type>();

        /// <summary>
        /// <see cref="IBLogic"/> NXɑΉ <see cref="Type"/> CX^X
        /// LbV擾܂B
        /// </summary>
        /// <value>
        /// <see cref="IBLogic"/> NXɑΉ <see cref="Type"/> CX^X̃LbVB
        /// </value>
        protected IDictionary<string, Type> BLogicTypes
        {
            get
            {
                return _blogicTypes;
            }
        }

        /// <summary>
        /// <c>RequestName</c> wb_̐ݒlɁAv
        /// <see cref="IHttpHandler"/> C^tF[XNX肵A
        /// ̃CX^XԂ܂B
        /// </summary>
        /// <param name="context"> 
        /// HTTP v邽߂ɎgpAgݍ݂̃T[o[ IuWFNg
        /// (Ƃ΁A Request A Response A Session A Server )
        /// ւ̎QƂ񋟂 <see cref="HttpContext"/> NX̃CX^XB</param>
        /// <param name="requestType"> NCAggp HTTP f[^]\bh ( GET ܂ POST )B</param>
        /// <param name="url">
        /// vꂽ\[X <see cref="HttpRequest.RawUrl"/> B
        /// </param>
        /// <param name="pathTranslated">vꂽ\[X <see cref="HttpRequest.PhysicalApplicationPath"/> B</param>
        /// <returns>vV <see cref="IHttpHandler"/> C^[tFCXNX̃CX^XB</returns>
        /// <remarks>
        /// <para><c>RequestName</c> wb_̒l <see cref="HttpContext.Items"/> 
        /// L[ <see cref="RequestControllerConstants.KEY_REQUEST_NAME"/> ֊i[܂B</para>
        /// <para><see cref="LoadAttributes"/> ĂяoAΉrWlXWbNNX܂B</para>
        /// <para><see cref="GetTargetRequestController"/> ĂяoAY郊NGXgRg[𐶐܂B</para>
        /// <para><see cref="GetTargetRequestController"/> ɂ郊NGXgRg[NX̐ɎsꍇAΏۂƂȂ郊NGXgRg[ʎq
        ///  <see cref="REQUEST_CTRL_UNKNOWN"/> ƂA<see cref="GetTargetRequestController"/> ēxĂяo܂B</para>
        /// <para>Oꍇ͑Sĕ⑫AO <see cref="HttpContext.Items"/> 
        /// L[ <see cref="RequestControllerConstants.KEY_CONTROLLER_FACTORY_ERR"/> ֊i[܂B</para>
        /// </remarks>
        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Design", "CA1062:ValidateArgumentsOfPublicMethods"), 
        System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Design", "CA1031:DoNotCatchGeneralExceptionTypes")]
        public virtual IHttpHandler GetHandler(HttpContext context, string requestType, string url, string pathTranslated)
        {
            IHttpHandler handler = null;

            try
            {
                string reqCtrlName = null;
                string requestName = context.Request.Headers[REQUEST_NAME];

                if (string.IsNullOrEmpty(requestName))
                {
                    // NGXgݒ̏ꍇ
                    InvalidRequestException exception = new InvalidRequestException(Properties.Resources.E_EMPTY_REQUEST_NAME);
                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(exception.Message, exception);
                    }
                    throw exception;
                }
                else
                {
                    // NGXĝێĂ
                    context.Items[RequestControllerConstants.KEY_REQUEST_NAME] = requestName;

                    // ΏۃrWlXWbÑ^CvIuWFNgѕt
                    // ǂݍ
                    LoadAttributes(context);
                    
                    reqCtrlName = context.Items[RequestControllerConstants.KEY_REQUEST_TYPE] as string;

                    handler = GetTargetRequestController(reqCtrlName);
                    
                }
            }
            catch (Exception ex)
            {
                Exception targetEx = ex;
                // NGXgݒ肳ĂȂA܂͗OđΏۂƂȂ郊NGXgRg[
                // łȂꍇ́ASUnknownRequestControllerŏB
                try
                {
                    handler = GetTargetRequestController(REQUEST_CTRL_UNKNOWN);
                }
                catch (Exception newEx)
                {
                    if (_log.IsWarnEnabled)
                    {
                        _log.Warn(string.Format(
                            Properties.Resources.W_NOT_CREATE_REQUEST_CONTROLLER, context.Items[RequestControllerConstants.KEY_REQUEST_NAME]));
                    }

                    // ̕sNGXgRg[𗘗pB
                    handler = new UnknownRequestController();
                    targetEx = newEx;
                }

                // NGXgRg[t@NgŔG[UnknownRequestController
                // 邽߁AHttpContext.Items Ɋi[ĂB
                context.Items[RequestControllerConstants.KEY_CONTROLLER_FACTORY_ERR] = targetEx;
            }

            return handler;
        }

        /// <summary>
        /// ΏۃrWlXWbÑ^CvIuWFNgѕt鑮ǂݍ݂܂B
        /// </summary>
        /// <param name="context"> 
        /// HTTP v邽߂ɎgpAgݍ݂̃T[o[ IuWFNg
        /// (Ƃ΁A Request A Response A Session A Server )
        /// ւ̎QƂ񋟂 <see cref="HttpContext"/> NX̃CX^XB</param>
        /// <remarks>
        /// vꂽ <see cref="IBLogic"/> NX <see cref="Type"/> ƁAw肳Ă
        ///  <see cref="ControllerInfoAttribute"/> A <see cref="ValidationInfoAttribute"/> 
        /// ݒl擾A<see cref="HttpContext.Items"/> ֈȉ̒ʂi[܂B
        /// <list type="table">
        /// <listheader>
        /// <term>L[</term>
        /// <description>ݒl</description>
        /// </listheader>
        /// <item>
        /// <term><see cref="RequestControllerConstants.KEY_BLOGIC_CLASS_TYPE"/></term>
        /// <description>vꂽrWlXWbNNX <see cref="Type"/>B</description>
        /// </item>
        /// <item>
        /// <term><see cref="RequestControllerConstants.KEY_REQUEST_TYPE"/></term>
        /// <description><see cref="ControllerInfoAttribute"/> 
        /// <see cref="ControllerInfoAttribute.RequestType"/> vpeBݒlB</description>
        /// </item>
        /// <item>
        /// <term><see cref="RequestControllerConstants.KEY_INPUT_DATASET_CLASS_TYPE"/></term>
        /// <description><see cref="ControllerInfoAttribute"/> 
        /// <see cref="ControllerInfoAttribute.InputDataSetType"/> vpeBݒlB</description>
        /// </item>
        /// <item>
        /// <term><see cref="RequestControllerConstants.KEY_VALIDATION_FILE_PATH"/></term>
        /// <description><see cref="ValidationInfoAttribute"/> 
        /// <see cref="ValidationInfoAttribute.ValidationFilePath"/> vpeBݒlB</description>
        /// </item>
        /// </list>
        /// </remarks>
        /// <exception cref="InvalidRequestException">
        /// <c>RequestName</c> wb_ ɐݒ肳ꂽlɑΉ
        /// rWlXWbNNX <see cref="Type"/> 擾ł܂B
        /// </exception>
        protected virtual void LoadAttributes(HttpContext context)
        {
            string requestName = context.Request.Headers[REQUEST_NAME];

            // ΏۃrWlXWbÑ^Cv擾
            Type blogicType = null;
            try
            {
                bool canGetValue = false;
                ICollection blogicTypesCollection = (ICollection)_blogicTypes;
                lock (blogicTypesCollection.SyncRoot)
                {
                    canGetValue = _blogicTypes.TryGetValue(requestName, out blogicType);
                }

                if (!canGetValue)
                {
                    // rWlXWbNݒt@CBLogic̃^Cv擾
                    string blogicTypeName = BLogicConfiguration.GetBLogicTypeName(requestName);

                    if (blogicTypeName == null)
                    {
                        TerasolunaException exception = new TerasolunaException(string.Format(
                            Properties.Resources.E_BLOGIC_TYPE_NOT_FOUND, requestName));
                        if (_log.IsErrorEnabled)
                        {
                            _log.Error(exception.Message, exception);
                        }
                        throw exception;
                    }

                    blogicType = Type.GetType(blogicTypeName);

                    if (blogicType == null)
                    {
                        TerasolunaException exception = new TerasolunaException(string.Format(
                            Properties.Resources.E_BLOGIC_IBLOGIC_TYPE_NOT_FOUND, blogicTypeName));
                        if (_log.IsErrorEnabled)
                        {
                            _log.Error(exception.Message, exception);
                        }
                        throw exception;
                    }

                    lock (blogicTypesCollection.SyncRoot)
                    {
                        _blogicTypes[requestName] = blogicType;
                    }

                }
            }
            catch (TypeLoadException tle)
            {
                InvalidRequestException exception = new InvalidRequestException(string.Format(
                    Properties.Resources.E_UNKNOWN_REQUEST_NAME, requestName), tle);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;
            }
            catch (ConfigurationErrorsException cee)
            {
                InvalidRequestException exception = new InvalidRequestException(string.Format(
                    Properties.Resources.E_UNKNOWN_REQUEST_NAME, requestName), cee);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;
            }

            context.Items[RequestControllerConstants.KEY_BLOGIC_CLASS_TYPE] = blogicType;

            // t鑮擾
            object[] attrArray = blogicType.GetCustomAttributes(true);
            // ̃NGXg^CvƂĒʏw肵Ă
            context.Items[RequestControllerConstants.KEY_REQUEST_TYPE] = RequestTypeNames.NORMAL;

            foreach (object attr in attrArray)
            {
                // rWlXWbNl擾
                ControllerInfoAttribute ctrlInfoAttr = attr as ControllerInfoAttribute;
                if (ctrlInfoAttr != null)
                {
                    string requestType = ctrlInfoAttr.RequestType;
                    if (requestType != null)
                    {
                        context.Items[RequestControllerConstants.KEY_REQUEST_TYPE] = requestType;
                    }
                    context.Items[RequestControllerConstants.KEY_INPUT_DATASET_CLASS_TYPE] = ctrlInfoAttr.InputDataSetType;
                }
                // ͒lؑl擾
                ValidationInfoAttribute validationAttr = attr as ValidationInfoAttribute;
                if (validationAttr != null)
                {
                    context.Items[RequestControllerConstants.KEY_VALIDATION_FILE_PATH] = validationAttr.ValidationFilePath;
                    context.Items[RequestControllerConstants.KEY_VALIDATION_RULESET] = validationAttr.RuleSet;
                }
            }
        }

        /// <summary>
        /// NGXgRg[̎ʎqΏۂƂȂ <see cref="IHttpHandler"/> C^tF[XNGXgRg[NX̃CX^X擾܂B
        /// </summary>
        /// <param name="reqCtrlName">NGXgRg[̎ʎqB</param>
        /// <returns>vV <see cref="IHttpHandler"/> C^tF[XNX̃CX^XB</returns>
        /// <remarks>
        /// <para><paramref name="reqCtrlName"/> L[ƂāAAvP[V\t@C(Web.config)ɐݒ肳ꂽA
        /// NGXgRg[NX̃CX^X𐶐܂B</para>
        /// <para>AvP[V\t@C(Web.config)ɑΉ <paramref name="reqCtrlName"/> L[Ƃݒ肪A
        /// sĂȂꍇ́A <see cref="RequestControllerFactory.CreateDefaultRequestController"/> sA
        /// ̃NGXgRg[NXΏۂƂĐ܂B</para>
        /// </remarks>
        /// <exception cref="ConfigurationErrorsException">
        /// ȉ̂悤ȏꍇɗOX[܂B
        /// <list type="bullet">
        /// <item>
        /// Web \t@CǂݍނƂł܂B
        /// </item>
        /// <item>
        /// Web \t@C( Web.config )ɐݒ肳ꂽNGXgRg[ݒɂāA <paramref name="reqCtrlName"/> ɑΉݒl󕶎ƂȂĂ܂B
        /// </item>
        /// <item>
        /// Web \t@C( Web.config )ɐݒ肳ꂽNGXgRg[ݒɂāA <paramref name="reqCtrlName"/> ɑΉݒl <see cref="Type"/> 擾ł܂B
        /// </item>
        /// </list>
        /// </exception>
        /// <exception cref="TerasolunaException">
        /// ȉ̂悤ȏꍇɗOX[܂B
        /// <list type="bullet">
        /// <item>
        /// Ώۂ̃RXgN^słB
        /// </item>
        /// <item>
        /// Ώۂ̃NX <see cref="IHttpHandler"/> C^tF[XĂ܂B
        /// </item>
        /// <item>
        /// <paramref name="requestControllerName"/>  <see cref="RequestTypeNames.NORMAL"/> A<see cref="RequestTypeNames.DOWNLOAD"/> A 
        /// <see cref="RequestTypeNames.UPLOAD"/> A <see cref="REQUEST_CTRL_UNKNOWN"/> ȊO̒lݒ肳Ă܂B
        /// </item>
        /// </list>
        /// </exception>
        protected virtual IHttpHandler GetTargetRequestController(string reqCtrlName)
        {
            IHttpHandler handler = null;
            string reqCtrlTypeName = RequestControllerConfiguration.GetRequestControllerType(reqCtrlName);

            if (reqCtrlTypeName == null)
            {
                // Ή郊NGXgRg[ݒ肳ĂȂꍇ
                // ̃NGXgRg[gpB
                handler = CreateDefaultRequestController(reqCtrlName);
                if (_log.IsDebugEnabled)
                {
                    _log.Debug(string.Format(Properties.Resources.D_CREATE_DEFAULT_REQUEST_CONTROLLER, reqCtrlName, handler.GetType().FullName));
                }
            }
            else if (reqCtrlTypeName.Length < 1)
            {
                // Ή郊NGXgRg[̃L[͐ݒ肳Ă邪l̏ꍇ
                ConfigurationErrorsException exception = new ConfigurationErrorsException(string.Format(
                    Properties.Resources.E_INVALID_REQUEST_CONTROLLER_CONFIG, reqCtrlName));
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;
            }
            else
            {
                Type handlerType = Type.GetType(reqCtrlTypeName);
                if (handlerType == null)
                {
                    // ݒ肳Ă郊NGXgRg[NXTypeCX^XłȂꍇB
                    ConfigurationErrorsException exception = new ConfigurationErrorsException(string.Format(
                        Properties.Resources.E_REQUEST_CONTROLLER_CLASS_NOT_FOUND, reqCtrlName, reqCtrlTypeName));
                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(exception.Message, exception);
                    }
                    throw exception;
                }
                else
                {
                    try
                    {
                        handler = Activator.CreateInstance(handlerType) as IHttpHandler;
                    }
                    catch (MemberAccessException mae)
                    {
                        //ANZX̕s
                        TerasolunaException exception = new TerasolunaException(string.Format(
                            Properties.Resources.E_INVALID_CONSTRUCTOR, handlerType.FullName), mae);
                        if (_log.IsErrorEnabled)
                        {
                            _log.Error(exception.Message, exception);
                        }
                        throw exception;
                    }
                    catch (TargetInvocationException tie)
                    {
                        //RXgN^ŗO
                        TerasolunaException exception = new TerasolunaException(string.Format(
                            Properties.Resources.E_INVALID_CONSTRUCTOR, handlerType.FullName), tie);
                        if (_log.IsErrorEnabled)
                        {
                            _log.Error(exception.Message, exception);
                        }
                        throw exception;
                    }
                    if (handler == null)
                    {
                        // IHttpHandlerĂȂNXNGXgRg[ƂĐݒ肳ĂꍇB
                        TerasolunaException exception = new TerasolunaException(string.Format(
                            Properties.Resources.E_NOT_IMPLEMENT, reqCtrlTypeName, typeof(IHttpHandler).FullName));
                        if (_log.IsErrorEnabled)
                        {
                            _log.Error(exception.Message, exception);
                        }
                        throw exception;
                    }
                    if (_log.IsDebugEnabled)
                    {
                        _log.Debug(string.Format(Properties.Resources.D_CREATE_REQUEST_CONTROLLER, reqCtrlName, handler.GetType().FullName));
                    }
                }
            }
            return handler;
        }

        /// <summary>
        /// ̃NGXgRg[𐶐܂B
        /// </summary>
        /// <param name="requestControllerName">Ώۂ̃NGXgRg[ʖB</param>
        /// <returns>
        /// <paramref name="requestControllerName"/> ɑΉ <see cref="IHttpHandler"/> C^tF[X
        /// ̃NGXgRg[NX̃CX^XB
        /// </returns>
        /// <remarks>
        /// <para><paramref name="requestControllerName"/> Ɛ郊NGXgRg[NX́A
        /// ȉ̂悤ɑΉÂĂ܂B</para>
        /// <list type="table">
        /// <listheader>
        /// <term><paramref name="requestControllerName"/></term>
        /// <description>NGXgRg[NX</description>
        /// </listheader>
        /// <item>
        /// <term><see cref="RequestTypeNames.NORMAL"/></term>
        /// <description><see cref="BLogicRequestController"/></description>
        /// </item>
        /// <item>
        /// <term><see cref="RequestTypeNames.DOWNLOAD"/></term>
        /// <description><see cref="FileDownloadRequestController"/></description>
        /// </item>
        /// <item>
        /// <term><see cref="RequestTypeNames.UPLOAD"/></term>
        /// <description><see cref="FileUploadRequestController"/></description>
        /// </item>
        /// <item>
        /// <term><see cref="REQUEST_CTRL_UNKNOWN"/></term>
        /// <description><see cref="UnknownRequestController"/></description>
        /// </item>
        /// </list>
        /// </remarks>
        /// <exception cref="TerasolunaException">
        /// <paramref name="requestControllerName"/>  <see cref="RequestTypeNames.NORMAL"/> A<see cref="RequestTypeNames.DOWNLOAD"/> A 
        /// <see cref="RequestTypeNames.UPLOAD"/> A <see cref="REQUEST_CTRL_UNKNOWN"/> ȊO̒lݒ肳Ă܂B
        /// </exception>
        protected virtual IHttpHandler CreateDefaultRequestController(string requestControllerName)
        {
            if(RequestTypeNames.NORMAL.Equals(requestControllerName, StringComparison.OrdinalIgnoreCase))
            {
                return new BLogicRequestController();
            }
            else if (RequestTypeNames.MULTIPART_UPLOAD.Equals(requestControllerName, StringComparison.OrdinalIgnoreCase))
            {
                return new MultipartUploadRequestController();
            }
            else if (RequestTypeNames.UPLOAD.Equals(requestControllerName, StringComparison.OrdinalIgnoreCase))
            {
                return new FileUploadRequestController();
            }
            else if (RequestTypeNames.DOWNLOAD.Equals(requestControllerName, StringComparison.OrdinalIgnoreCase))
            {
                return new FileDownloadRequestController();
            }
            else if (REQUEST_CTRL_UNKNOWN.Equals(requestControllerName, StringComparison.OrdinalIgnoreCase))
            {
                return new UnknownRequestController();
            }
            else
            {
                TerasolunaException exception = new TerasolunaException(string.Format(
                    Properties.Resources.E_UNKNOWN_REQUEST_CONTROLLER, requestControllerName));
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;
            }
        }

        /// <summary>
        ///  <see cref="IHttpHandler"/> NX̃CX^Xėp邽߂
        /// t@NgLɂ܂B
        /// </summary>
        /// <param name="handler">ėp <see cref="IHttpHandler"/> NX̃CX^XB</param>
        /// <remarks>
        /// <para>NGXgIۂ <see cref="HttpRuntime"/> Ăяo܂B</para>
        /// <para><paramref name="handler"/> Ɏw肳ꂽNX <see cref="IDisposable"/> Ă
        /// NXłꍇA<seealso cref="IDisposable.Dispose"/> Ăяo\[X̉s܂B</para>
        /// </remarks>
        public virtual void ReleaseHandler(IHttpHandler handler)
        {
            // nh̎\[XKvꍇ͂ŎKvB

            // IDisposablenh̏ꍇ͉sB
            IDisposable disposableHandler = handler as IDisposable;
            if (disposableHandler != null)
            {
                disposableHandler.Dispose();
            }
        }
    }
}
