// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Net;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.Client;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Configuration;
using TERASOLUNA.Fw.Common;
using System.IO;
using System.Data;
using TERASOLUNA.Fw.ClientUT.Communication.BinaryFileSenderData;
namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.Sender.BinaryFileSender のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.Sender.BinaryFileSender 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class BinaryFileSenderTest
    {
        private TestContext testContextInstance;

        ///<summary>
        /// 正常な通信先を表すURL文字列です。
        ///</summary>
        private static readonly string TEST_NORMAL_URL = ConfigurationManager.AppSettings["TestNormalUrl"];

        ///<summary>
        /// 正常な通信先を表すURL文字列です。
        ///</summary>
        private static readonly string TEST_NO_SERVER_URL = ConfigurationManager.AppSettings["TestNoServerUrl"];

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 request が null の場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest01()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = null;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("request", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<string>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "request"));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 request で設定したサーバが存在しない場合、
        /// WebException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest02()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NO_SERVER_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 param が null の場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest03()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = null;
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("paramData", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<string>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "paramData"));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 param が空文字列の場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest04()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = string.Empty;
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "filePath"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(BinaryFileSender), typeof(ArgumentException), "E_EMPTY_STRING", "filePath"));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 param に存在しないファイルパスを指定した場合、
        /// FileNotFoundException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest05()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "NotExist.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (FileNotFoundException e)
            {
                // 確認
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_FILE_NOT_FOUND", paramData), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(BinaryFileSender), typeof(FileNotFoundException), "E_COMMUNICATION_FILE_NOT_FOUND", paramData));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection が null の場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest06()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = null;
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("headerCollection", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<string>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "headerCollection"));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 0 個の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest07()  
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("ERROR", resultData.Tables["error"].Rows[0]["error-code"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest08()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success02");
            headerCollection.Add("Terasoluna01", "TERASOLUNA01");
            headerCollection.Add("Terasoluna02", "TERASOLUNA02");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個でありキー値に空文字列が含まれる場合、
        /// ArgumentException が発生することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest09()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            headerCollection.Add(string.Empty, "TERASOLUNA01");
            headerCollection.Add("Terasoluna02", "TERASOLUNA02");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個でありキー値に "Content-Type" が含まれる場合、
        /// ArgumentException が発生することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest10()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            headerCollection.Add("Content-Type", "text/xml");
            headerCollection.Add("Terasoluna02", "TERASOLUNA02");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個でありバリュー値に null が含まれる場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest11()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success03");
            headerCollection.Add("Terasoluna01", null);
            headerCollection.Add("Terasoluna02", "TERASOLUNA02");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個でありバリュー値に空文字列が含まれる場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest12()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success03");
            headerCollection.Add("Terasoluna01", string.Empty);
            headerCollection.Add("Terasoluna02", "TERASOLUNA02");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 reporter が not null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest13()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            BinaryFileSender_IProgressChangeReporterStub01 reporter = new BinaryFileSender_IProgressChangeReporterStub01();
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);
            Assert.AreEqual(50, reporter._eventArgs.ProgressPercentage);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// プロパティ BufferSize が 1 バイトの場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest14()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            BinaryFileSender_IProgressChangeReporterStub01 reporter = new BinaryFileSender_IProgressChangeReporterStub01();
            target.BufferSize = 1;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);
            Assert.AreEqual(50, reporter._eventArgs.ProgressPercentage);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// ファイル名のエンコードがUTF8の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest15()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success04");
            BinaryFileSender_IProgressChangeReporterStub01 reporter = new BinaryFileSender_IProgressChangeReporterStub01();
            target.BufferSize = 8192;
            target.FileNameEncoding = Encoding.UTF8;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);
            Assert.AreEqual(50, reporter._eventArgs.ProgressPercentage);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// SendBinary (HttpWebRequest, string, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 request がキャンセル(Abort)されている場合、
        /// WebException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileSenderData\単体試験.txt")]
        public void SendTest16()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NORMAL_URL) as HttpWebRequest;
            string paramData = "単体試験.txt";
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "UploadBinaryFileSender_Success01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;

            // リクエストのキャンセル
            request.Abort();

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// FileNameEncoding のテスト
        /// </summary>
        /// <remarks>
        /// null を設定した場合、
        /// 規定値のエンコードを取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void FileNameEncodingTest01()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            Encoding encoding = null;

            // 実施
            target.FileNameEncoding = encoding;
            Encoding actualEncoding = target.FileNameEncoding;

            // 確認
            Assert.AreEqual(Encoding.GetEncoding(TERASOLUNA_Fw_Client_Communication_BinaryFileSenderAccessor.ENCODE_CHAR), actualEncoding);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// FileNameEncoding のテスト
        /// </summary>
        /// <remarks>
        /// UTF8 を設定した場合、
        /// UTF8を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void FileNameEncodingTest02()
        {
            // 準備
            BinaryFileSender target = new BinaryFileSender();
            Encoding encoding = Encoding.UTF8;

            // 実施
            target.FileNameEncoding = encoding;
            Encoding actualEncoding = target.FileNameEncoding;

            // 確認
            Assert.AreEqual(Encoding.UTF8, actualEncoding);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<string>), LogLevel.INFO));
        }

        /// <summary>
        /// 電文ログの出力チェックを行い、成功時にtrueを返却するメソッド。
        /// </summary>
        /// <param name="request">リクエスト</param>
        /// <param name="paramData">データセット</param>
        /// <returns>成功の場合true</returns>
        private bool CheckRequestLogData(HttpWebRequest request, string filePath)
        {
            if (!LogUTUtil.CheckTraceUseMessageId(typeof(BinaryFileSender), "T_REQUEST_SEND_ADDRESS", request.Address.ToString()))
            {
                return false;
            }

            if (!LogUTUtil.CheckTraceUseMessageId(typeof(BinaryFileSender), "T_UPLOAD_FILE_PATH", filePath))
            {
                return false;
            }

            StringBuilder requestHeaders = new StringBuilder();
            requestHeaders.AppendLine(MessageManager.GetMessage("T_REQUEST_SEND_HEADER"));
            foreach (string key in request.Headers.AllKeys)
            {
                requestHeaders.AppendLine(MessageManager.GetMessage("T_DICTIONARY_KEY_VALUE", key, request.Headers[key]));
            }

            if (!LogUTUtil.CheckTrace(typeof(BinaryFileSender), requestHeaders.ToString().Trim()))
            {
                return false;
            }

            return true;
        }
    }


}
