﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.ClientUT.Communication.BinaryFileDownloadCommunicatorData;
using System.Data;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Reflection;
using TERASOLUNA.Fw.ClientUT.Communication.CommunicatorBaseData;
namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    /// BinaryFileDownloadCommunicator のテスト クラスです。
    /// </summary>
    [TestClass()]
    public class BinaryFileDownloadCommunicatorTest
    {

        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        /// <summary>
        ///BinaryFileDownloadCommunicator () のテスト
        ///</summary>
        [TestMethod()]
        public void ConstructorTest()
        {
            // 準備
            PropertyInfo senderProperty = typeof(BinaryFileDownloadCommunicator).GetProperty("Sender", BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance);
            PropertyInfo receiverProperty = typeof(BinaryFileDownloadCommunicator).GetProperty("Receiver", BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance);

            // 実施
            BinaryFileDownloadCommunicator target = new BinaryFileDownloadCommunicator();

            // 確認
            DataSetXmlSender actualSender = senderProperty.GetValue(target, null) as DataSetXmlSender;
            BinaryFileReceiver actualReceiver = receiverProperty.GetValue(target, null) as BinaryFileReceiver;
            Assert.IsInstanceOfType(actualSender, typeof(DataSetXmlSender));
            Assert.IsInstanceOfType(actualReceiver, typeof(BinaryFileReceiver));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileDownloadCommunicator), LogLevel.INFO));

        }

        /// <summary>
        /// Download01 () のテスト
        /// </summary>
        ///<remarks>
        /// <paramref name="paramData"/> に <c>null</c> が設定される場合 
        /// ArgumentNullExceptionが発生することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void DownloadTest01()
        {
            // 準備
            BinaryFileDownloadCommunicator target = new BinaryFileDownloadCommunicator();
            DataSet paramData = null;
            IDictionary<string, string> requestHeaders = new Dictionary<string, string>();

            try
            {
                // 実施
                DownloadResult actualDownloadResult = target.Download(paramData, requestHeaders);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("paramData", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileDownloadCommunicator), LogLevel.INFO));
        }


        /// <summary>
        /// Dawnload02 ()のテスト
        /// </summary>
        ///<remarks>
        /// <paramref name="requestHeaders"/> に <c>null</c> が設定される場合 
        /// ArgumentNullExceptionが発生することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void DownloadTest02()
        {
            // 準備
            BinaryFileDownloadCommunicator target = new BinaryFileDownloadCommunicator();
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeaders = null;

            try
            {
                // 実施
                DownloadResult actualDownloadResult = target.Download(paramData, requestHeaders);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("requestHeaders", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileDownloadCommunicator), LogLevel.INFO));
        }


        /// <summary>
        /// Dawnload03 ()のテスト
        /// </summary>
        ///<remarks>
        /// Receiverの戻り値がCommunicationResultインスタンスかつ、ヘッダの要素を0個設定した場合に、 
        /// 戻り値のCommunicationResultインスタンスのヘッダに要素が0個格納されていることを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void DownloadTest03()
        {
            // 準備
            BinaryFileDownloadCommunicatorStub01 target = new BinaryFileDownloadCommunicatorStub01();
            BinaryFileDownloadCommunicator_IHttpSenderStub01 senderStub = new BinaryFileDownloadCommunicator_IHttpSenderStub01();
            BinaryFileDownloadCommunicator_IHttpReceiverStub01 receiverStub = new BinaryFileDownloadCommunicator_IHttpReceiverStub01();

            CommunicationResult expectedCommunicationResult = new CommunicationResult();
            DataSet expectedResultData = new DataSet();
            expectedCommunicationResult.ResultData = expectedResultData;
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeaders = new Dictionary<string, string>();
            receiverStub._communicationResult = expectedCommunicationResult;

            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;

            // 実施
            DownloadResult actualDownloadResult = target.Download(paramData, requestHeaders);

            // 確認
            Assert.IsNull(actualDownloadResult.DownloadFilePath);
            Assert.AreEqual(0, actualDownloadResult.ResponseHeaders.Count);
            Assert.AreSame(expectedResultData, actualDownloadResult.ResultData);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeaders, senderStub._headerCollection);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileDownloadCommunicator), LogLevel.INFO));
        }


        /// <summary>
        /// Dawnload04 ()のテスト
        /// </summary>
        ///<remarks>
        /// Receiverの戻り値がDownloadResultインスタンスかつ、ヘッダの要素を1個設定した場合に、 
        /// 戻り値のDownloadResultインスタンスのヘッダに要素が1個格納されていることを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void DownloadTest04()
        {
            // 準備
            BinaryFileDownloadCommunicatorStub01 target = new BinaryFileDownloadCommunicatorStub01();
            BinaryFileDownloadCommunicator_IHttpSenderStub01 senderStub = new BinaryFileDownloadCommunicator_IHttpSenderStub01();
            BinaryFileDownloadCommunicator_IHttpReceiverStub02 receiverStub = new BinaryFileDownloadCommunicator_IHttpReceiverStub02();

            string expectedDownloadFilePath = "terasoluna.txt";
            DownloadResult expectedCommunicationResult = new DownloadResult(expectedDownloadFilePath);
            DataSet expectedResultData = new DataSet();
            expectedCommunicationResult.ResultData = expectedResultData;
            string key = "TERASOLUNA";
            string value = "terasoluna";
            expectedCommunicationResult.ResponseHeaders[key] = value;
            receiverStub._downloadResult = expectedCommunicationResult;

            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeaders = new Dictionary<string, string>();

            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;

            // 実施
            DownloadResult actualDownloadResult = target.Download(paramData, requestHeaders);

            // 確認
            Assert.AreEqual(1, actualDownloadResult.ResponseHeaders.Count);
            Assert.AreEqual(value, actualDownloadResult.ResponseHeaders[key]);
            Assert.AreSame(expectedCommunicationResult, actualDownloadResult);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeaders, senderStub._headerCollection);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileDownloadCommunicator), LogLevel.INFO));
        }


        /// <summary>
        /// Dawnload05 ()のテスト
        /// </summary>
        ///<remarks>
        /// Receiverの戻り値がCommunicationResultインスタンスかつ、ヘッダの要素を3個設定した場合に、 
        /// 戻り値のCommunicationResultインスタンスのヘッダに要素が3個格納されていることを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void DownloadTest05()
        {
            // 準備
            BinaryFileDownloadCommunicatorStub01 target = new BinaryFileDownloadCommunicatorStub01();
            BinaryFileDownloadCommunicator_IHttpSenderStub01 senderStub = new BinaryFileDownloadCommunicator_IHttpSenderStub01();
            BinaryFileDownloadCommunicator_IHttpReceiverStub01 receiverStub = new BinaryFileDownloadCommunicator_IHttpReceiverStub01();

            CommunicationResult expectedCommunicationResult = new CommunicationResult();
            DataSet expectedResultData = new DataSet();
            expectedCommunicationResult.ResultData = expectedResultData;
            string key01 = "TERASOLUNA1";
            string key02 = "TERASOLUNA2";
            string key03 = "TERASOLUNA3";
            string value01 = "terasoluna1";
            string value02 = "terasoluna2";
            string value03 = "terasoluna3";
            expectedCommunicationResult.ResponseHeaders[key01] = value01;
            expectedCommunicationResult.ResponseHeaders[key02] = value02;
            expectedCommunicationResult.ResponseHeaders[key03] = value03;
            receiverStub._communicationResult = expectedCommunicationResult;

            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeaders = new Dictionary<string, string>();

            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;

            // 実施
            DownloadResult actualDownloadResult = target.Download(paramData, requestHeaders);

            // 確認
            Assert.IsNull(actualDownloadResult.DownloadFilePath);
            Assert.AreEqual(3, actualDownloadResult.ResponseHeaders.Count);
            Assert.AreEqual(value01, actualDownloadResult.ResponseHeaders[key01]);
            Assert.AreEqual(value02, actualDownloadResult.ResponseHeaders[key02]);
            Assert.AreEqual(value03, actualDownloadResult.ResponseHeaders[key03]);
            Assert.AreSame(expectedResultData, actualDownloadResult.ResultData);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeaders, senderStub._headerCollection);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BinaryFileDownloadCommunicator), LogLevel.INFO));
        }
    }
}
