// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections;
using System.Collections.Generic;
using System.Configuration;
using TERASOLUNA.Fw.Common.Configuration.BLogic;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Common.BLogic
{
    /// <summary>
    /// <see cref="IBLogic"/> NX̃CX^X𐶐܂B
    /// </summary>
    /// <remarks>
    /// <see cref="IBLogic"/> NX̃CX^X𐶐邽߂ɂ́A
    /// rWlXWbNݒt@C blogic vf name w肵A
    ///  name ɑΉ type ̌^ <see cref="IBLogic"/> 𐶐܂B
    /// </remarks>
    public class BLogicFactory
    {
        /// <summary>
        /// <see cref="ILog"/> NX̃CX^XłB
        /// </summary>
        /// <remarks>
        /// Oo͂ɗp܂B
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(BLogicFactory));

        /// <summary>
        /// p <see cref="BLogicFactory"/> NX̌^AvP[V\t@C
        /// 擾邽߂̃L[łB
        /// </summary>
        /// <remarks>
        /// <para>
        /// AvP[V\t@C appSettings vfA̕L[ƂāA
        /// <see cref="BLogicFactory"/>NX̌^擾܂B
        /// </para>
        /// <para>
        /// ̒萔̒l "BLogicFactoryTypeName" łB
        /// </para>
        /// </remarks>
        protected static readonly string BLOGIC_FACTORY_TYPENAME = "BLogicFactoryTypeName";

        /// <summary>
        /// <see cref="BLogicFactory"/> NX̃CX^X̃LbVłB
        /// </summary>
        private static volatile BLogicFactory _factory = null;

        /// <summary>
        /// bNp̃IuWFNgłB
        /// </summary>
        private static object _syncRoot = new Object();

        /// <summary>
        /// <see cref="IBLogic"/> NXɑΉ <see cref="Type"/> CX^X
        /// i[LbVłB
        /// </summary>
        private IDictionary<string, Type> _blogicTypes = new Dictionary<string, Type>();

        /// <summary>
        /// <see cref="IBLogic"/> NXɑΉ <see cref="Type"/> CX^X
        /// LbV擾܂B
        /// </summary>
        /// <value>
        /// <see cref="IBLogic"/> NXɑΉ <see cref="Type"/> CX^X̃LbVłB
        /// </value>
        /// <remarks>
        /// ftHgĺA <see cref="Dictionary{String, Type}"/> łB
        /// </remarks>
        protected IDictionary<string, Type> BLogicTypes
        {
            get
            {
                return _blogicTypes;
            }
        }

        /// <summary>
        /// <see cref="BLogicFactory"/> ^̃NX̃CX^X擾܂B
        /// </summary>
        /// <remarks>
        /// App.config<see cref="BLOGIC_FACTORY_TYPENAME"/> L[ƂČ^
        /// LqĂꍇǍ^̃CX^X𐶐܂BLqȂꍇA
        ///  <see cref="BLogicFactory"/> ̃CX^X𐶐܂B
        /// </remarks>
        /// <value>
        /// <see cref="BLogicFactory"/> ^̃NX̃CX^XłB
        /// </value>
        /// <exception cref="ArgumentException">
        /// <see cref="BLogicFactory"/> hNX̃CX^X𐶐ł܂B
        /// </exception>
        /// <exception cref="TerasolunaException">
        /// <see cref="BLogicFactory"/> hNX̃CX^X𐶐ł܂B
        /// </exception>
        protected static BLogicFactory Factory
        {
            get
            {
                if (_factory == null)
                {
                    lock (_syncRoot)
                    {
                        if (_factory == null)
                        {
                            _factory = ClassUtil.CreateInstanceFromAppSettings<BLogicFactory>(
                              BLOGIC_FACTORY_TYPENAME, typeof(BLogicFactory));
                        }
                    }
                }
                return _factory;
            }
        }

        /// <summary>
        /// <see cref="BLogicFactory"/> NX̐VCX^X܂B
        /// </summary>
        /// <remarks>
        /// ftHgRXgN^łB
        /// </remarks>
        public BLogicFactory()
        {
        }

        /// <summary>
        /// <see cref="IBLogic"/> NX̃CX^X𐶐܂B
        /// </summary>
        /// <param name="blogicName"> <see cref="IBLogic"/> NXʂ邽߂̕B</param>
        /// <param name="genericTypeNames">WFlbN^̌^p[^ɓn^̉ϒB</param>
        /// <returns><see cref="IBLogic"/> NX̃CX^XB
        /// Ɏsꍇ́AOX[A null ͕Ԃ܂B</returns>
        /// <remarks>
        /// <para>
        /// <paramref name="blogicName"/> ɁArWlXWbNݒt@Cɐݒ肳ꂽ
        ///  <see cref="IBLogic"/> NX̌^擾AΉCX^X𐶐܂B
        /// </para>
        /// <para>
        /// <paramref name="blogicName"/>  null ܂͋󕶎̏ꍇ́AsȂ <see cref="NopBLogic"/> ԋp܂B
        /// </para>
        /// <para>
        /// <paramref name="blogicName"/> ̓WFlbN^Ō^p[^NX̌^w\ŁȀꍇ
        ///  <paramref name="genericTypeNames"/> Ɍ^p[^ɒu^̌^w肷Kv܂B
        /// </para>
        /// <para>
        /// u^p[^KvƂ鐔Ȃꍇ́AOX[܂B
        /// </para>
        /// </remarks>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="genericTypeNames"/>  null QƂłB
        /// </exception>
        /// <exception cref="TerasolunaException">
        /// ȉ̂悤ȏꍇɗOX[܂B
        /// <list type="bullet">
        /// <item>
        /// <see cref="BLogicFactory"/> hNX̃CX^X𐶐ł܂B
        /// </item>
        /// <item>
        /// <paramref name="blogicName"/> rWlXWbNƂrWlXWbNݒt@C blogic vf`łB
        /// </item>
        /// <item>
        /// <paramref name="blogicName"/> rWlXWbNƂrWlXWbNݒt@C blogic vf type ̒l
        /// ^ł܂B 
        /// </item>
        /// <item>
        /// <see cref="IBLogic"/> NX̃CX^X𐶐ł܂B
        /// </item>
        /// <item>
        /// <paramref name="genericTypeNames"/> ɐݒ肳ĂAZuC̐A
        /// <paramref name="blogicName"/> 琶ꂽWFlbN^ɕKvȌ^p[^菭ȂłB
        /// </item>
        /// </list>
        /// </exception>
        /// <exception cref="ConfigurationErrorsException">
        /// ȉ̂悤ȏꍇɗOX[܂B
        /// <list type="bullet">
        /// <item>
        /// \t@CǂݍނƂł܂B
        /// </item>
        /// <item>
        /// \t@C appSettings vf "blogicConfiguration" L[Ƃ
        /// 擾l琶ꂽCX^XA <see cref="BLogicConfigurationSection"/> ^ł͂܂B
        /// </item>
        /// </list>
        /// </exception>
        /// <example>
        /// rWlXWbNݒt@C̐ݒȉɎ܂B
        /// <code>
        /// &lt;blogicConfiguration xmlns="http://www.terasoluna.jp/schema/BLogicSchema.xsd"&gt;
        ///   &lt;!-- WFlbN^Ō^p[^rWlXWbN̏ꍇ --&gt;
        ///   &lt;blogic name="genericBLogicSample"
        ///              type="TERASOLUNA.Sample.GenericBLogicSample`1, TERASOLUNA.Sample" /&gt;
        ///   &lt;!-- WFlbN^Ō^p[^ȂBLogicBasȅꍇ --&gt;
        ///   &lt;blogic name="noGenericBLogicSample" 
        ///              type="TERASOLUNA.Sample.NoGenericBLogicSample, TERASOLUNA.Sample" /&gt;
        /// &lt;/blogicConfiguration&gt;
        /// </code>
        /// </example>
        public static IBLogic CreateBLogic(string blogicName, params string[] genericTypeNames)
        {
            if (genericTypeNames == null)
            {
                ArgumentNullException exception = new ArgumentNullException("genericTypeNames");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(Properties.Resources.E_NULL_ARGUMENT, "genericTypeNames"), exception);
                }
                throw exception;
            }

            IBLogic blogic = Factory.CreateInstance(blogicName, genericTypeNames);

            if (_log.IsDebugEnabled)
            {
                _log.Debug(string.Format(Common.Properties.Resources.D_CREATE_INSTANCE_SUCCESS,
                    typeof(IBLogic).FullName, blogic.GetType().FullName));
            }

            return blogic;
        }

        /// <summary>
        /// <see cref="IBLogic"/> NX̃CX^X𐶐܂B
        /// </summary>
        /// <param name="blogicName"> <see cref="IBLogic"/> hNXʂ邽߂̕B</param>
        /// <param name="genericTypeNames">WFlbN^̌^p[^ɒu^̌^̉ϒB</param>
        /// <returns><see cref="IBLogic"/> hNX̃CX^XB
        /// Ɏsꍇ́AOX[A null ͕Ԃ܂B</returns>
        /// <remarks>
        /// <para>
        /// <paramref name="blogicName"/> ɁArWlXWbNݒt@Cɐݒ肳ꂽ
        /// <see cref="IBLogic"/> NX̌^擾AΉCX^X𐶐܂B
        /// </para>
        /// <para>
        /// <paramref name="blogicName"/>  null ܂͋󕶎̏ꍇ́AsȂ <see cref="NopBLogic"/> ԋp܂B
        /// </para>
        /// <para>
        /// <paramref name="blogicName"/> ̓WFlbN^̌^p[^NX̌^w\ŁȀꍇ
        /// <paramref name="genericTypeNames"/> Ɍ^p[^ɒu^̌^w肷Kv܂B
        /// </para>
        /// <para>
        /// u^p[^KvƂ鐔Ȃꍇ́AOX[܂B
        /// </para>
        /// </remarks>
        /// <exception cref="TerasolunaException">
        /// ȉ̂悤ȏꍇɗOX[܂B
        /// <list type="bullet">
        /// <item>
        /// <paramref name="blogicName"/> rWlXWbNƂrWlXWbNݒt@C blogic vf`łB
        /// </item>
        /// <item>
        /// <paramref name="blogicName"/> rWlXWbNƂrWlXWbNݒt@C blogic vf type ̒l
        /// ^ł܂B 
        /// </item>
        /// <item>
        /// <see cref="IBLogic"/> NX̃CX^X𐶐ł܂B
        /// </item>
        /// <item>
        /// <paramref name="genericTypeNames"/> ɐݒ肳ĂAZuC̐A
        /// <paramref name="blogicName"/> 琶ꂽWFlbN^ɕKvȌ^p[^菭ȂłB
        /// </item>
        /// </list>
        /// </exception>
        /// <exception cref="ConfigurationErrorsException">
        /// ȉ̂悤ȏꍇɗOX[܂B
        /// <list type="bullet">
        /// <item>
        /// \t@CǂݍނƂł܂B
        /// </item>
        /// <item>
        /// \t@C appSettings vf blogicConfiguration L[Ƃ
        /// 擾l琶ꂽCX^XA <see cref="BLogicConfigurationSection"/> ^ł͂܂B
        /// </item>
        /// </list>
        /// </exception>
        /// <example>
        /// rWlXWbNݒt@C̐ݒȉɎ܂B
        /// <code>
        /// &lt;blogicConfiguration xmlns="http://www.terasoluna.jp/schema/BLogicSchema.xsd"&gt;
        ///   &lt;!-- WFlbN^Ō^p[^rWlXWbN̏ꍇ --&gt;
        ///   &lt;blogic name="genericBLogicSample"
        ///              type="TERASOLUNA.Sample.GenericBLogicSample`1, TERASOLUNA.Sample" /&gt;
        ///   &lt;!-- WFlbN^Ō^p[^ȂBLogicBasȅꍇ --&gt;
        ///   &lt;blogic name="noGenericBLogicSample" type="TERASOLUNA.Sample.NoGenericBLogicSample, TERASOLUNA.Sample" /&gt;
        /// &lt;/blogicConfiguration&gt;
        /// </code>
        /// </example>
        protected virtual IBLogic CreateInstance(string blogicName, params string[] genericTypeNames)
        {
            Type blogicType = null;

            if (string.IsNullOrEmpty(blogicName))
            {
                return new NopBLogic();
            }

            bool canGetValue = false;
            ICollection blogicTypesCollection = (ICollection)_blogicTypes;
            lock (blogicTypesCollection.SyncRoot)
            {
                canGetValue = _blogicTypes.TryGetValue(blogicName, out blogicType);
            }

            if (!canGetValue)
            {
                // rWlXWbNݒt@CBLogic̃^Cv擾
                string blogicTypeName = BLogicConfiguration.GetBLogicTypeName(blogicName);

                if (string.IsNullOrEmpty(blogicTypeName))
                {
                    TerasolunaException exception = new TerasolunaException(string.Format(
                        Properties.Resources.E_BLOGIC_TYPE_NOT_FOUND, blogicName));
                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(exception.Message, exception);
                    }
                    throw exception;
                }

                try
                {
                    blogicType = Type.GetType(blogicTypeName, true);
                }
                catch (Exception e)
                {
                    string message = string.Format(Properties.Resources.E_TYPE_LOAD_FAILED, blogicTypeName);
                    TerasolunaException exception = new TerasolunaException(message, e);

                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(message, exception);
                    }
                    throw exception;
                }

                lock (blogicTypesCollection.SyncRoot)
                {
                    _blogicTypes[blogicName] = blogicType;
                }
            }

            IBLogic blogic = ClassUtil.CreateInstanceWithGenericTypes<IBLogic>(blogicType.AssemblyQualifiedName, genericTypeNames);

            return blogic;
        }
    }
}