// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client;
using System.Data;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
namespace TERASOLUNA.Fw.ClientUT
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.ExecutionResult のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.ExecutionResult 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class ExecutionResultTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        ///ExecutionResult (string) のテスト
        ///</summary>
        [TestMethod()]
        public void ConstructorTest01()
        {
            // 準備
            string resultString = "TERASOLUNA";

            // 実施
            ExecutionResult target = new ExecutionResult(resultString);

            // 確認
            Assert.IsNotNull(target);
            Assert.AreEqual("TERASOLUNA", resultString);
            Assert.IsNull(target.BLogicResultData);
            Assert.AreEqual(0, target.Errors.Count);
        }

        /// <summary>
        ///ExecutionResult (string, DataSet) のテスト
        ///</summary>
        [TestMethod()]
        public void ConstructorTest02()
        {
            // 準備
            string resultString = "TERASOLUNA";
            DataSet blogicResultData = new DataSet();

            // 実施
            ExecutionResult target = new ExecutionResult(resultString, blogicResultData);

            // 確認
            Assert.IsNotNull(target);
            Assert.AreEqual("TERASOLUNA", target.ResultString);
            Assert.AreSame(blogicResultData, target.BLogicResultData);
            Assert.AreEqual(0, target.Errors.Count);
        }

        /// <summary>
        ///AddErrors (IList&lt;MessageInfo&gt;) のテスト
        ///</summary>
        /// <remarks>
        /// 引数 messagesが null の場合、ArgumentNullExceptionが発生することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void AddErrorsTest01()
        {
            // 準備
            IList<MessageInfo> expectedMessages = null;
            ExecutionResult target = new ExecutionResult();

            try
            {
                // 実施
                target.AddErrors(expectedMessages);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("messages", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ExecutionResult), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "messages"));
        }

        /// <summary>
        ///AddErrors (IList&lt;MessageInfo&gt;) のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値が設定できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void AddErrorsTest02()
        {
            // 準備
            IList<MessageInfo> expectedMessages = new List<MessageInfo>();
            ExecutionResult target = new ExecutionResult();
            target.Errors.Clear();

            // 実施
            target.AddErrors(expectedMessages);

            // 確認
            Assert.AreEqual(0, target.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        ///AddErrors (IList&lt;MessageInfo&gt;) のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値が設定できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void AddErrorsTest03()
        {
            // 準備
            IList<MessageInfo> expectedMessages = new List<MessageInfo>();
            MessageInfo expectedMessageInfo01 = new MessageInfo("TERASOLUNA01");
            expectedMessages.Add(expectedMessageInfo01);
            ExecutionResult target = new ExecutionResult();
            target.Errors.Clear();

            // 実施
            target.AddErrors(expectedMessages);

            // 確認
            Assert.AreEqual(1, target.Errors.Count);
            Assert.AreSame(expectedMessageInfo01, target.Errors[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        ///AddErrors (IList&lt;MessageInfo&gt;) のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値が設定できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void AddErrorsTest04()
        {
            // 準備
            IList<MessageInfo> expectedMessages = new List<MessageInfo>();
            MessageInfo expectedMessageInfo01 = new MessageInfo("TERASOLUNA01");
            MessageInfo expectedMessageInfo02 = new MessageInfo("TERASOLUNA02");
            MessageInfo expectedMessageInfo03 = new MessageInfo("TERASOLUNA03");
            expectedMessages.Add(expectedMessageInfo01);
            expectedMessages.Add(expectedMessageInfo02);
            expectedMessages.Add(expectedMessageInfo03);
            ExecutionResult target = new ExecutionResult();
            target.Errors.Clear();

            // 実施
            target.AddErrors(expectedMessages);

            // 確認
            Assert.AreEqual(3, target.Errors.Count);
            Assert.AreSame(expectedMessageInfo01, target.Errors[0]);
            Assert.AreSame(expectedMessageInfo02, target.Errors[1]);
            Assert.AreSame(expectedMessageInfo03, target.Errors[2]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        ///BLogicResultData のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値の取得・設定ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void BLogicResultDataTest01()
        {
            // 準備
            ExecutionResult target = new ExecutionResult();
            DataSet expectedBLogicResultData = new DataSet();

            // 実施
            target.BLogicResultData = expectedBLogicResultData;
            DataSet actualBLogicResultData = target.BLogicResultData;

            // 確認
            Assert.AreSame(expectedBLogicResultData, actualBLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        /// Errors のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値の取得ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ErrorsTest01()
        {
            // 準備
            ExecutionResult target = new ExecutionResult();

            // 実施
            IList<MessageInfo> actualErrors = target.Errors;

            // 確認
            Assert.IsInstanceOfType(actualErrors, typeof(List<MessageInfo>));
            Assert.AreEqual(0, actualErrors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        ///ResultString のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値の取得・設定ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ResultStringTest01()
        {
            // 準備
            ExecutionResult target = new ExecutionResult();
            string expectedResultString = "TERASOLUNA";

            // 実施
            target.ResultString = expectedResultString;
            string actualResultString = target.ResultString;

            // 確認
            Assert.AreSame(expectedResultString, actualResultString);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        /// Success のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値の取得ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SuccessTest01()
        {
            // 準備
            ExecutionResult target = new ExecutionResult();
            TERASOLUNA_Fw_Client_ExecutionResultAccessor accessor = new TERASOLUNA_Fw_Client_ExecutionResultAccessor(target);
            accessor._resultString = null;

            // 実施
            bool actualSuccess = target.Success;

            // 確認
            Assert.IsFalse(actualSuccess);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        /// Success のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値の取得ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SuccessTest02()
        {
            // 準備
            ExecutionResult target = new ExecutionResult();
            TERASOLUNA_Fw_Client_ExecutionResultAccessor accessor = new TERASOLUNA_Fw_Client_ExecutionResultAccessor(target);
            accessor._resultString = string.Empty;

            // 実施
            bool actualSuccess = target.Success;

            // 確認
            Assert.IsFalse(actualSuccess);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        /// Success のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値の取得ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SuccessTest03()
        {
            // 準備
            ExecutionResult target = new ExecutionResult();
            TERASOLUNA_Fw_Client_ExecutionResultAccessor accessor = new TERASOLUNA_Fw_Client_ExecutionResultAccessor(target);
            accessor._resultString = "TERASOLUNA";

            // 実施
            bool actualSuccess = target.Success;

            // 確認
            Assert.IsFalse(actualSuccess);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

        /// <summary>
        /// Success のテスト
        ///</summary>
        /// <remarks>
        /// 想定通りの値の取得ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SuccessTest04()
        {
            // 準備
            ExecutionResult target = new ExecutionResult();
            TERASOLUNA_Fw_Client_ExecutionResultAccessor accessor = new TERASOLUNA_Fw_Client_ExecutionResultAccessor(target);
            accessor._resultString = "success";

            // 実施
            bool actualSuccess = target.Success;

            // 確認
            Assert.IsTrue(actualSuccess);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecutionResult), LogLevel.INFO));
        }

    }
}
