// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Configuration.View;
using TERASOLUNA.Fw.Client.Configuration;
using System.Configuration;
using TERASOLUNA.Fw.Common.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
namespace TERASOLUNA.Fw.ClientUT.Configuration.View
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Configuration.View.ViewConfiguration のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Configuration.View.ViewConfiguration 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class ViewConfigurationTest
    {


        private TestContext testContextInstance;

        private const string CONFIG_FILE = "ViewConfiguration_ViewConfigSection01.config";

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用

        [TestInitialize()]
        public void MyTestInitialize()
        {
            PrivateType config = new PrivateType(typeof(TERASOLUNA.Fw.Client.Configuration.TerasolunaClientConfiguration));
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE);
            section.CreateViewElements();
            config.SetStaticField("_viewConfigSection", section);
            LogUTUtil.Clear();
        }

        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        [TestCleanup()]
        public void MyTestCleanup()
        {
            PrivateType config = new PrivateType(typeof(TERASOLUNA.Fw.Client.Configuration.TerasolunaClientConfiguration));
            config.SetStaticField("_viewConfigSection", null);
        }
        //
        #endregion


        ///<summary>
        /// GetViewTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        /// 構成ファイル、外部ファイルが正常な場合、タイプ名が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigSection01.config")]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigExternal01.config")]
        public void GetViewTypeNameTest01()
        {
            string id = "registerView";

            string expected = "ClientPrototypeApp.Form.RegisterForm, ClientPrototypeApp";

            string actual = ViewConfiguration.GetViewTypeName(id);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewConfiguration), LogLevel.INFO));
        }

        ///<summary>
        /// GetViewTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のidにnullを渡す場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigSection01.config")]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigExternal01.config")]
        public void GetViewTypeNameTest02()
        {
            try
            {
                TERASOLUNA.Fw.Client.Configuration.View.ViewConfiguration.GetViewTypeName(null);
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("id", e.ParamName);
            }
           
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ViewConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ViewConfiguration), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "id"));

        }

        ///<summary>
        /// GetViewTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のidに空文字を渡す場合、ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigSection01.config")]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigExternal01.config")]
        public void GetViewTypeNameTest03()
        {
            try
            {
                TERASOLUNA.Fw.Client.Configuration.View.ViewConfiguration.GetViewTypeName(string.Empty);
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "id"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ViewConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ViewConfiguration), typeof(ArgumentException), "E_EMPTY_STRING", "id"));

        }

        ///<summary>
        /// GetViewTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のidに存在しないidを指定すると、nullが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigSection01.config")]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigExternal01.config")]
        public void GetViewTypeNameTest04()
        {
            string id = "Nothing";

            string actual = TERASOLUNA.Fw.Client.Configuration.View.ViewConfiguration.GetViewTypeName(id);

            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewConfiguration), LogLevel.INFO));
        }

        ///<summary>
        /// GetViewTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        /// ViewConfigurationSectionがnullの場合に、nullが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigSection01.config")]
        [DeploymentItem(@"Configuration\View\ViewConfigurationTestData\ViewConfiguration_ViewConfigExternal01.config")]
        public void GetViewTypeNameTest05()
        {
            //アプリケーション構成ファイルの書き換え
            System.Configuration.Configuration conf = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
            ViewConfigurationSection oldConfiguration = conf.GetSection("viewConfiguration") as ViewConfigurationSection;
            conf.Sections.Remove("viewConfiguration");
            conf.Save();
            ConfigurationManager.RefreshSection("viewConfiguration");

            PrivateType config = new PrivateType(typeof(TerasolunaClientConfiguration));
            config.SetStaticField("_viewConfigSection", null);
            string actual = TERASOLUNA.Fw.Client.Configuration.View.ViewConfiguration.GetViewTypeName("terasoluna");
            Assert.IsNull(actual);

            //構成ファイルを元に戻す
            ViewConfigurationSectionEx section = new ViewConfigurationSectionEx();
            conf.Sections.Add("viewConfiguration", section);
            conf.Save();
            ConfigurationManager.RefreshSection("viewConfiguration");

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewConfiguration), LogLevel.INFO));
        }

        ///<summary>
        /// 設定ファイル構築用クラス
        ///</summary>
        private class ViewConfigurationSectionEx : ViewConfigurationSection
        {
            public ViewConfigurationSectionEx()
            {
                base["files"] = new FilesCollectionEx();
            }
        }

        ///<summary>
        /// 設定ファイル構築用クラス
        ///</summary>
        private class FilesCollectionEx : FilesCollection
        {
            public FilesCollectionEx()
            {
                FileElement element = new FileElementEx();
                BaseAdd(element);
            }
        }

        ///<summary>
        /// 設定ファイル構築用クラス
        ///</summary>
        private class FileElementEx : FileElement
        {
            public FileElementEx()
            {
                this["path"] = "ViewConfiguration_ViewConfigExternal01.config";
            }
        }
    }


}
