// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Data;
using TERASOLUNA.Fw.Client.BLogic;
using TERASOLUNA.Fw.ClientUT.BLogic.MultipartUploadBLogicData;
using TERASOLUNA.Fw.Common.BLogic;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
using System.Reflection;
namespace TERASOLUNA.Fw.ClientUT.BLogic
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.BLogic.MultipartUploadBLogic&lt;TResultData&gt; のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.BLogic.MultipartUploadBLogic&lt;TResultData&gt; 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class MultipartUploadBLogicTest
    {
        private TestContext testContextInstance;

        /// <summary>
        /// マルチパート送信を行うテキスト情報を保持している <see cref="DataSet"/> のテーブル名です。
        /// </summary>
        /// <remarks>この定数の値は "Text" です。</remarks>
        public static readonly string TEXT_TABLE_NAME = MultipartUploadBLogic<DataSet>.TEXT_TABLE_NAME;

        /// <summary>
        /// マルチパート送信を行うファイル情報を保持している <see cref="DataSet"/> のテーブル名です。
        /// </summary>
        /// <remarks>この定数の値は "File" です。</remarks>
        public static readonly string FILE_TABLE_NAME = MultipartUploadBLogic<DataSet>.FILE_TABLE_NAME;

        /// <summary>
        /// マルチパート送信を行うエレメント要素名を保持している <see cref="DataTable"/> のカラム名です。
        /// </summary>
        /// <remarks>この定数の値は "Name" です。</remarks>
        public static readonly string ELEMENT_NAME = MultipartUploadBLogic<DataSet>.ELEMENT_NAME;

        /// <summary>
        /// マルチパート送信を行うファイルパスを保持している <see cref="DataTable"/> のカラム名です。
        /// </summary>
        /// <remarks>この定数の値は "FilePath" です。</remarks>
        public static readonly string FILE_PATH = MultipartUploadBLogic<DataSet>.FILE_PATH;

        /// <summary>
        /// マルチパート送信を行うテキストの値を保持している <see cref="DataTable"/> のカラム名です。
        /// </summary>
        /// <remarks>この定数の値は "Value" です。</remarks>
        public static readonly string TEXT_VALUE = MultipartUploadBLogic<DataSet>.TEXT_VALUE;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用

        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// CreateCommunicator() のテスト
        /// </summary>
        /// <remarks>
        /// 通信クラスを正常に取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateCommunicator01()
        {
            // 準備
            MultipartUploadBLogicStub06<DataSet> target = new MultipartUploadBLogicStub06<DataSet>();

            // 実施
            ICommunicator<IList<MultipartElement>> actual = target.CreateCommunicator();

            // 確認
            Assert.IsInstanceOfType(actual, typeof(MultipartUploadCommunicator<DataSet>));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 0 個の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest01()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            blogicParam.ParamData = paramData;
            paramData.AcceptChanges();
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(0, expectedCommunicator._paramData.Count);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が Textであり行数が 0 行の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest02()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(0, expectedCommunicator._paramData.Count);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が Textであり行数が 1 行の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest03()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string value01 = "value01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が Textであり行数が 3 行の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest04()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string name02 = "name02";
            string name03 = "name03";
            string value01 = "value01";
            string value02 = "value02";
            string value03 = "value03";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name02, value02);
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name03, value03);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(3, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            MultipartValueElement element02 = expectedCommunicator._paramData[1] as MultipartValueElement;
            MultipartValueElement element03 = expectedCommunicator._paramData[2] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(name02, element02.Name);
            Assert.AreEqual(value02, element02.Value);
            Assert.AreEqual(name03, element03.Name);
            Assert.AreEqual(value03, element03.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が Textであり行数が 1 行でありカラム名Nameが存在しない場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest05()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string value01 = "value01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_COLUMN_NOT_FOUND", MultipartUploadBLogic<DataSet>.TEXT_TABLE_NAME, MultipartUploadBLogic<DataSet>.ELEMENT_NAME), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// Textテーブルの行数が 1 行、カラム Name の値が null の場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest06()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = null;
            string value01 = "value01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", TEXT_TABLE_NAME, ELEMENT_NAME), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// Textテーブルの行数が 1 行、カラム Name の値が 空文字列 の場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest07()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = string.Empty;
            string value01 = "value01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", TEXT_TABLE_NAME, ELEMENT_NAME), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が Textであり行数が 1 行でありカラム名 Value が存在しない場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest08()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            string name01 = "name01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_COLUMN_NOT_FOUND", MultipartUploadBLogic<DataSet>.TEXT_TABLE_NAME, MultipartUploadBLogic<DataSet>.TEXT_VALUE), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// Textテーブルの行数が 1 行、カラム Value の値が null の場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest09()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string value01 = null;
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", TEXT_TABLE_NAME, TEXT_VALUE), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// Textテーブルの行数が 1 行、カラム Value の値が 空文字列 の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest10()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string value01 = string.Empty;
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// Textテーブルの行数が 1 行、カラム Name,Value 以外の値が null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest11()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string name01 = "name01";
            string value01 = "value01";
            string filePath01 = null;
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01, filePath01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// Textテーブルの行数が 1 行、カラム Name,Value 以外の値が 空文字列 の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest12()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string name01 = "name01";
            string value01 = "value01";
            string filePath01 = string.Empty;
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01, filePath01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が File であり行数が 0 行の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest13()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(0, expectedCommunicator._paramData.Count);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が File であり行数が 1 行の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest14()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string name01 = "name01";
            string filePath01 = "filePath01";
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01, filePath01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartFileElement element01 = expectedCommunicator._paramData[0] as MultipartFileElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(filePath01, element01.UploadFilePath);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が File であり行数が 3 行の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest15()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string name01 = "name01";
            string name02 = "name02";
            string name03 = "name03";
            string filePath01 = "filePath01";
            string filePath02 = "filePath02";
            string filePath03 = "filePath03";
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01, filePath01);
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name02, filePath02);
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name03, filePath03);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(3, expectedCommunicator._paramData.Count);
            MultipartFileElement element01 = expectedCommunicator._paramData[0] as MultipartFileElement;
            MultipartFileElement element02 = expectedCommunicator._paramData[1] as MultipartFileElement;
            MultipartFileElement element03 = expectedCommunicator._paramData[2] as MultipartFileElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(filePath01, element01.UploadFilePath);
            Assert.AreEqual(name02, element02.Name);
            Assert.AreEqual(filePath02, element02.UploadFilePath);
            Assert.AreEqual(name03, element03.Name);
            Assert.AreEqual(filePath03, element03.UploadFilePath);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が File であり行数が 1 行でありカラム名 Name が存在しない場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest16()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string filePath01 = "filePath01";
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(filePath01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_COLUMN_NOT_FOUND", MultipartUploadBLogic<DataSet>.FILE_TABLE_NAME, MultipartUploadBLogic<DataSet>.ELEMENT_NAME
                    ), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// File テーブルの行数が 1 行、カラム Name の値が null の場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest17()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string name01 = null;
            string filePath01 = "filePath01";
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01, filePath01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", FILE_TABLE_NAME, ELEMENT_NAME), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// File テーブルの行数が 1 行、カラム Name の値が 空文字列 の場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest18()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string name01 = string.Empty;
            string filePath01 = "filePath01";
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01, filePath01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", FILE_TABLE_NAME, ELEMENT_NAME), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// テーブル名が File であり行数が 1 行でありカラム名 FilePath が存在しない場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest19()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            string name01 = "name01";
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_COLUMN_NOT_FOUND", MultipartUploadBLogic<DataSet>.FILE_TABLE_NAME, MultipartUploadBLogic<DataSet>.FILE_PATH), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// File テーブルの行数が 1 行、カラム FilePath の値が null の場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest20()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string name01 = "name01";
            string filePath01 = null;
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01, filePath01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", FILE_TABLE_NAME, FILE_PATH), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// File テーブルの行数が 1 行、カラム FilePath の値が 空文字列 の場合、
        /// ArgumentException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest21()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            string name01 = "name01";
            string filePath01 = string.Empty;
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01, filePath01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", FILE_TABLE_NAME, FILE_PATH), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// File テーブルの行数が 1 行、カラム Name,FilePath 以外の値が null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest22()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string filePath01 = "filePath01";
            string value01 = null;
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01, filePath01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartFileElement element01 = expectedCommunicator._paramData[0] as MultipartFileElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(filePath01, element01.UploadFilePath);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 1 個、
        /// File テーブルの行数が 1 行、カラム Name,FilePath 以外の値が 空文字列 の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest23()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string filePath01 = "filePath01";
            string value01 = string.Empty;
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name01, filePath01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartFileElement element01 = expectedCommunicator._paramData[0] as MultipartFileElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(filePath01, element01.UploadFilePath);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 blogicParam のプロパティ ParamData のテーブル数が 3 個、
        /// テーブル名 Text の行数が 1 行、
        /// テーブル名 File の行数が 1 行、
        /// テーブル名 Array の行数が 1 行、
        /// であり、それぞれのテーブルのカラム名には、Name, Value, FilePath が存在する場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest24()
        {
            // 準備
            MultipartUploadBLogicStub01<DataSet> target = new MultipartUploadBLogicStub01<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            paramData.Tables.Add(FILE_TABLE_NAME);
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(FILE_PATH, typeof(string));
            paramData.Tables[FILE_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            paramData.Tables.Add("Array");
            paramData.Tables["Array"].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables["Array"].Columns.Add(TEXT_VALUE, typeof(string));
            paramData.Tables["Array"].Columns.Add(FILE_PATH, typeof(string));

            string name01 = "name01";
            string name02 = "name02";
            string name03 = "name03";
            string value01 = "value01";
            string value02 = "value02";
            string value03 = "value03";
            string filePath01 = "filePath01";
            string filePath02 = "filePath02";
            string filePath03 = "filePath03";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01, filePath01);
            paramData.Tables[FILE_TABLE_NAME].Rows.Add(name02, filePath02, value02);
            paramData.Tables["Array"].Rows.Add(name03, value03, filePath03);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub01<DataSet> expectedCommunicator = target._communicator;
            DataSet expectedResultData = expectedCommunicator._result;
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreEqual(2, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            MultipartFileElement element02 = expectedCommunicator._paramData[1] as MultipartFileElement;
            Assert.AreEqual(name02, element02.Name);
            Assert.AreEqual(filePath02, element02.UploadFilePath);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// MultipartUploadBLogic のプロパティ Communicator の通信処理で ArgumentException が発生した場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest25()
        {
            // 準備
            MultipartUploadBLogicStub02<DataSet> target = new MultipartUploadBLogicStub02<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string value01 = "value01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(e.GetType().Name, typeof(ArgumentException).Name);
                Assert.AreEqual("MultipartUploadException", e.Message);
            }

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub02<DataSet> expectedCommunicator = target._communicator;
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// MultipartUploadBLogic のプロパティ Communicator の通信処理で
        /// ErrorType が "serverException" である ServerException が発生した場合、
        /// ServerException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest26()
        {
            // 準備
            MultipartUploadBLogicStub03<DataSet> target = new MultipartUploadBLogicStub03<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string value01 = "value01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual("serverException", e.ErrorType);
                Assert.AreEqual("MultipartUploadException", e.Message);
            }

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub03<DataSet> expectedCommunicator = target._communicator;
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ServerException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// MultipartUploadBLogic のプロパティ Communicator の通信処理で
        /// ErrorType が "serverValidateException" である ServerException が発生した場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest27()
        {
            // 準備
            MultipartUploadBLogicStub04<DataSet> target = new MultipartUploadBLogicStub04<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string value01 = "value01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub04<DataSet> expectedCommunicator = target._communicator;
            Assert.AreEqual("serverValidateException", blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);
            Assert.AreEqual(1, blogicResult.Errors.Count);
            Assert.AreEqual("key01", blogicResult.Errors[0].Key);
            Assert.AreEqual("value01", blogicResult.Errors[0].Message);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(ServerException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// MultipartUploadBLogic のプロパティ Communicator の通信処理で
        /// CommunicationException が発生した場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest28()
        {
            // 準備
            MultipartUploadBLogicStub05<DataSet> target = new MultipartUploadBLogicStub05<DataSet>();

            BLogicParam blogicParam = new BLogicParam();
            DataSet paramData = new DataSet();
            paramData.Tables.Add(TEXT_TABLE_NAME);
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(ELEMENT_NAME, typeof(string));
            paramData.Tables[TEXT_TABLE_NAME].Columns.Add(TEXT_VALUE, typeof(string));
            string name01 = "name01";
            string value01 = "value01";
            paramData.Tables[TEXT_TABLE_NAME].Rows.Add(name01, value01);
            paramData.AcceptChanges();

            blogicParam.ParamData = paramData;
            blogicParam.Items.Add("RequestName", "multipartUploadBLogic");

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            MultipartUploadBLogic_MultipartUploadCommunicatorStub05<DataSet> expectedCommunicator = target._communicator;
            Assert.AreEqual(CommunicateBLogicBase<DataSet>.COMMUNICATION_EXCEPTION, blogicResult.ResultString);
            Assert.AreEqual(1, expectedCommunicator._paramData.Count);
            MultipartValueElement element01 = expectedCommunicator._paramData[0] as MultipartValueElement;
            Assert.AreEqual(name01, element01.Name);
            Assert.AreEqual(value01, element01.Value);
            Assert.AreEqual(1, expectedCommunicator._requestHeaders.Count);
            Assert.AreEqual("multipartUploadBLogic", expectedCommunicator._requestHeaders["RequestName"]);
            Assert.AreEqual(1, blogicResult.Errors.Count);
            Assert.AreEqual("communicationException", blogicResult.Errors[0].Key);
            Assert.AreEqual("MultipartUploadException", blogicResult.Errors[0].Message);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartUploadBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<IList<MultipartElement>>), typeof(CommunicationException), "W_BLOGIC_EXCEPTION"));
        }
    }
}
