﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.  

using System;
using System.IO;
using System.Text;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Communication
{
    /// <summary>
    /// 文字列をマルチパートデータとして格納するためのデータ型です。
    /// </summary>
    public class MultipartValueElement : MultipartElement
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(MultipartValueElement));

        /// <summary>
        /// マルチパートデータのヘッダに Content-Type ヘッダとして設定する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>この定数の値は "application/x-www-form-urlencoded" です。</para>
        /// </remarks>
        protected static readonly string CONTENT_TYPE_VALUE = "application/x-www-form-urlencoded";

        /// <summary>
        /// マルチパートデータのヘッダに Content-Disposition ヘッダを設定する際に使用する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>この定数の値は "form-data; name={0}" です。</para>
        /// </remarks>
        protected static readonly string FORM_DATA_FORMAT = "form-data; name={0}";

        /// <summary>
        /// アップロード対象のテキストの値を設定する文字列です。
        /// </summary>
        private string _value = null;

        /// <summary>
        /// <see cref="MultipartElement"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// <para><paramref name="name"/>を引数とするコンストラクタです。</para>
        /// </remarks>
        /// <param name="name">マルチパートの要素名です。</param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="name"/>が <c>null</c> 参照です。
        /// </exception>
        /// <exception cref="ArgumentException">
        /// <paramref name="name"/>が空文字です。
        /// </exception>
        public MultipartValueElement(string name)
            : base(name)
        {
        }

        /// <summary>
        /// マルチパートデータの本文となる文字列を取得または設定します。
        /// </summary>
        /// <value>
        /// マルチパートデータの本文となる文字列。
        /// </value>
        public virtual string Value
        {
            get
            {
                return _value;
            }
            set
            {
                _value = value;
            }
        }

        /// <summary>
        /// <see cref="Value"/>を含むマルチパートデータのボディ部を読み出すための<see cref="Stream"/>を返却します。
        /// </summary>
        /// <returns><see cref="Value"/>文字列を読み出すための<see cref="Stream"/>です。</returns>
        /// <param name="encoding">マルチパートデータの本文となる文字列をエンコードするための<see cref="Encoding"/>です。</param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="encoding"/>が <c>null</c> 参照です。
        /// </exception>
        public override Stream CreateBodyStream(Encoding encoding)
        {
            if (encoding == null)
            {
                ArgumentNullException exception = new ArgumentNullException("encoding");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "encoding"), exception);
                }
                throw exception;
            }

            Stream streamBody = new MemoryStream();

            // null、または、空文字の時は、何もしません。
            if (!string.IsNullOrEmpty(this.Value))
            {
                // ボディにデータを設定します。
                byte[] byteBody = encoding.GetBytes(this.Value);
                streamBody.Write(byteBody, 0, byteBody.Length);
            }

            streamBody.Position = 0;
            return streamBody;
        }

        /// <summary>
        /// マルチパートデータのヘッダ情報を HTTP ヘッダ形式に整形して返却します。
        /// </summary>
        /// <remarks>
        /// <para>ヘッダには "Content-Disposition" 、 "Content-Type" が含まれます。</para>
        /// <para>Content-Disposition ヘッダに格納される値は、 "form-data; name=[マルチパート要素名]" となります。
        /// マルチパート要素名は、コンストラクタで与えられた文字列です。</para>
        /// <para>Content-Type ヘッダに格納される値は、 "application/x-www-form-urlencoded" となります。</para>
        /// <para>サーバ・アプリケーションで Commons FileUpload ライブラリ使用する場合には、使用するバージョンによっていくつかの
        /// 制約があります。</para>
        /// <para>バージョン 1.1.1 では、マルチパートデータのヘッダに以下の制約があります。</para>
        /// <para>
        /// <list type="bullet">
        /// <item>
        /// <description>ヘッダは一行(キー / 値)で1024バイトまでです。</description>
        /// </item>
        /// <item>
        /// <description>ヘッダ部は最大 10240 バイトまでです。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>バージョン 1.2 では、マルチパートデータのヘッダに以下の制約があります。</para>
        /// <para>
        /// <list type="bullet">
        /// <item>
        /// <description>ヘッダ部は最大 10240 バイトまでです。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        /// <returns>マルチパートデータのヘッダとなる文字列です。</returns>
        public override string CreateHeader()
        {
            StringBuilder sb = new StringBuilder();

            string headerFormat = HEADER_FORMAT_STRING;

            sb.Append(string.Format(headerFormat, CONTENT_DISPOSITION, string.Format(FORM_DATA_FORMAT, Name)));
            sb.Append(NEWLINE_STRING);
            sb.Append(string.Format(headerFormat, CONTENT_TYPE, CONTENT_TYPE_VALUE));

            return sb.ToString();
        }
    }
}
