/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.validation;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collection;
import java.util.Date;
import java.util.StringTokenizer;

import jp.terasoluna.fw.util.PropertyUtil;
import jp.terasoluna.fw.util.StringUtil;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.validator.GenericTypeValidator;
import org.apache.commons.validator.GenericValidator;
import org.apache.commons.validator.UrlValidator;

/**
 * ؃WbÑ[eBeBNXB
 *
 *
 */
public class ValidationUtil {
    /**
     * pJĩ`FbNɎgp镶B
     */
    protected static String hankakuKanaList =
        "¯֬ܦް";

    /**
     * SpJĩ`FbNɎgp镶B
     */
    protected static String zenkakuKanaList =
        "ACEGI@BDFHJLNPRKMOQSTVXZ\"
            + "UWY[]^`ceg_adfhijklmnqtwz"
            + "orux{psvy|}~"
            + "b[";

    /**
     * <code>ApplicationResources</code>
     * t@Cɒ`ꂽpe[u擾L[B
     */
    protected static final String HANKAKU_KANA_LIST_KEY
        = "validation.hankaku.kana.list";

    /**
     * <code>ApplicationResources</code>
     * t@Cɒ`ꂽSpe[u擾L[B
     */
    protected static final String ZENKAKU_KANA_LIST_KEY
        = "validation.zenkaku.kana.list";

    /**
     * UNICODER[h'&#165;u0000''&#165;u00ff'
     * ͈͓ɑ݂SpB
     */
    protected static final String ZENKAKU_BEGIN_U00_LIST =
        "_Nʁ}L~";

    static {
        // pJiESpJi`̕ύX
        setHankakuKanaList();
        setZenkakuKanaList();
    }

    /**
     * pJi`ݒ肷B
     */
    protected static void setHankakuKanaList() {
        String value = null;
        // pJie[u擾
        value = PropertyUtil.getProperty(HANKAKU_KANA_LIST_KEY);
        if (value != null) {
            hankakuKanaList = value;
        }
    }

    /**
     * SpJi`ݒ肷B
     */
    protected static void setZenkakuKanaList() {
        // SpJie[u擾
        String value = null;
        value = PropertyUtil.getProperty(ZENKAKU_KANA_LIST_KEY);
        if (value != null) {
            zenkakuKanaList = value;
        }
    }

    /**
     * w肳ꂽpJił邱Ƃ`FbNB
     *
     * @param c 
     * @return pJił true
     */
    protected static boolean isHankakuKanaChar(char c) {
        return hankakuKanaList.indexOf(c) >= 0;
    }

    /**
     * w肳ꂽpł邱Ƃ`FbNB
     *
     * @param c 
     * @return pł true
     */
    protected static boolean isHankakuChar(char c) {
        return (c <= '\u00ff' && ZENKAKU_BEGIN_U00_LIST.indexOf(c) < 0)
                || isHankakuKanaChar(c);
    }

    /**
     * w肳ꂽSpł邱Ƃ`FbNB
     *
     * @param c 
     * @return Spł true
     */
    protected static boolean isZenkakuChar(char c) {
        return !isHankakuChar(c);
    }

    /**
     * w肳ꂽSpJił邱Ƃ`FbNB
     *
     * @param c 
     * @return SpJił true
     */
    protected static boolean isZenkakuKanaChar(char c) {
        return zenkakuKanaList.indexOf(c) >= 0;
    }

    /**
     * ؒlK\ɍv邱Ƃ`FbNB
     *
     * @param value ؒl
     * @param mask K\
     * @return
     *            ؒlK\ɍvȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean matchRegexp(String value, String mask) {
        if (!StringUtils.isEmpty(value)
                && !GenericValidator.matchRegexp(value, mask)) {
            return false;
        }
        return true;
    }

    /**
     * ؒlp݂̂Ȃ镶ł邱Ƃ`FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @return
     *            ؒlp݂̂Ȃ镶łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isAlphaNumericString(String value) {
        return matchRegexp(value, "^([0-9]|[a-z]|[A-Z])*$");
    }

    /**
     * ؒl啶p݂̂Ȃ镶ł邱Ƃ`FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @return
     *            ؒl啶p݂̂Ȃ镶łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isUpperAlphaNumericString(String value) {
        return matchRegexp(value, "^([0-9]|[A-Z])*$");
    }

    /**
     * ؒl݂̂Ȃ镶ł邱Ƃ`FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @return
     *            ؒl݂̂Ȃ镶łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isNumericString(String value) {
        return matchRegexp(value, "^([0-9])*$");
    }

    /**
     * ؒlw肳ꂽł邱Ƃ`FbNB
     * <br>
     * `FbŃAƏɕāAȉ̂悤ɍsB
     * <ul>
     * <li>̌`FbN
     * <ol>
     * <li><code>isAccordedInteger</code><code>true</code>̏ꍇA
     * ̌A<code>integerLength</code>̒l
     * v邩ǂ`FbNB
     *
     * <li><code>isAccordedInteger</code><code>false</code>̏ꍇA
     * ̌A<code>integerLength</code>̒lȉł邱Ƃ
     * `FbNB
     * </ol>
     *
     * <li>̌`FbN
     * <ol>
     * <li><code>isAccordedScale</code><code>true</code>̏ꍇA
     * ̌A<code>scaleLength</code>̒l
     * v邩ǂ`FbNB
     *
     * <li><code>isAccordedScale</code><code>true</code>̏ꍇA
     * ̌A<code>scaleLength</code>̒lȉł邱Ƃ
     * `FbNB
     * </ol>
     * </ul>
     *
     * @param value ؒl
     * @param integerLength ̌
     * @param isAccordedInteger
     *           ̌v`FbNsꍇA
     *           <code>true</code>w肷B
     *           ̌ȓ`FbNsꍇA
     *           <code>false</code>w肷B
     * @param scaleLength ̌
     * @param isAccordedScale
     *           ̌v`FbNsꍇA
     *           <code>true</code>w肷B
     *           ̌ȓ`FbNsꍇA
     *           <code>false</code>w肷B
     *
     * @return
     *            ؒlw肳ꂽłȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isNumber(
            BigDecimal value, int integerLength, boolean isAccordedInteger,
            int scaleLength, boolean isAccordedScale) {

        // ؒlnull̎Atrueԋp
        if (value == null) {
            return true;
        }

        // `FbNs
        // Βl̂ݒo
        BigInteger bi = value.toBigInteger().abs();
        // 
        int length = bi.toString().length();
        if (!checkNumberFigures(length, integerLength, isAccordedInteger)) {
            return false;
        }

        // `FbNs
        int scale = value.scale();
        if (!checkNumberFigures(scale, scaleLength, isAccordedScale)) {
            return false;
        }

        return true;
    }

    /**
     * `FbNs߂̃wp\bhB
     *
     * @param length ۂ̌
     * @param checkLength rs߂̌
     * @param isAccorded
     *           v`FbNsꍇA<code>true</code>w肷B
     *           ȓ`FbNsꍇA<code>false</code>w肷B
     * @return
     *            ۂ̌w肳ꂽłȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    protected static boolean checkNumberFigures(
            int length, int checkLength, boolean isAccorded) {
        // I[óAfalseԋp
        if (length > checkLength) {
            return false;
        }

        // vw肳ĂƂ
        if (isAccorded) {
            // sv́Afalseԋp
            if (length != checkLength) {
                return false;
            }
        }
        return true;
    }

    /**
     * ؒlpJî݂Ȃ镶ł邩ǂ
     * `FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @return
     *            ؒlpJî݂Ȃ镶łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isHankakuKanaString(String value) {

        // ؒlnull܂͋󕶎̎Atrueԋp
        if (StringUtils.isEmpty(value)) {
            return true;
        }

        char[] chars = value.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            if (!isHankakuKanaChar(chars[i])) {
                return false;
            }
        }
        return true;

    }

    /**
     * ؒlp݂̂Ȃ镶ł邩ǂ
     * `FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @return
     *            ؒlp݂̂Ȃ镶łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isHankakuString(String value) {

        // ؒlnull܂͋󕶎̎Atrueԋp
        if (StringUtils.isEmpty(value)) {
            return true;
        }

        char[] chars = value.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            if (!isHankakuChar(chars[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * ؒlSp݂̂Ȃ镶ł邩ǂ
     * `FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @return
     *            ؒlSp݂̂Ȃ镶łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isZenkakuString(String value) {

        // ؒlnull܂͋󕶎̎Atrueԋp
        if (StringUtils.isEmpty(value)) {
            return true;
        }

        char[] chars = value.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            if (!isZenkakuChar(chars[i])) {
                return false;
            }
        }
        return true;

    }

    /**
     * ؒlSpJî݂Ȃ镶ł邩ǂ
     * `FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @return
     *            ؒlSpJî݂Ȃ镶łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
    */
    public static boolean isZenkakuKanaString(String value) {

        // ؒlnull܂͋󕶎̎Atrueԋp
        if (StringUtils.isEmpty(value)) {
            return true;
        }

        char[] chars = value.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            if (!isZenkakuKanaChar(chars[i])) {
                return false;
            }
        }
        return true;

    }

    /**
     * ؒl֎~܂܂ȂƂ`FbNB
     * GXP[vKvȕ́u\vtB
     * Ⴆ΃_uR[e[Vu"v֎~ɂꍇA
     * u\"vƃGXP[vtKvB
     *
     * ֎~<code>null</code>A܂͋󕶎͐Ƃ݂ȂB
     *
     * @param value ؒl
     * @param prohibitedChars ֎~̕
     * @return ؒl֎~܂łȂ<code>true</code>A
     * ȊO<code>false</code>ԂB
     */
    public static boolean hasNotProhibitedChar(
            String value, String prohibitedChars) {

        // ؒlnull܂͋󕶎̎Atrueԋp
        if (StringUtils.isEmpty(value)) {
            return true;
        }

        char[] chars = value.toCharArray();

        // ͋֎~񂪖ݒ̏ꍇA`FbNsȂ
        if (prohibitedChars == null || "".equals(prohibitedChars)) {
            return true;
        }

        // 
        for (int i = 0; i < chars.length; i++) {
            if (prohibitedChars.indexOf(chars[i]) >= 0) {
                return false;
            }
        }
        return true;
    }

    /**
     * zA܂́A<code>Collection</code>̗vfA
     * w肳ꂽőlEŏl͈͓̔ł邩ǂ`FbNB
     *
     * Ώۂ̔zE<code>Collection</code><code>null</code>
     * ꍇ́Avf0ƂďsB܂AΏۂ̒lzA
     * Collectionł͂Ȃꍇ́AIllegalArgumentExceptionX[B
     *
     * @param obj Ώۂ̔zE<code>Collection</code>
     * @param min vf̍ŏl
     * @param max vf̍ől
     * @return
     *            Ώۂ̔zE<code>Collection</code>
     *            w肳ꂽőlEŏl͈͓̔łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isArrayInRange(Object obj, int min, int max) {

        // ؒl̔z
        int targetLength = 0;
        if (obj == null) {
            targetLength = 0;
        } else if (obj instanceof Collection) {
            targetLength = ((Collection) obj).size();
        } else if (obj.getClass().isArray()) {
            targetLength = Array.getLength(obj);
        } else {
            // ؒlz^ł͂ȂꍇAIllegalArgumentExceptionX[
            throw new IllegalArgumentException(obj.getClass().getName() +
                    " is neither Array nor Collection.");
        }

        // ͂ꂽvfw͈͈ȊOȂfalseԋp
        if (!GenericValidator.isInRange(targetLength, min, max)) {
            return false;
        }
        return true;
    }


    /**
     * ؒlURL`̕ł邩ǂ`FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @param allowallschemes SẴXL[邩ǂw
     * @param allow2slashes _uXbV邩ǂw
     * @param nofragments URL֎~邩ǂw
     * @param schemesVar XL[B
     * ꍇ̓J}؂Ŏw肷B
     * @return
     *            ؒlURL`̕łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isUrl(
            String value, boolean allowallschemes, boolean allow2slashes,
            boolean nofragments, String schemesVar) {

        if (StringUtils.isEmpty(value)) {
            return true;
        }

        // IvV̐ݒ
        int options = 0;
        if (allowallschemes) {
            options += UrlValidator.ALLOW_ALL_SCHEMES ;
        }
        if (allow2slashes) {
            options += UrlValidator.ALLOW_2_SLASHES;
        }
        if (nofragments) {
            options += UrlValidator.NO_FRAGMENTS;
        }

        // IvVȂꍇ̓ftHgGenericValidatorgp
        if (options == 0 && schemesVar == null) {
            if (GenericValidator.isUrl(value)) {
                return true;
            }
            return false;
        }

        // XL[String[]ɕϊ
        String[] schemes = null;
        if (schemesVar != null) {

            StringTokenizer st = new StringTokenizer(schemesVar, ",");
            schemes = new String[st.countTokens()];

            int i = 0;
            while (st.hasMoreTokens()) {
                schemes[i++] = st.nextToken().trim();
            }
        }

        // IvV̏ꍇUrlValidatorgp
        UrlValidator urlValidator = new UrlValidator(schemes, options);
        if (urlValidator.isValid(value)) {
            return true;
        }
        return false;
    }

    /**
     * ؒl̃oCg񒷂w肵őlEŏl͈͓̔ł邩ǂ
     * `FbNBT|[gĂȂGR[fBOw肵ꍇA
     * OB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @param encoding `FbN̕<code>encoding</code>
     * @param min ől
     * @param max ŏl
     * @return
     *            ؒl̃oCg񒷂w肵őlEŏl
     *            ͈͓łȂ<code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isByteInRange(
            String value, String encoding, int min, int max) {

        // ؒlnull܂͋󕶎̎Atrueԋp
        if (StringUtils.isEmpty(value)) {
            return true;
        }

        // wGR[fBOŃoCg擾
        int bytesLength = 0;
        try {
            bytesLength = StringUtil.getByteLength(value, encoding);
        } catch (UnsupportedEncodingException e) {
            throw new IllegalArgumentException(e.getMessage());
        }

        // oCg`FbN
        if (!GenericValidator.isInRange(bytesLength, min, max)) {
            return false;
        }
        return true;
    }

    /**
     * tw肵͈͓ł邩ǂ`FbNB
     *
     * <code>null</code> A󕶎́AƂ݂ȂB
     *
     * @param value ؒl
     * @param startDateStr
     *            t͈͂̊Jn臒lƂȂtB
     *            <code>datePattern</code>A܂́A
     *            <code>datePatternStrict</code>Ŏw肵`Őݒ肷邱ƁB
     * @param endDateStr
     *            t͈͂̏I臒lƂȂtB
     *            <code>datePattern</code>A܂́A
     *            <code>datePatternStrict</code>Ŏw肵`Őݒ肷邱ƁB
     * @param datePattern tH[}bgtp^[B
     * @param datePatternStrict tH[}bgtp^[B
     * @return
     *            ؒlpJî݂Ȃ镶łȂ
     *            <code>true</code>ԂB
     *            ȊȌꍇA<code>false</code>ԂB
     */
    public static boolean isDateInRange(
            String value, String startDateStr, String endDateStr,
            String datePattern, String datePatternStrict) {

        // ؒlnull܂͋󕶎̎Atrueԋp
        if (StringUtils.isEmpty(value)) {
            return true;
        }

        // ͓t̑Ó`FbN
        Date result = toDate(value, datePattern, datePatternStrict);
        if (result == null) {
            return false;
        }

        if (GenericValidator.isBlankOrNull(startDateStr)
                && GenericValidator.isBlankOrNull(endDateStr)) {
            // t͈͂w肳ĂȂꍇ͐Ƃ݂Ȃ
            return true;
        }

        // Jntȍ~ǂ`FbN
        if (!GenericValidator.isBlankOrNull(startDateStr)) {
            Date startDate =
                toDate(startDateStr, datePattern, datePatternStrict);

            if (startDate == null) {
                throw new IllegalArgumentException("startDate is unparseable["
                    + startDateStr + "]");
            }

            if (result.before(startDate)) {
                return false;
            }
        }

        // ItȑOǂ`FbN
        if (!GenericValidator.isBlankOrNull(endDateStr)) {
            Date endDate = toDate(endDateStr, datePattern, datePatternStrict);

            if (endDate == null) {
                throw new IllegalArgumentException("endDate is unparseable["
                    + endDateStr + "]");
            }

            if (result.after(endDate)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Date^ɕϊB
     * <br>
     * ϊ͈ȉ̂悤ɍsB
     * ̏ꍇAiȓt`FbNs邽߁A
     * 2000/02/31̂悤ȁA݂Ȃt<code>value</code>
     * Ɏw肳ꂽꍇA<code>null</code>ԋpB
     * <ul>
     * <li>
     * <code>datePattern</code><code>null</code>AсA
     * 󕶎łȂꍇ<br>
     * lȂtϊsB
     * Ƃ΁A<code>datePattern</code>yyyy/MM/dd̏ꍇA
     * 2000/1/1ϊƁA2000/01/01\Date^ԋpB
     * <li>
     * <code>datePatternStrict</code><code>null</code>AсA
     * 󕶎łȂꍇ<br>
     * ltϊsB
     * Ƃ΁A<code>datePattern</code>yyyy/MM/dd̏ꍇA
     * 2000/1/1ϊƁAnullԋpB
     * <li>
     * <code>datePattern</code><code>datePatternStrict</code>
     * A<code>null</code>AсA 󕶎łȂꍇ<br>
     * <code>datePattern</code>D悵ėpB
     * <li>
     * <code>datePattern</code><code>datePatternStrict</code>
     * <code>null</code>A܂́A󕶎̏ꍇ<br>
     * OԋpB
     * </ul>
     * <li>
     * <code>value</code><code>null</code>AсA
     * 󕶎̏ꍇA<code>null</code>ԋpB
     *
     * @param value ϊΏۂ̕
     * @param datePattern tp^[ilȂp^[wj
     * @param datePatternStrict tp^[ilp^[wj
     * @return 񂩂ϊꂽDateCX^XBϊs\ȏꍇnullB
     */
    public static Date toDate(String value, String datePattern,
            String datePatternStrict) {

        if (StringUtils.isEmpty(value)) {
            return null;
        }

        Date result = null;
        
        // `FbNȂ̕ϊ
        if (datePattern != null && datePattern.length() > 0) {
            result = GenericTypeValidator.formatDate(value,
                            datePattern, false);

        // `FbN̕ϊ
        } else if (datePatternStrict != null
                && datePatternStrict.length() > 0) {
            result = GenericTypeValidator.formatDate(value,
                            datePatternStrict, true);

        // tp^[ݒ肳ĂȂ
        } else {
            throw new IllegalArgumentException(
                    "datePattern or datePatternStrict must be specified.");
        }
        
        return result;
    }
}
