/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.oxm.xsd.xerces;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URL;
import java.util.Properties;

import jp.terasoluna.utlib.LogUTUtil;
import jp.terasoluna.utlib.UTUtil;
import jp.terasoluna.fw.oxm.xsd.exception.NamespaceNotFoundException;
import jp.terasoluna.fw.oxm.xsd.exception.NamespaceNotUniqueException;
import jp.terasoluna.fw.oxm.xsd.exception.ParserIOException;
import jp.terasoluna.fw.oxm.xsd.exception.ParserNotSupportedException;
import jp.terasoluna.fw.oxm.xsd.exception.ParserSAXException;
import jp.terasoluna.fw.oxm.xsd.exception.SchemaFileNotFoundException;
import jp.terasoluna.fw.oxm.xsd.message.ErrorMessage;
import jp.terasoluna.fw.oxm.xsd.message.ErrorMessages;
import jp.terasoluna.fw.oxm.xsd.xerces.SchemaValidatorImpl;
import jp.terasoluna.fw.oxm.xsd.xerces.XMLErrorReporterEx;
import junit.framework.TestCase;

import org.apache.xerces.impl.Constants;
import org.apache.xerces.parsers.DOMParser;
import org.apache.xerces.parsers.XML11Configuration;
import org.apache.xerces.xni.parser.XMLParserConfiguration;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;

/**
 * {@link jp.terasoluna.fw.oxm.xsd.xerces.SchemaValidatorImpl} NX̃ubN{bNXeXgB
 * 
 * <p>
 * <h4>yNX̊Tvz</h4>
 * XMLf[^̌``FbNsSchemaValidatorNXB<br>
 * EO<br>
 * setCommonParserProperty\bḧerrorMessagesnullɂȂ邱Ƃ͂ȂB<br>
 * setCommonParserProperty\bḧparsernullɂȂ邱Ƃ͂ȂB<br>
 * setCommonParserFeature\bḧparsernullɂȂ邱Ƃ͂ȂB<br>
 * <p>
 * 
 * @see jp.terasoluna.fw.oxm.xsd.xerces.SchemaValidatorImpl
 */
public class SchemaValidatorImplTest extends TestCase {

    /**
     * setUp\bhɂA񐶐CX^X
     */
    private SchemaValidatorImpl setUpValidator = null;
    
    /**
     * ̃eXgP[Xsׂ
     * GUI AvP[VNB
     * 
     * @param args java R}hɐݒ肳ꂽp[^
     */
    public static void main(String[] args) {
        junit.swingui.TestRunner.run(SchemaValidatorImplTest.class);
    }

    /**
     * sB
     * 
     * @throws Exception ̃\bhŔO
     * @see junit.framework.TestCase#setUp()
     */
    @Override
    protected void setUp() throws Exception {
        super.setUp();
        this.setUpValidator = new SchemaValidatorImpl();
        
    }

    /**
     * IsB
     * 
     * @throws Exception ̃\bhŔO
     * @see junit.framework.TestCase#tearDown()
     */
    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
     * RXgN^B
     * 
     * @param name ̃eXgP[X̖OB
     */
    public SchemaValidatorImplTest(String name) {
        super(name);
    }

    /**
     * testSetCache01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() cache:true<br>
     *         () this.cache:false<br>
     *         
     * <br>
     * ҒlF(ԕω) this.cache:true<br>
     *         
     * <br>
     * ɐݒ肳邱ƂmFɂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testSetCache01() throws Exception {
        // O
        setUpValidator.cache = false;

        // eXg{
        setUpValidator.setCache(true);
        
        // 
        assertTrue(setUpValidator.cache);        

    }

    /**
     * testSetNamespace01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() namespaces:true<br>
     *         () this.namespaces:false<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespaces:true<br>
     *         
     * <br>
     * ɐݒ肳邱ƂmFɂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testSetNamespace01() throws Exception {
        // O
        setUpValidator.namespace = false;
        
        // eXg{
        setUpValidator.setNamespace(true);
        
        // 
        assertTrue(setUpValidator.namespace);        

    }

    /**
     * testSetNamespaceProperties01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() namespaceProperties:not nul<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespaceproperties:not null<br>
     *         
     * <br>
     * ɐݒ肳邱ƂmFɂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testSetNamespaceProperties01() throws Exception {
        // O
        setUpValidator.namespaceProperties = null;
        Properties prop = new Properties();
        
        // eXg{
        setUpValidator.setNamespaceProperties(prop);
        
        // 
        assertSame(prop, setUpValidator.namespaceProperties);        

    }

    /**
     * testSetNamespacePropertyFileName01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() namespacePropertyFileName:"abc"<br>
     *         () this.namespacePropertyFileName:null<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespacePropertyFileName:"abc"<br>
     *         
     * <br>
     * objectnull̏ꍇAOX[邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testSetNamespacePropertyFileName01() throws Exception {
        // O
        setUpValidator.namespacePropertyFileName = null;
        
        // eXg{
        setUpValidator.setNamespacePropertyFileName("abc");
        
        // 
        assertEquals("abc", setUpValidator.namespacePropertyFileName);        

    }

    /**
     * testInitNamespaceProperties01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() this.namespaceCheck:true<br>
     *         () loadNamespaceProperties():ĂяoꂽƂmF<br>
     *         () checkNamespaceProperties():ĂяoꂽƂmF<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * this.namespaceChecktruȅꍇAe\bhs邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testInitNamespaceProperties01() throws Exception {
        // O
        SchemaValidatorImpl_SchemaValidatorImplStub04 schemaValidator = new SchemaValidatorImpl_SchemaValidatorImplStub04();
        schemaValidator.setNamespaceCheck(true);

        // eXg{
        schemaValidator.initNamespaceProperties();

        // 
        assertTrue(schemaValidator.isLoadNamespaceProperties);
        assertTrue(schemaValidator.isCheckNamespaceProperties);
    }

    /**
     * testInitNamespaceProperties02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() this.namespaceCheck:false<br>
     *         () loadNamespaceProperties():ĂяoꂽƂmF<br>
     *         () checkNamespaceProperties():ĂяoȂƂmF<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * this.namespaceCheckfalsȅꍇAcheckNamespaceProperties\bhsȂƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testInitNamespaceProperties02() throws Exception {
        // O
        SchemaValidatorImpl_SchemaValidatorImplStub04 schemaValidator = new SchemaValidatorImpl_SchemaValidatorImplStub04();
        schemaValidator.setNamespaceCheck(false);
        
        // eXg{
        schemaValidator.initNamespaceProperties();

        // 
        assertTrue(schemaValidator.isLoadNamespaceProperties);
        assertFalse(schemaValidator.isCheckNamespaceProperties);
    }

    /**
     * testLoadNamespaceProperties01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() this.namespacePropertyFileName:"test_message_01_en_US.properties"<br>
     *         () vpeBt@C:ȉ̃vpeB݂邱<br>
     *                test_message_01_en_US.properties<br>
     *                @SystemExceptionHandlerTest.key={0}message<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespaceProperties:test_message_01_en_US.properties̒lPropertiesIuWFNg<br>
     *         
     * <br>
     * vpeBIuWFNg̒gP̎Aw肳ꂽvpeBt@C[hĂ邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testLoadNamespaceProperties01() throws Exception {
        // O
        setUpValidator.namespacePropertyFileName = "test_message_01_en_US.properties";
        
        // eXgΏۂ̎s
        setUpValidator.initNamespaceProperties();

        // ʊmF
        Properties result = setUpValidator.namespaceProperties;
        
        assertTrue(result.containsKey("SystemExceptionHandlerTest.key"));
        assertTrue(result.containsValue("{0}message"));
    }

    /**
     * testLoadNamespaceProperties02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() this.namespacePropertyFileName:"test_message_01.properties"<br>
     *         () vpeBt@C:ȉ̃vpeB݂邱<br>
     *                test_message_01.properties<br>
     *                @SystemExceptionHandlerTest.error.message=ObZ[W<br>
     *                @SystemExceptionHandlerTest.error.message.null=<br>
     *                @SystemExceptionHandlerTest.key={0}ftHgbZ[W<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespaceProperties:test_message_01.properties̒lPropertiesIuWFNg<br>
     *         
     * <br>
     * w肳ꂽvpeBt@C̒ǧ[hĂ邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testLoadNamespaceProperties02() throws Exception {
        // O
        setUpValidator.namespacePropertyFileName = "test_message_01.properties";
        
        // eXgΏۂ̎s
        setUpValidator.initNamespaceProperties();

        // ʊmF
        Properties result = setUpValidator.namespaceProperties;

        assertEquals("\u4f8b\u5916\u30e1\u30c3\u30bb\u30fc\u30b8", result.getProperty("SystemExceptionHandlerTest.error.message"));
        assertEquals("", result.getProperty("SystemExceptionHandlerTest.error.message.null"));
        assertEquals("{0}\u30c7\u30d5\u30a9\u30eb\u30c8\u30e1\u30c3\u30bb\u30fc\u30b8", result.getProperty("SystemExceptionHandlerTest.key"));
        
    }

    /**
     * testLoadNamespaceProperties03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,C
     * <br><br>
     * ͒lF() this.namespacePropertyFileName:null<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespaceProperties:null<br>
     *         
     * <br>
     * vpeBt@C݂ȂꍇAI邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testLoadNamespaceProperties03() throws Exception {
        // O
        setUpValidator.namespacePropertyFileName = null;
        
        // eXgΏۂ̎s
        setUpValidator.initNamespaceProperties();

        // ʊmF
        assertNull(setUpValidator.namespaceProperties);

    }

    /**
     * testLoadNamespaceProperties04()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,C
     * <br><br>
     * ͒lF() this.namespacePropertyFileName:""<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespaceProperties:null<br>
     *         
     * <br>
     * vpeBt@C݂ȂꍇAI邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testLoadNamespaceProperties04() throws Exception {
        // O
        setUpValidator.namespacePropertyFileName = "";
        
        // eXgΏۂ̎s
        setUpValidator.initNamespaceProperties();

        // ʊmF
        assertNull(setUpValidator.namespaceProperties);

    }

    /**
     * testLoadNamespaceProperties05()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() this.namespacePropertyFileName:"test_message_10.properties"<br>
     *         () vpeBt@C:ȉ̃vpeB݂邱<br>
     *                test_message_10.propertiesigȂj<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespaceProperties:PropertiesIuWFNg<br>
     *         
     * <br>
     * w肳ꂽvpeBt@C̒g̎ÃvpeBIuWFNgoĂ邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testLoadNamespaceProperties05() throws Exception {
        // O
        setUpValidator.namespacePropertyFileName = "test_message_10.properties";
        
        // eXgΏۂ̎s
        setUpValidator.initNamespaceProperties();

        // ʊmF
        Properties result = setUpValidator.namespaceProperties;
        assertTrue(result.isEmpty());
    }

    /**
     * testLoadNamespaceProperties06()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,C
     * <br><br>
     * ͒lF() this.namespacePropertyFileName:"test_me.properties"<br>
     *         () vpeBt@C:test_me.properties݂Ȃ<br>
     *         
     * <br>
     * ҒlF(ԕω) this.namespaceProperties:null<br>
     *         
     * <br>
     * ݂Ȃt@Cw肳ꂽAnull߂lƂďI邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testLoadNamespaceProperties06() throws Exception {
        // O
        setUpValidator.namespacePropertyFileName = "test_me.properties";
        
        // eXgΏۂ̎s
        setUpValidator.initNamespaceProperties();

        // ʊmF
        Properties result = setUpValidator.namespaceProperties;
        LogUTUtil.checkWarn("*** Can not find property-file" +
                " [test_me.properties] ***");
        assertNull(result);
    }

    /**
     * testCheckNamespaceProperties01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,C
     * <br><br>
     * ͒lF() namespaceProperties:null<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * namespacePropertiesnull̏ꍇAsȂƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCheckNamespaceProperties01() throws Exception {
        // O
        setUpValidator.setNamespaceProperties(null);
        
        // eXg{
        setUpValidator.checkNamespaceProperties();

    }

    /**
     * testCheckNamespaceProperties02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() namespaceProperties:Properties<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * namespaceProperties̏ꍇAsȂƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCheckNamespaceProperties02() throws Exception {
        // O
        Properties properties = new Properties();
        setUpValidator.setNamespaceProperties(properties);
        
        // eXg{
        setUpValidator.checkNamespaceProperties();

    }

    /**
     * testCheckNamespaceProperties03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() namespaceProperties:Properties{<br>
     *                  "abc", "123"<br>
     *                }<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * vpeBdĂȂꍇAsȂƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCheckNamespaceProperties03() throws Exception {
        // O
        Properties properties = new Properties();
        properties.put("abc", "123");
        setUpValidator.setNamespaceProperties(properties);
        
        // eXg{
        setUpValidator.checkNamespaceProperties();

    }

    /**
     * testCheckNamespaceProperties04()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() namespaceProperties:Properties{<br>
     *                  "abc", "123"<br>
     *                  "def", "456"<br>
     *                  "ghi", "789"<br>
     *                }<br>
     *         
     * <br>
     * ҒlF
     * <br>
     * vpeBdĂȂꍇAsȂƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCheckNamespaceProperties04() throws Exception {
        // O
        Properties properties = new Properties();
        properties.put("abc", "123");
        properties.put("def", "456");
        properties.put("ghi", "789");
        setUpValidator.setNamespaceProperties(properties);
        
        // eXg{
        setUpValidator.checkNamespaceProperties();

    }

    /**
     * testCheckNamespaceProperties05()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,D,G
     * <br><br>
     * ͒lF() namespaceProperties:Properties{<br>
     *                  "abc", "123"<br>
     *                  "def", "456"<br>
     *                  "ghi", "456"<br>
     *                }<br>
     *         
     * <br>
     * ҒlF(ԕω) O:PropertyNotUniqueException<br>
     *         (ԕω) O:OxFERROR<br>
     *                    Namespace name [456] is not unique. Namespace must be unique. (key = [ghi])<br>
     *         
     * <br>
     * vpeBdĂꍇAOX[邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCheckNamespaceProperties05() throws Exception {
        // O
        Properties properties = new Properties();
        properties.put("abc", "123");
        properties.put("def", "456");
        properties.put("ghi", "456");
        setUpValidator.setNamespaceProperties(properties);
        
        // eXg{
        try {
            setUpValidator.checkNamespaceProperties();
            fail();
        }catch (NamespaceNotUniqueException e) {
            // 
            String message = "Namespace name [456] is not unique. Namespace must be unique. (key = [ghi])";

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }

    }

    /**
     * testValidate01()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,C,G
     * <br><br>
     * ͒lF() in:null<br>
     *         () object:not null<br>
     *         () errorMessages:not null<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     *                      message = "InputStream is null."<br>
     *                    }<br>
     *         (ԕω) O:Ox:error<br>
     *                    InputStream is null.<br>
     *         
     * <br>
     * innull̏ꍇAOX[AOo͂s邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate01() throws Exception {

        // eXg{
        try{
            setUpValidator.validate(null, new Object(), new ErrorMessages());
            fail();
        }catch (IllegalArgumentException e) {
            // 
            String message = "InputStream is null.";
            assertSame(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }
        
    }

    /**
     * testValidate02()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,C,G
     * <br><br>
     * ͒lF() in:not null<br>
     *         () object:not null<br>
     *         () errorMessages:null<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     *                      message = "ErrorMessages is null."<br>
     *                    }<br>
     *         (ԕω) O:OxFerror<br>
     *                    ErrorMessages is null.<br>
     *         
     * <br>
     * errorMessagesnull̏ꍇAOX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate02() throws Exception {
        // O
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                new byte[] {});
        
        // eXg{
        try{
            setUpValidator.validate(inputStream, new Object(), null);
            fail();
        }catch (IllegalArgumentException e) {
            // 
            String message = "ErrorMessages is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }
        
    }

    /**
     * testValidate03()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,G
     * <br><br>
     * ͒lF() in:not null<br>
     *         () object:not null<br>
     *         () errorMessages:not null<br>
     *         () getUrl():ĂяoƂmFB<br>
     *         () createNonNamespaceDomParser():SAXNotRecognizedExceptionX[B<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(ԕω) O:ParserNotSupportedExceptiono<br>
     *                      cause=  SAXNotRecognizedException<br>
     *                    }<br>
     *         (ԕω) O:OxFerror<br>
     *                    Schema property error.<br>
     *         
     * <br>
     * DOMParseɁASAXNotRecognizedExceptionX[ꂽꍇASAXNotRecognizedExceptionbvParserNotSupportedExceptionX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate03() throws Exception {
        // O
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                new byte[] {});
        SchemaValidatorImpl_SchemaValidatorImplStub02 validator 
        	= new SchemaValidatorImpl_SchemaValidatorImplStub02();
        
        // eXg{
        try{
            validator.validate(inputStream, new Object(), new ErrorMessages());
            fail();
        }catch (ParserNotSupportedException e) {
            // 
            String message = "Schema property error.";
            assertSame(SAXNotRecognizedException.class, e.getCause().getClass());

            // OmF
            assertTrue(LogUTUtil.checkError(message, e.getCause()));
        }
    }

    /**
     * testValidate04()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,G
     * <br><br>
     * ͒lF() in:not null<br>
     *         () object:not null<br>
     *         () errorMessages:not null<br>
     *         () getUrl():ĂяoƂmFB<br>
     *         () createNonNamespaceDomParser():SAXNotSupportedExceptionX[B<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(ԕω) O:ParserNotSupportedExceptiono<br>
     *                      cause= SAXNotSupportedException<br>
     *                    }<br>
     *         (ԕω) O:OxFerror<br>
     *                    Schema property error.<br>
     *         
     * <br>
     * DOMParseɁASAXNotSupportedExceptionX[ꂽꍇASAXNotSupportedExceptionbvParserNotSupportedExceptionX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate04() throws Exception {
        // O
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                new byte[] {});
        SchemaValidatorImpl_SchemaValidatorImplStub03 validator 
        	= new SchemaValidatorImpl_SchemaValidatorImplStub03();
        
        // eXg{
        try{
            validator.validate(inputStream, new Object(), new ErrorMessages());
            fail();
        }catch (ParserNotSupportedException e) {
            // 
            String message = "Schema property error.";
            assertSame(SAXNotSupportedException.class, e.getCause().getClass());

            // OmF
            assertTrue(LogUTUtil.checkError(message, e.getCause()));
        }
    }

    /**
     * testValidate05()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() in:not null<br>
     *         () object:null<br>
     *         () errorMessages:not null<br>
     *         () getUrl():ĂяoƂmFB<br>
     *         () createNonNamespaceDomParser():ĂяoƂmFB<br>
     *                <br>
     *                errorMessagesɃG[l߂<br>
     *         () parser.perse():ĂяoƂmFB<br>
     *                nꂽƂmFB<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(߂l) Document:null<br>
     *         
     * <br>
     * ``FbNG[ꍇANullԋp邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate05() throws Exception {
        
        // O
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                new byte[] {});
        SchemaValidatorImpl_SchemaValidatorImplStub01 validator = new SchemaValidatorImpl_SchemaValidatorImplStub01();
        validator.domParser = new SchemaValidatorImpl_DOMParserStub01();

        // ``FbNG[𔭐邽߁AsOG[i[ĂB
        ErrorMessages errorMessages = new ErrorMessages();
        errorMessages.add(new ErrorMessage(null,null));
        
        // eXg{
        validator.validate(inputStream, new Object(), errorMessages);

        assertTrue(validator.isCreateDomParser);
        assertTrue(validator.isSetCommonParserProperty);
        assertTrue(validator.isSetCommonParserFeature);
        
        SchemaValidatorImpl_DOMParserStub01 parser 
        	= (SchemaValidatorImpl_DOMParserStub01) 
        	UTUtil.getPrivateField(validator, "domParser");
        assertTrue(parser.isParse);
        assertFalse(parser.isGetDocument);
        
    }

    /**
     * testValidate06()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,D
     * <br><br>
     * ͒lF() in:not null<br>
     *         () object:null<br>
     *         () errorMessages:not null<br>
     *         () getUrl():ĂяoƂmFB<br>
     *         () createNonNamespaceDomParser():ĂяoƂmFB<br>
     *         () parser.perse():ĂяoƂmFB<br>
     *                nꂽƂmFB<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(߂l) Document:DocumentCX^X<br>
     *                  iDOMc[j<br>
     *         
     * <br>
     * ``FbNsADocument邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate06() throws Exception {
        
        // O
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                new byte[] {});
        SchemaValidatorImpl_SchemaValidatorImplStub01 validator = new SchemaValidatorImpl_SchemaValidatorImplStub01();
        validator.domParser = new SchemaValidatorImpl_DOMParserStub01();
        
        // eXg{
        validator.validate(inputStream, new Object(), new ErrorMessages());

        assertTrue(validator.isCreateDomParser);
        assertTrue(validator.isSetCommonParserProperty);
        assertTrue(validator.isSetCommonParserFeature);
        
        SchemaValidatorImpl_DOMParserStub01 parser 
        	= (SchemaValidatorImpl_DOMParserStub01) 
        	UTUtil.getPrivateField(validator, "domParser");
        assertTrue(parser.isParse);
        assertTrue(parser.isGetDocument);
    }

    /**
     * testValidate07()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() in:not null<br>
     *         () object:null<br>
     *         () errorMessages:not null<br>
     *         () getUrl():ĂяoƂmFB<br>
     *         () getNamespaceName():ĂяoƂmFB<br>
     *         () creatNamespaceDomParser():ĂяoƂmFB<br>
     *         () setCommonParserProperty():ĂяoƂmFB<br>
     *         () setCommonParserFeature:ĂяoƂmFB<br>
     *         () parser.perse():ĂяoƂmFB<br>
     *                nꂽƂmFB<br>
     *         () this.namespace:true<br>
     *         
     * <br>
     * ҒlF(߂l) Document:DocumentCX^X<br>
     *                  iDOMc[j<br>
     *         
     * <br>
     * this.namespacetruȅꍇA``FbNsADocument邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate07() throws Exception {
        
        // O
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                new byte[] {});
        SchemaValidatorImpl_SchemaValidatorImplStub01 validator = new SchemaValidatorImpl_SchemaValidatorImplStub01();
        validator.domParser = new SchemaValidatorImpl_DOMParserStub01();
        validator.namespace = true;
        
        // eXg{
        validator.validate(inputStream, new Object(), new ErrorMessages());

        assertTrue(validator.isCreateDomParser);
        assertTrue(validator.isSetCommonParserProperty);
        assertTrue(validator.isSetCommonParserFeature);
        
        SchemaValidatorImpl_DOMParserStub01 parser 
        	= (SchemaValidatorImpl_DOMParserStub01) 
        	UTUtil.getPrivateField(validator, "domParser");
        assertTrue(parser.isParse);
        assertTrue(parser.isGetDocument);
        
    }

    /**
     * testValidate08()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,G
     * <br><br>
     * ͒lF() in:not null<br>
     *         () object:not null<br>
     *         () errorMessages:not null<br>
     *         () getUrl():ĂяoƂmFB<br>
     *         () createNonNamespaceDomParser():ĂяoƂmFB<br>
     *         () setCommonParserProperty():ĂяoƂmFB<br>
     *         () setCommonParserFeature:ĂяoƂmFB<br>
     *         () parser.perse():SAXExceptionX[<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(ԕω) O:ParserSAXException{<br>
     *                      cause = SAXException<br>
     *                    }<br>
     *         (ԕω) O:OxFerror<br>
     *                    Schema parse error.<br>
     *         
     * <br>
     * parse()\bhsɁASAXExceptionX[ꂽꍇASAXExceptionbvParserSAXExceptionX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate08() throws Exception {
        
        // O
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                new byte[] {});
        SchemaValidatorImpl_SchemaValidatorImplStub01 validator = new SchemaValidatorImpl_SchemaValidatorImplStub01();
        validator.domParser = new SchemaValidatorImpl_DOMParserStub02();
        validator.namespace = true;
        
        // eXg{
        try{
            validator.validate(inputStream, new Object(), new ErrorMessages());
            fail();
        }catch (ParserSAXException e) {
            // 
            String message = "Schema parse error.";
            assertSame(SAXException.class, e.getCause().getClass());

            // OmF
            assertTrue(LogUTUtil.checkError(message, e.getCause()));
        }
        assertTrue(validator.isCreateDomParser);
        assertTrue(validator.isSetCommonParserProperty);
        assertTrue(validator.isSetCommonParserFeature);
        
    }

    /**
     * testValidate09()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,G
     * <br><br>
     * ͒lF() in:not null<br>
     *         () object:not null<br>
     *         () errorMessages:not null<br>
     *         () getUrl():ĂяoƂmFB<br>
     *         () createNonNamespaceDomParser():ĂяoƂmFB<br>
     *         () setCommonParserProperty():ĂяoƂmFB<br>
     *         () setCommonParserFeature:ĂяoƂmFB<br>
     *         () parser.perse():IOExceptionX[<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(ԕω) O:ParserIOException{<br>
     *                      cause = IOException<br>
     *                    }<br>
     *         (ԕω) O:OxFerror<br>
     *                    Schema io error.<br>
     *         
     * <br>
     * parse()\bhsIOExceptionX[ꂽꍇAIOExceptionbvParserIOExceptionX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testValidate09() throws Exception {
        
        // O
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                new byte[] {});
        SchemaValidatorImpl_SchemaValidatorImplStub01 validator 
        	= new SchemaValidatorImpl_SchemaValidatorImplStub01();
        validator.domParser = new SchemaValidatorImpl_DOMParserStub03();
        validator.namespace = true;
        
        // eXg{
        try{
            validator.validate(inputStream, new Object(), new ErrorMessages());
            fail();
        }catch (ParserIOException e) {
            // 
            String message = "Schema io error.";
            assertSame(IOException.class, e.getCause().getClass());

            // OmF
            assertTrue(LogUTUtil.checkError(message, e.getCause()));
        }
        assertTrue(validator.isCreateDomParser);
        assertTrue(validator.isSetCommonParserProperty);
        assertTrue(validator.isSetCommonParserFeature);
        
    }

    /**
     * testCreateDomParser01()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FG
     * <br><br>
     * ͒lF() object:not null<br>
     *         () getUrl():null<br>
     *         
     * <br>
     * ҒlF(ԕω) O:SchemaFileNotFoundException<br>
     * 				Schema file is not found. Set schema file in [root-classpath]/java/lang/Object.xsd
     *         (ԕω) O:OxFerror<br>
     *                    Schema file is not found.<br>
     *         
     * <br>
     * XL[}t@CȂꍇ̃eXgBO𓊂B
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateDomParser01() throws Exception {
        // O
    	SchemaValidatorImpl_SchemaValidatorImplStub05 validator 
    		= new SchemaValidatorImpl_SchemaValidatorImplStub05();
    	validator.config = new XML11Configuration();
    	validator.url = null;
    	Object object = new Object();

        // eXg{
    	try {
    		validator.createDomParser(object);
    		fail();
    	} catch (SchemaFileNotFoundException e) {
    		assertTrue(validator.isGetUrl);
    		assertTrue(validator.isCreateXmlParserConfiguration);
    		assertTrue(LogUTUtil.checkError("Schema file is not found. " 
    			+ "Set schema file in [root-classpath]/java/lang/Object.xsd"));
    	}
    }

    /**
     * testCreateDomParser02()
     * <br><br>
     * 
     * (n) 
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() object:not null<br>
     *         () getUrl():http://www.nttdata.co.jp/<br>
     *         () this.namespace:true<br>
     *         
     * <br>
     * ҒlF(߂l) DOMParser:DOMParser<br>
     *         
     * <br>
     * OԎgpꍇ̃eXgBDOMp[TԂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateDomParser02() throws Exception {
    	// O
    	SchemaValidatorImpl_SchemaValidatorImplStub05 validator 
    		= new SchemaValidatorImpl_SchemaValidatorImplStub05();
    	validator.config = new XML11Configuration();
    	validator.url = new URL("http://www.nttdata.co.jp/");
    	validator.namespace = true;
    	validator.namespaceName = "NamespaceTest";
    	Object object = new Object();

        // eXg{
    	DOMParser result = validator.createDomParser(object);
    	Object location = result.getProperty(
    		"http://apache.org/xml/properties/schema/external-schemaLocation");
    	assertEquals("NamespaceTest http://www.nttdata.co.jp/", location);
    	assertTrue(validator.isGetUrl);
    	assertTrue(validator.isCreateXmlParserConfiguration);
    }

    /**
     * testCreateDomParser03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() object:not null<br>
     *         () getUrl():http://www.nttdata.co.jp/<br>
     *         () this.namespace:false<br>
     *         
     * <br>
     * ҒlF(߂l) DOMParser:DOMParser<br>
     *         
     * <br>
     * OԎgpȂꍇ̃eXgBDOMp[TԂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateDomParser03() throws Exception {
    	// O
    	SchemaValidatorImpl_SchemaValidatorImplStub05 validator 
    		= new SchemaValidatorImpl_SchemaValidatorImplStub05();
    	validator.config = new XML11Configuration();
    	validator.url = new URL("http://www.nttdata.co.jp/");
    	validator.namespace = false;
    	Object object = new Object();

        // eXg{
    	DOMParser result = validator.createDomParser(object);
    	Object location = result.getProperty(
    		"http://apache.org/xml/properties/schema/external-noNamespaceSchemaLocation");
    	assertEquals("http://www.nttdata.co.jp/", location);
    	assertTrue(validator.isGetUrl);
    	assertTrue(validator.isCreateXmlParserConfiguration);
    }
    
    /**
     * testCreateXmlParserConfiguration01()
     * <br><br>
     * 
     * (n) 
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() ths.namespace:true<br>
     *         () this.cache:true<br>
     *         
     * <br>
     * ҒlF(߂l) XMLParserConfiguration:XMLParserConfiguration<br>
     *                  (grammarPoolێ)<br>
     *         
     * <br>
     * OԂƃLbVLɂȂꍇ̃eXgB@v[ێConfigurationԂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateXmlParserConfiguration01() throws Exception {
        // O
    	SchemaValidatorImpl validator = new SchemaValidatorImpl();
    	validator.namespace = true;
    	validator.cache = true;

        // eXg{
    	XMLParserConfiguration result = validator.createXmlParserConfiguration();

        // 
    	assertSame(validator.grammarPool, UTUtil.getPrivateField(result, "fGrammarPool"));
    }

    /**
     * testCreateXmlParserConfiguration02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() ths.namespace:true<br>
     *         () this.cache:false<br>
     *         
     * <br>
     * ҒlF(߂l) XMLParserConfiguration:XMLParserConfiguration<br>
     *         
     * <br>
     * OԂ̂ݗLɂȂꍇ̃eXgBʏConfigurationԂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateXmlParserConfiguration02() throws Exception {
    	SchemaValidatorImpl validator = new SchemaValidatorImpl();
    	validator.namespace = true;
    	validator.cache = false;

        // eXg{
    	XMLParserConfiguration result = validator.createXmlParserConfiguration();

        // 
    	assertNotSame(validator.grammarPool, UTUtil.getPrivateField(result, "fGrammarPool"));
    }

    /**
     * testCreateXmlParserConfiguration03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() ths.namespace:false<br>
     *         () this.cache:true<br>
     *         
     * <br>
     * ҒlF(߂l) XMLParserConfiguration:XMLParserConfiguration<br>
     *         
     * <br>
     * LbV̂ݗLɂȂꍇ̃eXgBʏConfigurationԂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testCreateXmlParserConfiguration03() throws Exception {
    	SchemaValidatorImpl validator = new SchemaValidatorImpl();
    	validator.namespace = false;
    	validator.cache = true;

        // eXg{
    	XMLParserConfiguration result = validator.createXmlParserConfiguration();

        // 
    	assertNotSame(validator.grammarPool, UTUtil.getPrivateField(result, "fGrammarPool"));
    }
    
    /**
     * testSetCommonParserProperty01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() parser:not null<br>
     *         () errorMessages:not null<br>
     *         
     * <br>
     * ҒlF(߂l) DOMParser:íjDOMParserCX^X{<br>
     *                    fProperties{<br>
     *                      Constants.XERCES_PROPERTY_PREFIX<br>
     *                      + Constants.ERROR_REPORTER_PROPERTY= <br>
     *                         XMLErrorReporterExCX^X= {<br>
     *                           errorMessages= ErrorMessagesCX^X<br>
     *                         }<br>
     *                    }<br>
     *                  }<br>
     *         
     * <br>
     * DOMParser̃vpeBXMLErrorReporterExCX^Xݒ肳邱ƂmFB<br>
     * XMLErrorReporterExCX^XɈErrorMessagesCX^Xݒ肳邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testSetCommonParserProperty01() throws Exception {

        // O
        DOMParser parser = new DOMParser();
        ErrorMessages messages = new ErrorMessages();
        
        // eXg{
        setUpValidator.setCommonParserProperty(parser, messages);

        // 
        XMLErrorReporterEx reporter = (XMLErrorReporterEx) parser.getProperty(Constants.XERCES_PROPERTY_PREFIX
                + Constants.ERROR_REPORTER_PROPERTY);
        assertEquals(messages, reporter.getErrorMessages());
    }

    /**
     * testSetCommonParserFeature01()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() parser:not null<br>
     *         
     * <br>
     * ҒlF(߂l) DOMParser:íjDOMParserCX^X{<br>
     *                    fProperties{<br>
     *                      Constants.SAX_FEATURE_PREFIX<br>
     *                      + Constants.VALIDATION_FEATURE= true,<br>
     *                      Constants.XERCES_FEATURE_PREFIX<br>
     *                                  + Constants.SCHEMA_VALIDATION_FEATURE= true<br>
     *                    }<br>
     *                  }<br>
     *         
     * <br>
     * DOMParser̃vpeBɊevpeBݒ肳邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testSetCommonParserFeature01() throws Exception {

        // O
        DOMParser parser = new DOMParser();
        
        // eXg{
        setUpValidator.setCommonParserFeature(parser);

        // 
        assertTrue(((Boolean) parser.getFeature(Constants.SAX_FEATURE_PREFIX
                + Constants.VALIDATION_FEATURE)).booleanValue());
        assertTrue(((Boolean)(parser.getFeature(Constants.XERCES_FEATURE_PREFIX
                + Constants.SCHEMA_VALIDATION_FEATURE))).booleanValue());
        
    }

    /**
     * testGetUrl01()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,C,G
     * <br><br>
     * ͒lF() object:null<br>
     *         
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     *                      message = Argument object is null.<br>
     *                    }<br>
     *         (ԕω) O:Ox:error<br>
     *                    Argument object is null.<br>
     *         
     * <br>
     * objectnull̏ꍇAOX[邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetUrl01() throws Exception {
        // O
        Object object = null;

        // eXg{
        try {
            setUpValidator.getUrl(object);
            fail();
        } catch (IllegalArgumentException e) {
            // 
            String message = "Argument is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testGetUrl02()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() object:not null<br>
     *                it@C݂pXj<br>
     *         
     * <br>
     * ҒlF(߂l) URL:URLCX^X<br>
     *         
     * <br>
     * wpXɃt@C݂ꍇAURLCX^Xԋp邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetUrl02() throws Exception {
        // 
        // ݂pXiSchemaValidatorImplTest.xsdj
        Object object = this;

        // eXg{
        URL result = setUpValidator.getUrl(object);
        
        // 
        String fileName = this.getClass().getName().replace(
                SchemaValidatorImpl.NESTED_PACKAGE_SEPARATOR,
                SchemaValidatorImpl.NESTED_FOLDER_SEPARATOR)
                + SchemaValidatorImpl.XSD_FILE_SUFFIX;
        URL url = Thread.currentThread().getContextClassLoader().getResource(
                fileName);
        assertEquals(url.toExternalForm(), result.toExternalForm());        

    }

    /**
     * testGetUrl03()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() object:not null<br>
     *                it@C݂ȂpXj<br>
     *         
     * <br>
     * ҒlF(߂l) URL:null<br>
     *         
     * <br>
     * wpXɃt@C݂ȂꍇAnullԋp邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetUrl03() throws Exception {
        // 
        // ݂ȂpX
        Object object = setUpValidator;

        // eXg{
        URL result = setUpValidator.getUrl(object);
        
        // 
        assertNull(result);        

    }

    /**
     * testGetNamespaceName01()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FC,G
     * <br><br>
     * ͒lF() object:null<br>
     *         () this.namespaceProperties:not null<br>
     *         () this.NAME_SPACE_SUFFIX:".Namespace"<br>
     *         
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     *                      message = Argument is null.<br>
     *                    }<br>
     *         (ԕω) O:Ox:error<br>
     *                    Argument is null.<br>
     *         
     * <br>
     * objectnull̏ꍇAOX[邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetNamespaceName01() throws Exception {

        // eXg{
        try {
            setUpValidator.getNamespaceName(null);
            fail();
        } catch (IllegalArgumentException e) {
            // 
            String message = "Argument is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }

    }

    /**
     * testGetNamespaceName02()
     * <br><br>
     * 
     *  (ُn)
     * <br>
     * ϓ_FC,G
     * <br><br>
     * ͒lF() not null<br>
     *         () this.namespaceProperties:null<br>
     *         () this.NAME_SPACE_SUFFIX:".Namespace"<br>
     *         
     * <br>
     * ҒlF(ԕω) O:IllegalStateException{<br>
     *                      message= Namespace properties is null.<br>
     *                    }<br>
     *         (ԕω) O:Ox:error<br>
     *                    Namespace properties is null.<br>
     *         
     * <br>
     * this.namespacePropertiesnull̏ꍇAOX[邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetNamespaceName02() throws Exception {
        // O
        setUpValidator.namespaceProperties = null;
        setUpValidator.namespace = true;

        // eXg{
        try {
            setUpValidator.getNamespaceName(new Object());
            fail();
        } catch (IllegalStateException e) {
            // 
            String message = "Namespace property is not set. "
            	+ "Put namespaces.properties file on your classpath, and call SchemaValidatorImpl initNamespaceProperties() after instanciate.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }

    }

    /**
     * testGetNamespaceName03()
     * <br><br>
     * 
     * (ُn)
     * <br>
     * ϓ_FA,C
     * <br><br>
     * ͒lF() object:jp.terasoluna.fw.oxm.xsd.xerces.SchemaValidatorImplCX^X<br>
     *         () this.namespaceProperties:PropertiesCX^X<br>
     *         () this.NAME_SPACE_SUFFIX:".Namespace"<br>
     *         
     * <br>
     * ҒlF(߂l) NamespaceNotFoundException<br>
     *         
     * <br>
     * object̃NXɍvvpeB݂ȂꍇA
     * OX[邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetNamespaceName03() throws Exception {
        // O
        Properties prop = new Properties();
        setUpValidator.namespaceProperties = prop;
        setUpValidator.namespace = true;

        // eXg{
        try {
        	setUpValidator.getNamespaceName("dummy");
        	fail();
        } catch (NamespaceNotFoundException e) {
        	String message = "Schema namespace is not found. Set namespace key " +
        		"- java.lang.String.Namespace in namespaces.properties file.";
        	assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testGetNamespaceName04()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA,C,D
     * <br><br>
     * ͒lF() object:jp.terasoluna.fw.oxm.xsd.xerces.SchemaValidatorImplCX^X<br>
     *         () this.namespaceProperties:PropertiesCX^X{<br>
     *                  "jp.terasoluna.fw.oxm.xsd.xerces.SchemaValidatorImpl"= "123"<br>
     *                }<br>
     *         () this.NAME_SPACE_SUFFIX:".Namespace"<br>
     *         
     * <br>
     * ҒlF(߂l) String:"123"<br>
     *         
     * <br>
     * object̃NXɍvvpeB݂ꍇAlԋp邱ƂmFB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetNamespaceName04() throws Exception {
        // O
        Properties prop = new Properties();
        prop.setProperty(setUpValidator.getClass().getName() + ".Namespace", "123");
        setUpValidator.namespace = true;
        setUpValidator.namespaceProperties = prop;
        
        // eXg{
        assertEquals("123", setUpValidator.getNamespaceName(setUpValidator));
    }

    /**
     * testGetNamespaceName05()
     * <br><br>
     * 
     * (n)
     * <br>
     * ϓ_FA
     * <br><br>
     * ͒lF() object:not null<br>
     *         () this.namespaceProperties:-<br>
     *         () this.namespace:false<br>
     *         () this.NAME_SPACE_SUFFIX:-<br>
     *         
     * <br>
     * ҒlF(߂l) null<br>
     *         
     * <br>
     * OԂgȂꍇ̃eXgBNullԂB
     * <br>
     * 
     * @throws Exception ̃\bhŔO
     */
    public void testGetNamespaceName05() throws Exception {
        // O
        setUpValidator.namespace = false;
        
        // eXg{
        assertNull(setUpValidator.getNamespaceName(new Object()));
    }
}
