class TestEditingContextFetch < Test::Unit::TestCase
	include TapKit

	def setup
		#@app = Application.new MODELS
		@app = Application.new(MODELS, '.')
		@ec  = @app.create_editing_context
	end


	# fetch specification

	def test_fetchspec_object_values_for_properties
		qualifier = Qualifier.format "title like '*'"
		fetchspec = FetchSpec.new('cd', qualifier)
		fetchspec.limit = 1
		object = @ec.fetch(fetchspec).first

		assert_not_nil object
		assert_not_nil object['title']
		assert_not_nil object['artist']
		assert_not_nil object['release']
	end

	def test_fetchspec_limited
		limit = 1
		qualifier = Qualifier.format "title like '*'"
		fetchspec = FetchSpec.new('Book', qualifier)
		fetchspec.limit = limit

		objects = @ec.fetch fetchspec
		assert_equal(limit, objects.size)
	end

	def test_fetchspec_case_insensitive_like
		qualifier1 = Qualifier.format "title like 'a*'"
		qualifier2 = Qualifier.format "title cilike 'a*'"
		fetchspec1 = FetchSpec.new('cd', qualifier1)
		fetchspec2 = FetchSpec.new('cd', qualifier2)
		fetchspec1.limit = 5
		fetchspec2.limit = 5
		objects1   = @ec.fetch fetchspec1
		objects2   = @ec.fetch fetchspec2

		assert_not_equal(objects1, objects2)
	end

	def test_fetchspec_sort
		# @app.log_options[:sql] = true
		qualifier  = Qualifier.format "title like '*'"
		sort1      = SortOrdering.new('title', SortOrdering::ASC)
		sort2      = SortOrdering.new('title', SortOrdering::DESC)
		fetchspec1 = FetchSpec.new('cd', qualifier, [sort1])
		fetchspec2 = FetchSpec.new('cd', qualifier, [sort2])
		fetchspec1.limit = 10
		fetchspec2.limit = 10
		objects1   = @ec.fetch fetchspec1
		objects2   = @ec.fetch fetchspec2

		assert_not_equal(objects1, objects2)
	end

	def test_fetchspec_sort_case_insensitive
		# @app.log_options[:sql] = true
		qualifier  = Qualifier.format "title cilike 'a*'"
		sort1      = SortOrdering.new('title', SortOrdering::ASC)
		sort2      = SortOrdering.new('title', SortOrdering::CI_ASC)
		fetchspec1 = FetchSpec.new('cd', qualifier, [sort1])
		fetchspec2 = FetchSpec.new('cd', qualifier, [sort2])
		fetchspec1.limit = 10
		fetchspec2.limit = 10
		objects1   = @ec.fetch fetchspec1
		objects2   = @ec.fetch fetchspec2

		assert_not_equal(objects1, objects2)
	end

	def test_fetchspec_all_data
		fetchspec = FetchSpec.new 'cd'
		objects   = @ec.fetch fetchspec
		fetchspec.limit = 10

		assert_equal(false, objects.empty?)
	end

	def test_fetchspec_refresh_objects
		qualifier = Qualifier.format "title like '*'"
		fetchspec = FetchSpec.new('Book', qualifier)
		fetchspec.limit = 1

		objects  = @ec.fetch fetchspec
		object   = objects.first
		original1 = object['title'].dup
		original2 = object['publisher']['name'].dup
		object['title'] = 'changed'
		object['publisher']['name'] = 'changed'

		fetchspec.refresh = true
		objects = @ec.fetch fetchspec
		object  = objects.first

		assert_equal(original1, object['title'])
		assert_equal(original2, object['publisher']['name'])
	end

	def test_fetchspec_refresh_objects_for_to_many
		qualifier = Qualifier.format "name like '*'"
		fetchspec = FetchSpec.new('Publisher', qualifier)
		fetchspec.limit = 1
		objects = @ec.fetch fetchspec
		object  = objects.first

		original = object['books'][0]['title'].dup
		object['books'][0]['title'] = 'changed'

		fetchspec.refresh = true
		objects = @ec.fetch fetchspec
		object  = objects.first
		assert_equal(original, object['books'][0]['title'])
	end

	def test_fetchspec_qualifier_for_to_many
		qualifier = Qualifier.format "books.title like '*'"
		fetchspec = FetchSpec.new('Publisher', qualifier)
		fetchspec.limit = 1
		objects = @ec.fetch fetchspec
		assert_equal(false, objects.empty?)
	end


	# aggregate

	def test_aggregate_not_grouping
		# @app.log_options[:sql] = true
		qualifier = Qualifier.format "artist like 'The Beatles'"
		spec = AggregateSpec.new('cd', qualifier)
		spec.add('CDs', 'artist', AggregateSpec::COUNT)
		objects = @ec.fetch spec

		assert objects[0].key?('CDs')
	end


	# faulting

	def test_fault?
		qualifier = Qualifier.format "title == 'Programming Ruby'"
		fetchspec = FetchSpec.new('Book', qualifier)
		objects   = @ec.fetch fetchspec
		object    = objects.first

		publisher = object['publisher']
		assert publisher.fault?

		name   = publisher['name']
		assert_instance_of(String, name)
	end

	def test_to_one_faulting
		app = Application.new MODEL_TO_ONE
		ec  = app.create_editing_context

		qualifier = Qualifier.format("a_name == 'A_NAME'")
		fetchspec = FetchSpec.new('A', qualifier)
		objects   = ec.fetch(fetchspec)

		objects.each do |a|
			assert a['b'].fault?
			assert a['b.c'].fault?
			a['b.c.c_name'] # fire faulting
			assert_equal(false, a['b.c'].fault?)
		end
	end


	# snapshot

	def test_snapshot
		qualifier = Qualifier.format "title like '*'"
		fetchspec = FetchSpec.new('Book', qualifier)
		fetchspec.limit = 10
		objects   = @ec.fetch fetchspec
		model     = @app.model MODEL2
		database  = @app.database model

		objects.each do |object|
			gid = @ec.gid object
			snapshot = database.snapshot gid
			assert_instance_of(Snapshot, snapshot)
		end
	end


	# uniquing

	def test_compare_with_fetched_objects
		qualifier = Qualifier.format "title like '*'"
		fetchspec = FetchSpec.new('Book', qualifier)
		fetchspec.limit = 10
		objects1  = @ec.fetch fetchspec
		objects2  = @ec.fetch fetchspec

		assert_equal(objects1, objects2)
	end

	def test_compare_with_faulted_objects
		qualifier = Qualifier.format "title like 'The Ruby Way'"
		fetchspec = FetchSpec.new('Book', qualifier)
		fetchspec.limit = 10
		object1   = @ec.fetch(fetchspec).first
		object2   = @ec.fetch(fetchspec).first
		fault1    = object1['publisher']
		fault2    = object2['publisher']
		fault1.will_read
		fault2.will_read

		assert_equal(fault1, fault2)
	end

	def test_compare_with_objects_fetched_by_different_editing_contexts
		qualifier = Qualifier.format "title like '*'"
		fetchspec = FetchSpec.new('Book', qualifier)
		fetchspec.limit = 10

		ec1 = @app.create_editing_context
		ec2 = @app.create_editing_context

		objects1  = ec1.fetch fetchspec
		objects2  = ec2.fetch fetchspec

		assert_not_equal(objects1, objects2)
	end


	# data mapping

	def test_mapping
		qualifier = Qualifier.format "string like '*'"
		fetchspec = FetchSpec.new('Mapping', qualifier)
		objects   = @ec.fetch fetchspec
		object    = objects.first

		tests = {'string'=>String, 'integer'=>Fixnum, 'float'=>Float,
			'decimal'=>String, 'date'=>Date, 'time'=>Time, 'timestamp'=>Timestamp,
			'binary'=>String}

		tests.each do |key, expected|
			assert_kind_of(expected, object[key])
		end
	end

	def test_boolean1_insert_for_true
		_set_boolean
		obj = @ec.create 'Mapping'
		obj['integer'] = true
		@ec.save
	end

	def test_boolean2_fetch_for_true
		_set_boolean
		# @app.log_options[:sql] = true
		q = Qualifier.format("integer == true")
		fs = FetchSpec.new('Mapping', q)
		obj = @ec.fetch(fs).first
		assert_equal(true, obj['integer'])
		@ec.delete obj
		@ec.save
	end

	def test_boolean3_insert_for_false
		_set_boolean
		# @app.log_options[:sql] = true
		obj = @ec.create 'Mapping'
		obj['integer'] = false
		@ec.save
	end

	def test_boolean4_fetch_for_false
		_set_boolean
		# @app.log_options[:sql] = true
		q = Qualifier.format("integer == false")
		fs = FetchSpec.new('Mapping', q)
		obj = @ec.fetch(fs).first
		assert_equal(false, obj['integer'])
		@ec.delete obj
		@ec.save
	end

	def _set_boolean
		entity = @app.entity 'Mapping'
		attr_int = entity.attribute 'integer'
		attr_int.class_name = 'Boolean'
	end
end
