class TestDatabaseObject < Test::Unit::TestCase
	include TapKit

	def setup
		@app     = Application.new MODELS
		@ec      = @app.create_editing_context
		@object  = @ec.create 'Book'
		@object2 = @ec.create 'Publisher'
		@entity  = @app.model_group.entity 'Book'
	end


	# using class description methods

	def test_attribute_keys
		all_keys = ['book_id', 'title', 'publisher_id']
		keys     = ['title']

		assert_equal(all_keys, @object.all_attribute_keys)
		assert_equal(keys, @object.attribute_keys)
	end

	def test_to_one_keys
		all_keys = ['publisher']
		keys     = ['publisher']

		assert_equal(all_keys, @object.all_to_one_relationship_keys)
		assert_equal(keys, @object.to_one_relationship_keys)
	end

	def test_to_many_keys
		all_keys = ['books']
		keys     = ['books']

		assert_equal(all_keys, @object2.all_to_many_relationship_keys)
		assert_equal(keys, @object2.to_many_relationship_keys)
	end

	def test_property_keys
		all_keys = ['book_id', 'title', 'publisher_id', 'publisher']
		keys     = ['title', 'publisher']

		assert_equal(all_keys, @object.all_property_keys)
		assert_equal(keys, @object.property_keys)
	end

	def test_entity_name
		assert_equal(@entity.name, @object.entity_name)
	end

	def test_delete_rule
		keys = {
			'publisher' => ClassDescription::DELETE_RULE_NULLIFY
		}

		keys.each do |key, rule|
			assert_equal(rule, @object.delete_rule(key))
		end
	end

	def test_class_description
		# for destination key
		key1     = 'publisher'
		key2     = 'books'
		expected = 'Publisher'

		cd = @object.class_description key1
		assert_equal(expected, cd.entity_name)

		# class description object
		assert_kind_of(ClassDescription, @object.class_description)
	end


	# other methods

	def test_clear
		@object['title'] = 'test'
		@object.clear
		assert_equal(nil, @object['title'])
	end

	def test_to_many_key?
		assert_equal(true,  @object2.to_many_key?('books'))
		assert_equal(false, @object.to_many_key?('publisher'))
	end

	def test_snapshot
		@object['title'] = 'test'
		snapshot = Snapshot.new
		snapshot['title'] = 'test'
		snapshot['publisher'] = nil

		assert_equal(snapshot, @object.snapshot)
	end

	# Checks if you retrieve value for non-class property, UnknownKeyError is raised.
	def test_key_value_coding
		# book_id, title
		assert_raises(KeyValueCoding::UnknownKeyError) { @object['book_id'] }
		assert_nothing_thrown { @object['title'] }
	end
end


# CustomObject
# changes_from_snapshot, inverse, remove_from_property,
# update

class TestCustomObject < Test::Unit::TestCase
	include TapKit

	def setup
		@app    = Application.new MODELS
		@ec     = @app.create_editing_context
		@object = @ec.create 'Book'
		@entity = @app.model_group.entity 'Book'
	end

	def test_to_s
		@object['title'] = 'test'
		expected = '{title = "test", publisher = nil}'

		assert_equal(expected, @object.to_s)
	end

end
