$LOAD_PATH.unshift '../../../lib'
require 'tapkit'
require 'tapkit/access/adapters/csv'
require 'test/unit'
require 'test/unit/ui/console/testrunner'

MODEL = 'testadapter.yaml'

class TestCSVAdapter < Test::Unit::TestCase
	include TapKit

	def setup
		app = Application.new
		model = Model.new MODEL
		@adapter = CSVAdapter.new(model, app)
	end

	def test_using_classes
		assert(CSVExpression, CSVAdapter.expression_class)
		assert_kind_of(CSVExpressionFactory, @adapter.expression_factory)
	end

	def test_internal_type
		tests = { 'dummy'=>String }

		tests.each do |type, expected|
			assert_equal(expected, CSVAdapter.internal_type(type))
		end
	end
end


class TestCSVAccessing < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new [MODEL]
		@adapter = @app.adapter(@app.model(MODEL))
		@context = CSVContext.new @adapter
		@channel = @context.create_channel
		@ec = @app.ec
	end

	def test_accessing_date_and_time
		date = Date.today
		time = Time.now
		timestamp = Timestamp.now

		# insert
		ec = @app.ec
		obj = ec.create 'TestAdapter'
		obj['test_date'] = date
		obj['test_time'] = time
		obj['test_timestamp'] = timestamp
		assert_nothing_thrown { ec.save }

		# fetch
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		obj = ec.fetch(fs).first
		assert_equal(date, obj['test_date'])
		assert_equal(time, obj['test_time'])
		assert_equal(timestamp, obj['test_timestamp'])

		# update
		date += 1
		time += 1
		timestamp += 1
		obj['test_date'] = date
		obj['test_time'] = time
		obj['test_timestamp'] = timestamp
		assert_nothing_thrown { ec.save }

		# fetch again
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		obj = ec.fetch(fs).first
		assert_equal(date, obj['test_date'])
		assert_equal(time, obj['test_time'])
		assert_equal(timestamp, obj['test_timestamp'])

		# delete
		ec.delete obj
		assert_nothing_thrown { ec.save }

		# fetch again again
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		objs = ec.fetch(fs)
		assert objs.empty?
	end

	def fetchspec( date, time, timestamp )
		q = Qualifier.format(
			"(test_date = %@) and (test_time = %@) and (test_timestamp = %@)",
			[date, time, timestamp])
		FetchSpec.new('TestAdapter', q)
	end
end


class TestCSVBasicAccessing < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new [MODEL]
		@ec = @app.ec
	end

	def test_fetch_all
		fs = FetchSpec.new('TestAdapter')
		objs = @ec.fetch fs
		assert(objs.size > 1)
	end

	def test_fetch_nil
		q = Qualifier.format "test_string = 'nil test'"
		fs = FetchSpec.new('TestAdapter', q)
		objs = @ec.fetch fs
		obj, = objs
		assert_equal(nil, obj['test_number'])
		assert_equal(nil, obj['test_date'])
	end

	def test_fetch_with_limit
		fs = FetchSpec.new('TestAdapter')
		fs.limit = 1
		objs = @ec.fetch fs
		assert(objs.size == 1)
	end

	def test_accessing_relationship
		fs = FetchSpec.new('TestAdapter')
		objs = @ec.fetch fs
		obj, = objs
		assert(objs[0]['test_relationships'].size > 0)
	end

	def test_fetch_with_qualifier_for_destination
		q = Qualifier.format "test_relationships.test_string like '*'"
		fs = FetchSpec.new('TestAdapter', q)
		objs = @ec.fetch fs
		assert_equal(false, objs.empty?)
	end

	def test_01_insert
		# @app.log_options[:sql]=true
		@@data = [Time.now.to_s, 100, Date.today, Time.now, Timestamp.now]
		obj = @ec.create 'TestAdapter'
		obj['test_string'] = @@data[0]
		obj['test_number'] = @@data[1]
		obj['test_date'] = @@data[2]
		obj['test_time'] = @@data[3]
		obj['test_timestamp'] = @@data[4]
		@ec.save

		ec = @app.ec
		q = Qualifier.format("test_string = %@", [@@data[0]])
		fs = FetchSpec.new('TestAdapter', q)
		objs = ec.fetch fs
		obj, = objs
		assert_equal(@@data[0], obj['test_string'])
		assert_equal(@@data[1], obj['test_number'])
		assert_equal(@@data[2], obj['test_date'])
		assert_equal(@@data[3], obj['test_time'])
		assert_equal(@@data[4], obj['test_timestamp'])
	end

	def test_02_update
		# @app.log_options[:sql]=true
		ec = @app.ec
		q = Qualifier.format("test_string = %@", [@@data[0]])
		fs = FetchSpec.new('TestAdapter', q)
		obj, = ec.fetch fs
		@@data[0] = "test update #{Time.now}"
		obj['test_string'] = @@data[0]
		ec.save

		ec = @app.ec
		q = Qualifier.format("test_string = %@", [@@data[0]])
		fs = FetchSpec.new('TestAdapter', q)
		objs = ec.fetch fs
		obj, = objs
		assert_equal(@@data[0], obj['test_string'])
		assert_equal(@@data[1], obj['test_number'])
		assert_equal(@@data[2], obj['test_date'])
		assert_equal(@@data[3], obj['test_time'])
		assert_equal(@@data[4], obj['test_timestamp'])
	end

	def test_03_delete
		# @app.log_options[:sql]=true
		ec = @app.ec
		q = Qualifier.format("test_string = %@", [@@data[0]])
		fs = FetchSpec.new('TestAdapter', q)
		obj, = ec.fetch fs
		ec.delete obj
		ec.save

		ec = @app.ec
		q = Qualifier.format("test_string = %@", [@@data[0]])
		fs = FetchSpec.new('TestAdapter', q)
		objs = ec.fetch fs
		assert objs.empty?
	end
end


if __FILE__ == $0 then
	suite = Test::Unit::TestSuite.new 'TapKit TestSuite'
	suite << TestCSVAdapter.suite
	suite << TestCSVBasicAccessing.suite
	# suite << TestCSVAccessing.suite
	runner = Test::Unit::UI::Console::TestRunner.new suite
	runner.start
end
